/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.CONFLICT_RESOLUTION;
import static com.openexchange.usm.connector.commands.CommandConstants.CREATED;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.LIMIT;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.Arrays;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.SlowSyncRequiredException;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.api.session.SynchronizationConflictException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;

/**
 * Handler for the SyncInit USM-JSON-Command.
 * @author afe
 *
 */
public class SyncInitHandler extends SyncCommandHandler {
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDERID, CREATED, LIMIT, CONFLICT_RESOLUTION };

    public SyncInitHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        String folderUUID = getStringParameter(FOLDERID, null);
        String lockKey = (folderUUID == null) ? "<FolderHierarchy>" : folderUUID;
        String acquirer = "syncInit:" + lockKey;
        String previousAcquirer = _session.tryLock(lockKey, acquirer);
        if (previousAcquirer != null)
            return createCannotLockFolderResponse("syncInit", "", lockKey, previousAcquirer);
        try {
            readOptionalLimit();
            ConflictResolution conflictResolution = getOptionalConflictResolution();
            _syncResult = computeSyncInit(folderUUID, conflictResolution);
            initializeErrorMap(_syncResult);
            Folder folder = null;
            if (folderUUID != null)
                folder = getFolderByUUID(folderUUID);
            _cachedElements = new DataObjectSet(_syncResult.getNewState());
            long syncId = finishProcessingAndSaveNewState(_syncResult, folder, _syncResult.getTimestamp());
            JSONObject response = createResponseFromSyncResults(_syncResult, syncId);
            logSyncData(" syncInit", folderUUID, 0L, syncId, _syncResult.getTimestamp());
            return new ResponseObject(getSuccessResponseCode(), response);
        } finally {
            _session.unlock(lockKey);
        }
    }

	private SyncResult computeSyncInit(String folderID, ConflictResolution conflictResolution)
			throws USMJSONAPIException {
		try {
			if (folderID == null) {
				Folder[] clientFolders = readCreatedFolders(null);
				_allFolders = clientFolders;
				Arrays.sort(clientFolders, new FolderHierarchyComparator(clientFolders, _servlet.getJournal()));
				return _session.syncFoldersWithServer(_limit, null, true, conflictResolution, clientFolders);
			}
			setFolderToSync(folderID);
			_clientChangesArray = readCreatedFolderElements(_folderToSync, -1L, conflictResolution,
					EMPTY_DATAOBJECT_ARRAY);
			addModificationsForExceptions(SyncResult.EMPTY_DATA_OBJECT_ARRAY, SyncResult.EMPTY_DATA_OBJECT_ARRAY,
					_clientChangesArray); //create appointment exceptions in delayed calls
			return _session.syncWithServer(_folderToSync.getID(), _limit, getFilter(_folderToSync), true,
					conflictResolution, _clientChangesArray);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (SlowSyncRequiredException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SYNC_INIT_INVALID_SYNCID,
					ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_INIT_INTERNAL_ERROR, e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

	@Override
	protected long getOriginalSyncID() {
		return 0L;
	}
}
