/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.APPLICATION_OCTET_STREAM;
import static com.openexchange.usm.connector.commands.CommandConstants.ATTACHMENT;
import static com.openexchange.usm.connector.commands.CommandConstants.ATTACHMENTS;
import static com.openexchange.usm.connector.commands.CommandConstants.ATTACHMENTS_LAST_MODIFIED;
import static com.openexchange.usm.connector.commands.CommandConstants.BASE64;
import static com.openexchange.usm.connector.commands.CommandConstants.BODY;
import static com.openexchange.usm.connector.commands.CommandConstants.CHANGE_EXCEPTIONS;
import static com.openexchange.usm.connector.commands.CommandConstants.CHARSET;
import static com.openexchange.usm.connector.commands.CommandConstants.CLIENT;
import static com.openexchange.usm.connector.commands.CommandConstants.CONFIRMATIONS;
import static com.openexchange.usm.connector.commands.CommandConstants.CONFLICTS;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTACT_IMAGE_DATA;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTENT_DISPOSITION;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTENT_TRANSFER_ENCODING;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTENT_TYPE;
import static com.openexchange.usm.connector.commands.CommandConstants.CREATED;
import static com.openexchange.usm.connector.commands.CommandConstants.DATA_KEY;
import static com.openexchange.usm.connector.commands.CommandConstants.DELETED;
import static com.openexchange.usm.connector.commands.CommandConstants.DELETED_IF_EXIST;
import static com.openexchange.usm.connector.commands.CommandConstants.DISTRIBUTION_LIST;
import static com.openexchange.usm.connector.commands.CommandConstants.ERRORS;
import static com.openexchange.usm.connector.commands.CommandConstants.FILENAME;
import static com.openexchange.usm.connector.commands.CommandConstants.FLAGS;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDER_UUID;
import static com.openexchange.usm.connector.commands.CommandConstants.HEADERS;
import static com.openexchange.usm.connector.commands.CommandConstants.ID;
import static com.openexchange.usm.connector.commands.CommandConstants.IMAGE_JPEG;
import static com.openexchange.usm.connector.commands.CommandConstants.LIMIT;
import static com.openexchange.usm.connector.commands.CommandConstants.MODIFIED;
import static com.openexchange.usm.connector.commands.CommandConstants.MORE_AVAILABLE;
import static com.openexchange.usm.connector.commands.CommandConstants.MULTIPART_MIXED;
import static com.openexchange.usm.connector.commands.CommandConstants.NAME;
import static com.openexchange.usm.connector.commands.CommandConstants.NUMBER_OF_ATTACHMENTS;
import static com.openexchange.usm.connector.commands.CommandConstants.OBJECT_TYPE;
import static com.openexchange.usm.connector.commands.CommandConstants.ORIGINAL_MAIL_EML;
import static com.openexchange.usm.connector.commands.CommandConstants.PARAMS;
import static com.openexchange.usm.connector.commands.CommandConstants.RECURRENCE_ID;
import static com.openexchange.usm.connector.commands.CommandConstants.SERVER;
import static com.openexchange.usm.connector.commands.CommandConstants.SERVER_INFO_ERROR_ON_READING_EMAIL;
import static com.openexchange.usm.connector.commands.CommandConstants.SUBJECT;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import static com.openexchange.usm.connector.commands.CommandConstants.TEMPID;
import static com.openexchange.usm.connector.commands.CommandConstants.TEXT_PLAIN;
import static com.openexchange.usm.connector.commands.CommandConstants.TYPE;
import static com.openexchange.usm.connector.commands.CommandConstants.UID;
import static com.openexchange.usm.connector.commands.CommandConstants.UTF_8;
import static com.openexchange.usm.connector.commands.CommandConstants.UUID_KEY;
import static com.openexchange.usm.connector.commands.ErrorStatusCode.PIM_ATTACHMENT_CREATION_DENIED_COUNT;
import static com.openexchange.usm.connector.commands.ErrorStatusCode.PIM_ATTACHMENT_CREATION_DENIED_SIZE;
import static com.openexchange.usm.connector.commands.ErrorStatusCode.PIM_ATTACHMENT_CREATION_FAILED;
import static com.openexchange.usm.json.ConnectorBundleErrorCodes.COMMAND_FAILED_TO_ACQUIRE_LOCK_ON_FOLDER;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_STATUS;
import static com.openexchange.usm.json.response.ResponseStatusCode.TEMPORARY_NOT_AVAILABLE;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.ContactContentType;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.FolderConstants;
import com.openexchange.usm.api.contenttypes.MailContentType;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChange;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.OperationDeniedException;
import com.openexchange.usm.api.exceptions.PIMAttachmentCountLimitExceededException;
import com.openexchange.usm.api.exceptions.PIMAttachmentSizeLimitExceededException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.session.ChangeState;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.FolderType;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SlowSyncRequiredException;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.api.session.SynchronizationConflictException;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.InvalidUUIDException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.connector.sync.AppointmentFilter;
import com.openexchange.usm.connector.sync.EmailSizeFilter;
import com.openexchange.usm.connector.sync.PIMAtachmentsCountFilter;
import com.openexchange.usm.contenttypes.util.OXErrorConstants;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.datatypes.mail.MailAttachment;
import com.openexchange.usm.datatypes.mail.MeetingRequestResponseUtil;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.ConfirmingParticipant;
import com.openexchange.usm.datatypes.tasks.calendar.ParticipantIdentifier;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.json.streaming.MailAttachmentStreamingUtil;
import com.openexchange.usm.json.streaming.ServerTempId;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.mimemail.MimeMailBuilder;
import com.openexchange.usm.mimemail.StreamingMimeMailBuilder;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;

public abstract class SyncCommandHandler extends NormalCommandHandler {

    private static final String PARENT_UUID = "parent_uuid";

    private static final boolean _SORT_CHANGE_EXCEPTIONS_BY_START_DATE = true;

    private static final String SEQUENCE = "sequence";

    private static final String START_DATE = "start_date";

    private static final String INDEX = "index";

    private static final PIMAttachment[] EMPTY_PIMATTACHMENT_ARRAY = new PIMAttachment[0];

    private static final String COLOR_LABEL = "color_label";

    private static final long MILLIS_PER_DAY = 24L * 60L * 60L * 1000L;

    protected static final DataObject[] EMPTY_DATAOBJECT_ARRAY = new DataObject[0];

    private static final BitSet _REQUIRED_ALL_APPOINTMENT_FIELDS = new BitSet();

    private static final BitSet _EXTENDED_ALL_APPOINTMENT_FIELDS = new BitSet();

    static {
        _REQUIRED_ALL_APPOINTMENT_FIELDS.set(0, 2); // id and folder_id
    }

    public static int getAttendeeStatus(DataObject appointment, int userID) {
        Object o = appointment.getFieldContent(CommonCalendarTasksFieldNames.USERS);
        if (o instanceof Object[]) {
            Object[] list = (Object[]) o;
            for (Object u : list) {
                if (u instanceof UserParticipantObject) {
                    UserParticipantObject participant = (UserParticipantObject) u;
                    if (participant.getId() == userID)
                        return participant.getConfirmation();
                }
            }
        }
        return -1;
    }

    protected Folder _folderToSync;

    protected int _folderOwnerID;

    protected Map<DataObject, UUID> _recurrenceUUIDMap = new HashMap<DataObject, UUID>();

    protected Map<DataObject, UUID> _exceptionChangesForDelayedCall = new HashMap<DataObject, UUID>();

    protected DataObject[] _clientChangesArray;

    private List<JSONObject> _mailDiffsClientAndServer = new ArrayList<JSONObject>();

    private Map<String, PIMAttachments> _clientAttachmentsMap = new HashMap<String, PIMAttachments>();
    
    private Map<DataObject, UserParticipantObject> _appointmentsToConfirm = new HashMap<DataObject,UserParticipantObject>();

    protected DataObjectSet _newStateToSave;

    // This can be set to force the sending of the moreAvailable-flag even if pending server changes are reported in the SyncResult
    protected boolean _extraSyncRequired;

    protected boolean _newStateToSaveChanged = false;

    protected long _syncIdForUpdate = 0L;

    protected DataObjectSet _cachedElements;

    // Used to store all errors that occur during the sync or any of the post processing methods. As soon as an error
    // is stored for one DataObject, no further operations should be performed on that DataObject
    protected final Map<String, SyncErrorData> _errorMap = new HashMap<String, SyncErrorData>();

    // Stores an extra list of deletions to report to the client. This is used e.g. if a creation is requested for
    // an appointment due to a MeetingRequest, which has to be ignored (i.e. directly reported back as a deletion)
    protected final List<DataObject> _extraDeletions = new ArrayList<DataObject>();

    // Limit of the number of server operations to report to the client
    protected int _limit;

    // Used for MeetingRequests, to only retrieve once the current OX data for the default calendar folder if there are multiple
    // MeetingRequests to sync
    protected DataObjectSet _currentDefaultCalendarFolderContent = null;

    // Used for MeetingRequests, to only search once for the default calendar folder if there are multiple MeetingRequests to sync
    protected Folder _defaultCalendarFolder = null;

    protected DataObjectFilter _dataObjectFilter = null;

    protected SyncCommandHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
        long now = System.currentTimeMillis();
        _session.setStartDate((_servlet.getTimefilterLimitPast() <= 0) ? Long.MIN_VALUE : (now - _servlet.getTimefilterLimitPast() * MILLIS_PER_DAY));
        _session.setEndDate((_servlet.getTimefilterLimitFuture() <= 0) ? Long.MAX_VALUE : (now + _servlet.getTimefilterLimitFuture() * MILLIS_PER_DAY));
    }

    protected Folder[] readCreatedFolders(Folder[] originalFolders) throws USMJSONAPIException {
        return readFolders(CREATED, originalFolders, true);
    }

    protected Folder[] readModifiedFolders(Folder[] originalFolders) throws USMJSONAPIException {
        return readFolders(MODIFIED, originalFolders, false);
    }

    protected Folder[] readDeletedFolders(Folder[] originalFolders) throws USMJSONAPIException {
        return convertToFolderArray(readDeletedDataObjects(originalFolders));
    }

    protected Folder[] readFolders(String key, Folder[] originalFolders, boolean newObjects) throws USMJSONAPIException {
        return convertToFolderArray(readDataObjects(null, getFolderContentType(), key, originalFolders, newObjects));
    }

    protected DataObject[] readCreatedFolderElements(Folder folder, long syncId, ConflictResolution conflictResolution, DataObject[] originalObjects) throws USMJSONAPIException {
        ContentType elementsContentType = folder.getElementsContentType();
        if (elementsContentType == null)
            return EMPTY_DATAOBJECT_ARRAY;
        DataObject[] result = (elementsContentType.getCode() == DefaultContentTypes.MAIL_CODE) ? readCreatedMailDataObjects(
            folder,
            (MailContentType) elementsContentType,
            conflictResolution,
            syncId,
            originalObjects) : readDataObjects(folder, elementsContentType, CREATED, originalObjects, true);
        for (DataObject o : result)
            o.setParentFolder(folder);
        return result;
    }

    protected DataObject[] readModifiedFolderElements(Folder folder, DataObject[] originalElements, DataObject[] creations) throws USMJSONAPIException {
        DataObject[] modifications = readDataObjects(folder, folder.getElementsContentType(), MODIFIED, originalElements, false);
        return _recurrenceUUIDMap.isEmpty() ? modifications : addModificationsForExceptions(modifications, originalElements, creations);
    }

    // TODO Optimize ?
    protected DataObject[] addModificationsForExceptions(DataObject[] originalModifications, DataObject[] originalElements, DataObject[] creations) throws USMJSONAPIException {
        List<DataObject> modifications = new ArrayList<DataObject>(originalModifications.length + _recurrenceUUIDMap.size());
        for (int i = 0; i < originalModifications.length; i++) {
            modifications.add(originalModifications[i]);
        }
        List<DataObject> toRemoveFromReccurenceMap = new ArrayList<DataObject>();

        for (DataObject exceptionObject : _recurrenceUUIDMap.keySet()) {
            UUID seriesUUID = _recurrenceUUIDMap.get(exceptionObject);
            int count = 0;
            for (UUID uuid : _recurrenceUUIDMap.values()) {
                if (uuid.equals(seriesUUID))
                    count++;
            }
            boolean delayed = count > 0;
            DataObject seriesObjectFromExistingElements;
            try {
                seriesObjectFromExistingElements = getSeriesObjectByUUID(originalElements, seriesUUID);
                if (seriesObjectFromExistingElements.getChangeState() != ChangeState.UNMODIFIED) {
                    delayed = true;
                }

            } catch (USMJSONAPIException e) {
                seriesObjectFromExistingElements = getSeriesObjectByUUID(creations, seriesUUID); // find the series object from the current
                                                                                                 // client creations
                delayed = true;
            }
            String serieID = seriesObjectFromExistingElements.getID();
            DataObject seriesObject = seriesObjectFromExistingElements.createCopy(false);
            seriesObject.commitChanges(); // copy is needed because there may be more exception created in one call
            ContentTypeField[] fields = exceptionObject.getContentType().getFields();
            for (int i = 0; i < fields.length; i++) {
                seriesObject.setFieldContent(i, exceptionObject.getFieldContent(i));
            }
            seriesObject.setID(serieID);
            seriesObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(serieID));

            if (delayed) {
                seriesObject.setUUID(exceptionObject.getUUID());
                _exceptionChangesForDelayedCall.put(seriesObject, seriesUUID);
                toRemoveFromReccurenceMap.add(exceptionObject);
            } else {
                exceptionObject.setID(serieID);
                exceptionObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(serieID));
                modifications.add(seriesObject);
            }
        }
        for (DataObject dataObject : toRemoveFromReccurenceMap) {
            _recurrenceUUIDMap.remove(dataObject); // remove from recurrence_map all exception which were delayed for next sync calls
        }
        return modifications.toArray(new DataObject[modifications.size()]);
    }

    protected long finishProcessingAndSaveNewState(SyncResult result, Folder folder, long syncId) throws USMJSONAPIException {
        if (folder == null) {
            if (_newStateToSave == null)
                return syncId;
            return saveNewSyncState(result, folder, syncId);
        }
        return makeDelayedCallsAndSaveNewState(result, folder, syncId);
    }

    protected long makeDelayedCallsAndSaveNewState(SyncResult result, Folder folder, long syncId) throws USMJSONAPIException {

        if (folder.getElementsContentType() == null)
            return syncId;
        String folderID = folder.getID();
        if (folderID == null || folderID.equals(folder.getElementsContentTypeID())) // is Dummy content type folder
            return syncId;

        makeDelayedSyncForExceptions(syncId, folder);

        if (_newStateToSave == null) {
            try {
                // TODO Depending on syncId, we might already have the cached folder elements somewhere
                DataObject[] cachedFolderElements = _session.getCachedFolderElements(folderID, folder.getElementsContentType(), syncId);
                if(cachedFolderElements == null)
                    throw new USMException(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_READ_CACHED_ELEMENTS, "DB/Cache error: SyncState could not be read");
                _newStateToSave = new DataObjectSet(cachedFolderElements);
            } catch (USMException e) {
                throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_RETRIEVE_CACHED_ELEMENTS, e);
            }
        }

        _newStateToSaveChanged = false; // needed because the PIMAttachments are always compared just by timestamp, and we make changes also
                                        // to the attachments list and keep the old timestamp
        handlePIMAttachments(folder, result, syncId);
        checkMeetingRequestCreationConflicts();
        confirmForeignAppointments();
        return saveNewSyncState(result, folder, syncId);
    }
    
    private void confirmForeignAppointments() {
        if(_appointmentsToConfirm != null && _appointmentsToConfirm.size() > 0) {
            for (Entry<DataObject, UserParticipantObject> entry : _appointmentsToConfirm.entrySet()) {
                DataObject appointment = entry.getKey();
                AppointmentContentType contentType = (AppointmentContentType) appointment.getContentType();
                try {
                    if(appointment.getID() == null)
                        appointment.setID(contentType.resolveUID(_session, (String)appointment.getFieldContent(UID)));
                    if(appointment.getID() != null) 
                        contentType.confirm(appointment, entry.getValue().getConfirmation(), entry.getValue().getConfirmMessage());
                } catch (AuthenticationFailedException e) {
                    _servlet.getJournal().error(_session + " Could not confirm appointment: " + appointment, e);
                } catch (OXCommunicationException e) {
                    _servlet.getJournal().error(_session + " Could not confirm appointment: " + appointment, e);
                } catch (USMException e) {
                    _servlet.getJournal().error(_session + " Could not confirm appointment: " + appointment, e);
                }
            }
        }
    }

    protected long saveNewSyncState(SyncResult result, Folder folder, long syncId) throws USMJSONAPIException {
        if (!_newStateToSaveChanged && _newStateToSave.isEqualTo(result.getNewState()))
            return syncId;
        try {
            if (folder == null)
                return _session.storeSyncState(getOriginalSyncID(), syncId, _newStateToSave.toArray());
            return _session.storeSyncState(getOriginalSyncID(), syncId, folder.getID(), _newStateToSave.toArray());
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_SAVE_SYNC_STATE, e);
        }
    }

    /**
     * Removes error messages from the errors list if the error is on object creation from a meeting request mail. 
     * @throws USMJSONAPIException
     */
    private void checkMeetingRequestCreationConflicts() throws USMJSONAPIException {
        for (Iterator<SyncErrorData> i = _errorMap.values().iterator(); i.hasNext();) {
            SyncErrorData entry = i.next();
            if (!entry._errorStatus.isAttachmentError() && checkMeetingRequestCreationConflict(entry.getObject()))
                i.remove();
        }
    }

    protected void makeDelayedSyncForExceptions(long syncID, Folder folder) {
        ContentType elementsContentType = folder.getElementsContentType();
        for (Map.Entry<DataObject, UUID> entry : _exceptionChangesForDelayedCall.entrySet()) {
            DataObject exception = entry.getKey();
            UUID recurrenceUUID = entry.getValue();
            if (hasNoError(recurrenceUUID) && hasNoError(exception) && !hasExceptionBeenDeleted(exception, recurrenceUUID)) {
                try {
                    // TODO Optimize, if possible
                    DataObject[] currentFolderElements = elementsContentType.getTransferHandler().readFolderContent(folder);
                    updateUUIDs(currentFolderElements);
                    DataObject exceptionToSend = refreshExceptionFromCachedElements(currentFolderElements, exception, recurrenceUUID);
                    if (exceptionToSend != null) {
                        elementsContentType.getTransferHandler().writeUpdatedDataObject(exceptionToSend, exceptionToSend.getTimestamp());
                        _newStateToSave = getNewStateToSave(folder, syncID, exceptionToSend, recurrenceUUID);
                    }
                } catch (SlowSyncRequiredException e) {
                    addError(recurrenceUUID, exception, new USMJSONAPIException(
                        ConnectorBundleErrorCodes.SYNC_UPDATE_INVALID_SYNCID_2,
                        ResponseStatusCode.UNKNOWN_SYNCID,
                        "Unknown SyncID"));
                } catch (SynchronizationConflictException e) {
                    addError(recurrenceUUID, exception, generateConflictException(e));
                } catch (OXCommunicationException e) {
                    if (!isOXErrorForAlreadyDeletedException(e))
                        addError(
                            recurrenceUUID,
                            exception,
                            USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.OX_ERROR_FOR_SERIES_EXCEPTION, e));
                } catch (USMException e) {
                    addError(
                        recurrenceUUID,
                        exception,
                        USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_INTERNAL_ERROR_2, e));
                }
            }
        }
    }

    private static boolean isOXErrorForAlreadyDeletedException(OXCommunicationException e) {
        if (!e.isJSONErrorSet())
            return false;
        JSONObject errorData = e.getJSONError();
        return ("USER_INPUT".equals(errorData.optString("categories")) || errorData.optInt("category") == 1) && "APP-0098".equals(errorData.optString("code"));
    }

    private void updateUUIDs(DataObject[] currentFolderElements) throws DatabaseAccessException, USMSQLException {
        DataObjectSet savedData = _newStateToSave == null ? new DataObjectSet(_syncResult.getNewState()) : _newStateToSave;
        for (DataObject o : currentFolderElements) {
            DataObject saved = savedData.get(o.getID());
            if (saved != null)
                o.setUUID(saved.getUUID());
            else
                insertStoredUUID(o);
        }
    }

 private boolean hasExceptionBeenDeleted(DataObject exception, UUID recurrenceUUID) {
        if (exception == null || exception.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE)
            return true;
        DataObject seriesObject = getCurrentAppointmentSeriesObject(recurrenceUUID);
        if (seriesObject == null)
            return true; // Whole series could not be found -> series has been deleted
        Object recContent = exception.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
        if (!(recContent instanceof Number))
            return false;
        long recurrenceDatePosition = ((Number) recContent).longValue();
        Object delExceptions = seriesObject.getFieldContent(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
        if (!(delExceptions instanceof Object[]))
            return false;
        for (Object delException : ((Object[]) delExceptions)) {
            if (delException instanceof Number) {
                if (recurrenceDatePosition == ((Number) delException).longValue())
                    return true;
            }
        }
        return false;
    }

    private DataObject getCurrentAppointmentSeriesObject(UUID recurrenceUUID) {
        if (_newStateToSave != null)
            return _newStateToSave.get(recurrenceUUID);
        if (_syncResult != null)
            return DataObjectUtil.findDataObject(recurrenceUUID, _syncResult.getNewState());
        return null;
    }

    protected boolean hasNoError(UUID uuid) {
        return uuid != null && !_errorMap.containsKey(uuid.toString());
    }

    protected boolean hasNoError(DataObject o) {
        return hasNoError(o.getUUID());
    }

    protected void addError(UUID uuid, DataObject object, USMException error) {
        addError(uuid, object, error, ErrorStatusCode.OTHER);
    }

    protected void addError(UUID uuid, DataObject object, USMException error, ErrorStatusCode errorStatus) {
        if (uuid == null)
            uuid = object.getUUID();
        if (uuid != null)
            addError(uuid.toString(), object, error, errorStatus);
    }

    protected void addError(String uuid, DataObject object, USMException error, ErrorStatusCode errorStatus) {
        // PIM Attachment creation is a "low priority" error and may occur more than once. Only the first of its kind will be
        // stored and other errors will be reported if they occur afterwards (e.g. a DB error when storing the modified sync state)
        if (!errorStatus.isAttachmentError() || !_errorMap.containsKey(uuid)) {
            SyncErrorData errorData = _errorMap.get(uuid);
            if (errorData != null)
                errorData.initObject(object);
            else
                _errorMap.put(uuid, new SyncErrorData(error, errorStatus, object));
        }
    }

    protected void addError(DataObject object, USMException error) {
        addError(object, error, ErrorStatusCode.OTHER);
    }

    protected void addError(DataObject object, USMException error, ErrorStatusCode errorStatus) {
        addError((UUID) null, object, error, errorStatus);
    }

    /**
     * New state to save in case of delayed calls for appointment exceptions.
     * 
     * @param folder
     * @param syncID
     * @param exceptionToSend
     * @param seriesUUID
     * @return
     * @throws USMException
     */
    protected DataObjectSet getNewStateToSave(Folder folder, long syncID, DataObject exceptionToSend, UUID seriesUUID) throws USMException {
        // TODO Might need optimization
        _session.invalidateCachedData(folder.getID());
        SyncResult syncResult = _session.syncChangesWithServer(folder.getID(), syncID, Session.NO_LIMIT, null, false, null);
        DataObjectSet oldState;
        if (_newStateToSave == null || _newStateToSave.size() == 0)
            oldState = new DataObjectSet(_session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(), syncID));
        else
            oldState = _newStateToSave;

        setTheClientUUIDsInObjectsCreatedOnServer(syncResult, exceptionToSend, oldState);

        DataObjectSet result = new DataObjectSet(oldState);
        for (DataObject dataObject : syncResult.getNewState()) {
            if (exceptionToSend.getUUID().equals(dataObject.getUUID()) || seriesUUID.equals(dataObject.getUUID())) {
                // remove the old state of this object if existing
                result.remove(dataObject.getUUID());
                result.add(dataObject);
            }
        }
        return result;
    }

    private void setTheClientUUIDsInObjectsCreatedOnServer(SyncResult result, DataObject exception, DataObjectSet oldState) throws USMException, USMJSONAPIException {
        DataObject o = findDataObjectWithMatchingRecurrenceIdAndStartDate(exception, result.getNewState());
        if (o != null) {
            if (oldState == null || !oldState.contains(o.getID()) || !exception.getUUID().equals(oldState.get(o.getID()).getUUID())) {
                try {
                    _session.storeUUID(o.getContentType(), Integer.parseInt(o.getID()), exception.getUUID(), _folderOwnerID);
                } catch (NumberFormatException e) {
                    throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_NUMBER_ERROR, new USMException(
                        ConnectorBundleErrorCodes.SYNC_UPDATE_NUMBER_ERROR,
                        e));
                }
            }
            o.setUUID(exception.getUUID());
        }
    }

    private boolean matchRecurrenceIdAndStartDate(DataObject o1, DataObject o2) {
        Object recurrence_id1 = o1.getFieldContent(RECURRENCE_ID);
        if (recurrence_id1 == null)
            return false;
        Object startDate1 = o1.getFieldContent(START_DATE);
        if (startDate1 == null)
            return false;
        Object recurrence_id2 = o2.getFieldContent(RECURRENCE_ID);
        if (!recurrence_id1.equals(recurrence_id2))
            return false;
        Object startDate2 = o2.getFieldContent(START_DATE);
        return startDate1.equals(startDate2);
    }

    private DataObject findDataObjectWithMatchingRecurrenceIdAndStartDate(DataObject o, DataObject[] list) {
        for (DataObject object : list) {
            if (isAppException(object) && matchRecurrenceIdAndStartDate(object, o)) //
                return object;
        }
        return null;
    }

    private DataObject refreshExceptionFromCachedElements(DataObject[] cachedFolderElements, DataObject exception, UUID recurrenceUUID) throws NumberFormatException, DatabaseAccessException, USMSQLException {
        // TODO Optimize this, do not iterate over all elements (depending on optimizations of makeDelayedSyncForExceptions)
        UUID exceptionUUID = exception.getUUID();
        if (exceptionUUID == null)
            return null;
        // First, search if exception is still present in OX server (i.e. series change did not remove all exceptions)
        for (int i = 0; i < cachedFolderElements.length; i++) {
            if (cachedFolderElements[i] != null && exceptionUUID.equals(cachedFolderElements[i].getUUID())) {
                DataObject temp = DataObjectUtil.copyAndModify(cachedFolderElements[i], exception, false);
                temp.setID(cachedFolderElements[i].getID());
                temp.setParentFolderID(cachedFolderElements[i].getParentFolderID());
                temp.setTimestamp(cachedFolderElements[i].getTimestamp());
                return temp;
            }
        }
        // If we have no series or the exception was deleted by the server due to a major change to the series, do nothing
        // We do not try to recreate exceptions that were deleted by the OX server due to major changes to the series because that causes
        // too many problems e.g. with UUID mappings
        if (recurrenceUUID == null || getCachedElement(exception.getUUID()) != null)
            return null;
        // If exception was newly created, search for series object in order to create the exception (with the data sent from the client)
        for (int i = 0; i < cachedFolderElements.length; i++) {
            if (cachedFolderElements[i] != null && recurrenceUUID.equals(cachedFolderElements[i].getUUID())) {
                return buildDataObjectForNewException(cachedFolderElements[i], exception);
            }
        }
        return null;
    }

    private DataObject buildDataObjectForNewException(DataObject series, DataObject exception) {
        DataObject temp = DataObjectUtil.copyAndModify(series, exception, false);
        temp.setID(series.getID());
        temp.setParentFolderID(series.getParentFolderID());
        temp.setFieldContent(RECURRENCE_ID, Integer.parseInt(series.getID()));
        temp.setUUID(exception.getUUID());
        temp.setTimestamp(series.getTimestamp());
        return temp;
    }

    protected DataObject[] readDeletedFolderElements(DataObject[] originalElements) throws USMJSONAPIException {
        return readDeletedDataObjects(originalElements);
    }

    private Folder[] convertToFolderArray(DataObject[] dataObjects) {
        Folder[] result = new Folder[dataObjects.length];
        System.arraycopy(dataObjects, 0, result, 0, dataObjects.length);
        return result;
    }

    private DataObject[] readDataObjects(Folder folder, ContentType contentType, String key, DataObject[] originalObjects, boolean newObjects) throws USMJSONAPIException {
        if (!_parameters.has(key))
            return EMPTY_DATAOBJECT_ARRAY;
        final boolean isFolder = DefaultContentTypes.FOLDER_ID.equals(contentType.getID());
        DataObjectSet originalObjectSet = new DataObjectSet(originalObjects);
        JSONArray list = getJSONArray(_parameters, key);
        int size = list.length();
        DataObjectSet result = new DataObjectSet();
        Map<DataObject, UUID> parentFolderUUIDMap = new HashMap<DataObject, UUID>();

        for (int i = 0; i < size; i++) {
            JSONObject data = getJSONObject(list, i);
            DataObject object = null;
            String objectUUID = getUUIDString(data);
            try {
                object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, newObjects ? null : originalObjects);
                if (newObjects && originalObjectSet.contains(object.getUUID())) {
                    addDuplicateUUIDError(object);
                } else {
                    ContentTypeField[] fields = contentType.getFields();
                    for (String prop : JSONToolkit.keys(data)) {
                        final boolean isTitle = CommandConstants.TITLE.equals(prop);
                        if (OBJECT_TYPE.equals(prop)) {
                            checkObjectTypeInJSON(contentType, data);
                        } else if (isFolder && FOLDER_UUID.equals(prop)) {
                            String folder_uuid = getString(data, prop);
                            if (folder_uuid != null && folder_uuid.length() > 0)
                                parentFolderUUIDMap.put(object, UUID.fromString(folder_uuid));
                        } else if (ATTACHMENTS.equals(prop)) {
                            addAttachmentsToObject(object, getJSONArray(data, ATTACHMENTS), objectUUID);
                        } else if (CONTACT_IMAGE_DATA.equals(prop)) {
                            storeContactImage(object, data);
                        } else if (DISTRIBUTION_LIST.equals(prop)) {
                            addDistributionListToObject(object, data, contentType, fields);
                        } else if (CONFIRMATIONS.equals(prop) && MODIFIED.equals(key)) {
                            JSONArray confirmations = getJSONArray(data, prop);
                            ConfirmingParticipant newConfirmation = getModifiedConfirmationForExternalParticipant(object, confirmations);
                            if (newConfirmation != null)
                                updateConfirmationOnServer(newConfirmation, object);
                        } else if (isTitle && isFolder && MODIFIED.equals(key) && DefaultContentTypes.MAIL_ID.equals(((Folder) object).getElementsContentTypeID())) {
                            // when the title of a mail folder is changed than change the id too
                            // removed because of bug 27444 and 27525 - we should not change the id in the changes list before it is changed
                            // on the server!
                            storeField(object, fields, data, prop);
                        } else if (isTitle && isFolder && FolderType.SHARED.matches(object.getFieldContent(TYPE))) {
                            // Bug 18160: ignore titles of shared folders which are sent by the client
                        } else if (USMJSONVersion.supportsContactImageStreaming(_session) && TEMPID.equals(prop) && contentType.getCode() == DefaultContentTypes.CONTACTS_CODE) {
                            object.setFieldContent(CONTACT_IMAGE_DATA, new Image(data.optString(TEMPID), CommandConstants.IMAGE_JPEG));
                        } else if (!UUID_KEY.equals(prop) && !CHANGE_EXCEPTIONS.equals(prop)) {
                            storeField(object, fields, data, prop);
                        }
                    }
                    if (data.has(CHANGE_EXCEPTIONS)) {
                        addExceptionsToClientChanges(result, object, getJSONArray(data, CHANGE_EXCEPTIONS), originalObjects, contentType);
                    }
                    // Only add object to changes if anything was modified, e.g. if only exceptions to a series were created/modified, we do not
                    // need to add the series itself
                    if (object.getChangeState() != ChangeState.UNMODIFIED || _clientAttachmentsMap.containsKey(objectUUID))
                        result.add(object);
                    //remove the participant status for creations for foreign appointments
                    removeOwnConfirmationFromForeignAppointment(object);
                }
            } catch (DataObjectNotFoundException e) {
                addError(objectUUID, null, e, ErrorStatusCode.UNKNOWN_UUID);
            } catch (MultipleOperationsOnDataObjectException e) {
                addError(objectUUID, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
            } catch (InvalidUUIDException e) {
                addError(objectUUID, null, e, ErrorStatusCode.UNKNOWN_UUID);
            }
        }
        for (Map.Entry<DataObject, UUID> entry : parentFolderUUIDMap.entrySet()) {
            DataObject movedFolder = entry.getKey();
            setParentFolder(folder, movedFolder, entry.getValue(), result, originalObjectSet);
            if(!newObjects)
                result.add(movedFolder);
        }
        return result.toArray();
    }
    
    /**
     * Resets the own confirmation status to 0 for appointments created via external meeting requests. 
     * The confirmation ctatus should be than set using the calendar-confirm call in order to trigger a reply mail for the organizer.  
     * @param object
     */
    private void removeOwnConfirmationFromForeignAppointment(DataObject object) {
        if(object.getChangeState() != ChangeState.CREATED || object.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE)
            return;
        if(object.getFieldContent("organizerId") != null) //identifies external appointments
            return;
        ArrayList<UserParticipantObject> newUsersList = new ArrayList<UserParticipantObject>();
        Object[] oldUsersList = (Object[]) object.getFieldContent("users");
        for (int i = 0; i < oldUsersList.length; i++) {
            UserParticipantObject user = (UserParticipantObject) oldUsersList[i];
                if(user.getId() == _session.getUserIdentifier() && user.getConfirmation() > 0) {
                    UserParticipantObject changedParticipant = new UserParticipantObject(user.getId(), user.getDisplayName(), 0, "");
                    newUsersList.add(changedParticipant);
                    _appointmentsToConfirm.put(object, user);
                } else {
                    newUsersList.add(user);
                }
        }
        object.setFieldContent("users", newUsersList.toArray());
    }

    private ConfirmingParticipant getModifiedConfirmationForExternalParticipant(DataObject object, JSONArray confirmations) {
        Object[] oldConfirmations = (Object[]) object.getFieldContent(CommandConstants.CONFIRMATIONS);
        for (Object confirmationObject : oldConfirmations) {
            ConfirmingParticipant oldConfirmation = (ConfirmingParticipant) confirmationObject;
            if (oldConfirmation.getType() == 5)
                for (int i = 0; i < confirmations.length(); i++) {
                    try {
                        ConfirmingParticipant newConfirmation = new ConfirmingParticipant(confirmations.getJSONObject(i));
                        if (newConfirmation.getMail() != null && newConfirmation.getMail().equals(oldConfirmation.getMail())) {
                            if (newConfirmation.getStatus() != oldConfirmation.getStatus())
                                return newConfirmation;
                        }
                    } catch (JSONException e) {
                        // skip confirmation
                    }
                }
        }
        return null;
    }

    private void updateConfirmationOnServer(ConfirmingParticipant confirmation, DataObject appointment) throws USMJSONAPIException {
        try {
            AppointmentContentType contentType = (AppointmentContentType) appointment.getContentType();
            contentType.confirm(appointment, confirmation.getStatus(), confirmation.getConfirmMessage(), confirmation.getMail());
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_CAN_NOT_CHANGE_CONFIRMATIONS_ERR1, e);
        }
    }

    private void addDistributionListToObject(DataObject object, JSONObject data, ContentType contentType, ContentTypeField[] fields) throws USMJSONAPIException {
        try {
            JSONArray dataArray = getJSONArray(data, DISTRIBUTION_LIST);
            int length = dataArray.length();
            for (int i = 0; i < length; i++) {
                JSONObject memberObj = dataArray.getJSONObject(i);
                if (memberObj.has(UUID_KEY)) {
                    UUID uuid = UUID.fromString(memberObj.getString(UUID_KEY));
                    int id = _session.getMappedObjectId(contentType, uuid);
                    if (id != 0)
                        memberObj.put(CommandConstants.ID, String.valueOf(id));
                }
            }
            storeField(object, fields, data, DISTRIBUTION_LIST);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_BAD_DISTRIBUTION_LIST,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Illegal distribution_list structure ",
                e);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_CANNOT_RESOLVE_DISTRIBUTION_LIST_MEMBER, e);
        }
    }

    private void addAttachmentsToObject(DataObject object, JSONArray attachmentsArray, String objectUUID) throws USMJSONAPIException, InvalidUUIDException {
        Object attachmentsFieldContent = object.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
        long timestamp = 0L;
        PIMAttachment[] oldAttachments = EMPTY_PIMATTACHMENT_ARRAY;
        if (attachmentsFieldContent instanceof PIMAttachments) {
            timestamp = ((PIMAttachments) attachmentsFieldContent).getTimestamp();
            oldAttachments = ((PIMAttachments) attachmentsFieldContent).getAttachments();
        }
        int length = attachmentsArray.length();
        PIMAttachment[] attachments = new PIMAttachment[length];
        try {
            for (int i = 0; i < length; i++) {
                JSONObject attObject = attachmentsArray.getJSONObject(i);
                UUID uuid = extractUUIDFromJSONObject(attObject);
                byte[] data = null;
                if (attObject.has(CommandConstants.DATA_KEY))
                    data = Toolkit.decodeBase64(attObject.optString(CommandConstants.DATA_KEY));
                attachments[i] = new PIMAttachment(uuid, attObject, data);
            }
            if (!Arrays.equals(attachments, oldAttachments)) {
                _clientAttachmentsMap.put(objectUUID, new PIMAttachments(timestamp, true, attachments));
            }
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_READ_ATTACHMENTS,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid attachment data",
                e);
        }
    }

    private DataObject[] readCreatedMailDataObjects(Folder folder, MailContentType contentType, ConflictResolution conflictResolution, long syncId, DataObject[] originalObjectsArray) throws USMJSONAPIException {
        if (!_parameters.has(CREATED))
            return EMPTY_DATAOBJECT_ARRAY;
        boolean incrementalSync = syncId > 0;
        DataObjectSet originalObjects;
        DataObjectSet recentObjects;
        if (incrementalSync) {
            originalObjects = new DataObjectSet(originalObjectsArray);
            recentObjects = getOriginalMails(folder, contentType, 0);
        } else {
            originalObjects = getOriginalMails(folder, contentType, syncId);
            recentObjects = originalObjects;
            originalObjectsArray = originalObjects.toArray();
        }
        JSONArray list = getJSONArray(_parameters, CREATED);
        int size = list.length();
        List<DataObject> clientCreations = new ArrayList<DataObject>(size);
        List<DataObject> newState = new ArrayList<DataObject>();
        for (DataObject o : originalObjectsArray)
            newState.add(o);
        boolean newStateModified = false;
        if (conflictResolution == null)
            conflictResolution = _session.getConflictResolution();
        try {
            for (int i = 0; i < size; i++) {
                JSONObject data = getJSONObject(list, i);
                String uuidString = getUUIDString(data);
                UUID uuid;
                try {
                    uuid = extractUUIDFromString(uuidString);
                } catch (InvalidUUIDException e) {
                    addError(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                    continue;
                }
                if (incrementalSync && originalObjects.contains(uuid)) {
                    _servlet.getJournal().warn(_session + " Client sent repeated email creation for " + uuid);
                    newStateModified |= handleRepeatedClientMailCreation(
                        data,
                        originalObjects.get(uuid),
                        conflictResolution,
                        folder,
                        originalObjectsArray,
                        syncId,
                        clientCreations,
                        newState,
                        true);
                } else if (recentObjects.contains(uuid)) {
                    newStateModified |= handleRepeatedClientMailCreation(
                        data,
                        recentObjects.get(uuid),
                        conflictResolution,
                        folder,
                        originalObjectsArray,
                        syncId,
                        clientCreations,
                        newState,
                        false);
                } else {
                    newStateModified |= handleNewClientMailCreation(contentType, uuid, data, folder, newState);
                }
            }
            if (incrementalSync && newStateModified)
                _syncIdForUpdate = _session.storeSyncState(
                    getOriginalSyncID(),
                    syncId,
                    folder.getID(),
                    newState.toArray(new DataObject[newState.size()]));
        } catch (USMJSONAPIException e) {
            throw e;
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_SAVE_EMAILS_SYNC_STATE, e);
        }
        return clientCreations.toArray(new DataObject[clientCreations.size()]);
    }

    private void addDuplicateUUIDError(DataObject dataObject) {
        addError(dataObject, new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_DUPLICATE_UUID,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "UUID already in use: " + dataObject.getUUID()), ErrorStatusCode.OTHER);
    }

    private DataObjectSet getOriginalMails(Folder folder, ContentType contentType, long syncId) throws USMJSONAPIException {
        DataObjectSet result = new DataObjectSet();
        try {
            result.addAll((syncId > 0) ? _session.getCachedFolderElements(folder.getID(), contentType, syncId) : _session.getCachedFolderElements(
                folder.getID(),
                contentType));
        } catch (DatabaseAccessException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_DB_ERROR_ON_READ_EMAILS,
                ResponseStatusCode.DATABASE_ERROR,
                "DB Access Exception on reading cached mails");
        } catch (USMSQLException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_DB_ERROR_ON_READ_EMAILS_1,
                ResponseStatusCode.DATABASE_ERROR,
                "USMSQLException on reading cached mails");
        }
        return result;
    }

    private boolean handleRepeatedClientMailCreation(JSONObject clientData, DataObject serverObject, ConflictResolution conflictResolution, Folder folder, DataObject[] originalObjects, long syncId, List<DataObject> clientCreations, List<DataObject> newState, boolean mailExistsInOriginalSyncState) {
        boolean requiresNewState = !mailExistsInOriginalSyncState;
        try {
            JSONObject diffs = getDifferencesClientAndServerMail(clientData, serverObject);
            if (diffs != null) {
                // if client mail is the same as the server mail - do nothing
                switch (conflictResolution) {
                case ERROR:
                case ERROR_DELETE_OVER_CHANGE:
                    throw new USMJSONAPIException(
                        ConnectorBundleErrorCodes.COMMAND_SYNC_CONFLICT_2,
                        ResponseStatusCode.SYNC_FAILED,
                        "Conflicting changes",
                        diffs);
                case USE_CLIENT_DELETE_OVER_CHANGE:
                case USE_CLIENT:
                    DataObject[] tempArray = new DataObject[] { serverObject };
                    updateMailObject(folder, originalObjects, syncId, clientCreations, clientData, serverObject, syncId, tempArray);
                    newState.remove(serverObject);
                    serverObject = tempArray[0];
                    requiresNewState = true;
                    break;
                case USE_SERVER_DELETE_OVER_CHANGE:
                case USE_SERVER:
                    // ignore changes from client
                    _mailDiffsClientAndServer.add(diffs);
                    readMailFromJSONObjAndReturnFlags(folder, serverObject.getContentType(), clientData, serverObject);
                    if (serverObject.getChangeState() == ChangeState.UNMODIFIED)
                        serverObject.setTimestamp(serverObject.getTimestamp() + 1);
                    clientCreations.add(serverObject);
                    break;
                default:
                    throw new USMJSONAPIException(
                        ConnectorBundleErrorCodes.COMMAND_BAD_CONFLICT_RESOLUTION_PARAMETER,
                        ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                        "Unknown Conflict Resolution ");
                }
            }
            if (requiresNewState) {
                DataObject stateObject = serverObject.createCopy(true);
                stateObject.commitChanges();
                newState.add(stateObject);
                return true;
            }
        } catch (USMJSONAPIException e) {
            addError(serverObject, e);
        }
        return false;
    }

    private boolean handleNewClientMailCreation(MailContentType contentType, UUID uuid, JSONObject data, Folder folder, List<DataObject> modifiedStartState) throws DatabaseAccessException, USMSQLException {
        // object doesn't exist in cache, it must be created on the server
        DataObject object = contentType.newDataObject(_session);
        object.setUUID(uuid);
        try {
            int flags = readMailFromJSONObjAndReturnFlags(folder, contentType, data, object);
            object.setParentFolder(folder);
            createNewMIMEMailOnServer(contentType, data, object, flags);
            object = object.createCopy(true);
            object.commitChanges();
            modifiedStartState.add(object);
            return true;
        } catch (USMJSONAPIException e) {
            addError(object, e);
            return false;
        }
    }

    private JSONObject getDifferencesClientAndServerMail(JSONObject clientData, DataObject serverObject) throws USMJSONAPIException {
        JSONObject result = null;
        String uuid = null;
        JSONObject extraMailFieldsOnServer = null;
        ContentType contentType = serverObject.getContentType();
        if (existExternalMailFieldsInJSONObject(clientData)) {
            try {
                extraMailFieldsOnServer = readStructuredMail(serverObject);
            } catch (USMException e) {
                throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_READ_MAIL_FROM_SERVER, e);
            }
        }
        for (String prop : JSONToolkit.keys(clientData)) {
            if (OBJECT_TYPE.equals(prop)) {
                checkObjectTypeInJSON(contentType, clientData);
            } else if (FOLDER_UUID.equals(prop)) {
                String folder_uuid = getString(clientData, prop);
                if (folder_uuid != null && folder_uuid.length() > 0) {
                    Folder parentFolder = getFolderByUUID(folder_uuid);

                    if (!serverObject.getParentFolderID().equals(parentFolder.getID())) {
                        result = addDifferencesToJSON(result, folder_uuid, serverObject.getParentFolder().getUUID().toString(), FOLDER_UUID);
                    }
                }
            } else if (UUID_KEY.equals(prop)) {
                uuid = getString(clientData, prop);
            } else if (ExternalMailContentTypeFields.getFieldIndex(prop) < 0) {
                Object clientField;
                try {
                    clientField = clientData.get(prop);
                } catch (JSONException e) {
                    throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_1, e);
                }
                Object serverField = serverObject.getFieldContent(prop);
                if (!clientField.equals(serverField)) {
                    result = addDifferencesToJSON(result, clientField, serverField, prop);
                }
            } else if (ExternalMailContentTypeFields.getFieldIndex(prop) > 0) {
                Object clientField;
                try {
                    clientField = clientData.get(prop);
                    Object serverField = extraMailFieldsOnServer.get(prop);
                    if (clientField instanceof JSONObject && serverField instanceof JSONObject) {
                        if (!JSONToolkit.equals((JSONObject) clientField, (JSONObject) serverField))
                            result = addDifferencesToJSON(result, clientField, serverField, prop);
                    } else if (clientField instanceof JSONArray && serverField instanceof JSONArray) {
                        if (!JSONToolkit.equals((JSONArray) clientField, (JSONArray) serverField))
                            result = addDifferencesToJSON(result, clientField, serverField, prop);
                    } else {
                        if (!clientField.equals(serverField))
                            result = addDifferencesToJSON(result, clientField, serverField, prop);
                    }
                } catch (JSONException e) {
                    throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_2, e);
                }
            }
        }
        if (result != null)
            try {
                JSONObject details = ((JSONArray) result.get(CONFLICTS)).getJSONObject(0);
                details.put(UUID_KEY, uuid);
            } catch (JSONException e) {
                throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_2, e);
            }
        return result;
    }

    private JSONObject addDifferencesToJSON(JSONObject jo, Object clientValue, Object serverValue, String key) {
        try {
            if (jo == null)
                jo = new JSONObject();
            JSONArray conflicts;
            if (jo.has(CONFLICTS))
                conflicts = (JSONArray) jo.get(CONFLICTS);
            else
                conflicts = new JSONArray();
            JSONObject details;
            if (conflicts.length() > 0)
                details = conflicts.getJSONObject(0);
            else
                details = new JSONObject();
            JSONObject clientObject;
            if (details.has(CLIENT))
                clientObject = (JSONObject) details.get(CLIENT);
            else
                clientObject = new JSONObject();
            JSONObject serverObject;
            if (details.has(SERVER))
                serverObject = (JSONObject) details.get(SERVER);
            else
                serverObject = new JSONObject();

            clientObject.put(key, clientValue);
            serverObject.put(key, serverValue);
            details.put(CLIENT, clientObject);
            details.put(SERVER, serverObject);
            conflicts.put(0, details);
            jo.put(CONFLICTS, conflicts);
        } catch (JSONException ignored) {
            // do nothing
        }
        return jo;
    }

    protected boolean existExternalMailFieldsInJSONObject(JSONObject data) {
        String[] fields = ExternalMailContentTypeFields.getFields();
        for (int j = 0; j < fields.length; j++) {
            if (data.has(fields[j]))
                return true;
        }
        return false;
    }

    private static Set<String> getValidMailFields(MailContentType contentType) {
        Set<String> fields = new HashSet<String>();
        for (ContentTypeField field : contentType.getFields())
            fields.add(field.getFieldName());
        for (String s : ExternalMailContentTypeFields.getFields())
            fields.add(s);
        return fields;
    }

    protected long updateMailObject(Folder folder, DataObject[] originalObjects, long syncId, List<DataObject> result, JSONObject data, DataObject object, long syncIdForUpdate, DataObject[] syncStateToStore) throws USMJSONAPIException {
        MailContentType contentType = (MailContentType) object.getContentType();
        if (existExternalMailFieldsInJSONObject(data)) {
            checkElementCreationRights(folder);
            checkElementDeletionRights(folder);
            try {
                contentType.getTransferHandler().readDataObject(object, _session.getFieldFilter(contentType));
                JSONObject extraMailFieldsOnServer = readStructuredMail(object);
                Set<String> validFields = getValidMailFields(contentType);
                for (Iterator<?> iterator = extraMailFieldsOnServer.keys(); iterator.hasNext();) {
                    String key = (String) iterator.next();
                    if (validFields.contains(key) && !data.has(key))
                        // put the fields that are not modified from client but needed to create the new MIME mail
                        data.put(key, extraMailFieldsOnServer.get(key));
                }
                // create the new mime mail on the server
                DataObject newObj = contentType.newDataObject(_session);
                newObj.setUUID(object.getUUID());
                newObj.setParentFolderID(object.getParentFolderID());
                int flags = readMailFromJSONObjAndReturnFlags(folder, contentType, data, newObj);
                createNewMIMEMailOnServer(contentType, data, newObj, flags);
                Object color = newObj.getFieldContent(COLOR_LABEL);
                if (color != null && ((Integer) color).intValue() > 0) {
                    newObj.setFieldContent(COLOR_LABEL, 0);
                    newObj.commitChanges();
                    newObj.setFieldContent(COLOR_LABEL, color);
                    contentType.getTransferHandler().writeUpdatedDataObject(newObj, syncId);
                }
                for (int i = 0; i < syncStateToStore.length; i++) {
                    DataObject originalObject = syncStateToStore[i];
                    if (originalObject.getUUID().equals(object.getUUID())) {
                        syncStateToStore[i] = newObj;
                    } else {
                        syncStateToStore[i] = originalObject;
                    }
                }
                // delete the old mail on server
                contentType.getTransferHandler().writeDeletedDataObject(object);
                if (syncIdForUpdate < 0)
                    result.add(newObj); // _syncIDForUpdate is < 0 only from syncInit command

            } catch (USMException e) {
                if (ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE != _session.getConflictResolution()) {
                    throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_READ_MAIL_FROM_SERVER, e);
                }
            } catch (JSONException e) {
                throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_UPDATE_MAIL_FROM_SERVER, e);
            }
        } else {
            readMailFromJSONObjAndReturnFlags(folder, contentType, data, object);

            try {
                contentType.getTransferHandler().writeUpdatedDataObject(object, syncId);
            } catch (USMException e) {
                if (ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE != _session.getConflictResolution()) {
                    throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_UPDATE_EMAIL_ON_SERVER, e);
                }
            }
        }
        return syncIdForUpdate;
    }

    protected void checkElementDeletionRights(Folder folder) throws USMJSONAPIException {
        if (folder == null)
            return;
        int ownRights = getOwnRights(folder);
        if ((ownRights & 0xFE00000) < 0x400000)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SYNC_INSUFFICIENT_DELETION_RIGHTS,
                ResponseStatusCode.SYNC_FAILED,
                "No deletion rights for objects in folder");
    }

    protected void checkElementCreationRights(Folder folder) throws USMJSONAPIException {
        if (folder == null)
            return;
        int ownRights = getOwnRights(folder);
        if ((ownRights & 0x7F) < 2)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SYNC_INSUFFICIENT_CREATION_RIGHTS,
                ResponseStatusCode.SYNC_FAILED,
                "No creation rights for objects in folder");
    }

   

    protected void createNewMIMEMailOnServer(MailContentType contentType, JSONObject data, DataObject object, int flags) throws USMJSONAPIException {
        checkElementCreationRights(object.getParentFolder());
        try {
            if (USMJSONVersion.supportsAttachmentStreaming(_session)) {
                StreamingMimeMailBuilder mimeMailBuilder = new StreamingMimeMailBuilder(_session);
                InputStreamProvider streamProvider = mimeMailBuilder.convertJSONObjectToMimeMail(data);
                String mailID = contentType.createNewMailFromStream(object.getParentFolderID(), flags, streamProvider, _session);
                object.setID(mailID);
            } else {
                MimeMailBuilder mimeMailBuilder = new MimeMailBuilder();
                String mimeMail = mimeMailBuilder.convertJSONObjectToMimeMail(data);
                String mailID = contentType.createNewMail(object.getParentFolderID(), flags, mimeMail, _session);
                object.setID(mailID);
            }
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_CANNOT_CREATE_MAIL, e);
        }
    }

    boolean isReplyOrForward(int oldFlags, int newFlags) {
        return newFlags > oldFlags && (((newFlags & 0x1) == 0) || ((newFlags & 0x80) == 0));
    }

    protected int readMailFromJSONObjAndReturnFlags(Folder folder, ContentType contentType, JSONObject data, DataObject object) throws USMJSONAPIException {
        int flags = 0;
        ContentTypeField[] fields = contentType.getFields();
        for (String prop : JSONToolkit.keys(data)) {
            if (OBJECT_TYPE.equals(prop)) {
                checkObjectTypeInJSON(contentType, data);
            } else if (FOLDER_UUID.equals(prop)) {
                String folder_uuid = getString(data, prop);
                if (folder_uuid != null && folder_uuid.length() > 0) {
                    Folder parentFolder = getFolderByUUID(folder_uuid);
                    checkMatchSyncAndParentFolder(folder, parentFolder);
                    object.setParentFolder(parentFolder);
                }
            } else if (FLAGS.equals(prop)) {
                flags = getInt(data, prop);
                storeField(object, fields, data, prop);
            } else if (!UUID_KEY.equals(prop) && ExternalMailContentTypeFields.getFieldIndex(prop) < 0) {
                storeField(object, fields, data, prop);
            }
        }
        return flags;
    }

    private void checkObjectTypeInJSON(ContentType contentType, JSONObject data) throws USMJSONAPIException {
        String type = getString(data, OBJECT_TYPE);
        if (!contentType.getID().equals(type))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_DATA_OBJECT_TYPE_MISMATCH,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "DataObject type " + type + " not of expected type " + contentType.getID());
    }

    private void addExceptionsToClientChanges(DataObjectSet result, DataObject seriesObject, JSONArray exceptions, DataObject[] originalObjects, ContentType contentType) throws USMJSONAPIException, MultipleOperationsOnDataObjectException, InvalidUUIDException {
        int length = exceptions.length();
        for (int i = 0; i < length; i++) {
            JSONObject data = getJSONObject(exceptions, i);
            DataObject exceptionObject = getDataObjectFromJSONObjectAndOriginalObjectsIfExisting(contentType, data, originalObjects);
            for (String prop : JSONToolkit.keys(data)) {
                if (OBJECT_TYPE.equals(prop)) {
                    checkObjectTypeInJSON(contentType, data);
                } else if (ATTACHMENTS.equals(prop)) {
                    addAttachmentsToObject(exceptionObject, getJSONArray(data, ATTACHMENTS), getString(data, UUID_KEY));
                } else if (!UUID_KEY.equals(prop)) {
                    storeField(exceptionObject, contentType.getFields(), data, prop);
                }
            }
            if (seriesObject.getID() == null)
                seriesObject.setID(String.valueOf(Integer.MAX_VALUE)); // set dummy id, it will be overwritten from server with a real id
            exceptionObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(seriesObject.getID()));
            if (seriesObject.getParentFolderID() != null)
                exceptionObject.setParentFolderID(seriesObject.getParentFolderID());
            if (exceptionObject.getChangeState() != ChangeState.UNMODIFIED) {
                if (seriesObject.getChangeState() == ChangeState.UNMODIFIED && exceptionObject.getChangeState() != ChangeState.CREATED) {
                    result.add(exceptionObject);
                } else {
                    if (exceptionObject.getUUID().equals(seriesObject.getUUID()))
                        throw new MultipleOperationsOnDataObjectException(seriesObject);
                    checkForChangeExceptionsWithSameUUID(exceptionObject.getUUID());
                    _recurrenceUUIDMap.put(exceptionObject, seriesObject.getUUID());
                }
            }
        }
    }

    private void checkForChangeExceptionsWithSameUUID(UUID uuid) throws MultipleOperationsOnDataObjectException {
        if (uuid == null)
            return;
        for (Iterator<DataObject> i = _recurrenceUUIDMap.keySet().iterator(); i.hasNext();) {
            DataObject exception = i.next();
            if (uuid.equals(exception.getUUID())) {
                i.remove();
                throw new MultipleOperationsOnDataObjectException(exception);
            }
        }
    }

    private void setParentFolder(Folder folder, DataObject o, UUID uuid, DataObjectSet firstList, DataObjectSet secondList) throws USMJSONAPIException {
        if (setParentFolder(folder, o, uuid, firstList) && setParentFolder(folder, o, uuid, secondList))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_UNKNOWN_FOLDER_UUID,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Unknown folder_uuid " + uuid);
    }

    private boolean setParentFolder(Folder folder, DataObject o, UUID uuid, DataObjectSet list) throws USMJSONAPIException {
        if (list != null && list.contains(uuid)) {
            Folder f = (Folder) list.get(uuid);
            checkMatchSyncAndParentFolder(folder, f);
            o.setParentFolder(f);
            return false;
        }
        return true;
    }

    private void checkMatchSyncAndParentFolder(Folder syncFolder, Folder parentFolder) throws USMJSONAPIException {
        if (syncFolder != null && !syncFolder.equals(parentFolder))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_SYNC_AND_PARENT_FOLDER_NO_MATCH,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Objects have different parent folder as the one currently synchronized");
    }

    private DataObject[] readDeletedDataObjects(DataObject[] originalObjects) throws USMJSONAPIException {
        if (!(_parameters.has(DELETED) || _parameters.has(DELETED_IF_EXIST)))
            return EMPTY_DATAOBJECT_ARRAY;
        List<DataObject> result = new ArrayList<DataObject>();
        DataObjectSet originalObjectsSet = new DataObjectSet(originalObjects);

        readDeletedClientList(DELETED, result, originalObjectsSet, true);
        readDeletedClientList(DELETED_IF_EXIST, result, originalObjectsSet, false);
        return result.toArray(new DataObject[result.size()]);
    }

    private void readDeletedClientList(String parameterKey, List<DataObject> result, DataObjectSet originalObjectsSet, boolean addErrorForMissingUUID) throws USMJSONAPIException {
        if (!_parameters.has(parameterKey))
            return;
        JSONArray list = getJSONArray(_parameters, parameterKey);
        int size = list.length();
        for (int i = 0; i < size; i++) {
            String uuid = getString(list, i);
            try {
                DataObject object = getDataObjectByUUID(originalObjectsSet, extractUUIDFromString(uuid));
                object.setChangeState(ChangeState.DELETED);
                result.add(object);
            } catch (DataObjectNotFoundException e) {
                if (addErrorForMissingUUID)
                    addError(uuid, null, e, ErrorStatusCode.UNKNOWN_UUID);
            } catch (MultipleOperationsOnDataObjectException e) {
                addError(uuid, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
            } catch (InvalidUUIDException e) {
                addError(uuid, null, e, ErrorStatusCode.UNKNOWN_UUID);
            }
        }
    }

    protected DataObject getDataObjectFromJSONObjectAndOriginalObjects(ContentType contentType, JSONObject data, DataObject[] originalObjects) throws DataObjectNotFoundException, MultipleOperationsOnDataObjectException, InvalidUUIDException {
        UUID uuid = extractUUIDFromJSONObject(data);
        if (originalObjects != null)
            return getDataObjectByUUID(new DataObjectSet(originalObjects), uuid);
        DataObject result = contentType.newDataObject(_session);
        result.setUUID(uuid);
        return result;
    }

    private DataObject getDataObjectFromJSONObjectAndOriginalObjectsIfExisting(ContentType contentType, JSONObject data, DataObject[] originalObjects) throws MultipleOperationsOnDataObjectException, InvalidUUIDException {
        UUID uuid = extractUUIDFromJSONObject(data);
        DataObject result;
        if (originalObjects != null) {
            result = getExceptionObjectByUUID(originalObjects, uuid);
            if (result != null)
                return result;
        }
        result = contentType.newDataObject(_session);
        result.setUUID(uuid);
        return result;
    }

    private DataObject getExceptionObjectByUUID(DataObject[] objects, UUID uuid) throws MultipleOperationsOnDataObjectException {
        if (uuid != null) {
            for (DataObject o : objects) {
                if (uuid.equals(o.getUUID())) {
                    if (o.getChangeState() != ChangeState.UNMODIFIED)
                        throw new MultipleOperationsOnDataObjectException(o);
                    return o;
                }
            }
        }
        return null;
    }

    // TODO Merge with getDataObjectByUUID
    private DataObject getSeriesObjectByUUID(DataObject[] objects, UUID uuid) throws USMJSONAPIException {
        if (objects != null) {
            for (DataObject o : objects) {
                if (o.getUUID().equals(uuid)) {
                    return o;
                }
            }
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_DATA_OBJECT_NOT_FOUND_2,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "No DataObject with uuid " + uuid);
    }

    private void storeContactImage(DataObject object, JSONObject element) throws USMJSONAPIException {
        try {
            String picture = element.isNull(CONTACT_IMAGE_DATA) ? null : element.getString(CONTACT_IMAGE_DATA);
            if (picture == null) {
                Object oldImage = object.getFieldContent(CONTACT_IMAGE_DATA);
                if (!(oldImage instanceof Image))
                    return;
            }
            object.setFieldContent(CONTACT_IMAGE_DATA, new Image(System.currentTimeMillis(), IMAGE_JPEG, Toolkit.decodeBase64(picture)));
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_ILLEGAL_PROPERTY_VALUE,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Illegal value for property " + CONTACT_IMAGE_DATA,
                e);
        }
    }

    // TODO Possible optimization: Create key->index map for folder to synchronize
    private void storeField(DataObject object, ContentTypeField[] fields, JSONObject element, String key) throws USMJSONAPIException {
        for (ContentTypeField field : fields) {
            if (field.getFieldName().equals(key)) {
                try {
                    object.setFieldContent(key, field.getFieldType().extractFromJSONObject(_session, element, key));
                    return;
                } catch (JSONException e) {
                    throw new USMJSONAPIException(
                        ConnectorBundleErrorCodes.COMMAND_ILLEGAL_PROPERTY_VALUE,
                        ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                        "Illegal value for property " + key,
                        e);
                }
            }
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_UNKNOWN_PROPERTY,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Unknown property " + key);
    }

    protected JSONObject createResponseFromSyncResults(SyncResult result, Long syncID) throws USMJSONAPIException {
        try {
            DataObject[] newState = result.getNewState();
            _allFolders = newState;
            JSONObject response = new JSONObject();
            if (syncID == null)
                syncID = result.getTimestamp();
            response.put(SYNCID, syncID);
            if (_extraSyncRequired || result.isIncomplete())
                response.put(MORE_AVAILABLE, Boolean.TRUE);
            JSONArray created = new JSONArray();
            JSONArray modified = new JSONArray();
            JSONArray deleted = new JSONArray();
            DataObject[] changes = buildClientChangesArray(result.getChanges(), deleted);
            List<DataObject> unnecessaryChanges = new ArrayList<DataObject>();
            List<DataObject> unhandledCreations = new ArrayList<DataObject>();
            List<DataObject> unmodifiedObjects = new ArrayList<DataObject>();
            Map<DataObject, List<DataObject>> exceptionsMap = (_newStateToSave == null) ? createExceptionsMap(new DataObjectSet(newState)) : createExceptionsMap(_newStateToSave);
            int indexCreated = 0;
            int indexModified = 0;
            for (DataObject o : changes) {
                switch (determineClientResultChangeState(o)) {
                case CREATED:
                    JSONObject extraFieldsObject = addExtraFieldsToDataObject(o);
                    if (DefaultContentTypes.CONTACTS_ID.equals(o.getContentType().getID())) {
                        setImageByContacts(o);
                    }
                    if (!isAppException(o)) {
                        JSONObject jObj = storeDataObjectWithExtraFieldsToJSONObject(exceptionsMap, o, extraFieldsObject);
                        indexCreated = addJSONObjToNumberedList(created, jObj, indexCreated);
                    } else if (!masterAppExistsInChanges(o, changes)) {
                        JSONObject jObj = storeDataObjectWithExtraFieldsToJSONObject(exceptionsMap, o, extraFieldsObject);
                        addParentUUIDToObject(o, jObj, newState);
                        indexCreated = addJSONObjToNumberedList(created, jObj, indexCreated);
                    } else {
                        unhandledCreations.add(o);
                    }
                    break;
                case DELETED:
                    addServerDeletionToResponse(deleted, changes, o);
                    break;
                case MODIFIED:
                    if (hasRealChangesForClient(o)) {
                        if (o.getContentType().getCode() == DefaultContentTypes.CONTACTS_CODE)
                            setImageByContacts(o);
                        if (!isAppException(o)) {
                            JSONObject jo = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
                            if (jo.length() > 2) {
                                // only add element if any actual change is present (i.e. more than "uuid" and "objectType" is set)
                                indexModified = addJSONObjToNumberedList(modified, jo, indexModified);
                            } else {
                                unnecessaryChanges.add(o);
                            }
                        } else if (!masterAppExistsInChanges(o, changes)) {
                            JSONObject jo = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
                            if (jo.length() > 2) {
                                // only add element if any actual change is present (i.e. more than "uuid" and "objectType" is set)
                                addParentUUIDToObject(o, jo, newState);
                                indexModified = addJSONObjToNumberedList(modified, jo, indexModified);
                            } else {
                                unnecessaryChanges.add(o);
                            }
                        } else {
                            unnecessaryChanges.add(o);
                        }
                    } else {
                        unnecessaryChanges.add(o);
                    }
                    break;
                case UNMODIFIED:
                    // Should not happen
                    unmodifiedObjects.add(o);
                    break;
                }
            }
            if (_servlet.getJournal().isDebugEnabled()) {
                if (!unnecessaryChanges.isEmpty())
                    logUnnecessaryChanges(unnecessaryChanges);
                if (!unhandledCreations.isEmpty())
                    logUnhandledCreations(unhandledCreations);
                if (!unmodifiedObjects.isEmpty())
                    logUnmodifiedObjects(unmodifiedObjects);
            }
            for (DataObject o : _extraDeletions)
                addServerDeletionToResponse(deleted, changes, o);
            addMailModifsExtraFields(modified);
            if (created.length() > 0)
                response.put(CREATED, created);
            if (modified.length() > 0)
                response.put(MODIFIED, modified);
            if (deleted.length() > 0)
                response.put(DELETED, deleted);
            if (!_errorMap.isEmpty())
                addErrorsToResponse(response);
            return response;
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_SYNC_RESULT_JSON_ERROR, e);
        } catch (USMException e1) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_RESULT_USM_ERROR, e1);
        }
    }

    private void logUnmodifiedObjects(List<DataObject> unmodifiedObjects) {
        StringBuilder sb = new StringBuilder();
        sb.append(getSession()).append(" Reported unmodified objects:");
        char objectDelimiter = ' ';
        for (DataObject o : unmodifiedObjects) {
            sb.append(objectDelimiter).append(o.getID());
            objectDelimiter = ',';
        }
        _servlet.getJournal().debug(sb.toString());
    }

    private void logUnhandledCreations(List<DataObject> unhandledCreations) {
        StringBuilder sb = new StringBuilder();
        sb.append(getSession()).append(" Unreported creations:");
        char objectDelimiter = ' ';
        for (DataObject o : unhandledCreations) {
            sb.append(objectDelimiter).append(o.getID());
            objectDelimiter = ',';
        }
        _servlet.getJournal().debug(sb.toString());
    }

    private void logUnnecessaryChanges(List<DataObject> unnecessaryChanges) {
        if (unnecessaryChanges.isEmpty())
            return;
        ContentType contentType = unnecessaryChanges.get(0).getContentType();
        boolean hasAttachmentsLastModified = hasAttachmentsLastModified(contentType);
        ContentTypeField[] fields = contentType.getFields();
        StringBuilder sb = new StringBuilder();
        sb.append(getSession()).append(" Unnecessary changes:");
        char objectDelimiter = ' ';
        for (DataObject o : unnecessaryChanges) {
            char fieldDelimiter = '(';
            sb.append(objectDelimiter).append(o.getID());
            for (int i = 0; i < fields.length; i++) {
                String fieldName = fields[i].getFieldName();
                if (o.isFieldModified(fieldName)) {
                    sb.append(fieldDelimiter).append(fieldName);
                    fieldDelimiter = ',';
                }
            }
            if (fieldDelimiter == ',')
                sb.append(')');
            objectDelimiter = ',';
        }
        _servlet.getJournal().debug(sb.toString());
        if (hasAttachmentsLastModified) {
            for (DataObject o : unnecessaryChanges) {
                if (o.isFieldModified(ATTACHMENTS_LAST_MODIFIED)) {
                    _servlet.getJournal().debug(
                        _session.toString() + ' ' + o.getID() + '.' + ATTACHMENTS_LAST_MODIFIED + ':' + o.getFieldContent(ATTACHMENTS_LAST_MODIFIED));
                }
            }
        }
    }

    /**
     * Checks if other field than the ones filtered out are modified.
     * 
     * @param o
     * @return
     */
    private boolean hasRealChangesForClient(DataObject o) {
        ContentTypeField[] fields = o.getContentType().getFields();
        for (int i = 0; i < fields.length; i++) {
            String fieldName = fields[i].getFieldName();
            if (o.isFieldModified(i) && isFieldSynchronized(o, i) && !shouldFilterOutField(fieldName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Adds special index field to the JSONObject before adding it to the passed JSONArray.
     * 
     * @param array
     * @param obj
     * @param index
     * @return
     * @throws JSONException
     */
    private int addJSONObjToNumberedList(JSONArray array, JSONObject obj, int index) throws JSONException {
        index++;
        obj.put(INDEX, index);
        array.put(obj);
        return index;
    }

    protected ChangeState determineClientResultChangeState(DataObject o) {
        return o.getChangeState();
    }

    protected DataObject[] buildClientChangesArray(DataObject[] changes, JSONArray deleted) {
        // TODO Expand changes by objects in refreshUUIDs collection, if present in new state (or better in _newStateToSave)
        // TODO Add deletion for every object in refreshUUIDs that is not present in new state
        if (changes.length == 0)
            return changes;
        Folder[] folderChanges = new Folder[changes.length];
        for (int i = 0; i < changes.length; i++) {
            if (!(changes[i] instanceof Folder))
                return changes;
            folderChanges[i] = (Folder) changes[i];
        }
        Arrays.sort(folderChanges, new FolderHierarchyComparator(folderChanges, _servlet.getJournal()));
        return folderChanges;
    }

    private void addServerDeletionToResponse(JSONArray deleted, DataObject[] changes, DataObject o) {
        if ((!isAppException(o) || !masterAppExistsInChanges(o, changes))) {
            deleted.put(o.getUUID().toString());
        }
    }

    private void addParentUUIDToObject(DataObject o, JSONObject jObj, DataObject[] newState) throws USMJSONAPIException {
        int recID = (Integer) o.getFieldContent(RECURRENCE_ID);
        DataObject parentApp = null;
        for (DataObject dataObject : newState) {
            if (String.valueOf(recID).equals(dataObject.getID())) {
                parentApp = dataObject;
                break;
            }
        }
        if (parentApp == null)
            return;
        try {
            String parentUUID = parentApp.getUUID().toString();
            jObj.put(PARENT_UUID, parentUUID);
        } catch (JSONException e) {
            // if new string can not be added to the json object
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.PARENT_UUID_CAN_NOT_BE_ADDED, e);
        }
    }

    private boolean masterAppExistsInChanges(DataObject exception, DataObject[] changes) {
        int recID = (Integer) exception.getFieldContent(RECURRENCE_ID);
        for (DataObject change : changes) {
            if (String.valueOf(recID).equals(change.getID()))
                return true;
        }
        return false;
    }

    private void addErrorsToResponse(JSONObject response) throws USMJSONAPIException, JSONException {
        JSONObject errors = new JSONObject();
        for (Map.Entry<String, SyncErrorData> entry : _errorMap.entrySet()) {
            SyncErrorData data = entry.getValue();
            DataObject o = data.getObject();
            JSONObject objectData = null;
            if (o != null) {
                DataObject object = o.createCopy(false);
                if (object.getChangeState() == ChangeState.MODIFIED)
                    object.rollbackChanges();
                object.setChangeState(ChangeState.CREATED);
                objectData = storeDataObjectWithExtraFieldsToJSONObject(null, object, addExtraFieldsToDataObject(object));
            }
            ResponseObject errorData = new ResponseObject(null, objectData, data._error);
            errorData.put(ERROR_STATUS, computeErrorStatus(data).getStatusCode());
            errors.put(entry.getKey(), errorData);
        }
        if (errors.length() > 0)
            response.put(ERRORS, errors);
    }

    private ErrorStatusCode computeErrorStatus(SyncErrorData data) {
        if (data._errorStatus != null)
            return data._errorStatus;
        DataObject object = data.getObject();
        if (object == null)
            return ErrorStatusCode.UNKNOWN_UUID;

        if (data._error != null && data._error.getOxErrorForJSONResponse() != null) {
            // check possible UID error
            JSONObject oxError = data._error.getOxErrorForJSONResponse();
            try {
                int category = oxError.optInt(UtilConstants.CATEGORY);
                String code = oxError.getString(UtilConstants.CODE);
                if (category == OXErrorConstants.CATEGORY_WRONG_INPUT && OXErrorConstants.APP_0100.equals(code))
                    return ErrorStatusCode.UID_ALREADY_EXISTS;
            } catch (JSONException e) {
                // do nothing
            }
        }

        if (object instanceof Folder) {
            if (object.getChangeState() == ChangeState.CREATED) {
                Folder parent = object.getParentFolder();
                if (parent != null) {
                    if ((getOwnRights(parent) & 0x7F) < 4)
                        return ErrorStatusCode.NO_CREATE_PERMISSION;
                    Folder copy = parent.createCopy(false);
                    try {
                        copy.getContentType().getTransferHandler().readDataObject(parent, new BitSet());
                    } catch (USMException e) {
                        return ErrorStatusCode.PARENT_FOLDER_NOT_FOUND;
                    }
                }
            } else if ((getOwnRights((Folder) object) & 0x10000000) == 0) {
                return (object.getChangeState() == ChangeState.DELETED) ? ErrorStatusCode.NO_DELETE_PERMISSION : ErrorStatusCode.NO_CHANGE_PERMISSION;
            } else if (object.getChangeState() == ChangeState.MODIFIED && data._error instanceof FolderNotFoundException) {
                return ErrorStatusCode.MAIL_FOLDER_NOT_FOUND;
            } else if (isPermissionsErrorFromServer(data._error.getOxErrorForJSONResponse())) {
                return ErrorStatusCode.PERMISSION_DENIED_BY_SERVER;
            }
        } else if (_folderToSync != null) {
            switch (object.getChangeState()) {
            case CREATED:
                if (!checkCreatePermissionsAllObjects(getOwnRights(_folderToSync)))
                    return ErrorStatusCode.NO_CREATE_PERMISSION;
                break;
            case DELETED:
                if (!checkDeletePermissions(getOwnRights(_folderToSync), object))
                    return ErrorStatusCode.NO_DELETE_PERMISSION;
                break;
            case MODIFIED:
                if (!checkModifyPermissions(getOwnRights(_folderToSync), object))
                    return ErrorStatusCode.NO_CHANGE_PERMISSION;
                break;
            case UNMODIFIED: // Should not be possible
                break;
            }
        }
        return ErrorStatusCode.OTHER;
    }

    private boolean isPermissionsErrorFromServer(JSONObject oxError) {
        if (oxError == null)
            return false;
        try {
            if (oxError.getInt("category") == 3)
                return true;
        } catch (JSONException e) {
            return false;
        }
        return false;
    }

    /**
     * Returns true if the error should be removed from the error map (i.e. this method applied some alternative algorithm to handle it)
     * The alternative algorithm has the following steps: 
     * 1. search for the appointment ID on the server using the calendar/resolveuid call
     * 2. search for the appointment object on the server using the GET calendar/all call
     * 3. read the current state of the found appointment
     * 4. update the appointment on the server (the usual change here is change on the confirmation status of the participant)
     * 
     * If step 1- 2 fail (or return no result), than the original error should not be removed from the errors map and proceeded to the server
     * If step 3 - 4 fail than the original error will be removed and not proceeded to the client. This may lead to lost of the client change and the client will have to repeat the change.  
     * @param object
     * @return
     * @throws USMJSONAPIException
     */
    private boolean checkMeetingRequestCreationConflict(DataObject object) throws USMJSONAPIException {
        if (object == null || object.getChangeState() != ChangeState.CREATED || !DefaultContentTypes.CALENDAR_ID.equals(object.getContentType().getID()))
            return false;
        String uid = (String) object.getFieldContent(CommandConstants.UID);
        if (uid == null)
            return false;
        Session session = object.getSession();
        AppointmentContentType contentType = (AppointmentContentType) object.getContentType();
        
        //step1: find the ox id based on the uid send from client
        String oxID = null;
        try {
            oxID = contentType.resolveUID(session, uid);
        } catch (USMException ignored) {
            //exception if appointment can not be found ignored
                _servlet.getJournal().error(_session + " Can not find appointment with UID " + uid, ignored);
        }
        if (oxID == null) 
            return false; // return false if the UID doesn't exist on the server
        
        //step 2: find the appointment object on the server by reading all appointments for this user
        DataObject app = null;
        try {
            DataObjectSet allApp = getAllAppointments();
            app = allApp.get(oxID);
            DataObject orphanInstance = getOccurrenceOf(oxID, allApp, object);
            if (orphanInstance != null)
                app = orphanInstance;
        } catch (USMException e) {
            _servlet.getJournal().error(_session + " USMException when trying to read all appointments to find app with ox id "+ oxID, e);
            return false; // report the error if the read of all appointments has failed - this means we can not find the appointment
        }
        if(app == null) {
            _servlet.getJournal().info(_session + " Can not find appointment with id " + oxID);
            return false; // report the original error if the appointment could not be found. It is possible that the appointment exists for some other user and the current user doesn't see it. 
        }
        //step 3 - 4: read and update the appointment - retry up to 10 times to change the appointment- because of possible failure on concurrent changes from other participants
        int retryCount = 0;
        boolean shouldRetry = true;
        while (shouldRetry) {
            try {
                contentType.getTransferHandler().readDataObject(app, _session.getFieldFilter(contentType));
                DataObject appCopy = DataObjectUtil.copyAndModify(app, object, true);
                appCopy.setTimestamp(app.getTimestamp());
                appCopy.setID(app.getID());
                appCopy.setParentFolderID(app.getParentFolderID());
                contentType.getTransferHandler().writeUpdatedDataObject(appCopy, appCopy.getTimestamp());
                if (_limit < 0) {
                    DataObject copy = object.createCopy(true);
                    copy.commitChanges();
                    _newStateToSave.add(copy);
                    _newStateToSaveChanged = true;
                } else {
                    _extraDeletions.add(object);
                }
                shouldRetry = false;
            } catch (USMException e) {
                _servlet.getJournal().error(_session + " USMException when trying to read/update existing app with uid " + uid, e);
                // do not report the error if changing the appointment or readig its current state failed
                retryCount++;
                if (retryCount > 10)
                    return false; // report the error to the client if after 10 retries there are still conflicting changes with other participants
            }
        }
        return true;
    }

    /**
     * Workaround for bug 20779 - client is not sending recurrence position and we have to find the exception based on the
     * recurrence_date_position field
     * 
     * @param oxID
     * @param allApp
     * @param createdObject
     * @return {@link DataObject}
     * @throws USMException
     */
    private DataObject getOccurrenceOf(String oxID, DataObjectSet allApp, DataObject createdObject) throws USMException {
        DataObject excObject = null;
        Number recDatePos = (Number) createdObject.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
        if (recDatePos != null && recDatePos.longValue() > 0) {
            for (DataObject dataObject : allApp) {
                Number currentObjRecDatePos = (Number) dataObject.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
                String recurrenceID = String.valueOf((Number) dataObject.getFieldContent(RECURRENCE_ID));
                if (oxID.equals(recurrenceID) && recDatePos.equals(currentObjRecDatePos)) {
                    excObject = dataObject;
                    break;
                }
            }
        }
        return excObject;
    }

    private JSONObject storeDataObjectWithExtraFieldsToJSONObject(Map<DataObject, List<DataObject>> exceptionsMap, DataObject o, JSONObject extraFieldsObject) throws JSONException, USMJSONAPIException {
        JSONObject jObj = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
        if (extraFieldsObject != null)
            storeExtraFieldsObjectInJSONObject(jObj, extraFieldsObject);
        return jObj;
    }

    // TODO Add description of the purpose of this method
    private void addMailModifsExtraFields(JSONArray modified) throws JSONException {
        for (JSONObject diffs : _mailDiffsClientAndServer) {
            JSONArray conflicts = (JSONArray) diffs.get(CONFLICTS);
            JSONObject details = conflicts.getJSONObject(0);
            JSONObject serverObject = (JSONObject) details.get(SERVER);
            String uuidDiffs = details.getString(UUID_KEY);
            // check if already exists in modifications
            JSONObject modObj = null;
            int length = modified.length();
            int originalIndex = length;
            for (int i = 0; i < length; i++) {
                JSONObject modObjFromArray = modified.getJSONObject(i);
                String uuid = modObjFromArray.getString(UUID_KEY);
                if (uuid.equals(uuidDiffs)) {
                    modObj = modObjFromArray;
                    originalIndex = i;
                }
            }
            if (modObj == null)
                modObj = new JSONObject();
            for (String prop : JSONToolkit.keys(serverObject)) {
                modObj.put(prop, serverObject.get(prop));
            }
            modObj.put(OBJECT_TYPE, "mail");
            modObj.put(UUID_KEY, uuidDiffs);
            modified.put(originalIndex, modObj);
        }
    }

    private void setImageByContacts(DataObject o) {
        if (shouldIncludeContactPicture(o)) {
            if (USMJSONVersion.supportsContactImageStreaming(_session)) {
                Object value = o.getFieldContent(CONTACT_IMAGE_DATA);
                if (value instanceof Image) {
                    Image i = (Image) value;
                    o.setFieldContent(CONTACT_IMAGE_DATA, new Image(ServerTempId.forContactImage(o).toString(), CommandConstants.IMAGE_JPEG));
                }
            } else {
                try {
                    byte[] pictureBytes = ((ContactContentType) o.getContentType()).getPictureData(o, "jpeg", Integer.MAX_VALUE);
                    if (pictureBytes != null)
                        o.setFieldContent(CONTACT_IMAGE_DATA, new Image(o.getTimestamp(), IMAGE_JPEG, pictureBytes));
                } catch (USMException e) {
                    _servlet.getJournal().warn(
                        _session + " Error reading image of contact " + o.getID() + " - " + e.getErrorCode() + ": " + e.getMessage(),
                        e);
                }
            }
        }
    }

    protected boolean shouldIncludeContactPicture(DataObject o) {
        if (o.getChangeState() == ChangeState.MODIFIED)
            return o.isFieldModified(CONTACT_IMAGE_DATA);
        return o.getFieldContent(CONTACT_IMAGE_DATA) instanceof Image;
    }

    private Map<DataObject, List<DataObject>> createExceptionsMap(DataObjectSet changes) {
        Map<DataObject, List<DataObject>> result = new HashMap<DataObject, List<DataObject>>();
        for (DataObject dataObject : changes) {
            if (isAppException(dataObject)) {
                addExceptionToChangeExceptionsList(dataObject, changes, result);
            }
        }
        if (_SORT_CHANGE_EXCEPTIONS_BY_START_DATE) {
            for (List<DataObject> exceptions : result.values()) {
                Collections.sort(exceptions, new Comparator<DataObject>() {

                    public int compare(DataObject o1, DataObject o2) {
                        Object v1 = o1.getFieldContent(CommonCalendarTasksFieldNames.START_DATE);
                        Object v2 = o2.getFieldContent(CommonCalendarTasksFieldNames.START_DATE);
                        if (v1 instanceof Number) {
                            if (!(v2 instanceof Number))
                                return 1;
                            long diff = ((Number) v1).longValue() - ((Number) v2).longValue();
                            if (diff < 0L)
                                return -1;
                            if (diff > 0L)
                                return 1;
                            return 0;
                        } else if (v2 instanceof Number) {
                            return -1;
                        } else {
                            return 0;
                        }
                    }
                });
            }
        }
        return result;
    }

    private void addExceptionToChangeExceptionsList(DataObject exception, DataObjectSet changes, Map<DataObject, List<DataObject>> result) {
        String recurrence_id = String.valueOf(exception.getFieldContent(RECURRENCE_ID));
        if (changes.contains(recurrence_id)) {
            DataObject dataObject = changes.get(recurrence_id);
            if (dataObject.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE) {
                return; // should not happen
            }
            List<DataObject> exceptionsList = result.get(dataObject);
            if (exceptionsList == null)
                exceptionsList = new ArrayList<DataObject>();
            if (!exceptionsList.contains(exception))
                exceptionsList.add(exception);
            result.put(dataObject, exceptionsList);
        }
    }

    private JSONObject addExtraFieldsToDataObject(DataObject o) {
        ContentType type = o.getContentType();
        BitSet extraFields = getExtraFields(type);
        if (!extraFields.isEmpty()) {
            try {
                if (type instanceof MailContentType)
                    return readStructuredMailWithMeetingRequests(o, (MailContentType) type);
                type.getTransferHandler().readDataObject(o, extraFields);
            } catch (USMException e) {
                if (hasNoError(o))
                    addError(o, new USMJSONAPIException(
                        ConnectorBundleErrorCodes.COMMAND_CANNOT_READ_EXTRA_FIELDS,
                        ResponseStatusCode.INTERNAL_ERROR,
                        "Extra information for new object could not be read from OX server",
                        e), ErrorStatusCode.OTHER);
                else
                    _servlet.getJournal().info(
                        _session + " Couldn't determine extra fields in object with errors, type=" + o.getContentType().getID() + ", id=" + o.getID(),
                        e);
            }
        }
        return null;
    }

    // Used to store all currently available appointments in case more than 1 MeetingRequest is executed in 1 sync
    private DataObjectSet _allAppointments = null;

    private DataObjectSet _extendedAllAppointments = null;

    protected SyncResult _syncResult;

    private DataObjectSet getAllAppointments() throws USMException {
        if (_extendedAllAppointments != null)
            return _extendedAllAppointments;
        if (_allAppointments == null) {
            AppointmentContentType appointmentType = (AppointmentContentType) _servlet.getContentTypeManager().getContentType(
                DefaultContentTypes.CALENDAR_ID);
            _allAppointments = new DataObjectSet(appointmentType.getAllAppointments(_session, _REQUIRED_ALL_APPOINTMENT_FIELDS));
        }
        return _allAppointments;
    }

    private DataObjectSet getExtendedAllAppointments() throws USMException {
        if (_extendedAllAppointments == null) {
            AppointmentContentType appointmentType = (AppointmentContentType) _servlet.getContentTypeManager().getContentType(
                DefaultContentTypes.CALENDAR_ID);
            if (_EXTENDED_ALL_APPOINTMENT_FIELDS.isEmpty()) {
                BitSet set = Toolkit.buildFieldBitSet(
                    appointmentType,
                    UtilConstants.ID,
                    UtilConstants.FOLDER_ID,
                    CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION,
                    RECURRENCE_ID);
                synchronized (_EXTENDED_ALL_APPOINTMENT_FIELDS) {
                    _EXTENDED_ALL_APPOINTMENT_FIELDS.or(set);
                }
            }
            _extendedAllAppointments = new DataObjectSet(appointmentType.getAllAppointments(_session, _EXTENDED_ALL_APPOINTMENT_FIELDS));
        }
        _allAppointments = null;
        return _extendedAllAppointments;
    }

    private JSONObject readStructuredMailWithMeetingRequests(DataObject o, MailContentType contentType) throws USMJSONAPIException {
        JSONObject structuredMail = null;
        try {
            structuredMail = readStructuredMail(o);
            // meeting requests
            JSONObject headers = structuredMail.optJSONObject(CommandConstants.MAIL_HEADERS);
            if (headers != null && o.getChangeState() == ChangeState.CREATED) {
                String module = headers.optString(CommandConstants.MAIL_HEADER_OX_MODULE);
                String type = headers.optString(CommandConstants.MAIL_HEADER_OX_TYPE);
                String id = headers.optString(CommandConstants.MAIL_HEADER_OX_OBJ_ID);
                String uid = headers.optString(CommandConstants.MAIL_HEADER_OX_OBJ_UID);
                String rdate = headers.optString(CommandConstants.MAIL_HEADER_OX_OBJ_RDATE);
                String reminder = headers.optString(CommandConstants.MAIL_HEADER_OX_REMINDER);
                if (CommandConstants.MAIL_HEADER_OX_MODULE_APPOINTMENTS.equals(module) && id != null) {
                    addMeetingRequestOrUpdateFieldToMail(structuredMail, id, type, uid, rdate);
                } else if (CommandConstants.MAIL_HEADER_OX_MODULE_TASKS.equals(module) && id != null && reminder != null) {
                    addTaskRequestOrUpdateFieldToMail(structuredMail, id, reminder, type);
                } else {
                    addExternalMeetingRequestToMail(structuredMail, contentType, o);
                }
            }
        } catch (OXCommunicationException e) {
            // the server returned an error: create one multipart email which describes the error and contains the original mail as
            // attachment
            return getErrorMailStructure(o, e, contentType);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_GET_MAIL_OR_MEETING_REQUEST, e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_ADD_MEETING_REQUEST_TO_MAIL, e);
        }
        return structuredMail;
    }

    private void addExternalMeetingRequestToMail(JSONObject originalMail, MailContentType contentType, DataObject mailObject) throws USMException, JSONException {
        mailObject = _session.readDataObject(mailObject.getParentFolderID(), mailObject.getID(), ATTACHMENTS, HEADERS);
        MailAttachment attachment = MeetingRequestResponseUtil.getICalFromExternalAppInvitationMail(mailObject);
        if (attachment != null) {
            DataObject appDataObject = MeetingRequestResponseUtil.getAppointmentFromExternalInvitation(mailObject, attachment, _session);
            if (appDataObject == null)
                return;
            insertUUIDInMeeting(appDataObject);
            ChangeState originalChangeState = appDataObject.getChangeState();
            appDataObject.commitChanges();
            JSONObject meetingRequestJSON = storeCompleteDataObjectInJSONObject(appDataObject);
            String reccurenceId = String.valueOf(appDataObject.getFieldContent(CommandConstants.RECURRENCE_ID));
            if (isAppException(appDataObject)) {
                addSeriesStartTimeAndTimezone(reccurenceId, meetingRequestJSON);
            }

            if (meetingRequestJSON != null) {
                switch (originalChangeState) {
                case CREATED:
                    originalMail.put(CommandConstants.MEETING_REQUEST, meetingRequestJSON);
                    break;
                case DELETED:
                    originalMail.put(CommandConstants.MEETING_CANCELATION, meetingRequestJSON);
                    break;
                case MODIFIED:
                    originalMail.put(CommandConstants.MEETING_UPDATE, meetingRequestJSON);
                    break;
                case UNMODIFIED:
                    originalMail.put(CommandConstants.MEETING_REPLY, meetingRequestJSON);
                    break;
                }
            }
        }
    }

    private void insertUUIDInMeeting(DataObject appDataObject) throws USMException {
        if (appDataObject.getChangeState() == ChangeState.CREATED) {
            return;
        } else if (appDataObject.getID() != null) {
            insertStoredUUID(appDataObject);
        } else {
            String uid = String.valueOf(appDataObject.getFieldContent(UID));
            String oxAppId = ((AppointmentContentType) appDataObject.getContentType()).resolveUID(_session, uid);
            appDataObject.setID(oxAppId);
            insertStoredUUID(appDataObject);
        }
    }

    private void addMeetingRequestOrUpdateFieldToMail(JSONObject structuredMail, String id, String type, String uid, String rdate) throws DatabaseAccessException, USMSQLException, USMException, JSONException, USMJSONAPIException {
        if (CommandConstants.MAIL_HEADER_OX_TYPE_DELETED.equals(type)) {
            addMeetingCancelationFieldToMail(structuredMail, uid, rdate);
            return;
        }
        String fieldName = "";
        if (CommandConstants.MAIL_HEADER_OX_TYPE_NEW.equals(type)) {
            fieldName = CommandConstants.MEETING_REQUEST;
        } else if (CommandConstants.MAIL_HEADER_OX_TYPE_MODIFIED.equals(type)) {
            fieldName = CommandConstants.MEETING_UPDATE;
        } else {
            return; // mail from different type
        }

        DataObject appointment = getAppointmentFromDefaultCalendarFolderOrAllApp(id);
        if (appointment != null) {
            insertStoredUUID(appointment);
            addServerAttachmentsToDataObject(appointment);
            JSONObject resultObject = storeCompleteDataObjectInJSONObject(appointment);
            String reccurenceId = String.valueOf(appointment.getFieldContent(CommandConstants.RECURRENCE_ID));
            if (isAppException(appointment))
                addSeriesStartTimeAndTimezone(reccurenceId, resultObject);
            structuredMail.put(fieldName, resultObject);
        }
    }

    private void addMeetingCancelationFieldToMail(JSONObject structuredMail, String uid, String rdate) throws JSONException {
        JSONObject result = new JSONObject();
        result.put(UID, uid);
        try {
            if (!Toolkit.isNullOrEmpty(rdate)) {
                result.put(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION, rdate);
                AppointmentContentType appContentType = (AppointmentContentType) _servlet.getContentTypeManager().getContentType(
                    DefaultContentTypes.CALENDAR_ID);
                String seriesID = appContentType.resolveUID(_session, uid);
                if (seriesID != null) {
                    addSeriesStartTimeAndTimezone(seriesID, result);
                }
            }
            structuredMail.put(CommandConstants.MEETING_CANCELATION, result);
        } catch (USMException e) {
            // ignore - do nothing if the uid can not be resolved - the mail should be displayed anyway without meeting_cancel field 
        }
    }

    private void addSeriesStartTimeAndTimezone(String seriesID, JSONObject jo) throws JSONException, USMException {
        DataObject seriesToCopy = getAppointmentFromDefaultCalendarFolderOrAllApp(seriesID);
        if (seriesToCopy == null)
            return;
        DataObject series = seriesToCopy.createCopy(true);
        series.getContentType().getTransferHandler().readDataObject(series, series.getSession().getFieldFilter(series.getContentType()));
        ContentTypeField[] fields = series.getContentType().getFields();
        for (int i = 0; i < fields.length; i++) {
            DataType<?> fieldType = fields[i].getFieldType();
            String fieldName = fields[i].getFieldName();
            if (CommandConstants.TIME_ZONE.equals(fieldName) || CommonCalendarTasksFieldNames.START_DATE.equals(fieldName) || SEQUENCE.equals(fieldName)) {
                Object v = series.getFieldContent(i);
                fieldType.storeInJSONObject(series.getSession(), jo, "series_" + fieldName, v);
            }
        }
    }

    private void addServerAttachmentsToDataObject(DataObject object) throws USMException {
        Object field = object.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
        if (field instanceof PIMAttachments) {
            object.setFieldContent(ATTACHMENTS_LAST_MODIFIED, object.getContentType().getTransferHandler().getAllAttachments(object));
        }
    }

    private DataObject getAppointmentFromDefaultCalendarFolderOrAllApp(String id) throws USMException {
        DataObject o = findAppointmentByOxId(id);
        if (o == null)
            return null;
        try {
            o.getContentType().getTransferHandler().readDataObject(o, _session.getFieldFilter(o.getContentType()));
            o.setChangeState(ChangeState.CREATED);
            return o;
        } catch (USMException e) {
            _servlet.getJournal().warn("Could not read current appointment data for meeting_request (may have been removed)", e);
            return null;
        }
    }

    private DataObject findAppointmentByOxId(String id) throws USMException {
        Folder f = getDefaultCalendarFolder();
        if (f != null) {
            ContentType elementsType = f.getElementsContentType();
            DataObject appointment = DataObjectUtil.findDataObject(id, _session.getCachedFolderElements(f.getID(), elementsType));
            if (appointment != null)
                return appointment;
        }
        DataObjectSet allApp = getCurrentContentOfDefaultCalendarFolder();
        if (allApp.contains(id))
            return allApp.get(id);
        allApp = getAllAppointments();
        DataObject o = allApp.get(id);
        return (o == null) ? null : o.createCopy(false);
    }

    private void addTaskRequestOrUpdateFieldToMail(JSONObject structuredMail, String id, String reminder, String type) throws DatabaseAccessException, USMSQLException, USMException, JSONException, USMJSONAPIException {
        String fieldName = "";
        if (CommandConstants.MAIL_HEADER_OX_TYPE_NEW.equals(type)) {
            fieldName = CommandConstants.TASK_REQUEST;
        } else if (CommandConstants.MAIL_HEADER_OX_TYPE_MODIFIED.equals(type)) {
            fieldName = CommandConstants.TASK_UPDATE;
        } else
            return; // mail from different type
        ContentType tasksContentType = _servlet.getContentTypeManager().getContentType(DefaultContentTypes.TASK_ID);
        String[] reminderValues = reminder.split(",");
        String folderID = reminderValues[1];
        Folder folder = _session.getCachedFolder(folderID);
        if (folder != null) {
            DataObject taskObj = DataObjectUtil.findDataObject(id, tasksContentType.getTransferHandler().readFolderContent(folder));
            if (taskObj != null) {
                insertStoredUUID(taskObj);
                addServerAttachmentsToDataObject(taskObj);
                structuredMail.put(fieldName, storeCompleteDataObjectInJSONObject(taskObj));
            }
        }
    }

    protected DataObjectSet getCurrentContentOfDefaultCalendarFolder() throws USMException {
        if (_currentDefaultCalendarFolderContent == null) {
            Folder f = getDefaultCalendarFolder();
            if (f != null) {
                ContentType elementsType = f.getElementsContentType();
                _currentDefaultCalendarFolderContent = new DataObjectSet(elementsType.getTransferHandler().readFolderContent(f));
            }
        }
        return _currentDefaultCalendarFolderContent;
    }

    protected Folder getDefaultCalendarFolder() throws DatabaseAccessException, USMSQLException {
        if (_defaultCalendarFolder == null)
            _defaultCalendarFolder = determineDefaultCalendarFolder();
        return _defaultCalendarFolder;
    }

    private Folder determineDefaultCalendarFolder() throws DatabaseAccessException, USMSQLException {
        Folder[] folders = _session.getCachedFolders();
        for (Folder f : folders) {
            if (DefaultContentTypes.CALENDAR_ID.equals(f.getElementsContentTypeID())) {
                Object v = f.getFieldContent(FolderConstants.STANDARD_FOLDER);
                if (v instanceof Boolean) {
                    if (((Boolean) v).booleanValue())
                        return f;
                } else {
                    v = f.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
                    if ((v instanceof Number) && ((Number) v).intValue() == FolderConstants.DEFAULT_CALENDAR_FOLDER)
                        return f;
                }
            }
        }
        return null;
    }

    private JSONObject getErrorMailStructure(DataObject o, OXCommunicationException e, MailContentType contentType) {
        JSONObject error = new JSONObject();
        JSONObject header = new JSONObject();
        JSONArray body = new JSONArray();
        JSONObject contentTypeMail = new JSONObject();
        try {
            contentTypeMail.put(TYPE, MULTIPART_MIXED);
            header.put(CONTENT_TYPE, contentTypeMail);
            header.put(SUBJECT, SERVER_INFO_ERROR_ON_READING_EMAIL);
            error.put(HEADERS, header);
            // add first part
            JSONObject errorPart = new JSONObject();
            JSONObject headerErrorPart = new JSONObject();
            JSONObject bodyErrorPart = new JSONObject();
            JSONObject contentTypeErrorPart = new JSONObject();
            contentTypeErrorPart.put(TYPE, TEXT_PLAIN);
            JSONObject paramsObject = new JSONObject();
            paramsObject.put(CHARSET, UTF_8);
            contentTypeErrorPart.put(PARAMS, paramsObject);
            headerErrorPart.put(CONTENT_TYPE, contentTypeErrorPart);
            bodyErrorPart.put(DATA_KEY, e.getMessage() + ": " + e.getJSONError());
            bodyErrorPart.put(ID, "1.1");
            errorPart.put(HEADERS, headerErrorPart);
            errorPart.put(BODY, bodyErrorPart);
            body.put(errorPart);
            try {
                byte[] originalMail = contentType.readMailSource(o, _session.getFieldFilter(contentType));
                // add original email as attachment part
                JSONObject mailPart = new JSONObject();
                JSONObject headerMailPart = new JSONObject();
                JSONObject bodyMailPart = new JSONObject();
                JSONObject contentTypeMailPart = new JSONObject();
                contentTypeMailPart.put(TYPE, APPLICATION_OCTET_STREAM);
                paramsObject = new JSONObject();
                paramsObject.put(NAME, ORIGINAL_MAIL_EML);
                contentTypeMailPart.put(PARAMS, paramsObject);
                headerMailPart.put(CONTENT_TYPE, contentTypeMailPart);
                JSONObject contentDispMailPart = new JSONObject();
                contentDispMailPart.put(TYPE, ATTACHMENT);
                paramsObject = new JSONObject();
                paramsObject.put(FILENAME, ORIGINAL_MAIL_EML);
                contentDispMailPart.put(CommandConstants.PARAMS, paramsObject);
                headerMailPart.put(CONTENT_DISPOSITION, contentDispMailPart);
                headerMailPart.put(CONTENT_TRANSFER_ENCODING, BASE64);
                bodyMailPart.put(DATA_KEY, Toolkit.encodeBase64(originalMail));
                bodyMailPart.put(ID, "1.2");
                mailPart.put(HEADERS, headerMailPart);
                mailPart.put(BODY, bodyMailPart);
                body.put(mailPart);
            } catch (USMException ignored) {
                // can not read email, create just the error part
            }
            error.put(CommandConstants.BODY, body);
        } catch (JSONException ignored) {
            // error on creating error mail; return empty object //XXX: or exception???
        }
        return error;
    }

    @SuppressWarnings("unchecked")
    private void storeExtraFieldsObjectInJSONObject(JSONObject jObj, JSONObject extraFieldsObject) throws JSONException {
        for (Iterator iterator = extraFieldsObject.keys(); iterator.hasNext();) {
            String key = (String) iterator.next();
            jObj.put(key, extraFieldsObject.get(key));
        }
    }

    // TODO Optimize, use extra field map which is initialized once at the beginning of the syncInit/Update for the folder
    // TODO The info if the ContentType has attachments_last_modified_utc can be stored in a boolean field so that it has to be computed
    // only once
    // TODO This whole method can probably be replaced with contentType.supportsPIMAttachments()
    protected boolean hasAttachmentsLastModified(ContentType contentType) {
        if (contentType != null && contentType.supportsPIMAttachments()) {
            ContentTypeField[] fields = contentType.getFields();
            for (int i = 0; i < fields.length; i++) {
                if (ATTACHMENTS_LAST_MODIFIED.equals(fields[i].getFieldName()))
                    return _session.getFieldFilter(contentType).get(i);
            }
        }
        return false;
    }

    // TODO Is it possible to only retrieve current server attachments once, maybe use a map for already retrieved data ?
    protected void handlePIMAttachments(Folder folder, SyncResult result, long newSyncId) throws USMJSONAPIException {
        ContentType elementsContentType = folder.getElementsContentType();
        if (!hasAttachmentsLastModified(elementsContentType))
            return;

        // handle the client changes
        for (DataObject clientObject : _clientChangesArray)
            handlePIMAttClientChanges(newSyncId, folder, elementsContentType, clientObject, null);
        for (Map.Entry<DataObject, UUID> entry : _exceptionChangesForDelayedCall.entrySet())
            handlePIMAttForAppExceptions(newSyncId, folder, elementsContentType, entry.getKey(), entry.getValue());
        for (Map.Entry<DataObject, UUID> entry : _recurrenceUUIDMap.entrySet())
            handlePIMAttForAppExceptions(newSyncId, folder, elementsContentType, entry.getKey(), entry.getValue());

        // handle the server changes
        try {
            handlePIMAttachmentsServerChanges(result);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_INTERNAL_ERROR_ON_CREATE_DELETE_ATTACHMENTS, e);
        }
    }

    private void handlePIMAttForAppExceptions(long newSyncId, Folder folder, ContentType elementsContentType, DataObject exceptionObject, UUID seriesUUID) {
        DataObject o = _newStateToSave.get(exceptionObject.getUUID());
        if (o != null)
            handlePIMAttClientChanges(newSyncId, folder, elementsContentType, o, seriesUUID);
    }

    private void handlePIMAttClientChanges(long newSyncId, Folder folder, ContentType elementsContentType, DataObject clientObject, UUID seriesUUID) {
        if (clientObject.getChangeState() == ChangeState.DELETED || !hasNoError(clientObject))
            return;
        PIMAttachments clientAttachments = _clientAttachmentsMap.get(clientObject.getUUID().toString());
        // if (clientAttachments == null)
        // return;
        try {
            PIMAttachments serverAttachments = elementsContentType.getTransferHandler().getAllAttachments(clientObject);
            PIMAttachments oldServerAttachments = null;
            if (clientObject.getID() != null) {
                DataObject cachedObjectState = getCachedElement(clientObject.getID());
                if (cachedObjectState != null) {
                    oldServerAttachments = (PIMAttachments) cachedObjectState.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                } else if (seriesUUID != null) {
                    if (null != serverAttachments) {
                        oldServerAttachments = new PIMAttachments(serverAttachments.getTimestamp(), serverAttachments.getAttachments());
                    }
                }
            }
            if (oldServerAttachments == null)
                oldServerAttachments = new PIMAttachments(0, EMPTY_PIMATTACHMENT_ARRAY);
            serverAttachments = handlePIMAttClientCreations(oldServerAttachments, serverAttachments, clientObject, clientAttachments);
            serverAttachments = handlePIMAttClientDeletions(
                oldServerAttachments,
                serverAttachments,
                clientObject,
                clientAttachments,
                newSyncId);
            DataObject serverObject = _newStateToSave.get(clientObject.getUUID());
            addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);
        } catch (USMException e) {
            addError(
                clientObject,
                USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_ON_CREATE_DELETE_ATTACHMENTS, e),
                ErrorStatusCode.OTHER);
        }
    }

    private void handlePIMAttachmentsServerChanges(SyncResult result) throws DatabaseAccessException, USMSQLException {
        for (DataObject serverObject : result.getChanges())
            handlePIMAttachmentsServerChanges(serverObject);
        for (DataObject serverObject : _newStateToSave) {
            ContentType contentType = serverObject.getContentType();
            if (serverObject.getChangeState() == ChangeState.UNMODIFIED && contentType.supportsPIMAttachments()) {
                // save the old attachments (if existing) in the _newStateToSave
                DataObject cachedObjectState = getCachedElement(serverObject.getID());
                if (cachedObjectState != null) {
                    PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                    PIMAttachments currentAttachments = (PIMAttachments) serverObject.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                    if (oldSavedAttachments != null && oldSavedAttachments.size() > 0 && (currentAttachments == null || currentAttachments.size() == 0))
                        addTheServerAttachmentsToNewSyncState(serverObject, new PIMAttachments(
                            oldSavedAttachments.getTimestamp(),
                            true,
                            oldSavedAttachments.getAttachments()));
                }
            }
        }
    }

    private void handlePIMAttachmentsServerChanges(DataObject serverObject) {
        ContentType contentType = serverObject.getContentType();
        if (serverObject.getChangeState() == ChangeState.DELETED || !contentType.supportsPIMAttachments())
            return;
        try {
            if (serverObject.getChangeState() == ChangeState.CREATED) {
                PIMAttachments serverAttachments = null;
                Number numOfAtt = (Number) serverObject.getFieldContent(CommandConstants.NUMBER_OF_ATTACHMENTS);
                if (numOfAtt != null && numOfAtt.intValue() > 0)
                    serverAttachments = contentType.getTransferHandler().getAllAttachments(serverObject);
                if (serverAttachments != null)
                    serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
                // save the new attachments in _newStateToSave
                addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);
            } else if (serverObject.isFieldModified(ATTACHMENTS_LAST_MODIFIED)) {
                DataObject cachedObjectState = getCachedElement(serverObject.getID());
                PIMAttachments currentAttachments = (PIMAttachments) serverObject.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                if (cachedObjectState != null) {
                    PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                    if (oldSavedAttachments != null) { // Previous SyncState had attachments -> copy them to current object with new
                                                       // timestamp
                        if (currentAttachments != null)
                            serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, new PIMAttachments(
                                currentAttachments.getTimestamp(),
                                true,
                                oldSavedAttachments.getAttachments()));
                        else {
                            PIMAttachments att = new PIMAttachments(0, true, EMPTY_PIMATTACHMENT_ARRAY);
                            // empty attachments field - means all attachments have been deleted on the server
                            serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, att);
                            addTheServerAttachmentsToNewSyncState(serverObject, att);
                        }
                    }
                }
                PIMAttachments serverAttachments = contentType.getTransferHandler().getAllAttachments(serverObject);
                if (serverAttachments != null)
                    serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
                // save the new attachments in _newStateToSave
                addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);
            } else if (!hasClientSentChange(serverObject.getUUID())) {
                if (isNumOfAttachmentsChanged(serverObject)) {
                    // some attachment has been deleted, but is it not the newest one and lastModifiedOfNewesAttachmentUTC is not changed
                    PIMAttachments serverAttachments = contentType.getTransferHandler().getAllAttachments(serverObject);
                    if (serverAttachments != null)
                        serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
                    // save the new attachments in _newStateToSave
                    addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);

                } else {
                    // save the old attachments (if existing) in the _newStateToSave
                    DataObject cachedObjectState = getCachedElement(serverObject.getID());
                    if (cachedObjectState != null) {
                        PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
                        addTheServerAttachmentsToNewSyncState(serverObject, oldSavedAttachments);
                    }
                }
            }
        } catch (USMException e) {
            _servlet.getJournal().warn(_session + " Error processing server changes on PIM attachment(s) for " + serverObject.getUUID(), e);
            // addError(
            // serverObject,
            // USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_ON_RETRIEVE_ATTACHMENTS_FROM_SERVER, e),
            // ErrorStatusCode.OTHER);
        }
    }

    private boolean hasClientSentChange(UUID uuid) {
        for (DataObject o : _clientChangesArray) {
            if (o.getUUID().equals(uuid))
                return true;
        }
        return false;
    }

    private boolean isNumOfAttachmentsChanged(DataObject serverObject) {
        Number originalValue = (Number) serverObject.getOriginalFieldContent(CommandConstants.NUMBER_OF_ATTACHMENTS);
        Number value = (Number) serverObject.getFieldContent(CommandConstants.NUMBER_OF_ATTACHMENTS);
        if ((originalValue == null || originalValue.intValue() == 0) && (value == null || value.intValue() == 0))
            return false;
        return serverObject.isFieldModified(CommandConstants.NUMBER_OF_ATTACHMENTS);
    }

    private void addTheServerAttachmentsToNewSyncState(DataObject serverObject, PIMAttachments serverAttachments) {
        if (serverAttachments == null || _newStateToSave == null || serverObject == null)
            return;
        DataObject o = _newStateToSave.get(serverObject.getID());
        if (o != null) {
            serverAttachments.setForceModified(true);
            o.setFieldContent(ATTACHMENTS_LAST_MODIFIED, (serverAttachments.size() == 0) ? null : serverAttachments);
            _newStateToSaveChanged = true;
        }
    }

    /**
     * Checks if the server still has some attachments which don't exist on the client any more : delete on server
     * 
     * @param oldServerState
     * @param newServerState
     * @param clientObject
     * @param clientAttachments
     * @throws USMException
     * @return new modified list of attachments stored on server
     */
    private PIMAttachments handlePIMAttClientDeletions(PIMAttachments oldServerAttachments, PIMAttachments serverAttachments, DataObject clientObject, PIMAttachments clientAttachments, long syncId) throws USMException {
        if (serverAttachments == null)
            return serverAttachments;
        for (PIMAttachment serverAttachment : serverAttachments.getAttachments()) {
            UUID uuid = serverAttachment.getUUID();
            if (clientAttachments == null || !clientAttachments.containsAttachment(uuid)) { // (oldServerAttachments == null ||
                                                                                            // oldServerAttachments.containsAttachment(uuid))
                                                                                            // &&
                long timestamp = clientObject.getContentType().getTransferHandler().deleteAttachments(clientObject, serverAttachment);
                serverAttachments = removePIMAttachmentFromNewServerState(clientObject, serverAttachment, timestamp, serverAttachments);
            }
        }
        return serverAttachments;
    }

    /**
     * Checks if the client has sent some attachments which still don't exist on server and create them on the server.
     * 
     * @param newServerAttachmentsState
     * @param clientObject
     * @param clientAttachments
     * @param oldServerState
     * @throws USMException
     * @return new modified list of attachments stored on server
     */
    private PIMAttachments handlePIMAttClientCreations(PIMAttachments serverAttachments, PIMAttachments newServerAttachmentsState, DataObject clientObject, PIMAttachments clientAttachments) throws USMException {
        if (clientAttachments == null)
            return newServerAttachmentsState;
        for (PIMAttachment clientAttachment : clientAttachments.getAttachments()) {
            if (serverAttachments == null || !serverAttachments.containsAttachment(clientAttachment.getUUID())) {
                boolean existsInNewServerState = newServerAttachmentsState != null && newServerAttachmentsState.containsAttachment(clientAttachment.getUUID());
                PIMAttachment identicalAttWithDiffUUID = findIdenticalAttachmentWithDiffUUID(clientAttachment, newServerAttachmentsState);
                if (newServerAttachmentsState != null && identicalAttWithDiffUUID != null) {
                    serverAttachments = addPIMAttachmentToNewServerState(
                        clientObject,
                        new PIMAttachment(identicalAttWithDiffUUID, clientAttachment.getUUID()),
                        newServerAttachmentsState.getTimestamp(),
                        serverAttachments,
                        identicalAttWithDiffUUID.getUUID());
                } else if (!existsInNewServerState) {
                    try {
                        long timestamp = clientObject.getContentType().getTransferHandler().createNewAttachment(clientObject, clientAttachment);
                        serverAttachments = addPIMAttachmentToNewServerState(clientObject, clientAttachment, timestamp, serverAttachments);
                    } catch (PIMAttachmentCountLimitExceededException e) {
                        addClientPIMAttachmentCreationError(clientObject, clientAttachment, PIM_ATTACHMENT_CREATION_DENIED_COUNT, e);
                    } catch (PIMAttachmentSizeLimitExceededException e) {
                        addClientPIMAttachmentCreationError(clientObject, clientAttachment, PIM_ATTACHMENT_CREATION_DENIED_SIZE, e);
                    } catch (USMException e) {
                        addClientPIMAttachmentCreationError(clientObject, clientAttachment, PIM_ATTACHMENT_CREATION_FAILED, e);
                    }
                }
            }
        }
        return serverAttachments;
    }

    protected void addClientPIMAttachmentCreationError(DataObject clientObject, PIMAttachment clientAttachment, ErrorStatusCode errorCode, USMException e) {
        addError(clientObject, new USMJSONAPIException(
            ConnectorBundleErrorCodes.SYNC_PIM_ATTACHMENT_CREATION_FAILED,
            ResponseStatusCode.OX_SERVER_ERROR,
            "PIM attachment creation failed for " + clientAttachment,
            e.getOxErrorForJSONResponse(),
            e.getErrorDetailsForJSONResponse(),
            e), errorCode);
    }

    private PIMAttachment findIdenticalAttachmentWithDiffUUID(PIMAttachment attachment, PIMAttachments attachmentsList) {
        if (attachmentsList != null && attachment != null) {
            for (PIMAttachment a : attachmentsList.getAttachments()) {
                if (attachment.equalsNoUUIDAndServerFields(a))
                    return a;
            }
        }
        return null;
    }

    private PIMAttachments addPIMAttachmentToNewServerState(DataObject clientObject, PIMAttachment clientAttachment, long timestamp, PIMAttachments oldAttachments) throws USMException {
        return addPIMAttachmentToNewServerState(clientObject, clientAttachment, timestamp, oldAttachments, null);
    }

    private PIMAttachments addPIMAttachmentToNewServerState(DataObject clientObject, PIMAttachment clientAttachment, long timestamp, PIMAttachments oldAttachments, UUID uuidToRemove) throws USMException {
        DataObject o = _newStateToSave.get(clientObject.getID());
        if (o == null)
            return oldAttachments;
        List<PIMAttachment> newAttachmentsList = new ArrayList<PIMAttachment>();
        UUID uuid = clientAttachment.getUUID();
        if (oldAttachments != null) {
            for (PIMAttachment a : oldAttachments.getAttachments()) {
                UUID u = a.getUUID();
                if (u != null && !u.equals(uuid) && !u.equals(uuidToRemove))
                    newAttachmentsList.add(a);
            }
        }
        newAttachmentsList.add(clientAttachment);
        return refreshNewStateToSaveWithNewPIMAttachments(timestamp, o, newAttachmentsList);
    }

    private PIMAttachments refreshNewStateToSaveWithNewPIMAttachments(long timestamp, DataObject o, List<PIMAttachment> newAttachmentsList) throws USMException {
        int numberOfAttachments = newAttachmentsList.size();
        PIMAttachments modifiedAttachments = (timestamp == 0 || numberOfAttachments == 0) ? null : new PIMAttachments(
            timestamp,
            true,
            newAttachmentsList.toArray(new PIMAttachment[numberOfAttachments]));
        o.setFieldContent(ATTACHMENTS_LAST_MODIFIED, modifiedAttachments);
        o.setFieldContent(NUMBER_OF_ATTACHMENTS, numberOfAttachments);
        _newStateToSaveChanged = true;
        return modifiedAttachments;
    }

    private PIMAttachments removePIMAttachmentFromNewServerState(DataObject objectOnServer, PIMAttachment clientAttachment, long timestamp, PIMAttachments serverAttachments) throws USMException {
        DataObject o = _newStateToSave.get(objectOnServer.getID());
        UUID uuid = clientAttachment.getUUID();
        if (o == null || uuid == null)
            return serverAttachments;
        List<PIMAttachment> newAttList = new ArrayList<PIMAttachment>();
        for (PIMAttachment attachment : serverAttachments.getAttachments()) {
            if (!uuid.equals(attachment.getUUID()))
                newAttList.add(attachment);
        }
        return refreshNewStateToSaveWithNewPIMAttachments(timestamp, o, newAttList);
    }

    protected USMJSONAPIException generateConflictException(SynchronizationConflictException e) {
        try {
            JSONObject details = new JSONObject();
            JSONArray conflicts = new JSONArray();
            for (ConflictingChange conflict : e.getConflictingChanges()) {
                JSONObject data = new JSONObject();
                DataObject clientObject = conflict.getClientChange();
                DataObject serverObject = conflict.getServerChange();
                data.put(UUID_KEY, clientObject.getUUID().toString());
                data.put(CLIENT, getDataObjectDifferences(clientObject, serverObject));
                data.put(SERVER, getDataObjectDifferences(serverObject, clientObject));
                conflicts.put(data);
            }
            details.put(CONFLICTS, conflicts);
            return new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_SYNC_CONFLICT,
                ResponseStatusCode.SYNC_FAILED,
                e.getMessage(),
                details);
        } catch (USMJSONAPIException e1) {
            return e1;
        } catch (JSONException e1) {
            return USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_JSON_ERROR, e1);
        }
    }

    private Object getDataObjectDifferences(DataObject object, DataObject other) throws JSONException, USMJSONAPIException {
        if (object.getChangeState() == ChangeState.DELETED)
            return DELETED;
        JSONObject diff = storeDataObjectFieldsInJSONObject(DataObjectUtil.copyAndModify(other, object, true), null);
        if (object.getUUID() != null && other.getUUID() != null && !object.getUUID().equals(other.getUUID()))
            diff.put(UUID_KEY, object.getUUID().toString());
        addOptionalFolderUUID(object, diff);
        return diff;
    }

    protected DataObjectSet getCachedElements() throws DatabaseAccessException, USMSQLException {
        if (_cachedElements == null) {
            _cachedElements = new DataObjectSet();
            initializeCachedElements();
        }
        return _cachedElements;
    }

    /**
     * Retrieves a DataObject as stored in the SyncState that was referenced by the client (for syncUpdate), or null
     * 
     * @param id
     * @return
     * @throws DatabaseAccessException
     * @throws USMSQLException
     */
    protected DataObject getCachedElement(String id) throws DatabaseAccessException, USMSQLException {
        return getCachedElements().get(id);
    }

    /**
     * Retrieves a DataObject as stored in the SyncState that was referenced by the client (for syncUpdate), or null
     * 
     * @param uuid
     * @return
     * @throws DatabaseAccessException
     * @throws USMSQLException
     */
    protected DataObject getCachedElement(UUID uuid) throws DatabaseAccessException, USMSQLException {
        return getCachedElements().get(uuid);
    }

    protected void initializeCachedElements() throws DatabaseAccessException, USMSQLException {
        // May be overwritten by sub-classes
    }

    protected DataObjectFilter getFilter(Folder folder) {
        if (_dataObjectFilter == null)
            _dataObjectFilter = createFilter(folder);
        return _dataObjectFilter;
    }

    private DataObjectFilter createFilter(Folder folder) {
        if (DefaultContentTypes.MAIL_ID.equals(folder.getElementsContentTypeID()))
            return new EmailSizeFilter(getOriginalSyncID(), _servlet.getEmailSizeLimit(), _servlet.getMaxEmailSize());
        if (DefaultContentTypes.CALENDAR_ID.equals(folder.getElementsContentTypeID()) && !FolderType.SHARED.matches(folder.getFieldContent(FolderConstants.TYPE)))
            return new AppointmentFilter(getOriginalSyncID(), _servlet.getPimAttachmentCountLimit());
        return new PIMAtachmentsCountFilter(getOriginalSyncID(), _servlet.getPimAttachmentCountLimit());
    }

    protected void initializeErrorMap(SyncResult result) {
        for (Map.Entry<DataObject, USMException> entry : result.getErrors().entrySet()) {
            DataObject object = entry.getKey();
            USMException error = entry.getValue();
            if ((error instanceof OperationDeniedException) && object.getChangeState() == ChangeState.DELETED) {
                removeFromNewSyncState(object, result);
                addError(object, error, ErrorStatusCode.NO_DELETE_PERMISSION);
            } else {
                addError(object, error);
            }
        }
    }

    private void removeFromNewSyncState(DataObject object, SyncResult result) {
        if (_newStateToSave == null)
            _newStateToSave = new DataObjectSet(result.getNewState());
        _newStateToSave.remove(object.getUUID());
        _extraSyncRequired = true;
        _session.invalidateCachedData(_folderToSync == null ? null : _folderToSync.getID());
    }

    protected void readOptionalLimit() throws USMJSONAPIException {
        int limit = retrieveOptionalLimit();
        int maxLimit = _servlet.getMaxLimit();
        _limit = (maxLimit > 0 && (limit == Session.NO_LIMIT || limit > maxLimit)) ? maxLimit : limit;
    }

    private int retrieveOptionalLimit() throws USMJSONAPIException {
        try {
            if (!_parameters.has(LIMIT))
                return Session.NO_LIMIT;
            int limit = _parameters.getInt(LIMIT);
            if (limit > 0)
                return limit;
            if (limit == 0)
                return -1;
        } catch (JSONException ignored) {
            // fall through
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_BAD_LIMIT_PARAMETER,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Bad value for parameter " + LIMIT);
    }

    protected void setFolderToSync(String folderID) throws USMJSONAPIException {
        _folderToSync = getFolderByUUID(folderID);
        _folderOwnerID = _folderToSync.getOwnerID();
    }

   

    protected void insertStoredUUID(DataObject dataObject) throws DatabaseAccessException, USMSQLException {
        _session.insertStoredUUID(dataObject);
    }

   

    protected ResponseStatusCode getSuccessResponseCode() {
        if (_dataObjectFilter instanceof EmailSizeFilter) {
            if (((EmailSizeFilter) _dataObjectFilter).hadTooBigEmails() && USMJSONVersion.supportsFilteredOutLargeObjectsStatusCode(_session))
                return ResponseStatusCode.SUCCESS_FILTERED_OUT_LARGE_OBJECTS;
        }
        return ResponseStatusCode.SUCCESS;
    }

    protected DataObject[] filterDataObjectsWithErrors(DataObject[] objects) {
        Map<UUID, DataObject> map = new HashMap<UUID, DataObject>();
        for (DataObject o : objects) {
            checkForMultipleOperations(map, o);
        }
        return map.values().toArray(new DataObject[map.size()]);
    }

    protected Folder[] filterFoldersWithErrors(Folder[] folders) {
        Map<UUID, Folder> map = new HashMap<UUID, Folder>();
        for (Folder f : folders) {
            checkForMultipleOperations(map, f);
        }
        return map.values().toArray(new Folder[map.size()]);
    }

    protected <K extends DataObject> boolean checkForMultipleOperations(Map<UUID, K> map, K o) {
        UUID uuid = o.getUUID();
        String s = uuid.toString();
        SyncErrorData data = _errorMap.get(s);
        if (data == null) {
            if (!map.containsKey(uuid)) {
                map.put(uuid, o);
                return true;
            }
            map.remove(uuid);
            addError(s, o, new MultipleOperationsOnDataObjectException(o), ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
        } else if (data._errorStatus == ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID) {
            data.initObject(o);
        }
        return false;
    }

    protected JSONObject readStructuredMail(DataObject mail) throws USMException {
        ContentType contentType = mail.getContentType();
        if (!(contentType instanceof MailContentType))
            return new JSONObject();
        boolean supportsAttachmentStreaming = USMJSONVersion.supportsAttachmentStreaming(_session);
        int maxAttachmentSize = supportsAttachmentStreaming ? _servlet.getMaxInlineAttachmentSize() : -1;
        JSONObject structuredMail = ((MailContentType) contentType).readStructuredMail(mail, maxAttachmentSize);
        MailAttachmentStreamingUtil.replaceEmptyDataWithTempId(mail, structuredMail, supportsAttachmentStreaming);
        return structuredMail;
    }

    @Override
    public void disposeResources() {
        if (_session != null)
            TempFileStorage.deleteTempFiles(_session);
    }

    protected abstract long getOriginalSyncID();
}
