/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.CONTACT_IMAGE_DATA;
import static com.openexchange.usm.connector.commands.CommandConstants.LENGTH;
import static com.openexchange.usm.connector.commands.CommandConstants.MORE_AVAILABLE;
import static com.openexchange.usm.connector.commands.CommandConstants.OFFSET;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.TEMPID;
import java.io.IOException;
import java.io.Writer;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.ContactContentType;
import com.openexchange.usm.api.contenttypes.MailContentType;
import com.openexchange.usm.api.contenttypes.ResourceInputStream;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.ServletConstants;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.json.streaming.ServerTempId;
import com.openexchange.usm.json.streaming.ServerTempIdType;
import com.openexchange.usm.util.Toolkit;

/**
 * {@link DownloadDataHandler} Sends Mail and PIM attachments to the client. Directly streams the data provided by the request to the OX
 * server to the HttpResponse of the client request
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class DownloadDataHandler extends NormalCommandHandler {

    private static final int _BUFFER_SIZE = 16383; // multiple of 3 for correct Base64-encoding of smaller parts

    private static final int _FIXED_RESPONSE_OVERHEAD = 32; // JSON-structure, "data" & "status"-fields, status-value as 2 bytes

    private static final String _FIXED_HEADER = "{\"data\":{";

    private static final String _MORE_AVAILABLE_JSON_DATA = "\"moreAvailable\":\"true\",";

    private static final String _DATA_PREFIX = "\"data\":\"";

    private static final String _DATA_SUFFIX = "\"},\"status\":";

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, TEMPID };

    private static final String[] OPTIONAL_PARAMETERS = { OFFSET, LENGTH };

    private final HttpServletResponse _response;

    private final long _offset;

    private final long _requestedLength;

    private long _realLength;

    private boolean _moreAvailable;

    private ServerTempId _id;

    private DataObject _object;

    private ResponseStatusCode _responseStatus = ResponseStatusCode.SUCCESS;

    public DownloadDataHandler(USMJSONServlet servlet, HttpServletRequest request, HttpServletResponse response) throws USMJSONAPIException {
        super(servlet, request);
        USMJSONVersion.verifyProtocolVersion(_session, USMJSONVersion.SUPPORTS_ATTACHMENT_STREAMING);
        _response = response;
        _offset = getLongParameter(OFFSET, 0L);
        _requestedLength = getLongParameter(LENGTH, Long.MAX_VALUE);
        if (_offset < 0L)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_INVALID_OFFSET,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid offset " + _offset);
        if (_requestedLength < 0L)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_INVALID_LENGTH,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid length " + _requestedLength);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        _id = ServerTempId.fromString(getStringParameter(TEMPID));
        initializeDataObjectForDownload();
        if (_id.getType() == ServerTempIdType.IMAGE)
            return handleRequestForContactImage();
        ResourceInputStream is = getResourceInputStream();
        try {
            Writer w = null;
            try {
                if (_id.getType() == ServerTempIdType.MAIL)
                    skipOffset(is);
                _response.setContentType(ServletConstants.USM_JSON_CONTENT_TYPE);
                _response.setContentLength(computeContentLength());
                w = _response.getWriter();
                writeResponse(is, w);
                return null;
            } catch (IOException e) {
                throw new USMJSONAPIException(
                    ConnectorBundleErrorCodes.DOWNLOAD_DATA_CAN_NOT_CREATE_RESPONSE_WRITER,
                    ResponseStatusCode.INTERNAL_ERROR,
                    "Can't write response",
                    e);
            } finally {
                Toolkit.close(w);
            }
        } finally {
            Toolkit.close(is);
        }
    }

    private ResponseObject handleRequestForContactImage() throws USMJSONAPIException {
        _object.setFieldContent(CONTACT_IMAGE_DATA, new Image());
        try {
            byte[] content = ((ContactContentType) _object.getContentType()).getPictureData(_object, "jpeg", Integer.MAX_VALUE);
            _realLength = Math.max(0L, Math.min(_requestedLength, content.length - _offset));
            _moreAvailable = _offset + _realLength < content.length;
            if (content.length != _realLength) {
                byte[] shortened = new byte[(int) _realLength];
                if (_realLength > 0)
                    System.arraycopy(content, (int) _offset, shortened, 0, (int) _realLength);
                content = shortened;
            }
            JSONObject data = new JSONObject();
            data.put("data", Toolkit.encodeBase64(content));
            if (_moreAvailable)
                data.put(MORE_AVAILABLE, true);
            return new ResponseObject(_responseStatus, data);
        } catch (USMException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_IMAGE_NOT_FOUND,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Image not found for '" + _id + '\'',
                e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.DOWNLOAD_DATA_JSON_ERROR, e);
        }
    }

    private void skipOffset(ResourceInputStream is) throws IOException {
        for (long remaining = _offset; remaining > 0; remaining -= is.skip(remaining))
            ;
    }

    private void writeResponse(ResourceInputStream is, Writer w) throws IOException {
        w.write(_FIXED_HEADER);
        if (_moreAvailable)
            w.write(_MORE_AVAILABLE_JSON_DATA);
        w.write(_DATA_PREFIX);
        writeContent(is, w);
        w.write(_DATA_SUFFIX);
        if (_responseStatus.getStatusCode() >= 10)
            w.write('0' + _responseStatus.getStatusCode() / 10);
        else
            w.write(' ');
        w.write('0' + _responseStatus.getStatusCode() % 10);
        w.write('}');
    }

    private void writeContent(ResourceInputStream is, Writer w) throws IOException {
        long remaining = _realLength;
        if (remaining >= _BUFFER_SIZE) {
            byte[] buffer = new byte[_BUFFER_SIZE];
            for (; remaining >= _BUFFER_SIZE; remaining -= _BUFFER_SIZE)
                writeContentBlock(is, w, buffer);
        }
        if (remaining > 0)
            writeContentBlock(is, w, new byte[(int) remaining]);
    }

    private void writeContentBlock(ResourceInputStream is, Writer w, byte[] buffer) throws IOException {
        int readPos = 0;
        while (readPos < buffer.length && _responseStatus == ResponseStatusCode.SUCCESS) {
            try {
                int read = is.read(buffer, readPos, buffer.length - readPos);
                if (read < 0)
                    _responseStatus = ResponseStatusCode.OX_SERVER_ERROR; // TODO use new status code for an unexpected short read ?
                else
                    readPos += read;
            } catch (IOException e) {
                _responseStatus = ResponseStatusCode.OX_SERVER_ERROR; // TODO use new status code for an unexpected short read ?
            }
        }
        while (readPos < buffer.length)
            buffer[readPos++] = 0;
        w.write(Toolkit.encodeBase64(buffer));
    }

    private int computeContentLength() {
        int length = _FIXED_RESPONSE_OVERHEAD;
        if (_moreAvailable)
            length += _MORE_AVAILABLE_JSON_DATA.length(); // No special characters, so number of bytes equals number of characters in UTF-8
        length += ((_realLength + 2) / 3) * 4;
        return length;
    }

    private Folder getFolderOfDataObject() throws USMJSONAPIException {
        try {
            return _session.getCachedFolder(_id.getFolderId());
        } catch (DatabaseAccessException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_DB_ERROR_FOR_FOLDER,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "DB error while accessing folder for '" + _id + '\'',
                e);
        } catch (USMSQLException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_SQL_ERROR_FOR_FOLDER,
                ResponseStatusCode.INTERNAL_ERROR,
                "SQL error while accessing folder for '" + _id + '\'',
                e);
        }
    }

    private void initializeDataObjectForDownload() throws USMJSONAPIException {
        Folder folder = getFolderOfDataObject();
        if (folder == null)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_FOLDER_NOT_FOUND,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Folder not found for '" + _id + '\'');
        _object = folder.getElementsContentType().newDataObject(_session);
        _object.setParentFolder(folder);
        _object.setID(_id.getObjectId());
        checkCorrectContentType();
    }

    private ResourceInputStream getResourceInputStream() throws USMJSONAPIException {
        try {
            _realLength = Math.max(0L, Math.min(_requestedLength, _id.getSize() - _offset));
            _moreAvailable = _offset + _realLength < _id.getSize();
            if (_id.getType() == ServerTempIdType.PIM) {
                int attachmentId = Integer.parseInt(_id.getAttachmentId());
                return _object.getContentType().getTransferHandler().getAttachmentDataStream(_object, attachmentId, _offset, _realLength);
            }
            MailContentType mailContentType = (MailContentType) _object.getContentType();
            return mailContentType.getMailAttachmentAsStream(_session, _id.getFolderId(), _id.getObjectId(), _id.getAttachmentId());
        } catch (NumberFormatException ignored) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.DOWNLOAD_DATA_INVALID_ATTACHMENT_ID,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid attachment id for '" + _id + '\'');
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.DOWNLOAD_DATA_ERROR_ACCESSING_PIM_ATTACHMENT, e);
        }
    }

    private void checkCorrectContentType() throws USMJSONAPIException {
        switch (_id.getType()) {
        case MAIL:
            if (_object.getContentType() instanceof MailContentType)
                return;
            break;
        case IMAGE:
            if (_object.getContentType() instanceof ContactContentType)
                return;
            break;
        case PIM:
            if (!(_object.getContentType() instanceof MailContentType))
                return;
            break;
        default:
            break;
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.DOWNLOAD_DATA_INVALID_OBJECT_TYPE,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Object associated with '" + _id + "' is of wrong ContentType " + _object.getContentType().getID());
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.connector.commands.BaseCommandHandler#getRequiredParameters()
     */
    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.connector.commands.BaseCommandHandler#getOptionalParameters()
     */
    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }
}
