/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.session.Session;

/**
 * Interface that specifies the methods a DataType has to provide.
 * 
 * @author afe
 * @param <T>
 */
public interface DataType<T extends Serializable> {

    /**
     * @return Unique name for this DataType
     */
    String getName();

    /**
     * Generate an Object of type T from the content of the specified field in the provided JSONObject. The provided Session object can be
     * used to perform necessary adjustments/conversions of the value for the communication with the OX server (e.g. converting time into
     * the timezone of the user).
     * 
     * @param session
     * @param jsonObject
     * @return
     * @throws JSONException
     */
    T extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException;

    /**
     * Generate an Object of type T from the content at the specified index in the provided JSONArray The provided Session object can be
     * used to perform necessary adjustments/conversions of the value for the communication with the OX server (e.g. converting time into
     * the timezone of the user).
     * 
     * @param session
     * @param jsonObject
     * @return
     * @throws JSONException
     */
    T extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException;

    /**
     * Store an Object of type T in the specified field in the provided JSONObject. It must be possible to restore the Object using
     * extractFromJSONObject(), i.e. the JSON format used in both methods must be compatible. Since outer objects may not know about the
     * exact java type of the value, the parameter is given as Object, but the DataType can assume that it is of the correct type. The
     * provided Session object can be used to perform necessary adjustments/conversions of the value for the communication with the OX
     * server (e.g. converting time into the timezone of the user).
     * 
     * @param session
     * @param parent
     * @param field
     * @param value
     * @throws JSONException
     */
    void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException;

    /**
     * Add an Object of type T to the provided JSONArray. It must be possible to restore the Object using extractFromJSONArray(), i.e. the
     * JSON format used in both methods must be compatible. Since outer objects may not know about the exact java type of the value, the
     * parameter is given as Object, but the DataType can assume that it is of the correct type. The provided Session object can be used to
     * perform necessary adjustments/conversions of the value for the communication with the OX server (e.g. converting time into the
     * timezone of the user).
     * 
     * @param session
     * @param parent
     * @param value
     * @throws JSONException
     */
    void addToJSONArray(Session session, JSONArray parent, Object value) throws JSONException;

    /**
     * Create a new / empty Object of type T, may also return null if the other methods of this DataType handle that value correctly.
     * 
     * @return
     */
    T createNewObject();

    /**
     * Create a copy of the provided Object. It is valid to return the same object if the Object is immutable. Since outer objects may not
     * know about the exact java type of the value, the parameter is given as Object, but the DataType can assume that it is of the correct
     * type.
     * 
     * @param value
     * @return
     */
    T createCopy(Object value);

    /**
     * Create a serialized version of the provided Object. It is valid to return the same object if the Object is immutable and contains no
     * transient data. Since outer objects may not know about the exact java type of the value, the parameter is given as Object, but the
     * DataType can assume that it is of the correct type.
     * 
     * @param value
     * @return
     */
    Serializable serialize(Object value);

    /**
     * Deserialize a value from a previously serialized (using serialized(Object)) version of the object.
     * 
     * @param value
     * @return
     */
    T deserialize(Serializable value);

    /**
     * Check that the provided Object is of the correct type. This method may also return another Object of type T that correctly represents
     * the given Object.
     * 
     * @param o
     * @return
     * @throws IllegalArgumentException if the Object is not of the correct type (or cannot be converted to that type if this DataType
     *             performs automatic conversion)
     */
    T checkValue(Object o);

    /**
     * Compares two value for equality. Since this sometimes can not be performed by the objects themselves (e.g. if the are plain java
     * arrays), it must be performed by the DataType itself. All checks for equality on field content must use the method of the DataType of
     * the field. The DataType can assume that the provided objects are of the correct java type.
     * 
     * @param object
     * @param object2
     * @return true if both values should be considered equal, false if not
     */
    boolean isEqual(Object object, Object object2);

    /**
     * Performs basically the same as isEqual(object, createNewObject()), i.e. returns true if the given value is the same as the default
     * (new) value of an object of this DataType.
     * 
     * @param object
     * @return
     */
    boolean isDefaultValue(Object object);
}
