/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.CREATED;
import static com.openexchange.usm.connector.commands.CommandConstants.DELETED;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.MODIFIED;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SlowSyncRequiredException;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * Handler for the SyncEstimate USM-JSON-Command.
 * @author afe
 *
 */
public class SyncEstimateHandler extends NormalCommandHandler {
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDERID, SYNCID };

    public SyncEstimateHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		String folderID = getStringParameter(FOLDERID, null);
		SyncResult result = computeSyncEstimate(folderID);
		int created = 0;
		int modified = 0;
		int deleted = 0;
		for (DataObject d : result.getChanges()) {
			switch (d.getChangeState()) {
				case CREATED:
					created++;
					break;
				case DELETED:
					deleted++;
					break;
				case MODIFIED:
					modified++;
					break;
				case UNMODIFIED:
					// Should never happen
					break;
			}
		}
		try {
			JSONObject resultData = new JSONObject();
			resultData.put(CREATED, created);
			resultData.put(MODIFIED, modified);
			resultData.put(DELETED, deleted);
			return new ResponseObject(ResponseStatusCode.SUCCESS, resultData);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.SYNC_ESTIMATE_JSON_ERROR, e);
		}
	}

	private SyncResult computeSyncEstimate(String folderID) throws USMJSONAPIException {
		try {
			if (folderID != null) {
				folderID = getFolderByUUID(folderID).getID();
				if (hasSyncID()) {
					// Compute estimate for incremental sync of a folder
					return _session.syncChangesWithServer(folderID, getSyncID(), Session.NO_LIMIT, null, false, null);
				}
				// Compute estimate for slow sync of a folder
				return _session.syncWithServer(folderID, Session.NO_LIMIT, null, false, null);
			}
			if (hasSyncID()) {
				// Compute estimate for incremental sync of folder hierarchy
				return _session.syncFolderChangesWithServer(getSyncID(), Session.NO_LIMIT, null, false, null);
			}
			// Compute estimate for slow sync of folder hierarchy
			return _session.syncFoldersWithServer(Session.NO_LIMIT, null, false, null);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (SlowSyncRequiredException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SYNC_ESTIMATE_INVALID_SYNCID,
					ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_ESTIMATE_INTERNAL_ERROR, e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}
}
