/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.sync;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;
import com.openexchange.usm.session.impl.SessionManagerImpl;

public class ContentSyncerSupport {
	private static final String TITLE = "title";

    private static final String ID = "id";

	private static final boolean READ_SERVER_CHANGES_ON_CREATION = true;

	private static final boolean READ_SERVER_CHANGES_ON_MODIFICATION = true;

	private static interface UpdateHandler {
		void update(DataObject update, DataObject serverObject, boolean isFolderUpdate) throws USMException;
	}

	public static SyncResult storeDataObjectsAndBuildClientResult(boolean incomplete,
			ContentSyncerStorage serverInterface, DataObjectSet serverDataObjects, List<DataObject> clientResults,
			Map<DataObject, USMException> errorMap, DataObject... extraDataObjects) throws DatabaseAccessException,
			USMSQLException {
		long mt = computeTimestamp(serverDataObjects);
		createMissingUUIDs(serverDataObjects);
		mt = serverInterface.storeServerData(mt, serverDataObjects, incomplete);
		setTimestampAndCommitChanges(mt, serverDataObjects);
		DataObject[] dataToReturn = serverDataObjects.toArray();
		for (DataObject f : clientResults)
			f.setTimestamp(mt);
		for (DataObject f : extraDataObjects)
			f.setTimestamp(mt);
		return new SyncResult(incomplete, mt, dataToReturn, errorMap, DataObjectUtil.toArray(clientResults));
	}

	private static void createMissingUUIDs(Collection<DataObject> serverDataObjects) throws DatabaseAccessException,
			USMSQLException {
		for (DataObject o : serverDataObjects) {
			if (o.getUUID() == null) {
				throw new IllegalStateException("DataObject without UUID");
			}
		}
	}

	public static long updateTimestampAndCommitChanges(Collection<DataObject> serverDataObjects)
			throws DatabaseAccessException, USMSQLException {
		long timestamp = computeTimestamp(serverDataObjects);
		createMissingUUIDs(serverDataObjects);
		setTimestampAndCommitChanges(timestamp, serverDataObjects);
		return timestamp;
	}

	public static void setTimestampAndCommitChanges(long t, Collection<DataObject> serverDataObjects) {
		for (DataObject f : serverDataObjects) {
			f.setTimestamp(t);
			f.commitChanges();
		}
	}

	private static long computeTimestamp(Collection<DataObject> serverDataObjects) {
		long mt = 0;
		for (DataObject f : serverDataObjects)
			mt = Math.max(mt, f.getTimestamp());
		// If we have no objects and therefore no timestamp, use current system time as timestamp
		if (mt == 0)
			mt = System.currentTimeMillis();
		return mt;
	}

	public static void executeServerUpdates(SessionManagerImpl sessionManager, DataObjectSet serverObjects,
			List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions,
			Map<DataObject, USMException> resultMap) {
		try {
			executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap,
					false, null);
		} catch (ConflictingChangeException e) {
			throw new IllegalStateException("Unexpected exception on update objects", e);
		}
	}

	public static void updateOnServer(SessionManagerImpl sessionManager, DataObjectSet serverObjects,
			List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions,
			Map<DataObject, USMException> resultMap, List<DataObject> serverChanges) throws ConflictingChangeException {
		executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap, true,
				serverChanges);
	}
	
	private static String getLatestMappedID(String originalID, Map<String,String> folderIdMapping) {
            String newMappedID = folderIdMapping.get(originalID);
            if(!folderIdMapping.containsKey(newMappedID))
                return newMappedID;
            
            return  getLatestMappedID(newMappedID, folderIdMapping);
	}

	private static void executeServerUpdatesInternal(final SessionManagerImpl sessionManager,
			final DataObjectSet serverObjects, List<DataObject> creations, List<DataObject> changes,
			List<DataObject> deletions, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions,
			final List<DataObject> serverChanges) throws ConflictingChangeException {
		final Map<String, String> folderIdMapping = new HashMap<String, String>();
		// perform changes on already existing objects
		performUpdates(serverObjects, changes, resultMap, throwConflictExceptions, new UpdateHandler() {
			public void update(DataObject change, DataObject serverObject, boolean isFolder) throws USMException {
				if (folderIdMapping.containsKey(change.getOriginalParentFolderID())
						|| folderIdMapping.containsKey(change.getOriginalID())) {
					DataObject serverObjectCopy = serverObject.createCopy(true);
					DataObjectUtil.copyModifiedFields(serverObjectCopy, change);
					if (folderIdMapping.containsKey(serverObjectCopy.getParentFolderID()))
						serverObjectCopy.setParentFolderID(getLatestMappedID(serverObjectCopy.getParentFolderID(), folderIdMapping)); 
					if (folderIdMapping.containsKey(serverObjectCopy.getID()))
						serverObjectCopy.setID(getLatestMappedID(serverObjectCopy.getID(), folderIdMapping)); 
					
					serverObjectCopy.getContentType().getTransferHandler().writeUpdatedDataObject(serverObjectCopy,
							serverObjectCopy.getTimestamp());
					if (READ_SERVER_CHANGES_ON_MODIFICATION && isFolder) {
					    readServerChanges(sessionManager, serverChanges, serverObjectCopy, (Folder) serverObjectCopy, serverObjectCopy.getContentType());
	                }
					 DataObjectUtil.copyModifiedFields(change, serverObjectCopy);
					 DataObjectUtil.copyModifiedFields(serverObject, serverObjectCopy);
                     serverObject.commitChanges();
				} else {
					change.getContentType().getTransferHandler().writeUpdatedDataObject(change, change.getTimestamp());
					if (READ_SERVER_CHANGES_ON_MODIFICATION && isFolder) {
	                    readServerChanges(sessionManager, serverChanges, change, (Folder) change, change.getContentType());
	                }
					if (serverObject != null) {
					    DataObjectUtil.copyModifiedFields(serverObject, change);
					    serverObject.commitChanges();
					}
				}
				if (isFolder && serverObject != null && change.isFieldModified(change.getFieldIndex(ID))) {
					//this means that we have renamed/moved a mail folder and the id is changed and we have to remap all cached data to the new id
				    String originalID = change.getOriginalID();
				    String newID = serverObject.getID();
				    
				    String newOriginalID = getLatestMappedID(originalID, folderIdMapping); //if the parent folder has also been renamed in the same update and we already have a new id mapped
				    if(newOriginalID != null) 
				        originalID = newOriginalID;
				    folderIdMapping.put(originalID, newID); //old to new
				    
				   
					Session session = serverObject.getSession();
					session.remapCachedData(session.getShortFolderID(originalID), session
							.getShortFolderID(newID));
					for (DataObject childFolder : serverObjects) {
					    String childFolderID = childFolder.getID();
						if (childFolderID.startsWith(originalID + "/")) {
                            // folder is child (or grand-child) of modified folder
                            // beginIndex is position where sub-path of folder (child) begins (after /)
                            int beginIndex = originalID.length() + 1;
                            String childSubPath = childFolderID.substring(beginIndex);
							String newParentFolderID = newID;
							int childParentDelimiterIndex = childSubPath.lastIndexOf('/');
							if(childParentDelimiterIndex >= 0) {
	                            // folder is grand-child of modified folder, add intermediate folder structure
							    newParentFolderID += "/" + childSubPath.substring(0, childParentDelimiterIndex);
							}
							String newChildID = newID + "/" + childSubPath;
							folderIdMapping.put(childFolderID, newChildID); //old to new
							childFolder.setParentFolderID(newParentFolderID);
							childFolder.setID(newChildID);
							session = childFolder.getSession();
							session.remapCachedData(session.getShortFolderID(childFolderID), session
									.getShortFolderID(newChildID));
							childFolder.commitChanges();
						}
					}
				}
				
			}
		});
		// perform deletions of already existing objects
		performUpdates(serverObjects, deletions, resultMap, throwConflictExceptions, new UpdateHandler() {
			public void update(DataObject deletion, DataObject serverObject, boolean isFolder) throws USMException {
				boolean executeDeletion = true;
				if (isFolder && serverObject != null) {
					Folder f = serverObject.getParentFolder();
					executeDeletion = f == null || serverObjects.contains(f);
				}
				if (executeDeletion)
					deletion.getContentType().getTransferHandler().writeDeletedDataObject(deletion);
				if (serverObject != null)
					serverObjects.remove(serverObject);
			}
		});
		// perform creations of new objects
		performUpdates(null, creations, resultMap, throwConflictExceptions, new UpdateHandler() {
			private DataObjectSet _newestCachedData = null;

			public void update(DataObject creation, DataObject serverObject, boolean isFolder) throws USMException {
				Folder f = creation.getParentFolder();
				if (f != null)
					creation.setParentFolder(f);
				ContentType contentType = creation.getContentType();
				contentType.getTransferHandler().writeNewDataObject(creation);
				DataObject creationCopy = creation.createCopy(true);
				if (READ_SERVER_CHANGES_ON_CREATION) {
					readServerChanges(sessionManager, serverChanges, creation, f, contentType);
				}
				serverObjects.add(creationCopy);
				if (creation.getUUID() == null) {
					_newestCachedData = sessionManager.insertStoredUUID(creation, null, _newestCachedData);
					if (creation.getUUID() == null)
						creation.setUUID(UUID.randomUUID());
				} else {
					sessionManager.storeUUID(creation);
				}
			}
		});
	}

	private static void readServerChanges(final SessionManagerImpl sessionManager,
			final List<DataObject> serverChanges, DataObject creation, Folder f, ContentType contentType) {
		if (mayHaveRightsToReadElements(f)) {
			try {
				DataObject creationCopy2 = creation.createCopy(true);
				creationCopy2.commitChanges();
				contentType.getTransferHandler().readDataObject(creationCopy2,
						creationCopy2.getSession().getFieldFilter(contentType));
				if (creationCopy2.isModified() && serverChanges != null)
					serverChanges.add(creationCopy2);
			} catch (USMException e) {
				sessionManager.getJournal().debug(
						creation.getSession() + " Couldn't read implicit server changes for client creation", e);
			}
		}
	}

	/**
	 * Checks given folder if it may have read rights for the current user.
	 * - If f is null, it returns false
	 * - if the field "own_rights" is not set,. it returns true (since the rights have not been read from the OX server for that folder)
	 * - otherwise it returns true, if any read rights exist (for own or for all objects), i.e. if bits 7-13 are not all 0.
	 *  
	 * @param f
	 * @return
	 */
	private static boolean mayHaveRightsToReadElements(Folder f) {
		if (f == null)
			return false;
		Object v = f.getFieldContent("own_rights");
		if (!(v instanceof Number))
			return true;
		int rights = ((Number) v).intValue();
		return (rights & 0x3F80) != 0;
	}

	private static void performUpdates(DataObjectSet serverObjects, List<DataObject> updates,
			Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler)
			throws ConflictingChangeException {
		if (updates == null || updates.isEmpty())
			return;
		if (performFolderUpdates(serverObjects, updates, resultMap, throwConflictExceptions, handler))
			return;
		for (DataObject update : updates)
			performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, update, false);
	}

	private static boolean performFolderUpdates(DataObjectSet serverObjects, List<DataObject> updates,
			Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler)
			throws ConflictingChangeException {
		if (updates.get(0).getContentType().getCode() != DefaultContentTypes.FOLDER_CODE)
			return false;
		Folder[] folders = FolderHierarchyComparator.convertToSortedFoldersArray(updates);
		if (folders == null)
			return false;
		for (Folder f : folders)
			performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, f, true);
		return true;
	}

	private static void performOneUpdate(DataObjectSet serverObjects, Map<DataObject, USMException> resultMap,
			boolean throwConflictExceptions, UpdateHandler handler, DataObject update, boolean isFolderUpdate)
			throws ConflictingChangeException {
		try {
			DataObject serverObject = null;
			if (serverObjects != null) {
				serverObject = serverObjects.get(update.getID());
				if (serverObject != null)
					update.setTimestamp(serverObject.getTimestamp());
			}
			handler.update(update, serverObject, isFolderUpdate);
		} catch (ConflictingChangeException e) {
			if (throwConflictExceptions)
				throw e;
			resultMap.put(update, e);
		} catch (USMException e) {
			resultMap.put(update, e);
		}
	}
}