/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.json.streaming;

import java.io.Closeable;
import java.io.IOException;
import java.io.Reader;
import org.json.JSONException;

/**
 * {@link SimpleJSONReader} Provides simple stream based parsing for JSON-Data (only methods required by uploadData-requests are provided)
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class SimpleJSONReader implements Closeable {

    // Internal security constraint: Limit the number of character per value to avoid reading huge amounts of data for bad requests
    private static final int _MAX_PART_LENGTH = 16384;

    public static boolean isQuotationMark(int c) {
        return c == '"' || c == '\'';
    }

    private final Reader _reader;

    public SimpleJSONReader(Reader reader) {
        _reader = reader;
    }

    @Override
    public void close() throws IOException {
        _reader.close();
    }

    public int read() throws JSONException {
        try {
            return _reader.read();
        } catch (IOException e) {
            throw new JSONException(e);
        }
    }

    public int readNonWhitespace() throws JSONException {
        for (;;) {
            int c = read();
            if (c < 0 || c > ' ')
                return c;
        }
    }

    public void read(int expected) throws JSONException {
        int c = readNonWhitespace();
        if (c != expected)
            throw new JSONException("Encountered unexpected character '" + ((char) c) + '\'');
    }

    public String readString(String identifier, int quotationMark) throws JSONException {
        if (!isQuotationMark(quotationMark))
            throw new JSONException("No valid quotation mark for string " + identifier);
        int c;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < _MAX_PART_LENGTH; i++) {
            c = read();
            switch (c) {
            case -1:
            case 0:
            case '\n':
            case '\r':
                throw new JSONException("Unterminated string for " + identifier);
            case '\\':
                c = read();
                switch (c) {
                case 'b':
                    sb.append('\b');
                    break;
                case 't':
                    sb.append('\t');
                    break;
                case 'n':
                    sb.append('\n');
                    break;
                case 'f':
                    sb.append('\f');
                    break;
                case 'r':
                    sb.append('\r');
                    break;
                case 'u':
                    sb.append((char) readHexCode(4));
                    break;
                case 'x':
                    sb.append((char) readHexCode(2));
                    break;
                default:
                    sb.append(c);
                }
                break;
            default:
                if (c == quotationMark) {
                    return sb.toString();
                }
                sb.append((char)c);
            }
        }
        throw new JSONException("String too long for " + identifier);
    }

    public long readLong(String identifier, int initialCharacter, char expectedDelimiter) throws JSONException {
        boolean negative = false;
        long value = 0L;
        if (initialCharacter == '-') {
            negative = true;
        } else {
            if (initialCharacter < '0' || initialCharacter > '9')
                throw new JSONException(identifier + " starts with non-digit character '" + ((char) initialCharacter) + '\'');
            value = initialCharacter - '0';
        }
        for (int i = 0; i < _MAX_PART_LENGTH; i++) {
            int c = read();
            if (c == expectedDelimiter)
                return negative ? -value : value;
            if (c < '0' || c > '9')
                throw new JSONException(identifier + " contains non-digit character '" + ((char) c) + '\'');
            value = value * 10 + (c - '0');
        }
        throw new JSONException("Number too long for " + identifier);
    }

    private int readHexCode(int length) throws JSONException {
        int value = 0;
        for (int i = 0; i < length; i++) {
            int c = read();
            if (c >= '0' && c <= '9')
                value = value * 16 + (c - '0');
            else if (c >= 'a' && c <= 'f')
                value = value * 16 + (c + 10 - 'a');
            else if (c >= 'A' && c <= 'F')
                value = value * 16 + (c + 10 - 'A');
            else
                throw new JSONException("No valid hex code character '" + ((char) c) + "' for encoded character");
        }
        return value;
    }
}
