/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.ox_json.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.HttpsURL;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.URI;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.httpclient.methods.ByteArrayRequestEntity;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.methods.multipart.ByteArrayPartSource;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMIllegalStateException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.ox_json.JSONResult;
import com.openexchange.usm.ox_json.JSONResultType;
import com.openexchange.usm.ox_json.OXJSONAccess;
import com.openexchange.usm.ox_json.OXJSONPropertyNames;
import com.openexchange.usm.ox_json.OXResource;

/**
 * @author afe
 */
public class OXJSONAccessImpl implements OXJSONAccess {

    private static final String OX_REQUEST_INFORMATION_FIELD = "OXJSONAccessImpl.OXRequestInformation";

    private static final int MAX_RETRY = 10;

    private static final int MAX_REDIRECT_COUNT = 10;

    private static final String IMAGE = "image/";

    private static final String TEXT_HTML = "text/html";

    private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

    private static final String APPLICATION_OCTET_STREAM_UTF_8 = "application/octet-stream;charset=UTF-8";

    private final static boolean ALWAYS_COLLECT_STATISTICS = true;

    private final static int RESOURCE_BUFFER_DEFAULT_SIZE = 40000;

    private final static String ACTION_KEY = "**action**";

    private final static String JSON_KEY = "**json**";

    private final static String FILE_PART_NAME = "file";

    private final static String STRING_PART_NAME = "json";

    private final static String BINARY_ENCODING = "binary";

    private final static String TEXT_JAVASCRIPT_CONTENTTYPE = "text/javascript";

    private final static String CHARSET_UTF8 = "UTF-8";

    private final static String CONFIG = "config/";

    private final static String DATA = "data";

    private final static String HTTPS_PREFIX = "https:";

    // private final static String UDATE_ACTION = "update";

    // Check after 5 minutes if redirect still active, restricts number of log messages and redirect tests
    private static final long REDIRECT_RETEST_INTERVAL = 300000L;

    private static final long LOCK_TIMEOUT = 100;

    private static final String USM_USER_AGENT = "Open-Xchange USM HTTP Client";

    private Log _journal;

    private MultiThreadedHttpConnectionManager _httpManager;

    private Thread _backgroundConnectionCloser;

    private int _connectionTimeout;

    private String _oxJSONBaseURLasString;

    private HttpURL _oxJSONBaseURL;

    private String _postResultPattern;

    private int _actionIndex;

    private int _jsonIndex;

    private long _lastRedirect = 0L;

    public void login(Session session) throws AuthenticationFailedException, OXCommunicationException {
        logout(session, false);
        try {
            OXConnectionData result = obtainConnectionData(session);
            if (_journal.isDebugEnabled())
                _journal.debug(session + " Logged in, session id = " + result.getSessionID());
        } catch (AuthenticationFailedException e) {
            if (_journal.isDebugEnabled())
                _journal.debug(session + " Authentication failed", e);
            throw e;
        } catch (OXCommunicationException e) {
            _journal.warn(session + " Error while authenticating", e);
            throw e;
        }
    }

    public JSONObject getConfiguration(Session session, String... path) throws AuthenticationFailedException, OXCommunicationException {
        JSONObject config = readConfiguration(session, path);
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Retrieved OX ajax configuration");
        return config;
    }

    private JSONObject readConfiguration(Session session, String... path) throws AuthenticationFailedException, OXCommunicationException {
        String accessPath = CONFIG;
        if (path.length > 0) {
            StringBuilder sb = new StringBuilder(CONFIG);
            for (String part : path)
                sb.append(part).append('/');
            accessPath = sb.toString();
        }
        JSONResult result = doGet(accessPath, null, session, null);
        if (result.getResultType() != JSONResultType.JSONObject) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_ERROR_WHILE_AUTHENTICATING_NUMBER_3,
                "Unknown OX response reading configuration",
                result.toString());
        }
        try {
            Object object = result.getJSONObject().get(DATA);
            if (object instanceof String)
                return new JSONObject().put(DATA, object);
            else if (object instanceof JSONObject)
                return (JSONObject) object;
            if (object instanceof JSONArray)
                return new JSONObject().put(DATA, object);
            else if (object instanceof Integer)
                return new JSONObject().put(DATA, String.valueOf(object));
            else
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_ILLEGAL_DATA_BY_OXCONFIG_RESPONSE_NUMBER_4,
                    "OX configuration response contains illegal data",
                    result.toString());
        } catch (JSONException e) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_ILLEGAL_DATA_BY_OXCONFIG_RESPONSE_NUMBER_4,
                "OX configuration response contains illegal data",
                e,
                result.toString());
        }
    }

    private OXConnectionData performLogin(Session session) throws AuthenticationFailedException, OXCommunicationException {
        String user = session.getUser();
        String password = session.getPassword();
        HttpClient client = new HttpClient(_httpManager);
        client.getParams().setParameter("http.protocol.content-charset", "UTF-8");
        client.getParams().setParameter(HttpMethodParams.USER_AGENT, USM_USER_AGENT);
        PostMethod method = new PostMethod(
            getBaseURL() + "login?client=" + urlEncode("USM-" + session.getProtocol()) + "&version=" + urlEncode(USMVersion.VERSION_BUILD) + "&authId=" + UUID.randomUUID() + createClientIPParameter(session));
        method.addParameter("action", "login");
        method.addParameter("name", user);
        method.addParameter("password", password);
        for (String header : session.getXRequestHeaders().keySet()) {
            method.setRequestHeader(header, session.getXRequestHeaders().get(header));
        }
        JSONResult result = executeMethodOnce(session, client, method);
        if (result.getResultType() == JSONResultType.Error)
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_AUTHENTICATION_FAILED_NUMBER_5,
                "Authentication failed",
                result.getJSONObject());
        else if (result.getResultType() != JSONResultType.JSONObject)
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_INVALID_RESULT_FROM_SERVER,
                "Server responce does not contain JSONObject",
                result.toString());
        try {
            String sessionID = result.getJSONObject().getString("session");
            OXConnectionData data = new OXConnectionData(sessionID, client);
            session.setOXConnectionData(data);
            return data;
        } catch (JSONException e) {
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_RESPONSE_NOT_CONTAIN_SESSIONID_NUMBER_6,
                "Server response did not contain session ID",
                result.getJSONObject());
        }
    }

    private String createClientIPParameter(Session session) {
        return ("127.0.0.1".equals(session.getClientIp()) ? "" : ("&clientIP=" + session.getClientIp()));
    }

    public void logout(Session session) {
        logout(session, true);
    }

    private void logout(Session session, boolean doLog) {
        if (doLog && _journal.isDebugEnabled())
            _journal.debug(session + " Logging out");
        if (!obtainOptionalLock(session, "login?action=logout")) {
            _journal.warn("Timeout while waiting to log out session " + session);
            session.setOXConnectionData(null);
            return;
        }
        try {
            OXConnectionInformation info = session.getOXConnectionData();
            session.setOXConnectionData(null);
            performLogout(info);
        } finally {
            releaseLock(session);
        }
    }

    public void logout(OXConnectionInformation info) {
        if (_journal.isDebugEnabled())
            _journal.debug("Logging out " + info);
        performLogout(info);
    }

    private void performLogout(OXConnectionInformation info) {
        if (!(info instanceof OXConnectionData))
            return;
        OXConnectionData data = (OXConnectionData) info;
        GetMethod method = new GetMethod(getBaseURL() + "login?action=logout&session=" + urlEncode(data.getSessionID()));
        try {
            executeMethod(data.getHttpClient(), method);
        } catch (HttpException ignored) {
        } catch (IOException ignored) {
        } finally {
            method.releaseConnection();
        }
    }

    public JSONResult doGet(String path, String action, Session session, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (_journal.isDebugEnabled())
            _journal.debug(session + " GET " + url);
        GetMethod method = new GetMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONArray array) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, array.toString());
    }

    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONObject object) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, object.toString());
    }

    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String str) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (_journal.isDebugEnabled()) {
            _journal.debug(session + " PUT " + url + '\n' + str);
        }
        PutMethod method = new PutMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        try {
            method.setRequestEntity(new StringRequestEntity(str, TEXT_JAVASCRIPT_CONTENTTYPE, CHARSET_UTF8));
        } catch (UnsupportedEncodingException e1) {
            throw new USMIllegalStateException(
                OXJSONErrorCode.ACCESSIMPL_UNSUPPORTED_ENCODING_NUMBER_7,
                "UnsupportedEncoding: Character encoding UTF-8 not found");
        }
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, byte[] bytes) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (_journal.isDebugEnabled())
            _journal.debug(session + " PUT " + url + " with byte array");
        PutMethod method = new PutMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        method.setRequestEntity(new ByteArrayRequestEntity(bytes));
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    public void activate(Log log, String oxJSONURL, String postResultPattern, int maxConnections, int connectionTimeout, boolean performStaleChecking) {
        String error = doActivate(log, oxJSONURL, postResultPattern, maxConnections, connectionTimeout, performStaleChecking);
        if (error != null) {
            error = "OXJSONAccess not activated, reason: " + error;
            log.error(error);
            deactivate(null);
            throw new USMStartupException(OXJSONErrorCode.ACCESSIMPL_OXAJAXACCESS_NOTACTIVATED_ENCODING_NUMBER_8, error);
        }
    }

    public void deactivate() {
        deactivate("USM OXJSONAccess deactivated");
    }

    private String doActivate(Log log, String oxJSONURL, String postResultPattern, int maxConnections, int connectionTimeout, boolean performStaleChecking) {
        _journal = log;
        _oxJSONBaseURLasString = oxJSONURL;
        try {
            _oxJSONBaseURL = getURLFromString(oxJSONURL);
        } catch (URIException e) {
            return "OXJSONAccess base URL is no valid http(s) URL: " + _oxJSONBaseURLasString;
        }
        _postResultPattern = postResultPattern.trim();
        _actionIndex = _postResultPattern.indexOf(ACTION_KEY);
        if (_actionIndex < 0)
            return ACTION_KEY + " not found in post result pattern";
        _jsonIndex = _postResultPattern.indexOf(JSON_KEY);
        if (_jsonIndex < 0)
            return JSON_KEY + " not found in post result pattern";
        _connectionTimeout = connectionTimeout;
        _httpManager = new MultiThreadedHttpConnectionManager();
        HttpConnectionManagerParams httpParams = _httpManager.getParams();
        httpParams.setDefaultMaxConnectionsPerHost(maxConnections);
        httpParams.setMaxTotalConnections(maxConnections);
        httpParams.setConnectionTimeout(_connectionTimeout);
        httpParams.setStaleCheckingEnabled(performStaleChecking);
        _backgroundConnectionCloser = new Thread("USM-OX-HttpConnectionCloser") {

            @Override
            public void run() {
                try {
                    while (!isInterrupted()) {
                        Thread.sleep(_connectionTimeout);
                        _httpManager.closeIdleConnections(_connectionTimeout);
                    }
                } catch (InterruptedException ignored) {
                    // We get interrupted to indicate that this thread has to stop, so simply stop
                }
            }
        };
        _backgroundConnectionCloser.setPriority(Thread.MIN_PRIORITY);
        _backgroundConnectionCloser.setDaemon(true);
        _backgroundConnectionCloser.start();
        _journal.info("USM OXJSONAccess activated, OX-URL=" + _oxJSONBaseURLasString);
        return null;
    }

    private HttpURL getURLFromString(String urlString) throws URIException {
        return (urlString.toLowerCase().startsWith(HTTPS_PREFIX)) ? new HttpsURL(urlString, CHARSET_UTF8) : new HttpURL(
            urlString,
            CHARSET_UTF8);
    }

    private void deactivate(String message) {
        if (_backgroundConnectionCloser != null && _backgroundConnectionCloser.isAlive())
            _backgroundConnectionCloser.interrupt();
        _backgroundConnectionCloser = null;
        _oxJSONBaseURL = null;
        _oxJSONBaseURLasString = null;
        _postResultPattern = null;
        if (message != null)
            _journal.info(message);
        _journal = null;
    }

    private OXConnectionData obtainConnectionData(Session session) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, "<obtain OX connection data>");
        try {
            OXConnectionInformation o = session.getOXConnectionData();
            _journal.debug(session + " Retrieving OX session with ID: " + o);
            return (o instanceof OXConnectionData) ? ((OXConnectionData) o) : performLogin(session);
        } finally {
            releaseLock(session);
        }
    }

    private NameValuePair[] generateQueryParameters(String action, String sessionID, Map<String, String> params) {
        List<NameValuePair> list = new ArrayList<NameValuePair>();
        if (action != null)
            list.add(new NameValuePair("action", action));
        if (sessionID != null)
            list.add(new NameValuePair("session", sessionID));
        if (params != null) {
            for (Map.Entry<String, String> entry : params.entrySet())
                list.add(new NameValuePair(entry.getKey(), entry.getValue()));
        }
        list.add(new NameValuePair("timezone", "UTC"));
        return list.toArray(new NameValuePair[list.size()]);
    }

    private String getBaseURL() {
        return _oxJSONBaseURLasString;
    }

    private String urlEncode(String s) {
        try {
            return URLEncoder.encode(s, CHARSET_UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new USMIllegalStateException(
                OXJSONErrorCode.ACCESSIMPL_UTF8_ENCODING_NOTFOUND_NUMBER_10,
                "Character encoding UTF-8 not found");
        }
    }

    private JSONResult executeMethodAndExtractResult(Session session, OXConnectionData data, HttpMethodBase method, String action, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, method.getName() + ' ' + method.getPath() + '?' + method.getQueryString());
        try {
            JSONResult result = executeMethodOnce(session, data.getHttpClient(), method);
            // On specific error for expired/unknown session, do login and repeat
            if (shouldPerformNewLogin(result)) {
                if (_journal.isDebugEnabled())
                    _journal.debug(session + " Session with sessionid " + data.getSessionID() + " unknown to OX server, performing login and repeat");
                login(session);
                data = obtainConnectionData(session);
                method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
                setRequestInformation(session, method.getName() + ' ' + method.getPath() + '?' + method.getQueryString());
                result = executeMethodOnce(session, data.getHttpClient(), method);
            }
            for (int i = 0; i < MAX_RETRY && shouldRetryRequest(result); i++) {
                if (_journal.isDebugEnabled())
                    _journal.debug("Server returned error from category: " + OXJSONSupport.getJSONErrorCategory(result.getJSONObject()) + ". Retrying request..");
                result = executeMethodOnce(session, data.getHttpClient(), method);
            }
            if (shouldRetryRequest(result))
                throw new TemporaryDownOrBusyException(
                    OXJSONErrorCode.ACCESSIMPL_TOO_MANY_DOWN_OR_BUSY_RETRIES,
                    "Server/Resource is temporary down or busy.",
                    result.getJSONObject());
            return result;
        } finally {
            releaseLock(session);
        }
    }

    private JSONResult executeMethodOnce(Session session, HttpClient client, HttpMethodBase method) throws OXCommunicationException {
        boolean collectOXStatistics = ALWAYS_COLLECT_STATISTICS || _journal.isDebugEnabled();
        long startTime = collectOXStatistics ? System.currentTimeMillis() : 0L;
        try {
            int statusCode = executeMethod(client, method);
            if (statusCode != HttpStatus.SC_OK)
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTP_CONNECTION_ERROR_11, statusCode);
            JSONResult result = OXJSONSupport.parseResult(extractResponseBodyAsString(method), null);
            if (isSynchronizationConflictError(result))
                throw new ConflictingChangeException(
                    OXJSONErrorCode.ACCESSIMPL_CONFLICT_CHANGE_ON_OXACCESS_NUMBER_12,
                    "Conflicting change on OX access",
                    result.getJSONObject());
            if (_journal.isDebugEnabled())
                _journal.debug("RESULT: " + result);
            return result;
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTPEXCEPTION_OCCURED_NUMBER_13, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_EXECUTE_METHOD_ERROR_NUMBER_14, e);
        } finally {
            method.releaseConnection();
            if (collectOXStatistics) {
                long duration = System.currentTimeMillis() - startTime;
                long number = getSessionProperty(session, OXJSONPropertyNames.NUMBER_OF_CALLS);
                session.setCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS, Long.valueOf(number + 1L));
                long accTime = getSessionProperty(session, OXJSONPropertyNames.ACCUMULATED_TIME);
                session.setCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME, Long.valueOf(accTime + duration));
            }
        }
    }

    private long getSessionProperty(Session session, String name) {
        Object o = session.getCustomProperty(name);
        return (o instanceof Long) ? ((Long) o).longValue() : 0L;
    }

    private boolean isSynchronizationConflictError(JSONResult result) {
        return result.getResultType() == JSONResultType.Error && OXJSONSupport.ERROR_CODE_OBJECT_CHANGED.equals(OXJSONSupport.getJSONErrorCode(result.getJSONObject()));
    }

    private boolean shouldPerformNewLogin(JSONResult result) {
        if (result.getResultType() != JSONResultType.Error)
            return false;
        String errorCode = OXJSONSupport.getJSONErrorCode(result.getJSONObject());
        return OXJSONSupport.ERROR_CODE_UNKNOWN_SESSION.equals(errorCode) || OXJSONSupport.ERROR_CODE_EXPIRED_SESSION.equals(errorCode);
    }

    private boolean shouldRetryRequest(JSONResult result) {
        if (result.getResultType() != JSONResultType.Error)
            return false;
        String errorCategory = OXJSONSupport.getJSONErrorCategory(result.getJSONObject());
        return OXJSONSupport.CATEGORY_RESOURCE_TEMPORARY_DOWN.equals(errorCategory)
        // || OXJSONSupport.CATEGORY_RESOURCE_FULL_OR_BUSY.equals(errorCategory)
        || OXJSONSupport.CATEGORY_SOCKET_CONNECTION_CORRUPT.equals(errorCategory) || OXJSONSupport.CATEGORY_SUBSYSTEM_DOWN.equals(errorCategory);
    }

    public OXResource getResource(Session session, String path, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        GetMethod method = null;
        try {
            if (_journal.isDebugEnabled())
                _journal.debug(session + " GET " + _oxJSONBaseURLasString + " + " + path);
            OXConnectionData connectionData = obtainConnectionData(session);
            if (parameters == null) {
                HttpURL url = getURLFromString(path);
                if (url.isRelativeURI()) {
                    url = (_oxJSONBaseURL instanceof HttpsURL) ? new HttpsURL((HttpsURL) _oxJSONBaseURL, path) : new HttpURL(
                        _oxJSONBaseURL,
                        url);
                }
                method = new GetMethod();
                method.setURI(url);
            } else {
                String url = getBaseURL() + path;
                method = new GetMethod(url);
                method.setQueryString(generateQueryParameters(null, connectionData.getSessionID(), parameters));
            }
            int statusCode = executeMethod(connectionData.getHttpClient(), method);
            if (statusCode != HttpStatus.SC_OK)
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_15, statusCode);
            Header responseHeader = method.getResponseHeader("Content-Type");
            if (responseHeader == null) {
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_NO_CONTENT_TYPE_RECEIVED,
                    "No content type in response header for path: " + method.getURI().getPathQuery());
            }
            String contentType = responseHeader.getValue();
            if (APPLICATION_OCTET_STREAM.equals(contentType) || APPLICATION_OCTET_STREAM_UTF_8.equals(contentType))
                return new OXResource(contentType, extractResponseBody(method));
            else if (contentType != null && contentType.startsWith(IMAGE))
                return new OXResource(contentType, extractResponseBody(method));
            else if (contentType != null && contentType.startsWith(TEXT_HTML)) {
                String httpResult = extractResponseBodyAsString(method);
                int startIndex = httpResult.indexOf("({");
                int endIndex = httpResult.lastIndexOf("})");
                JSONResult json = new JSONErrorResult(new JSONObject());
                if (startIndex > 0 && endIndex > startIndex) {
                    json = checkPostResult(httpResult.substring(startIndex + 1, endIndex + 1));
                }
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_ERROR_ON_READING_RESOURCE,
                    "Error on getting resource from Server",
                    json.getJSONObject());
            } else {
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_INVALID_CONTENT_TYPE_RECEIVED,
                    "Invalid content type in response: " + contentType);
            }
        } catch (URIException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_16, "Invalid URL specified: " + path, e);
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_17, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_18, e);
        } finally {
            if (method != null)
                method.releaseConnection();
        }
    }

    public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, byte[] imageByteData, String fileContentType) throws AuthenticationFailedException, OXCommunicationException {
        return storeResource(path, action, session, parameters, jsonData, imageByteData, fileContentType, STRING_PART_NAME, FILE_PART_NAME);
    }

    public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, byte[] imageByteData, String fileContentType, String jsonPartName, String filePartName) throws AuthenticationFailedException, OXCommunicationException {
        PostMethod multiPartPostMethod = null;
        try {
            OXConnectionData connectionData = obtainConnectionData(session);
            String url = getBaseURL() + path;

            if (_journal.isDebugEnabled())
                _journal.debug(session + " POST " + url);

            multiPartPostMethod = new PostMethod(url);
            multiPartPostMethod.setQueryString(generateQueryParameters(action, connectionData.getSessionID(), parameters));

            // "text/javascript", "UTF-8"
            List<Part> parts = new ArrayList<Part>();

            if (jsonData != null) {
                StringPart jsonPart = new StringPart(jsonPartName, jsonData.toString(), CHARSET_UTF8);
                jsonPart.setContentType(TEXT_JAVASCRIPT_CONTENTTYPE);
                parts.add(jsonPart);
            }

            FilePart filePart = new FilePart(
                filePartName,
                new ByteArrayPartSource(filePartName, imageByteData),
                fileContentType,
                BINARY_ENCODING);
            filePart.setTransferEncoding(BINARY_ENCODING);
            parts.add(filePart);

            multiPartPostMethod.setRequestEntity(new MultipartRequestEntity(
                parts.toArray(new Part[parts.size()]),
                multiPartPostMethod.getParams()));

            int statusCode = executeMethod(connectionData.getHttpClient(), multiPartPostMethod);
            if (statusCode != HttpStatus.SC_OK)
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_19, statusCode);
            String httpResult = extractResponseBodyAsString(multiPartPostMethod);
            return checkAndExtractResult(httpResult);
        } catch (URIException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_20, "Invalid URL specified", e);
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_21, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_22, e);
        } finally {
            if (multiPartPostMethod != null)
                multiPartPostMethod.releaseConnection();
        }
    }

    private JSONResult checkAndExtractResult(String httpResult) throws OXCommunicationException {
        httpResult = httpResult.trim();
        int startIndex = httpResult.indexOf("({");
        int endIndex = httpResult.lastIndexOf("})");
        if (startIndex > 0 && endIndex > startIndex) {
            String json = httpResult.substring(startIndex + 1, endIndex + 1);
            return checkPostResult(json);
        }
        throw new OXCommunicationException(
            OXJSONErrorCode.ACCESSIMPL_INVALID_RESULTOF_POST_REQUEST_NUMBER_23,
            "Result of POST request doesn't match configured pattern",
            httpResult);
    }

    private JSONResult checkPostResult(String json) throws OXCommunicationException {
        try {
            return OXJSONSupport.generateJSONResultForJSONObject(new JSONObject(json), JSONResultType.JSONObject);
        } catch (JSONException e) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_RESULTOF_POST_NOTCONTAIN_VALID_JSONOBJECT_NUMBER_24,
                "Result of POST request doesn't contain valid JSONObject: " + json,
                e);
        }
    }

    private byte[] extractResponseBody(HttpMethod method) throws IOException {
        ByteArrayOutputStream byteArrayStream = new ByteArrayOutputStream(RESOURCE_BUFFER_DEFAULT_SIZE);
        InputStream inputStream = method.getResponseBodyAsStream();
        byte[] data = new byte[4096];
        for (int count = inputStream.read(data); count > 0; count = inputStream.read(data))
            byteArrayStream.write(data, 0, count);
        byteArrayStream.close();
        return byteArrayStream.toByteArray();
    }

    private String extractResponseBodyAsString(HttpMethodBase method) throws IOException {
        return new String(extractResponseBody(method), method.getResponseCharSet());
    }

    private int executeMethod(HttpClient client, HttpMethod method) throws HttpException, IOException {
        if (_journal.isDebugEnabled())
            _journal.debug(method.getName() + method.getPath() + "&" + method.getQueryString());
        for (Header header : method.getRequestHeaders()) {
            _journal.debug("Header: " + header.getName() + ":" + header.getValue());
        }
        int status = HttpStatus.SC_NOT_FOUND;
        method.setFollowRedirects(false);
        String redirectFromServer = null;
        // override potentially set secure mode for secret cookie (when com.openexchange.forceHTTPS=true)
        for (org.apache.commons.httpclient.Cookie c : client.getState().getCookies()) {
            if (c.getName().startsWith("open-xchange-secret")) {
                c.setSecure(false);
            }
        }
        for (int i = 0; i < MAX_REDIRECT_COUNT; i++) {
            status = client.executeMethod(method);
            Header location = getRedirection(method, status);
            if (location == null) {
                if (i > 0 && _lastRedirect < System.currentTimeMillis() - REDIRECT_RETEST_INTERVAL) {
                    _lastRedirect = System.currentTimeMillis();
                    _journal.info("HTTP Access to JSON interface of OX-Server was redirected. Redirect from server: '" + redirectFromServer + "', used URL: '" + method.getURI() + "'");
                }
                return status;
            }
            redirectFromServer = location.getValue();
            URI newURI = new URI(redirectFromServer, true, method.getParams().getUriCharset());
            if (newURI.isRelativeURI())
                newURI = new URI(method.getURI(), newURI);
            method.setURI(newURI);
        }
        _journal.error("Too many HTTP redirects accessing JSON interface of OX-Server");
        return status;
    }

    private Header getRedirection(HttpMethod method, int status) {
        return (status == HttpStatus.SC_MOVED_PERMANENTLY || status == HttpStatus.SC_SEE_OTHER || status == HttpStatus.SC_TEMPORARY_REDIRECT) ? method.getResponseHeader("location") : null;
    }

    public JSONResult doPost(String path, String action, Session session, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {

        OXConnectionData data = obtainConnectionData(session);
        PostMethod method = new PostMethod(getBaseURL() + path);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));

        JSONResult result = executeMethodOnce(session, data.getHttpClient(), method);
        if (result.getResultType() == JSONResultType.Error)
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_AUTHENTICATION_FAILED,
                "Authentication failed",
                result.getJSONObject());
        else if (result.getResultType() != JSONResultType.JSONObject)
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_INVALID_RESULT_FROM_SERVER_2,
                "Server responce does not contain JSONObject",
                result.toString());
        return result;
    }

    private boolean obtainOptionalLock(Session session, String requestInformation) {
        try {
            return tryLock(session, requestInformation);
        } catch (InterruptedException e) {
            return false;
        }
    }

    private void obtainRequiredLock(Session session, String requestInformation) throws OXCommunicationException {
        try {
            if (tryLock(session, requestInformation))
                return;
        } catch (InterruptedException e) {
            // fall through
        }
        throw new OXCommunicationException(
            OXJSONErrorCode.FAILED_TO_GAIN_EXCLUSIVE_OX_ACCESS,
            "Timeout while waiting for exclusive access to OX server, locked by " + session + '/' + session.getCustomProperty(OX_REQUEST_INFORMATION_FIELD),
            (String) null);
    }

    private boolean tryLock(Session session, String requestInformation) throws InterruptedException {
        boolean success = session.getOXConnectionLock().tryLock(LOCK_TIMEOUT, TimeUnit.SECONDS);
        if (success)
            setRequestInformation(session, requestInformation);
        return success;
    }

    private void releaseLock(Session session) {
        setRequestInformation(session, null);
        session.getOXConnectionLock().unlock();
    }

    private void setRequestInformation(Session session, String requestInformation) {
        session.setCustomProperty(OX_REQUEST_INFORMATION_FIELD, requestInformation);
    }
}
