/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.ox_event.impl;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventHandler;

import com.openexchange.event.CommonEvent;
import com.openexchange.event.RemoteEvent;
import com.openexchange.groupware.Types;
import com.openexchange.groupware.container.DataObject;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.ox_event.OXEventListener;
import com.openexchange.usm.ox_event.OXEventManager;
import com.openexchange.group.*;
import com.openexchange.resource.*;

/**
 * 
 * @author afe
 *
 */
public class OXEventManagerImpl implements OXEventManager, EventHandler {
	private static final boolean USE_NEW_AFFECTED_USERS_METHOD = true;
	
	private static final String GROUP_TOPICS_PREFIX = "com/openexchange/groupware/group/";
	private static final String RESOURCES_TOPIC_PREFIX = "com/openexchange/groupware/resource/";

	/*
	 * The constants were copied from com.openexchange.push.PushEventConstants
	 * to avoid a requirement on that package which would need the appriopriate push-bundle.
	 */

	private static final class PushEventConstants {

		/**
		 * Initializes a new {@link PushEventConstants}.
		 */
		private PushEventConstants() {
			super();
		}

		/**
		 * The topic of push events.
		 */
		public static final String TOPIC = "com/openexchange/push";

		/**
		 * The context ID property of a push event. Property value is of type <code>java.lang.Integer</code>.
		 */
		public static final String PROPERTY_CONTEXT = "com.openexchange.push.context";

		/**
		 * The user ID property of a push event. Property value is of type <code>java.lang.Integer</code>.
		 */
		public static final String PROPERTY_USER = "com.openexchange.push.user";

		/**
		 * The folder fullname property of a push event. Property value is of type <code>java.lang.String</code>.
		 */
		public static final String PROPERTY_FOLDER = "com.openexchange.push.folder";

		/**
		 * The session property of a push event. Property value is of type <code>com.openexchange.session.Session</code>.
		 */
		public static final String PROPERTY_SESSION = "com.openexchange.push.session";

		/**
		 * If true (or not set), the content elements have been changed, if false, the folder or a sub-folder has changed
		 */
		public static final String PROPERTY_CONTENT_RELATED = "com.openexchange.push.content-related";
	}

	private Log _journal;

	private final List<OXEventListener> _listeners = new CopyOnWriteArrayList<OXEventListener>();

	public void activate(Log log) {
		_journal = log;
		_journal.info("USM OXEventManager activated");
	}

	public void deactivate() {
		_journal.info("USM OXEventManager deactivated");
		_journal = null;
	}

	public void addEventListener(OXEventListener listener) {
		if (_journal != null && _journal.isDebugEnabled())
			_journal.debug("Adding USM OXEventListener " + listener);
		_listeners.add(listener);
	}

	public void removeEventListener(OXEventListener listener) {
		if (_journal != null && _journal.isDebugEnabled())
			_journal.debug("Removing USM OXEventListener " + listener);
		_listeners.remove(listener);
	}

	@SuppressWarnings("restriction")
    public void handleEvent(Event event) {
        if (null == event) {
            return;
        }
        try {
            Object oxEventData = event.getProperty(CommonEvent.EVENT_KEY);
            if (event.getTopic().startsWith(GROUP_TOPICS_PREFIX)) {
                handleGroupEvent(event);
            } else if (event.getTopic().startsWith(RESOURCES_TOPIC_PREFIX)) {
                handleResourceEvent(event);
            } else if (oxEventData instanceof CommonEvent) {
                handleCommonEvent((CommonEvent) oxEventData);
            } else {
                oxEventData = event.getProperty(RemoteEvent.EVENT_KEY);
                if (oxEventData instanceof RemoteEvent) {
                    handleRemoteEvent((RemoteEvent) oxEventData);
                } else if (event.getTopic().equals(PushEventConstants.TOPIC)) {
                    handlePushEvent(event);
                }
            }
        } catch (Exception e) {
            _journal.error("Error while handling event \"" + event.getTopic() + "\".", e);
        }
    }

	private void handlePushEvent(Event event) {
		String folder = (String) event.getProperty(PushEventConstants.PROPERTY_FOLDER);
		int contextId = getIntProperty(event, PushEventConstants.PROPERTY_CONTEXT);
		int userId = getIntProperty(event, PushEventConstants.PROPERTY_USER);
		Boolean contentRelated = (Boolean) event.getProperty(PushEventConstants.PROPERTY_CONTENT_RELATED);
		boolean folderChange = contentRelated != null && !contentRelated.booleanValue();
		if (_journal.isDebugEnabled())
			_journal.debug("PushEvent " + folderChange + ":" + contextId + ":" + userId + ":" + folder);
		if (folder != null && contextId != 0 && userId != 0)
			notifyListeners(folderChange, contextId, userId, folder);
	}
	
	
	@SuppressWarnings("restriction")
    private void handleGroupEvent(Event event) {
        int contextId = getIntProperty(event, GroupEventConstants.PROPERTY_CONTEXT_ID);
        int userId = getIntProperty(event, GroupEventConstants.PROPERTY_USER_ID);
        String folderID = DefaultContentTypes.GROUPS_ID;
        boolean folderChange = false;
        if (_journal.isDebugEnabled())
            _journal.debug("Group Event " + event.getTopic() +  " in context :" + contextId + " for user :" + userId );
        if (contextId != 0 && userId != 0)
                notifyListeners(folderChange, contextId, userId, folderID);
    }
	
	
	@SuppressWarnings("restriction")
    private void handleResourceEvent(Event event) {
	    int contextId = getIntProperty(event, ResourceEventConstants.PROPERTY_CONTEXT_ID);
        int userId = getIntProperty(event, ResourceEventConstants.PROPERTY_USER_ID);
        String folderID = DefaultContentTypes.RESOURCES_ID;
        boolean folderChange = false;
        if (_journal.isDebugEnabled())
            _journal.debug("Resources Event " + event.getTopic() +  " in context :" + contextId + " for user :" + userId );
        if (contextId != 0 && userId != 0)
                notifyListeners(folderChange, contextId, userId, folderID);
    }

	private int getIntProperty(Event event, String property) {
		Object o = event.getProperty(property);
		return (o instanceof Integer) ? ((Integer) o) : 0;
	}

	private String extractFolderID(Object folder) {
		if (folder instanceof String)
			return (String) folder;
		if (folder instanceof DataObject)
			return String.valueOf(((DataObject) folder).getObjectID());
		return null;
	}

	private void handleCommonEvent(CommonEvent oxEvent) {
		int userId = oxEvent.getUserId();
		int contextId = oxEvent.getContextId();
		int module = oxEvent.getModule();
		String sourceFolderID = extractFolderID(oxEvent.getSourceFolder());
		String destFolderID = extractFolderID(oxEvent.getDestinationFolder());
		boolean isFolderChange = (module == Types.FOLDER);
		if (_journal.isDebugEnabled())
			_journal.debug("CommonEvent " + isFolderChange + ":" + contextId + ":" + userId + ":" + sourceFolderID
					+ ":" + destFolderID);
		if (USE_NEW_AFFECTED_USERS_METHOD && !oxEvent.getAffectedUsersWithFolder().isEmpty()) {
			Map<Integer, Set<Integer>> userMap = oxEvent.getAffectedUsersWithFolder();
			if (_journal.isDebugEnabled())
				_journal.debug("Affected users with folder map length: " + userMap.size());
			if (userMap != null) {
				for (Map.Entry<Integer, Set<Integer>> entry : userMap.entrySet()) {
					int affectedUser = entry.getKey();
					if (_journal.isDebugEnabled())
						_journal.debug("Affected user " + affectedUser);
					Set<Integer> affectedFolders = entry.getValue();
					if (_journal.isDebugEnabled())
						_journal.debug("Affected folders size: " + affectedFolders.size());
					if (affectedFolders != null && !affectedFolders.isEmpty()) {
						for (int folderID : affectedFolders) {
							notifyListeners(isFolderChange, contextId, affectedUser, String.valueOf(folderID));
							if (_journal.isDebugEnabled())
								_journal.debug("Affected folder: " + folderID);
						}
					} else {
						notifyListeners(isFolderChange, contextId, affectedUser, null);
						if (_journal.isDebugEnabled())
							_journal.debug("Else branch addected folders, notify listeners");
					}
				}
			}
		} else {
			if (sourceFolderID != null) {
				notifyListeners(isFolderChange, contextId, userId, sourceFolderID);
				if (destFolderID != null && !sourceFolderID.equals(destFolderID))
					notifyListeners(isFolderChange, contextId, userId, destFolderID);
			} else if (destFolderID != null) {
				notifyListeners(isFolderChange, contextId, userId, destFolderID);
			} else {
				notifyListeners(isFolderChange, contextId, userId, null);
			}
		}
		if (oxEvent.getAction() == CommonEvent.DELETE) {
			int contentType = 0;
			Object o = oxEvent.getActionObj();
			if (o instanceof DataObject) {
				DataObject actionObj = (DataObject) o;
				switch (module) {
					case Types.FOLDER:
						contentType = DefaultContentTypes.FOLDER_CODE;
						break;
					case Types.CONTACT:
						contentType = DefaultContentTypes.CONTACTS_CODE;
						break;
					case Types.APPOINTMENT:
						contentType = DefaultContentTypes.CALENDAR_CODE;
						break;
					case Types.TASK:
						contentType = DefaultContentTypes.TASK_CODE;
						break;
					case Types.EMAIL:
						contentType = DefaultContentTypes.MAIL_CODE;
						break;
					default:
						break;
				}
				if (contentType != 0)
					notifyListeners(contextId, actionObj.getObjectID(), contentType);
			}
		}
	}

	private void handleRemoteEvent(RemoteEvent oxEvent) {
		int userId = oxEvent.getUserId();
		int contextId = oxEvent.getContextId();
		int module = oxEvent.getModule();
		int folderId = oxEvent.getFolderId();
		boolean isFolderChange = (module == Types.FOLDER);
		if (_journal.isDebugEnabled())
			_journal.debug("RemoteEvent " + isFolderChange + ":" + contextId + ":" + userId + ":" + folderId);
		notifyListeners(isFolderChange, contextId, userId, String.valueOf(folderId));
	}

	private void notifyListeners(boolean isFolderChange, int contextId, int userId, String folderId) {
		if (isFolderChange) {
			for (OXEventListener listener : _listeners) {
				if (_journal.isDebugEnabled())
					_journal.debug("folderChanged"  + ":" + contextId + ":" + userId + ":" + folderId);
				listener.folderChanged(contextId, userId, folderId, 0);
			}
		} else {
			for (OXEventListener listener : _listeners) {
				listener.folderContentChanged(contextId, userId, folderId, 0);
			}
		}
	}

	private void notifyListeners(int contextID, int objectID, int contentType) {
		for (OXEventListener listener : _listeners) {
			listener.objectDeleted(contextID, objectID, contentType);
		}
	}
}
