/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.util;

import java.io.Serializable;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.session.dataobject.SimpleDataObject;

/**
 * Abstract base class for typical ContentType implementations.
 * 
 * @author afe
 *
 */
public abstract class AbstractContentType implements ContentType {

	private static final int IDENTICAL_RATING_VALUE = 5;
	private static final int SIMILAR_RATING_VALUE = 2;
	private static final int FIELD_MATCH_PERCENTAGE = 50; // from 0 to 100

	/*
	 * This method can be overwritten by ContentTypes to provide ContentType-specific match algorithms
	 * (non-Javadoc)
	 * @see com.openexchange.usm.api.contenttypes.ContentType#getMatchRating(com.openexchange.usm.api.session.DataObject, com.openexchange.usm.api.session.DataObject)
	 */
	public int getMatchRating(DataObject client, DataObject server) {
		// TODO Add some default heuristic to determine possible match ?
		return -1;
	}

	/**
	 * Default method of matching content types, can be called by sub-classes in the getMatchRating() method if
	 * no special handling is implemented.
	 * 
	 * @param client
	 * @param server
	 * @return
	 */
	protected int getDefaultMatchRating(DataObject client, DataObject server) {
		int matchingFields = 0;
		int nonEmptyFields = 0;
		boolean forceMatchRating = false;
		int matchValue = 0;
		ContentTypeField[] fields = getFields();
		for (int i = 0; i < fields.length; i++) {
			ContentTypeField field = fields[i];
			int weight = field.getFieldWeight();
			if (weight != 0) {
				DataType<?> dataType = field.getFieldType();
				Object value1 = client.getFieldContent(i);
				Object value2 = server.getFieldContent(i);
				if (!dataType.isDefaultValue(value1) || !dataType.isDefaultValue(value2)) {
					nonEmptyFields++;
					int factor = performSingleMatch(dataType, value1, value2);
					if (factor > 0) {
						if (weight < 0) {
							forceMatchRating = true;
							weight = -weight;
						}
						matchingFields++;
						matchValue += factor * weight;
					}
				}
			}
		}
		return (forceMatchRating || nonEmptyFields * FIELD_MATCH_PERCENTAGE <= matchingFields * 100) ? matchValue : -1;
	}

	/**
	 * Performs a single match between two field values. At least one of the two values is not at the default value for the DataType
	 * 
	 * @param dataType
	 * @param value1
	 * @param value2
	 * @return 0 if no match, value > 0 indicating the quality of the match
	 */
	protected int performSingleMatch(DataType<?> dataType, Object value1, Object value2) {
		if (dataType.isEqual(value1, value2))
			return IDENTICAL_RATING_VALUE;
		// TODO Do we want to delegate this to the DataType ?
		if (value1 != null && value2 != null && value1.toString().trim().equalsIgnoreCase(value2.toString().trim()))
			return SIMILAR_RATING_VALUE;
		return 0;
	}

	public DataObject newDataObject(Session session) {
		return new SimpleDataObject(session, this);
	}

	/**
	 * Default implementation of deserializeData() that automatically handles the addition of new ContentTypeFields at the
	 * end of the ContentType field list by setting them to their default value if not present in the deserialized data.
	 */
	public void deserializeData(Serializable[] data, int offset, Object[] fieldContent)
			throws DeserializationFailedException {
		ContentTypeField[] fields = getFields();
		int limit = Math.min(data.length - offset, fieldContent.length);
		int i = 0;
		try {
			for (; i < limit; i++)
				fieldContent[i] = fields[i].getFieldType().createCopy(data[i + offset]);
		} catch (Exception e) {
			throw new DeserializationFailedException(
					USMContentTypesUtilErrorCodes.DESERIALIZATION_FAILED_INVALID_FIELD_CONTENT,
					"Invalid serialized data, probably for field " + fields[i].getFieldName(), e);
		}
		for (; i < fieldContent.length; i++)
			fieldContent[i] = fields[i].getFieldType().createNewObject();
	}

	public boolean canBeFolderElementsContentType() {
		return true;
	}
}
