/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.util;

import java.util.UUID;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.api.datatypes.PIMAttachment;

/**
 * Helper class with support methods to read in PIMAttachment objects from JSON objects and to generate
 * JSONObjects from PIMAttachment objects.
 * 
 * @author afe
 *
 */
public class PIMAttachmentToolkit {
	public static final String UUID_FIELD = "uuid";
	public static final String CREATED_BY_FIELD = "created_by";
	public static final String CREATION_DATE_FIELD = "creation_date";
	public static final String FILENAME_FIELD = "filename";
	public static final String FILE_SIZE_FIELD = "file_size";
	public static final String FILE_MIMETYPE_FIELD = "file_mimetype";
	public static final String RTF_FLAG_FIELD = "rtf_flag";
	public static final String DATA_FIELD = "data";

	/**
	 * Reads PIMAttachment from JSONObject in a format as expected from the USM JSON API.
	 * This method can only be used if "data" and all associated required fields are present.
	 * If data is not present, the ox id and uuid should be read directly and a PIMAttachment
	 * object should be created from those two values.
	 * TODO Expand this method (add another method?) so that it can also be used when
	 * reading a PIMAttachment from the OX JSON API ?
	 * 
	 * @param o
	 * @return
	 * @throws JSONException
	 */
	public static PIMAttachment fromJSONObject(JSONObject o) throws JSONException {
		UUID uuid = UUID.fromString(o.getString(UUID_FIELD));
		String fileName = o.getString(FILENAME_FIELD);
		String mimeType = o.getString(FILE_MIMETYPE_FIELD);
		boolean rtfFlag = o.getBoolean(RTF_FLAG_FIELD);
		long fileSize = o.getLong(FILE_SIZE_FIELD);
		String createdBy = null;
		long creationDate = 0L;
		int expectedLength = 6;
		if (o.has(CREATED_BY_FIELD)) {
			createdBy = o.getString(CREATED_BY_FIELD);
			expectedLength++;
		}
		if (o.has(CREATION_DATE_FIELD)) {
			creationDate = o.getLong(CREATION_DATE_FIELD);
			expectedLength++;
		}
		if (o.length() != expectedLength)
			throw new JSONException("Unknown fields in PIM attachment");
		return new PIMAttachment(uuid, fileName, mimeType, fileSize, createdBy, creationDate, rtfFlag, Toolkit
				.decodeBase64(o.getString(DATA_FIELD)));
	}

	public static JSONObject toJSONObject(PIMAttachment a) throws JSONException {
		JSONObject o = new JSONObject();
		o.put(UUID_FIELD, a.getUUID().toString());
		byte[] data = a.getData();
		if (data != null) {
			o.put(DATA_FIELD, Toolkit.encodeBase64(data));
			o.put(FILENAME_FIELD, a.getFileName());
			o.put(FILE_MIMETYPE_FIELD, a.getMimeType());
			o.put(RTF_FLAG_FIELD, a.isRtfFlag());
			o.put(FILE_SIZE_FIELD, a.getFileSize());
			if (a.getCreatedBy() != null)
				o.put(CREATED_BY_FIELD, a.getCreatedBy());
			if (a.getCreationDate() != 0L)
				o.put(CREATION_DATE_FIELD, a.getCreationDate());
		}
		return o;
	}
}
