/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.ContentTypeTransferHandler;
import com.openexchange.usm.api.contenttypes.FolderContentType;
import com.openexchange.usm.api.contenttypes.UnsupportedContentOperationException;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMIllegalArgumentException;
import com.openexchange.usm.api.session.ChangeState;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.contenttypes.attachments.PIMAttachmentTransferHandler;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject;
import com.openexchange.usm.ox_json.JSONResult;
import com.openexchange.usm.ox_json.JSONResultType;
import com.openexchange.usm.ox_json.OXJSONAccess;

/**
 * Abstract base class for Transfer handler implementations. Contains utility methods
 * and simple default implementations for standard OX access calls.
 *
 * @author afe
 *
 */
public abstract class AbstractTransferHandler implements ContentTypeTransferHandler {
	private static final String RIGHT_HAND_LIMIT = "right_hand_limit";

    private static final String LEFT_HAND_LIMIT = "left_hand_limit";

    private static final String ERROR_PARAMS = "error_params";

	public static final boolean CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ = true;

	private static final String OWN_RIGHTS = "own_rights";

	protected final ContentType _contentType;
	protected final Log _journal;
	protected final OXJSONAccess _ajaxAccess;

	private final PIMAttachmentTransferHandler _pimAttachmentsTransferHandler;

	protected AbstractTransferHandler(ContentType contentType, Log journal, OXJSONAccess ajaxAccess) {
		_contentType = contentType;
		_journal = journal;
		_ajaxAccess = ajaxAccess;
		_pimAttachmentsTransferHandler = new PIMAttachmentTransferHandler(journal, ajaxAccess);
	}

	/**
	 * Creates a comma separated list of columns for the given ContentType.
	 * Each column is specified by a numeric identifier. This method assumes
	 * that the first field is the ID field of the ContentType and makes sure
	 * that it is always added to the list.
	 *
	 * @param requestedFields
	 * @return
	 * @throws UnsupportedContentOperationException
	 */
	protected String createColumnsParameter(BitSet requestedFields) throws UnsupportedContentOperationException {
		int len = checkAndGetRequestedFieldLength(requestedFields);
		StringBuilder columns = new StringBuilder(100);
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < len; i++) {
			if (requestedFields.get(i)) {
				if (fields[i].getFieldID() < 0) {
					throw new UnsupportedContentOperationException(USMContentTypesUtilErrorCodes.NOT_SUPPORTED_FIELD,
							"Field " + fields[i].getFieldName() + " is not supported");
				}
		        if (columns.length() > 0)
		            columns.append(',');
				appendColumns(columns, fields[i]);
			}
		}
		return columns.toString();
	}

    /**
     * Adds columns of a field to the requested fields. Can be overwritten by implementation for special handling of connected data.
     * 
     * @param columns
     * @param field the index
     */
    protected void appendColumns(StringBuilder columns, ContentTypeField field) {
        columns.append(field.getFieldID());
    }

	public void confirm(DataObject object, int response) throws AuthenticationFailedException, OXCommunicationException {
		confirm(object, response, "");
	}

	public void confirm(DataObject object, int response, String confirmMessage) throws AuthenticationFailedException,
			OXCommunicationException {

		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Confirming appointment or task " + object.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));

		JSONObject requestBody = new JSONObject();
		try {
			requestBody.put("confirmation", response);
			requestBody.put("confirmmessage", confirmMessage);
			requestBody.put("id", computeFolderOwnerID(object));
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER1, object,
					"can not create request body for confirming appoinment or task", e);
		}
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_CONFIRM, object.getSession(),
				parameters, requestBody);

		if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER2,
					"Could not confirm appointment or task with id " + object.getID(), result.getJSONObject());
		} else if (result.getResultType() != JSONResultType.JSONObject) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER3,
					"Invalid result type returned, expected JSONObject, was " + result.getResultType());
		} else {
			readOptionalTimestamp(result.getJSONObject(), object);
		}
	}

	public void confirm(DataObject object, int response, String confirmMessage, String participantId)
			throws AuthenticationFailedException, OXCommunicationException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Confirming appointment " + object.getID() + " for participant: "
					+ participantId);
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));

		JSONObject requestBody = new JSONObject();
		try {
			requestBody.put("confirmation", response);
			requestBody.put("confirmmessage", confirmMessage);
			requestBody.put("mail", participantId);
			requestBody.put("type", 5);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER4, object,
					"can not create request body for confirming appoinment for participant", e);
		}
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_CONFIRM, object.getSession(),
				parameters, requestBody);

		if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER5,
					"Could not confirm appointment " + object.getID() + " for participant: " + participantId,
					result.getJSONObject());
		} else if (result.getResultType() != JSONResultType.JSONObject) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER6,
					"Invalid result type returned, expected JSONObject, was " + result.getResultType());
		} else {
			readOptionalTimestamp(result.getJSONObject(), object);
		}
	}

	protected int computeFolderOwnerID(DataObject o) {
		int id = o.getParentFolderOwnerID();
		return (id == 0) ? o.getSession().getUserIdentifier() : id;
	}

	protected int checkAndGetRequestedFieldLength(BitSet requestedFields) {
		int len = requestedFields.length();
		if (len > _contentType.getFields().length)
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.INVALID_FIELDS_REQUESTED,
					"Invalid fields requested");
		return len;
	}

	protected void updateDataObjectFromJSONResult(DataObject destination, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
		int len = checkAndGetRequestedFieldLength(requestedFields);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject jObject = result.getJSONObject();

		JSONObject obj;
		try {
			obj = jObject.getJSONObject(UtilConstants.RESULT_DATA);
		} catch (JSONException e1) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER5, destination,
					"update object: data not present", e1);
		}
		readOptionalTimestamp(jObject, destination);
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < len; i++) {
			if (requestedFields.get(i))
				updateDataObjectFieldFromJSONObject(obj, fields[i], destination);
		}
	}

	protected void readOptionalTimestamp(JSONObject object, DataObject destination) throws OXCommunicationException {
		if (object.has(UtilConstants.RESULT_TIMESTAMP)) {
			try {
				long timestamp = object.getLong(UtilConstants.RESULT_TIMESTAMP);
				destination.setTimestamp(timestamp);
			} catch (JSONException e1) {
				throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER1, destination,
						"update object: invalid timestamp", e1);
			}
		} else {
			// TODO Check if this is the preferred way to handle mail updates
			destination.setTimestamp(System.currentTimeMillis());
		}
	}

	protected void updateDataObjectFromJSONArray(DataObject destination, BitSet requestedFields, JSONArray array)
			throws InternalUSMException {
		int len = checkAndGetRequestedFieldLength(requestedFields);
		int j = 0;
		ContentTypeField[] fields = destination.getContentType().getFields();
		for (int i = 0; i < len; i++) {
			if (requestedFields.get(i)) {
				updateDataObjectFieldFromJSONArray(array, j, fields[i], destination);
				j++;
			}
		}
	}

	/**
	 *
	 * @param source
	 * @param field
	 * @param destination
	 * @throws InternalUSMException
	 * @throws JSONException
	 */
	protected void updateDataObjectFieldFromJSONObject(JSONObject source, ContentTypeField field, DataObject destination)
			throws InternalUSMException {
		String fieldName = field.getFieldName();
		DataType<?> fieldType = field.getFieldType();
		try {
			destination.setFieldContent(fieldName,
					fieldType.extractFromJSONObject(destination.getSession(), source, fieldName));
		} catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER5, "reading", destination,
					fieldName, fieldType, e);
		}
	}

	/**
	 *
	 * @param source
	 * @param index
	 * @param field
	 * @param destination
	 * @throws InternalUSMException
	 */
	protected void updateDataObjectFieldFromJSONArray(JSONArray source, int index, ContentTypeField field,
			DataObject destination) throws InternalUSMException {
		DataType<?> fieldType = field.getFieldType();
		String fieldName = field.getFieldName();
		try {
			destination.setFieldContent(fieldName,
					fieldType.extractFromJSONArray(destination.getSession(), source, index));
		} catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER4, "reading", destination,
					fieldName, fieldType, e);
		}
	}

	protected void throwInternalUSMException(int errorCode, String action, DataObject destination, String fieldName,
			DataType<?> fieldType, JSONException e) throws InternalUSMException {
		String message = "Error while " + action + ' ' + fieldName + " of type " + fieldType.getName() + " in "
				+ destination.getContentType().getID();
		//		_journal.error(destination.getSession() + " " + message, e);
		throw new InternalUSMException(errorCode, message, e);
	}

	protected JSONObject createRequestBody(DataObject object, boolean isOnlyModifiedFields,
			boolean isIdshouldNotBePresent) throws InternalUSMException {
		JSONObject requestBody = new JSONObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		for (int i = 0; i < fields.length; i++) {
			addFieldToRequestBody(object, isOnlyModifiedFields, isIdshouldNotBePresent, requestBody, fields[i], i);
		}
		return requestBody;
	}

	protected void addFieldToRequestBody(DataObject object, boolean isOnlyModifiedFields,
			boolean isIdshouldNotBePresent, JSONObject requestBody, ContentTypeField field, int fieldIndex)
			throws InternalUSMException {

		DataType<?> fieldType = field.getFieldType();
		String fieldName = field.getFieldName();
		// If explicitly forbidden, do not send ID of object
		if (isIdshouldNotBePresent && field.isIDField())
			return;
		// If only modified fields are requested, do not send unmodified fields
		if (isOnlyModifiedFields && !field.isRequiredForTransferToServer() && !object.isFieldModified(fieldIndex))
			return;
		//String fieldType = field.getFieldType();
		Object obj = object.getFieldContent(fieldName);
		if (!isOnlyModifiedFields) {
			// If all fields are requested, do not send those fields that are not present (i.e. null or empty array)
			if (obj == null)
				return;
			if ((obj instanceof Object[]) && ((Object[]) obj).length == 0)
				return;
		}
		try {
			fieldType.storeInJSONObject(object.getSession(), requestBody, fieldName, obj);
		} catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER3, "writing", object,
					fieldName, fieldType, e);
		}
	}

	protected void checkForNotDeletedObjects(DataObject object, JSONResult result) throws USMException {
		//An array with object IDs of folders which were modified after the specified timestamp and were therefore not deleted
		JSONArray notDeletedArray = extractArrayResult(result);
		if (notDeletedArray != null && notDeletedArray.length() > 0) {
			int len = notDeletedArray.length();
			for (int i = 0; i < len; i++) {
				if (_journal.isDebugEnabled()) {
					try {
						_journal.debug(object.getSession() + " " + notDeletedArray.getString(i)
								+ " not deleted, modified after " + object.getTimestamp());
					} catch (JSONException e) {
						_journal.error(object.getSession() + " Can't read ID of not deleted object " + object.getID()
								+ " from JSONResult", e);
					}
				}
			}
			throw new ConflictingChangeException(USMContentTypesUtilErrorCodes.DELETE_ERROR, "Deletion failed",
					result.getJSONObject());
		}
	}

	protected DataObject createDataObject(JSONArray columnsArray, Folder folder, BitSet requestedFields, int idIndex)
			throws InternalUSMException, OXCommunicationException {
		String objId;
		try {
			objId = columnsArray.getString(idIndex);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER3, folder,
					"read/update folder: no id for new " + _contentType.getID() + " in folder " + folder.getID(), e);
		}
		DataObject dataObj = _contentType.newDataObject(folder.getSession());
		dataObj.setID(objId);
		updateDataObjectFromJSONArray(dataObj, requestedFields, columnsArray);
		return dataObj;
	}

	/**
	 * Returns array of DataObjects as returned in the JSON Result.
	 * Updates and deleted objects are handled separately.
	 * @param folder
	 * @param requestedFields
	 * @param result
	 * @return
	 * @throws InternalUSMException
	 * @throws OXCommunicationException
	 */
	protected DataObject[] getListOfResults(Folder folder, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
		JSONArray jsonar = null;
		long timestamp = 0L;
		checkResult(result, JSONResultType.JSONObject);
		JSONObject jsonObject = result.getJSONObject();
		try {
			jsonar = jsonObject.getJSONArray(UtilConstants.RESULT_DATA);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER4, folder,
					"getListOfResults: data array not present", e);
		}
		int length = jsonar.length();
		if (length == 0)
			return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
		try {
			if (jsonObject.has(UtilConstants.RESULT_TIMESTAMP))
				timestamp = jsonObject.getLong(UtilConstants.RESULT_TIMESTAMP);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER2, folder,
					"getListOfResults: invalid timestamp", e);
		}
		DataObject[] dataObjects = new DataObject[length];
		for (int j = 0; j < length; j++) {
			// get array which contains the information specified by the corresponding identifiers in the columns parameter
			try {
				JSONArray columnsArray = jsonar.getJSONArray(j);
				DataObject newDataObject = createDataObject(columnsArray, folder, requestedFields, 0);
				dataObjects[j] = newDataObject;
				newDataObject.setTimestamp(timestamp);
			} catch (JSONException je) {
				throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER1, folder,
						"getListOfResults: error while reading index " + j + " of folder " + folder.getID(), je);
			}
		}
		return dataObjects;
	}

	/**
	 * Returns array of DataObjects as returned in the JSON Result.
	 * Updates and deleted objects are handled separately.
	 * @param folder
	 * @param requestedFields
	 * @param result
	 * @return
	 * @throws InternalUSMException
	 * @throws OXCommunicationException
	 */
	protected DataObject[] getListOfUpdates(Folder folder, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
		JSONArray jsonar = extractArrayResult(result);
		long timestamp;
		try {
			timestamp = result.getJSONObject().getLong(UtilConstants.RESULT_TIMESTAMP);
		} catch (JSONException e) {
			if (_journal.isDebugEnabled())
				_journal.debug(folder.getSession() + " Folder updates: could not read timestamp from result");
			timestamp = 0;
		}
		List<DataObject> dataObjects = new ArrayList<DataObject>();
		int length = jsonar.length();
		for (int j = 0; j < length; j++) {
			try {
				Object o = jsonar.get(j);
				//  case updated object
				if (o instanceof JSONArray) {
					// new object or modified object
					// get array which contain the information specified by the corresponding identifiers in the columns parameter
					JSONArray columnsArray = jsonar.getJSONArray(j);
					DataObject newDataObject = createDataObject(columnsArray, folder, requestedFields, 0);
					dataObjects.add(newDataObject);
					if (timestamp > 0)
						newDataObject.setTimestamp(timestamp);
				} else if (o instanceof String) {
					// deleted object
					String deletedObjectId = jsonar.getString(j);
					DataObject dataObj = _contentType.newDataObject(folder.getSession());
					dataObj.setID(deletedObjectId);
					if (timestamp > 0)
						dataObj.setTimestamp(timestamp);
					dataObj.setChangeState(ChangeState.DELETED);
					dataObjects.add(dataObj);
				} else {
					throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER2, folder,
							"Folder updates contains elements other than JSONArray or String at index " + j
									+ " for folder " + folder.getID(), null);
				}
			} catch (JSONException je) {
				throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER3, folder,
						"Folder updates: error while parsing index " + j + " for folder " + folder.getID(), je);
			}
		}
		return dataObjects.toArray(new DataObject[dataObjects.size()]);
	}

	protected void checkTypeToWrite(DataObject object, String contentTypeID) {
		if (!object.getContentType().getID().equals(contentTypeID))
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.INVALID_CONTENT_TYPE_TO_WRITE,
					"DataObject is no " + contentTypeID);
	}

	protected JSONArray extractArrayResult(JSONResult result) throws OXCommunicationException, InternalUSMException {
		checkResult(result, JSONResultType.JSONObject);
		try {
			return result.getJSONObject().getJSONArray(UtilConstants.RESULT_DATA);
		} catch (JSONException e) {
			_journal.error("OX server did not send data array: " + result.toString());
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER3,
					"OX server did not send data array", e, result.toString());
		}
	}

	protected void checkResult(JSONResult result, JSONResultType expected) throws OXCommunicationException,
			InternalUSMException {
		if (result.getResultType() != expected) {
			switch (result.getResultType()) {
				case Error:
					checkConflictingChange(result.getJSONObject());
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER6,
							result.getJSONObject());
				case JSONArray:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER7,
							"OX server sent array, expected " + expected, result.getJSONArray().toString());
				case JSONObject:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER8,
							"OX server sent object, expected " + expected, result.getJSONObject().toString());
				case String:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER9,
							"OX server sent unknown response, expected " + expected, result.getString());
			}
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER10,
					"Unhandled JSONResultType " + result.getResultType());
		} else {
			checkIMAPDownWarning(result);
		}
	}

	protected void checkIMAPDownWarning(JSONResult result) throws TemporaryDownOrBusyException {
		if (result.getResultType() == JSONResultType.JSONObject && result.getJSONObject().has("code")) {
			try {
				JSONObject errorObject = result.getJSONObject();
				int category = errorObject.optInt(UtilConstants.CATEGORY);
				String message = errorObject.getString(UtilConstants.ERROR);
				String code = errorObject.getString(UtilConstants.CODE);
				JSONArray errorParams = errorObject.getJSONArray(ERROR_PARAMS);
				JSONObject errorDetails = new JSONObject();
				errorDetails.put(UtilConstants.CATEGORY, category);
				errorDetails.put(UtilConstants.ERROR, message);
				errorDetails.put(UtilConstants.CODE, code);
				errorDetails.put(ERROR_PARAMS, errorParams);
				if (OXErrorConstants.MSG_1016.equals(code) || OXErrorConstants.IMAP_1016.equals(code))
					throw new TemporaryDownOrBusyException(USMContentTypesUtilErrorCodes.IMAP_DOWN, message,
							errorDetails);
			} catch (JSONException e) {

			}
		}
	}

	protected void checkConflictingChange(JSONObject errorObject) throws ConflictingChangeException,
			InternalUSMException {
		try {
			int category = errorObject.optInt(UtilConstants.CATEGORY);
			String message = errorObject.getString(UtilConstants.ERROR);
			String code = errorObject.getString(UtilConstants.CODE);
			if (category == OXErrorConstants.CATEGORY_CONFLICTING_CHANGE
					|| (category == OXErrorConstants.CATEGORY_PERMISSION && OXErrorConstants.CON_0119.equals(code))
					|| (category == OXErrorConstants.CATEGORY_TRY_AGAIN && OXErrorConstants.CON_0240.equals(code))) {
				throw new ConflictingChangeException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER11, message,
						errorObject);
			}
		} catch (JSONException e) {
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER12,
					"OX server sent an invalid  error object " + errorObject);
		}
	}

	public void readDataObject(DataObject object, BitSet requestedFields) throws USMException {
		readDataObject(object, requestedFields, new HashMap<String, String>());
	}

	protected void readDataObject(DataObject object, BitSet requestedFields, Map<String, String> parameters)
			throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Read " + _contentType.getID() + " with id " + object.getID());
		checkTypeToWrite(object, _contentType.getID());
		parameters.put(UtilConstants.ID, object.getID());
		if (_contentType.canBeFolderElementsContentType())
			parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_GET, object.getSession(),
				parameters);
		checkResult(result, JSONResultType.JSONObject);
		updateDataObjectFromJSONResult(object, requestedFields, result);
	}

	public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
		return readStandardFolderContent(UtilConstants.FOLDER, UtilConstants.ACTION_ALL, folder, requestedFields, null,
				null);
	}

	protected DataObject[] readStandardFolderContent(String folderIDKey, String action, Folder folder,
			BitSet requestedFields) throws USMException {
		return readStandardFolderContent(folderIDKey, action, folder, requestedFields, null, null);
	}

	protected DataObject[] readStandardFolderContent(String folderIDKey, String action, Folder folder,
			BitSet requestedFields, String sort, String order) throws USMException {
		return readStandardFolderContent(folderIDKey, action, folder, requestedFields, sort, order, null, null);
	}

	protected DataObject[] readStandardFolderContent(String folderIDKey, String action, Folder folder,
			BitSet requestedFields, String sort, String order, String start, String end) throws USMException {
		return readStandardFolderContent(folderIDKey, action, folder, requestedFields, sort, order, start, end, null);
	}

	protected DataObject[] readStandardFolderContent(String folderIDKey, String action, Folder folder,
			BitSet requestedFields, String sort, String order, String start, String end, String folderTree)
			throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(folderIDKey, folder.getID());
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		if (sort != null) {
			parameters.put(UtilConstants.SORT, sort);
			if (order != null)
				parameters.put(UtilConstants.ORDER, order);
		}
		if (start != null) {
			parameters.put(UtilConstants.START, start);
			if (end != null)
				parameters.put(UtilConstants.END, end);
		}
		if (folderTree != null) {
			parameters.put(UtilConstants.TREE, folderTree);
		}
		if(folder.getSession().getObjectsLimit() > 0) {
            addObjectsLimitParameters(folder.getSession().getObjectsLimit(), parameters);
        }
		return performActionAndCheckRights(action, folder, requestedFields, parameters);
	}

	protected DataObject[] performActionAndCheckRights(String action, Folder folder, BitSet requestedFields,
			Map<String, String> parameters) throws AuthenticationFailedException, InternalUSMException, USMException,
			OXCommunicationException {
		if (CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ && !hasFolderReadRights(folder))
			return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
		try {
			JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), action, folder.getSession(), parameters);
			return getListOfResults(folder, requestedFields, result);
		} catch (OXCommunicationException e) {
			//catch the OXCommunication exception and than get the access rights
			if (hasFolderReadRights(folder))
				throw e;
			return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
		}
	}

	protected boolean hasFolderReadRights(Folder folder) throws USMException {
		BitSet requestedFields = new BitSet();
		int fieldIndex = folder.getFieldIndex(OWN_RIGHTS);
		requestedFields.set(fieldIndex);
		folder.getContentType().getTransferHandler().readFolder(folder, requestedFields);
		int own_rights = ((Number) folder.getFieldContent(fieldIndex)).intValue();
		return (own_rights & 0x3F80) != 0;
	}

	public void writeDeletedDataObject(DataObject object) throws USMException {
		writeExtendedDeleteDataObject(object, UtilConstants.FOLDER, object.getParentFolderID());
	}

	protected void writeExtendedDeleteDataObject(DataObject object, String... extraKeyValuePairs) throws USMException {
		if ((extraKeyValuePairs.length & 1) != 0)
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER4,
					"Extra key identifier without following field name");
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Delete " + _contentType.getID() + " with id " + object.getID()
					+ Arrays.toString(extraKeyValuePairs));
		checkTypeToWrite(object, _contentType.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
		JSONObject jObj = new JSONObject();
		try {
			jObj.put(UtilConstants.ID, object.getID());
			for (int i = 0; i < extraKeyValuePairs.length; i += 2) {
				jObj.put(extraKeyValuePairs[i], extraKeyValuePairs[i + 1]);
			}
		} catch (JSONException e) {
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER5,
					"Duplicate or null entry in provided key/value pairs");
		}
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_DELETE, object.getSession(),
				parameters, jObj);
		checkForNotDeletedObjects(object, result);
		readOptionalTimestamp(result.getJSONObject(), object);
	}

	public void writeNewDataObject(DataObject object) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " New " + _contentType.getID());
		checkTypeToWrite(object, _contentType.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		JSONObject requestBody = createRequestBody(object, false, true);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_NEW, object.getSession(),
				parameters, requestBody);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		readOptionalTimestamp(o, object);
		try {
			object.setID(o.getJSONObject(UtilConstants.RESULT_DATA).getString(UtilConstants.RESULT_ID));
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER4, object,
					"Missing result ID on new " + _contentType.getID(), e);
		}
	}

	public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
		writeStandardUpdateDataObject(object, timestamp, false);
	}

	protected void writeStandardUpdateDataObject(DataObject object, long timestamp, boolean isFolder)
			throws InternalUSMException, AuthenticationFailedException, OXCommunicationException,
			FolderNotFoundException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Updating " + _contentType.getID());
		checkTypeToWrite(object, _contentType.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		String folderTree = object.getSession().getPersistentField(FolderContentType.FOLDER_TREE);
		if (!isFolder) {
			parameters.put(UtilConstants.FOLDER, object.getOriginalParentFolderID());
		} else if (folderTree != null) {
			parameters.put(UtilConstants.TREE, folderTree);
		}
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
		JSONObject requestBody = createRequestBody(object, true, false);
		//		printTimeValues("Change", requestBody);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_UPDATE, object.getSession(),
				parameters, requestBody);
		if (isFolder) {
			if (result.getResultType() == JSONResultType.Error) {
				if (isFolderNotExistError(result.getJSONObject()))
					throw new FolderNotFoundException(USMContentTypesUtilErrorCodes.FOLDER_NOT_FOUND,
							"Folder not found: " + object.getUUID() + '(' + object.getID() + ')');
			}
			checkResult(result, JSONResultType.JSONObject);
			JSONObject o = result.getJSONObject();
			if (o.has(UtilConstants.RESULT_DATA)) {
				try {
					object.setID(o.getString(UtilConstants.RESULT_DATA));
				} catch (JSONException e) {
					throw generateException(USMContentTypesUtilErrorCodes.DATA_INVALID, object,
							"update object: invalid folder id", e);
				}
			}
			readOptionalTimestamp(o, object);
		} else {
			checkResult(result, JSONResultType.JSONObject);
			// XXX Shouldn't we use the timestamp returned from the OX server ?
			object.setTimestamp(timestamp);
		}
	}

	private boolean isFolderNotExistError(JSONObject errorObject) throws InternalUSMException {
		try {
			int category = errorObject.optInt(UtilConstants.CATEGORY);
			String code = errorObject.getString(UtilConstants.CODE);
			if (category == OXErrorConstants.CATEGORY_CODE_ERROR && OXErrorConstants.IMAP_1002.equals(code)) {
				return true;
			}
		} catch (JSONException e) {
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER13,
					"OX server sent an invalid  error object " + errorObject);
		}
		return false;
	}

	// For debugging purposes only
	//	protected void printTimeValues(String action, JSONObject requestBody) {
	//		printTimeValue(action, requestBody, "start_date");
	//		printTimeValue(action, requestBody, "end_date");
	//	}

	//	private void printTimeValue(String action, JSONObject requestBody, String key) {
	//		Long st = requestBody.optLong(key);
	//		if (st != null) {
	//			SimpleDateFormat format = new SimpleDateFormat("EEEE, dd.MM.yyyy HH:mm:ss.SSS z '('Z')'");
	//			double d = ((double) st) / 86400000.0;
	//			System.out.println(action + ": " + st + "(" + d + ")" + " -> " + format.format(new Date(st)));
	//		}
	//	}

	public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp)
			throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, UtilConstants.ACTION_UPDATES,
				UtilConstants.FOLDER);
	}

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey) throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, action, folderKey, null, null,
				null, null);
	}

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey, String sort, String order, String start, String end) throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, action, folderKey, sort, order,
				start, end, null);
	}

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey, String sort, String order, String start, String end, String folderTree)
			throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(folder.getSession() + " Read updated " + _contentType.getID() + " after " + timestamp
					+ " in folder " + folder.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(folderKey, folder.getID()); //object ID of the folder
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
		if (sort != null) {
			parameters.put(UtilConstants.SORT, sort);
			if (order != null)
				parameters.put(UtilConstants.ORDER, order);
		}
		if (start != null) {
			parameters.put(UtilConstants.START, start);
			if (end != null)
				parameters.put(UtilConstants.END, end);
		}
		if (folderTree != null) {
			parameters.put(UtilConstants.TREE, folderTree);
		}
        if (folder.getSession().getObjectsLimit() > 0) {
            addObjectsLimitParameters(folder.getSession().getObjectsLimit(), parameters);
        }
		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), action, folder.getSession(), parameters);
		return getListOfUpdates(folder, requestedFields, result);
	}

	protected OXCommunicationException generateException(int errorCode, DataObject obj, String message, Throwable cause) {
		//		_journal.error(obj.getSession() + " " + errorCode + ":" + message, cause);
		return new OXCommunicationException(errorCode, message, cause);
	}

	protected void resetRecurrenceTypeBasedFields(DataObject[] results) {
		/**recurrentType
		 * 0 	 none (single event)
		   1 	daily
		   2 	weekly
		   3 	monthly
		   4 	yearly
		 *
		 */
		for (DataObject data : results) {
			int recurrenceType = 0;
			if (data.getFieldContent("recurrence_type") != null)
				recurrenceType = ((Number) data.getFieldContent("recurrence_type")).intValue();
			//single event
			if (recurrenceType == 0) {
				data.setFieldContent("until", null);
				data.setFieldContent("interval", null);
				//				data.setFieldContent("recurrence_position", null);
				//				data.setFieldContent("recurrence_date_position", null);
				data.setFieldContent("occurrences", null);
			}

			if (recurrenceType <= 1) {
				data.setFieldContent("days", null);
			}
			//			else {
			//				if (data.getFieldContent("days") == null)
			//					data.setFieldContent("days", 127);
			//			}

			if (recurrenceType <= 2) {
				data.setFieldContent("day_in_month", null);
			}

			if (recurrenceType == 4) {
				if (recurrenceType > 0)
					data.setFieldContent("interval", 1);
			} else {
				data.setFieldContent("month", null);
			}
		}
	}

	protected void setAppOrTaskConfirmations(DataObject object) throws AuthenticationFailedException,
			OXCommunicationException {
		if (object.isFieldModified(CommonCalendarTasksFieldNames.USERS)) {
			int folderOwner = computeFolderOwnerID(object);
			Object[] users = (Object[]) object.getFieldContent(CommonCalendarTasksFieldNames.USERS);
			Object[] usersOld = (Object[]) object.getOriginalFieldContent(CommonCalendarTasksFieldNames.USERS);
			for (Object user : users) {
				UserParticipantObject participant = (UserParticipantObject) user;
				int confirmation = participant.getConfirmation();
				int id = participant.getId();
				if (/*confirmation != 0 &&*/id == folderOwner) {
					for (Object userOld : usersOld) {
						int idOld = ((UserParticipantObject) userOld).getId();
						if (id == idOld) {
							int confirmationOld = ((UserParticipantObject) userOld).getConfirmation();
							if (confirmationOld != confirmation) {
								confirm(object, confirmation);
								object.commitChanges();
							}
							break;
						}
					}
				}
			}
		}
	}

	protected abstract String getOXAjaxAccessPath();

	public long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException {
		return _pimAttachmentsTransferHandler.createNewAttachment(object, attachment);
	}

	public PIMAttachments getAllAttachments(DataObject object, String sort, String order) throws USMException {
		return _pimAttachmentsTransferHandler.getAllAttachments(object, sort, order);
	}

	public PIMAttachments getAllAttachments(DataObject object) throws USMException {
		return _pimAttachmentsTransferHandler.getAllAttachments(object);
	}

	public long deleteAttachments(DataObject object, PIMAttachment[] attachmentsToDelete) throws USMException {
		return _pimAttachmentsTransferHandler.deleteAttachments(object, attachmentsToDelete);
	}

	public byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException {
		return _pimAttachmentsTransferHandler.getAttachmentData(object, attachmentId);
	}
	
    protected void addObjectsLimitParameters(int limit, Map<String, String> parameters) {
        parameters.put(LEFT_HAND_LIMIT, "0");
        parameters.put(RIGHT_HAND_LIMIT, String.valueOf(limit));
        if (!parameters.containsKey(UtilConstants.SORT)) {
            parameters.put(UtilConstants.SORT, "5"); // sort by last modified
            parameters.put(UtilConstants.ORDER, "desc");
        }
    }
}
