/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.USMSessionID;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.Version;

/**
 * Handler for the Login USM-JSON-Command.
 * 
 * @author ldo
 */
public class LoginHandler extends BaseCommandHandler {

    private static final String USM_JSON_BUILD = "usmJSONBuild";

    private static final String USM_JSON_VERSION = "usmJSONVersion";

    private static final String USM_BUILD = "usmBuild";

    private static final String USM_VERSION = "usmVersion";

    private static final String[] REQUIRED_PARAMETERS = { USER, DEVICE, PASSWORD };

    private static final String[] OPTIONAL_PARAMETERS = { CONFIGURATION, AVAILABLE_CONTENT_TYPES, SYNC_CONTENT_TYPES, VERSION, LOCAL_TIME };

    private static final long ONE_HOUR = 60 * 60 * 1000L;

    private static SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

    public LoginHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, httpSession, parameters, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        JSONObject resultObject = new JSONObject();
        checkClientTime();
        retrieveSession();
        String randomHttpSessionID = generateRandomSessionID();
        USMSessionID usmSessionID = new USMSessionID(_session.getUser(), _session.getDevice(), _session.getPassword());
        JSONObject configuration = getBooleanParameter(CONFIGURATION) ? getConfiguration() : null;
        Version version = checkVersion();
        try {
            resultObject.put(SESSIONID, randomHttpSessionID); // put the random generated session id in the response object
            resultObject.put(VERSION, version.toString());
            resultObject.put(USERID, _session.getUserIdentifier());
            if (configuration != null)
                resultObject.put(CONFIGURATION, configuration);
            resultObject.put(USM_VERSION, USMVersion.VERSION);
            resultObject.put(USM_BUILD, USMVersion.BUILD_NUMBER);
            resultObject.put(USM_JSON_VERSION, USMJSONVersion.VERSION);
            resultObject.put(USM_JSON_BUILD, USMJSONVersion.BUILD_NUMBER);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.LOGIN_JSON_ERROR, e);
        }
        _httpSession.setAttribute(randomHttpSessionID, usmSessionID); // save the USM session in the HttpSession
        return new ResponseObject(ResponseStatusCode.SUCCESS, resultObject);
    }

    private Version checkVersion() throws USMJSONAPIException {
        Version requestedVersion = getVersionParameter();
        Version availableVersion = Version.fromString(USMJSONVersion.API_VERSION);
        if (requestedVersion != null) {
            if (requestedVersion.getMajor() != availableVersion.getMajor() || requestedVersion.getMinor() > availableVersion.getMinor()) {
                JSONObject details = new JSONObject();
                try {
                    details.put("available", availableVersion.toString());
                    details.put("requested", requestedVersion.toString());
                } catch (JSONException ignored) {
                    // can not happen
                }
                throw new USMJSONAPIException(
                    ConnectorBundleErrorCodes.LOGIN_UNSUPPORTED_VERSION,
                    ResponseStatusCode.UNSUPPORTED_VERSION,
                    "Unsupported API version",
                    details);
            }
        }
        return availableVersion;
    }

    private void checkClientTime() throws USMJSONAPIException {
        Date clientTime = getClientTimeParameter();
        if (clientTime == null)
            return;
        if (clientTime.getTime() >= (System.currentTimeMillis() + ONE_HOUR))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.LOGIN_CLIENT_TIME_TOO_NEW,
                ResponseStatusCode.UNALLOWED_CLIENT_TIME,
                "Local time is at least one hour later than server time.");
        if (clientTime.getTime() + ONE_HOUR <= (System.currentTimeMillis()))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.LOGIN_CLIENT_TIME_TOO_OLD,
                ResponseStatusCode.UNALLOWED_CLIENT_TIME,
                "Local time is at least one hour earlier than server time.");
    }

    private Date getClientTimeParameter() throws USMJSONAPIException {
        String timeParam = getStringParameter(LOCAL_TIME);
        if (timeParam == null)
            return null;
        try {
            DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("GMT"));
            Date clientTime = DATE_FORMAT.parse(timeParam);
            return clientTime;
        } catch (ParseException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.LOGIN_INVALID_DATE,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid local time",
                e);
        }
    }

    private Version getVersionParameter() throws USMJSONAPIException {
        try {
            String s = getStringParameter(VERSION);
            return (s == null) ? null : Version.parse(s);
        } catch (IllegalArgumentException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.LOGIN_INVALID_VERSION,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Invalid version",
                e);
        }
    }

    private void retrieveSession() throws USMJSONAPIException {
        try {
            setSession(_servlet.getSession(
                getStringParameter(USER),
                getStringParameter(PASSWORD),
                getStringParameter(DEVICE),
                getClientIP(),
                getXHeadersFromCurrentRequest()));
            _session.setXRequestHeaders(getXHeadersFromCurrentRequest());
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(e.getErrorCode(), e);
        }
    }

    /**
     * Generates a session-id to be returned to the client.
     * 
     * @return
     */
    private String generateRandomSessionID() {
        return new StringBuilder(48).append(UUID.randomUUID()).append('_').append(_session.getContextId()).append('_').append(
            _session.getSessionId()).toString();
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }
}
