/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.clt;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXServiceURL;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

/**
 * {@link USMSessionTool} Tool that displays information about USM sessions and allows enabling/disabling of the EAS debug logging.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMSessionTool {

    private static final String _USM_PROTOCOL_JSON = "JSON";

    private static final String _USM_PROTOCOL_EAS = "EAS";

    private static final String _USM_FIELD_EAS_DEBUG_LOG = "eas.DebugLog";

    private static final Object _EAS_USER_PARAMETER = "User";

    private static final Object _EAS_DEVICE_ID_PARAMETER = "DeviceId";

    private static final Options toolOptions;

    static {
        final Options opts = new Options();
        opts.addOption("h", "help", false, "Prints a help text");
        opts.addOption("l", "login", true, "The optional JMX login (if JMX has authentication enabled)");
        opts.addOption("p", "port", true, "The optional JMX port (default:9999)");
        opts.addOption("s", "password", true, "The optional JMX password (if JMX has authentication enabled)");
        OptionGroup easLogging = new OptionGroup();
        easLogging.addOption(new Option(
            "e",
            "enableEASDebugLog",
            true,
            "Enable EAS debug logging with the given keyword in the log messages"));
        easLogging.addOption(new Option("d", "disableEASDebugLog", false, "Disable EAS debug logging"));
        easLogging.addOption(new Option("r", "reportEASDebugLog", false, "Report if (and under which keyword) EAS debug logging is enabled"));
        easLogging.setRequired(false);
        opts.addOptionGroup(easLogging);
        OptionGroup verbosity = new OptionGroup();
        verbosity.addOption(new Option("v", "verbose", false, "Verbose output: Report all persistent fields for matching USM sessions"));
        verbosity.addOption(new Option(
            "q",
            "quiet",
            false,
            "Quiet mode: Only print errors/warnings. If -r is specified, print only EAS debug logging keyword (or empty line if EAS debug logging is disabled)"));
        verbosity.setRequired(false);
        opts.addOptionGroup(verbosity);
        toolOptions = opts;
    }

    private static void printHelpAndExit() {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.printHelp(
            "usmsessions [USM-Request-URL-parameters or USM-Session-filter]+",
            "",
            toolOptions,
            "Options -d,-e,-r are mutually exclusive, as are options -q,-v.\nArguments may be given as EAS-POST-URL-parameters, OLOX2-PUT-URL-parameters, or an USM-session-filter.\nAn USM-session-filter consists of 5 parts, separated by colon(\":\"):\nContext-ID:OX-User-ID:USM-Session-ID:USM-Protocol:Device-ID\nEach part may be left empty to match any value.\nCurrently available USM-protocols are: EAS, JSON.");
        System.exit(0);
    }

    private static String getOptionStringValue(CommandLine cmd, char opt) throws ParseException {
        String val = cmd.getOptionValue(opt);
        if (!provided(val))
            throw new ParseException("Option \"" + opt + "\" requires a non-empty string value");
        return val;
    }

    private static int getOptionIntValue(CommandLine cmd, char opt, int minValue, int maxValue) throws ParseException {
        String val = cmd.getOptionValue(opt);
        if (val != null) {
            try {
                int intVal = Integer.parseInt(val);
                if (intVal >= minValue && intVal <= maxValue)
                    return intVal;
            } catch (NumberFormatException ignored) {
                // fall through
            }
        }
        throw new ParseException("Option \"" + opt + "\" requires an integer value between " + minValue + " and " + maxValue);
    }

    private static int parseIntegerValue(String part) {
        return provided(part) ? Integer.parseInt(part) : 0;
    }

    private static boolean provided(String s) {
        return s != null && s.length() > 0;
    }

    public static void main(String[] args) {
        new USMSessionTool(args).execute();
    }

    private String _jmxLogin;

    private String _jmxPassword;

    private int _jmxPort = 9999;

    private boolean _verbose;

    private String _enableEASDebugLogging;

    private boolean _disableLogging;

    private boolean _quiet;

    private boolean _reportEasDebugLog;

    private List<String> _data = new ArrayList<String>();

    private MBeanServerConnection _connection;

    private USMSessionTool(String[] args) {
        CommandLineParser parser = new PosixParser();
        CommandLine cmd;
        try {
            cmd = parser.parse(toolOptions, args);
            if (cmd.hasOption('h')) {
                printHelpAndExit();
            }
            if (cmd.hasOption('l'))
                _jmxLogin = getOptionStringValue(cmd, 'l');
            if (cmd.hasOption('p'))
                _jmxPort = getOptionIntValue(cmd, 'p', 1, 65535);
            if (cmd.hasOption('s'))
                _jmxPassword = cmd.getOptionValue('s');
            if (cmd.hasOption('e'))
                _enableEASDebugLogging = cmd.getOptionValue('e');
            else if (cmd.hasOption('d'))
                _disableLogging = true;
            else if (cmd.hasOption('r'))
                _reportEasDebugLog = true;
            if (cmd.hasOption('v'))
                _verbose = true;
            else if (cmd.hasOption('q'))
                _quiet = true;
            String[] data = cmd.getArgs();
            if (data == null || data.length == 0) {
                System.err.println("No arguments specified");
                printHelpAndExit();
            }
            for (String entry : data)
                _data.add(entry);
        } catch (ParseException e) {
            System.err.println("Unable to parse command line: " + e.getMessage());
            printHelpAndExit();
        }
    }

    private void execute() {
        try {
            final JMXServiceURL url = new JMXServiceURL(
                new StringBuilder("service:jmx:rmi:///jndi/rmi://localhost:").append(_jmxPort).append("/server").toString());
            final JMXConnector jmxConnector = JMXConnectorFactory.connect(url, createJMXEnvironment());
            try {
                _connection = jmxConnector.getMBeanServerConnection();
                for (String data : _data) {
                    if (handleCustomData(data))
                        continue;
                    Map<String, String> urlParameters = extractURLParameters(data);
                    if (handleActiveSyncData(urlParameters))
                        continue;
                    if (handleOlox2Data(urlParameters))
                        continue;
                    // Option: Add more parsers here
                    System.err.println("Ignoring unrecognized data: " + data);
                }
            } finally {
                jmxConnector.close();
            }
        } catch (Exception e) {
            System.err.println("An error occurred while executing your request:");
            e.printStackTrace();
            printHelpAndExit();
        }
    }

    private boolean handleCustomData(String data) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        String[] parts = data.split(":", -1);
        if (parts.length == 5) {
            try {
                int cid = parseIntegerValue(parts[0]);
                int id = parseIntegerValue(parts[1]);
                int usmSessionId = parseIntegerValue(parts[2]);
                workOnUSMSession(cid, id, usmSessionId, parts[3], parts[4]);
                return true;
            } catch (NumberFormatException nfe) {
                System.err.println("Warning: Found possible custom parameter with correct structure but invalid data: " + nfe.getMessage());
            }
        }
        return false;
    }

    private boolean handleActiveSyncData(Map<String, String> urlParameters) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        String user = urlParameters.get(_EAS_USER_PARAMETER);
        String deviceId = urlParameters.get(_EAS_DEVICE_ID_PARAMETER);
        if (!provided(user) || !provided(deviceId))
            return false;
        workOnUSMSession(0, 0, 0, _USM_PROTOCOL_EAS, deviceId);
        return true;
    }

    private boolean handleOlox2Data(Map<String, String> urlParameters) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        for (String paramValue : urlParameters.values()) {
            String[] parts = paramValue.split("_");
            if (parts.length == 3) {
                int cid, usmSessionId;
                try {
                    UUID.fromString(parts[0]); // To check for correct syntax
                    cid = Integer.parseInt(parts[1]);
                    usmSessionId = Integer.parseInt(parts[2]);
                } catch (Exception e) {
                    System.err.println("Warning: Found possible OLOX2-parameter with correct structure but invalid data: " + e.getMessage());
                    continue;
                }
                workOnUSMSession(cid, 0, usmSessionId, _USM_PROTOCOL_JSON, null);
                return true;
            }
        }
        return false;
    }

    private void workOnUSMSession(int cid, int userId, int usmSessionId, String protocol, String deviceId) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object[] params = new Object[] { cid, userId, protocol, deviceId, null };
        String[] signature = new String[] {
            int.class.getName(), int.class.getName(), String.class.getName(), String.class.getName(), String.class.getName() };
        String[] result = (String[]) _connection.invoke(
            JMXObjectNames.USM_SESSION_INFO_MBEAN,
            "getPropertiesOfAllSessions",
            params,
            signature);
        List<JMXUSMSessionInfo> sessions = JMXUSMSessionInfo.fromJMXResult(result);
        for (JMXUSMSessionInfo session : sessions) {
            if (usmSessionId == 0 || session.getUsmSessionId() == usmSessionId) {
                if (_verbose) {
                    System.out.println("Session: " + session);
                    reportEASDebugLogging(session);
                    for (Entry<String, String> entry : session.getFields().entrySet()) {
                        System.out.println("   " + entry.getKey() + " = " + entry.getValue());
                    }
                } else if (_quiet) {
                    reportEASDebugLogging(session);
                } else {
                    System.out.println(session);
                    reportEASDebugLogging(session);
                }
                if (provided(_enableEASDebugLogging)) {
                    if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
                        setSessionProperty(session, _USM_FIELD_EAS_DEBUG_LOG, _enableEASDebugLogging);
                        if (_verbose)
                            System.out.println("--> EAS-Debug-Logging activated with keyword \"" + _enableEASDebugLogging + "\"");
                    } else
                        System.err.println("Warning: Ignoring option -e for non-EAS-session " + session);
                } else if (_disableLogging) {
                    if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
                        setSessionProperty(session, _USM_FIELD_EAS_DEBUG_LOG, "");
                        if (_verbose)
                            System.out.println("--> EAS-Debug-Logging deactivated");
                    } else
                        System.err.println("Warning: Ignoring option -d for non-EAS-session " + session);
                }
                if (_verbose)
                    System.out.println();
            }
        }
    }

    private void reportEASDebugLogging(JMXUSMSessionInfo session) {
        if (!_reportEasDebugLog)
            return;
        if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
            String easLogging = session.getFields().get(_USM_FIELD_EAS_DEBUG_LOG);
            if (_quiet)
                System.out.println(provided(easLogging) ? easLogging : "");
            else if (provided(easLogging))
                System.out.println("   EAS Debug Logging is activated with keyword \"" + easLogging + "\"");
            else
                System.out.println("   EAS Debug Logging is not activated");
        } else
            System.err.println("Warning: Ignoring option -r for non-EAS-session " + session);
    }

    private void setSessionProperty(JMXUSMSessionInfo session, String field, String value) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object[] params = new Object[] { session.getCID(), session.getID(), session.getProtocol(), session.getDevice(), field, value };
        String[] signature = new String[] {
            int.class.getName(), int.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(),
            String.class.getName() };
        String[] result = (String[]) _connection.invoke(
            JMXObjectNames.USM_SESSION_INFO_MBEAN,
            "updatePropertiesOfAllSessions",
            params,
            signature);
        if (result.length == 1) {
            String expectedResult = session.getCID() + ": >> 1 sessions updated <<";
            if (result[0].equals(expectedResult))
                return;
        }
        if (result.length == 0) {
            System.err.println("Warning: Empty result received");
        } else {
            System.err.println("Warning: Received unexpected result from update:");
            for (String line : result)
                System.err.println(line);
        }
    }

    private static Map<String, String> extractURLParameters(String data) {
        Map<String, String> result = new HashMap<String, String>();
        String params = cropToParameters(data);
        for (String entry : params.split("&")) {
            String[] parts = entry.split("=", 2);
            if (parts.length >= 2) {
                try {
                    result.put(parts[0], URLDecoder.decode(parts[1], "UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    throw new IllegalStateException("Required encoding UTF-8 is not supported", e);
                }
            } else if (parts.length >= 1) {
                result.put(parts[0], "");
            }
        }
        return result;
    }

    private static String cropToParameters(String data) {
        int paramStart = data.indexOf('?');
        if (paramStart >= 0)
            data = data.substring(paramStart + 1);
        int paramEnd1 = data.indexOf('#');
        int paramEnd2 = data.indexOf(' ');
        if (paramEnd1 >= 0 && paramEnd2 >= paramEnd1)
            return data.substring(0, paramEnd1);
        if (paramEnd2 < 0)
            return data;
        return data.substring(0, paramEnd2);
    }

    private Map<String, Object> createJMXEnvironment() {
        Map<String, Object> environment = new HashMap<String, Object>();
        if (_jmxLogin != null && _jmxPassword != null)
            environment.put(JMXConnectorServer.AUTHENTICATOR, new JMXAuthenticatorImpl(_jmxLogin, _jmxPassword));
        return environment;
    }
}
