/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2013 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.git.data;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Parses a single conflict in a {@link File} and provides methods to handle the conflict.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class Conflict {

    private static final String CONFLICT_START = "^<<<<<<< ([\\w-]*)$";
    private static final String CONFLICT_MIDDLE = "^=======$";
    private static final String CONFLICT_END = "^>>>>>>> ([\\w-]*)$";

    private static final Pattern PATTERN_START = Pattern.compile(CONFLICT_START);
    private static final Pattern PATTERN_MIDDLE = Pattern.compile(CONFLICT_MIDDLE);
    private static final Pattern PATTERN_END = Pattern.compile(CONFLICT_END);

    private final List<String> here = new ArrayList<String>();
    private final List<String> there = new ArrayList<String>();
    private int startPos;
    private int endPos;
    private String hereName;
    private String thereName;

    public Conflict() {
        super();
    }

    public static Conflict parseConflict(List<String> lines, int startPos) {
        int pos = startPos;
        Conflict retval = new Conflict();
        while (pos < lines.size()) {
            Matcher startMatcher = PATTERN_START.matcher(lines.get(pos));
            if (startMatcher.matches()) {
                retval.hereName = startMatcher.group(1);
                retval.startPos = pos++;
                Matcher middleMatcher = PATTERN_MIDDLE.matcher(lines.get(pos));
                while (!middleMatcher.matches()) {
                    retval.here.add(lines.get(pos++));
                    middleMatcher = PATTERN_MIDDLE.matcher(lines.get(pos));
                }
                pos++;
                Matcher endMatcher = PATTERN_END.matcher(lines.get(pos));
                while (!endMatcher.matches()) {
                    retval.there.add(lines.get(pos++));
                    endMatcher = PATTERN_END.matcher(lines.get(pos));
                }
                retval.endPos = pos;
                retval.thereName = endMatcher.group(1);
                return retval;
            }
            pos++;
        }
        return null;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("<<<<<<< ");
        sb.append(hereName);
        sb.append("\n");
        for (String line : here) {
            sb.append(line);
            sb.append('\n');
        }
        sb.append("=======\n");
        for (String line : there) {
            sb.append(line);
            sb.append('\n');
        }
        sb.append(">>>>>>> ");
        sb.append(thereName);
        sb.append('\n');
        return sb.toString();
    }

    public int getLastLine() {
        return endPos;
    }

    public List<String> getHere() {
        return here;
    }

    public List<String> getThere() {
        return there;
    }

    public int getFirstLine() {
        return startPos;
    }

    public String getThereName() {
        return thereName;
    }
}
