/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * {@link USMCacheInformation}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMCacheInformation {

    private final SessionManagerImpl _sessionManager;

    private final AtomicInteger _newSessions = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedDueToInactivity = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedByJVM = new AtomicInteger();

    private final AtomicInteger _syncStatesRemovedDueToInactivity = new AtomicInteger();

    private final AtomicInteger _syncStateSavedToDatabase = new AtomicInteger();

    private final AtomicInteger _syncStatesLoadedFromDatabase = new AtomicInteger();

    public USMCacheInformation(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    public int getUSMSesssionCacheInactivityTimeout() {
        return _sessionManager.getSessionCacheTimeout();
    }

    public void setUSMSesssionCacheInactivityTimeout(int timeout) {
        _sessionManager.setSessionCacheTimeout(timeout);
    }

    public int getSyncStateCacheInactivityTimeout() {
        return _sessionManager.getSyncStateCacheTimeout();
    }

    public void setSyncStateCacheInactivityTimeout(int timeout) {
        _sessionManager.setSyncStateCacheTimeout(timeout);
    }

    public int getTotalUSMSessionsInMemory() {
        return _sessionManager.getSessionStorage().getSessionCount();
    }

    public int getTotalSyncStatesInMemory() {
        int count = 0;
        for (SessionReference reference : _sessionManager.getSessionStorage().getSessionReferences()) {
            SessionImpl session = reference.poll();
            if (session != null)
                count += session.getDataObjectCache().getSyncStateCount();
        }
        return count;
    }

    public void clearCacheCounters() {
        _newSessions.set(0);
        _sessionsRemovedDueToInactivity.set(0);
        _sessionsRemovedByJVM.set(0);
        _syncStatesRemovedDueToInactivity.set(0);
        _syncStateSavedToDatabase.set(0);
        _syncStatesLoadedFromDatabase.set(0);
    }

    public int getNewUSMSessionsInMemory() {
        return _newSessions.get();
    }

    void newSessionCreated() {
        _newSessions.incrementAndGet();
    }

    public int getUSMSessionsRemovedDueToInactivity() {
        return _sessionsRemovedDueToInactivity.get();
    }

    void sessionsRemovedDueToInactivity(int count) {
        _sessionsRemovedDueToInactivity.addAndGet(count);
    }

    public int getUSMSessionsRemovedByJVM() {
        return _sessionsRemovedByJVM.get();
    }

    public void sessionsRemovedByJVM(int jvmRemovals) {
        _sessionsRemovedByJVM.addAndGet(jvmRemovals);
    }

    public int getSyncStatesRemovedDueToInactivity() {
        return _syncStatesRemovedDueToInactivity.get();
    }

    public void syncStatesRemovedDueToInactivity(int removedSyncStates) {
        _syncStatesRemovedDueToInactivity.addAndGet(removedSyncStates);
    }

    public int getSyncStatesLoadedFromDatabase() {
        return _syncStatesLoadedFromDatabase.get();
    }

    public void syncStatesLoadedFromDatabase(int count) {
        _syncStatesLoadedFromDatabase.addAndGet(count);
    }

    public int getSyncStatesSavedToDatabase() {
        return _syncStateSavedToDatabase.get();
    }

    public void syncStateSavedToDatabase() {
        _syncStateSavedToDatabase.incrementAndGet();
    }
}
