/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.calendar.impl;

import static com.openexchange.usm.api.datatypes.DataTypes.*;

import java.util.BitSet;

import com.openexchange.usm.api.contenttypes.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.ContentTypeTransferHandler;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.datatypes.PIMAttachmentsDataType;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.AbstractContentType;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.ConfirmingParticipantDataType;
import com.openexchange.usm.datatypes.tasks.calendar.ParticipantIdentifierDataType;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObjectDataType;

public class AppointmentContentTypeImpl extends AbstractContentType implements AppointmentContentType {

	private final static ContentTypeField[] FIELDS = {
			//Common object data
			new ContentTypeField(1, UtilConstants.FIELD_ID, STRING, false, true, false, false), //Object ID
			new ContentTypeField(20, UtilConstants.FOLDER_ID, STRING, false, false, true, false), //Object ID of the parent folder.
			new ContentTypeField(2, UtilConstants.CREATED_BY, STRING), //User ID of the user who created this object.
			new ContentTypeField(3, UtilConstants.MODIFIED_BY, STRING), //User ID of the user who last modified this object.
			new ContentTypeField(4, UtilConstants.CREATION_DATE, TIME), //Date and time of creation.
			new ContentTypeField(5, UtilConstants.LAST_MODIFIED, TIME), //Date and time of the last modification.
			new ContentTypeField(100, UtilConstants.CATEGORIES, STRING), //Each element is a string, naming a category. Order is preserved. Changing the order counts as modification of the object.
			new ContentTypeField(101, UtilConstants.PRIVATE_FLAG, BOOLEAN), //Overrides folder permissions in shared private folders: When true, this object is not visible to anyone except the owner.
			new ContentTypeField(104, UtilConstants.NUMBER_OF_ATTACHMENTS, NUMBER), //Number of attachments
			//Detailed task and appointment data
			new ContentTypeField(200, CommonCalendarTasksFieldNames.TITLE, STRING), //Short description.
			new ContentTypeField(201, CommonCalendarTasksFieldNames.START_DATE, TIME), //Inclusive start of the event as Date for tasks and whole day appointments and Time for normal appointments
			new ContentTypeField(202, CommonCalendarTasksFieldNames.END_DATE, TIME), //Exclusive end of the event as Date for tasks and whole day appointments and as Time for normal appointments
			new ContentTypeField(203, CommonCalendarTasksFieldNames.NOTE, STRING), //Long description
			new ContentTypeField(204, CommonCalendarTasksFieldNames.ALARM, NUMBER), //Specifies when to notify the participants as the number of minutes before the end of the task or before the start of the appointment
			new ContentTypeField(207, CommonCalendarTasksFieldNames.RECURRENCE_POSITION, NUMBER), //1-based position of an individual appointment in a sequence. Present if and only if recurrence_type > 0
			new ContentTypeField(208, CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION, NUMBER), //Date of an individual appointment in a sequence. Present if and only if recurrence_type > 0.
			new ContentTypeField(209, CommonCalendarTasksFieldNames.RECURRENCE_TYPE, NUMBER_NOT_NULL), //Specifies the type of the recurrence for a task sequence: 0 -none(single event); 1-daily; 2-weekly;3-monthly; 4-yearly;
			new ContentTypeField(210, CommonCalendarTasksFieldNames.CHANGE_EXCEPTIONS, NUMBER, true), //An array of Dates, representing all change exceptions of a sequence.
			new ContentTypeField(211, CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS, NUMBER, true), //An array of Dates, representing all delete exceptions of a sequence.
			new ContentTypeField(212, CommonCalendarTasksFieldNames.DAYS, NUMBER), //Specifies which days of the week are part of a sequence. The value is a bit field with bit 0 indicating sunday, bit 1 indicating monday and so on
			new ContentTypeField(213, CommonCalendarTasksFieldNames.DAY_IN_MONTH, NUMBER), //Specifies which day of a month is part of the sequence. Counting starts with 1
			new ContentTypeField(214, CommonCalendarTasksFieldNames.MONTH, NUMBER), //Month of the year in yearly sequences. 0 represents January, 1 represents February and so on
			new ContentTypeField(215, CommonCalendarTasksFieldNames.INTERVAL, NUMBER), //Specifies an integer multiplier to the interval specified by recurrence_type. Present if and only if recurrence_type > 0
			new ContentTypeField(216, CommonCalendarTasksFieldNames.UNTIL, DATE), //Exclusive end date of a sequence. May be present only if recurrence_type > 0
			new ContentTypeField(217, CommonCalendarTasksFieldNames.NOTIFICATION, BOOLEAN), // If true, all participants are notified of any changes to this object. This flag is valid for the current change only, i. e. it is not stored in the database and is never sent by the server to the client
			new ContentTypeField(220, CommonCalendarTasksFieldNames.PARTICIPANTS, new ParticipantIdentifierDataType(),
					true, false), //Each element identifies a participant, user group or booked resource
			new ContentTypeField(221, CommonCalendarTasksFieldNames.USERS, new UserParticipantObjectDataType(), true,
					false), //Each element represents a participant. User groups are resolved and are represented by their members. Any user can occur only once.
			new ContentTypeField(222, CommonCalendarTasksFieldNames.OCCURENCES, NUMBER), //Specifies how often a recurrence should appear. May be present only if recurrence_type > 0

			//Detailed appointment data
			new ContentTypeField(206, CalendarConstants.RECURRENCE_ID, NUMBER), //Object ID of the entire appointment sequence. Present on series and change exception appointments.
			//Equals to object identifier on series appointment and is different to object identifier on change exceptions
			new ContentTypeField(400, CalendarConstants.LOCATION, STRING), // Location
			new ContentTypeField(401, CalendarConstants.FULL_TIME, BOOLEAN, false, false, false, true), // True if the appointment is a whole day appointment, false otherwise.
			new ContentTypeField(402, CalendarConstants.SHOWN_AS, NUMBER), //Describes, how this appointment appears in availability queries: 1 reserved 2 temporary 3 absent 4 free
			new ContentTypeField(102, CalendarConstants.COLOR_LABEL, NUMBER), //Color number used by Outlook to label the appointment. The assignment of colors to numbers is arbitrary and specified by the client. The numbers are integer numbers
			new ContentTypeField(105, UtilConstants.ATTACHMENTS_LAST_MODIFIED, new PIMAttachmentsDataType()), // New complex field that contains attachments_last_modified for updates and additionally an array PIMAttachment for persistent storing in the SyncState
			new ContentTypeField(223, CommonCalendarTasksFieldNames.UID, STRING), //the UID of the appointment
			new ContentTypeField(224, CalendarConstants.ORGANIZER, STRING), //the Organizer of the appointment
			new ContentTypeField(225, CalendarConstants.SEQUENCE, NUMBER), //iCal sequence number.
			new ContentTypeField(226, CalendarConstants.CONFIRMATIONS, new ConfirmingParticipantDataType(), true, false), //Each element represents a confirming participant.This can be internal and external user.
			new ContentTypeField(408, CalendarConstants.TIMEZONE, STRING), //timezone
			new ContentTypeField(227, CalendarConstants.ORGANIZER_ID, NUMBER), //Contains the userIId of the appointment organizer if it is an internal user.
			new ContentTypeField(228, CalendarConstants.PRINCIPAL, STRING), //Contains the email address of the appointment principal which is not necessarily an internal user.
			new ContentTypeField(229, CalendarConstants.PRINCIPAL_ID, NUMBER),//Contains the userIId of the appointment principal if it is an internal user.
	};

	private AppointmentContentTypeTransferHandler _transferHandler;

	/**
	 * Constructor is package private to discourage access from outside the bundle (but to allow usage in test cases)
	 */
	AppointmentContentTypeImpl() {
	}

	public ContentTypeField[] getFields() {
		return FIELDS;
	}

	public String getID() {
		return DefaultContentTypes.CALENDAR_ID;
	}

	public ContentTypeTransferHandler getTransferHandler() {
		return _transferHandler;
	}

	public void setTransferHandler(AppointmentContentTypeTransferHandler transferHandler) {
		_transferHandler = transferHandler;
	}

	@Override
	public int hashCode() {
		return 131;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof AppointmentContentType))
			return false;
		return true;
	}

	public void confirm(DataObject object, int response) throws AuthenticationFailedException, OXCommunicationException {
		_transferHandler.confirm(object, response);
	}

	public void confirm(DataObject object, int response, String confirmMessage) throws AuthenticationFailedException,
			OXCommunicationException {
		_transferHandler.confirm(object, response, confirmMessage);
	}

	public DataObject[] getAllAppointments(Session session, BitSet requestedFields) throws USMException {
		return _transferHandler.getAllAppointments(session, requestedFields);
	}

	public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject)
			throws USMException {
		return _transferHandler.getAllRecurrencesAsAppointments(session, requestedFields, seriesObject);
	}

	public int getCode() {
		return DefaultContentTypes.CALENDAR_CODE;
	}

	public long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException {
		return _transferHandler.createNewAttachment(object, attachment);
	}

	public long deleteAttachments(DataObject object, PIMAttachment... attachmentsToDelete) throws USMException {
		return _transferHandler.deleteAttachments(object, attachmentsToDelete);
	}

	public PIMAttachments getAllAttachments(DataObject object, String sort, String order) throws USMException {
		return _transferHandler.getAllAttachments(object, sort, order);
	}

	public PIMAttachments getAllAttachments(DataObject object) throws USMException {
		return _transferHandler.getAllAttachments(object);
	}

	public byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException {
		return _transferHandler.getAttachmentData(object, attachmentId);
	}

	public boolean supportsPIMAttachments() {
		return true;
	}

	public Object getObjectGroupOwner(DataObject object) {
		Object recurrence_id = object.getFieldContent(CalendarConstants.RECURRENCE_ID);
		if (recurrence_id == null || ((Number) recurrence_id).intValue() == 0)
			return object.getID();
		else
			return String.valueOf(recurrence_id);
	}

	public String resolveUID(Session session, String uid) throws USMException {
		return _transferHandler.resolveUID(session, uid);
	}

	public void confirm(DataObject object, int response, String confirmMessage, String participantId)
			throws AuthenticationFailedException, OXCommunicationException {
		_transferHandler.confirm(object, response, confirmMessage, participantId);
	}
}
