/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.ant.tasks;

import java.net.MalformedURLException;
import java.util.HashSet;
import java.util.Set;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import com.openexchange.obs.api.BuildServiceClient;
import com.openexchange.obs.api.BuildServiceException;
import com.openexchange.obs.api.Project;
import com.openexchange.obs.api.Repositories;
import com.openexchange.obs.api.Repository;

/**
 * Creates a new project on the open build service. First a check is done if the project already exists. If the project already exists
 * nothing is done.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class CreateProject extends Task {

    private String url;
    private String login;
    private String password;
    private String projectName;
    private String obsConfPrefer;
    private String[] repositories;
    private String[] disabledRepositories;
    private String[] dependingProjectNames;

    public CreateProject() {
        super();
    }

    public void setUrl(final String url) {
        this.url = url;
    }

    public void setLogin(final String login) {
        this.login = login;
    }

    public void setPassword(final String password) {
        this.password = password;
    }

    public void setProjectName(final String projectName) {
        this.projectName = projectName;
    }

    public void setRepositories(final String repositories){
        if (repositories.length() != 0){
            this.repositories = repositories.split(",");
        }
    }

    public void setDisabledRepositories(String disabledRepositories) {
        if (disabledRepositories.length() != 0) {
            this.disabledRepositories = disabledRepositories.split(",");
        }
    }

    public void setDependingProjectNames(final String dependingProjectNames){
        if (dependingProjectNames.length() != 0) {
            this.dependingProjectNames = dependingProjectNames.split(",");
        }
    }

    public void setObsConfPrefer(final String obsConfPrefer){
        if (obsConfPrefer.length() != 0) {
            this.obsConfPrefer = obsConfPrefer;
        }
    }

    @Override
    public void execute() throws BuildException {
        if (projectName == null) {
            throw new BuildException("Please define project.");
        }
        if (repositories == null) {
            throw new BuildException("Please define repositories.");
        }
        //dependingProjectNames is allowed to be null
        final BuildServiceClient client;
        try {
            client = new BuildServiceClient(url, login, password);
            final Project obsProject = createProject(projectName, login, repositories, disabledRepositories, dependingProjectNames, obsConfPrefer);
            if (!client.existsProject(obsProject)) {
                getProject().log("Creating project \"" + obsProject.getName() + "\" on build service.", org.apache.tools.ant.Project.MSG_INFO);
                client.createProject(obsProject);
            } else if (!client.getProjectConfig(obsProject.getName()).equals(obsProject.getBuildConfiguration())){
                getProject().log("Project \"" + obsProject.getName() + "\" already exists on build service but has different config.", org.apache.tools.ant.Project.MSG_INFO);
                getProject().log("Project \"" + obsProject.getName() + "\" updated on build service.", org.apache.tools.ant.Project.MSG_INFO);
                client.createProject(obsProject);
            } else {
                getProject().log("Project \"" + obsProject.getName() + "\" already exists on build service.", org.apache.tools.ant.Project.MSG_INFO);

                final Repository[] currentRepos = client.getRepositoryList(projectName);
                final Set<String> currentRepoNamesHash = new HashSet<String>();
                System.out.println("currently used repositories on obs: ");
                for (Repository r : currentRepos) {
                    System.out.println(r.toString());
                    currentRepoNamesHash.add(r.toString());
                }

                System.out.println("needed repositories: ");
                final Repository[] neededRepos = obsProject.getRepositories();
                final Set<String> neededRepoNamesHash = new HashSet<String>();
                for (Repository r : neededRepos) {
                    System.out.println(r.toString());
                    neededRepoNamesHash.add(r.toString());
                }

                if (!currentRepoNamesHash.equals(neededRepoNamesHash)){
                    getProject().log("Project \"" + obsProject.getName() + "\" updated on build service.", org.apache.tools.ant.Project.MSG_INFO);
                    client.createProject(obsProject);
                }
            }
        } catch (final MalformedURLException e) {
            throw new BuildException(e.getMessage(), e);
        } catch (final BuildServiceException e) {
            throw new BuildException(e.getMessage(), e);
        }
    }

    private static Project createProject(String projectName, String user, String[] repositories, String[] disabledRepositories, String[] dependingProjectNames, String preferString) throws BuildServiceException {
        final Project project = new Project(projectName, projectName, user);
        if (null != disabledRepositories) {
            for (final String s : disabledRepositories) {
                project.addDisabledRepository(s);
            }
        }
        if (dependingProjectNames != null) {
            for (final String s : repositories) {
                project.addRepository(Repository.getNewRepository(Repositories.valueOf(s), dependingProjectNames));
            }
        } else {
            for (final String s : repositories) {
                project.addRepository(Repository.getNewRepository(Repositories.valueOf(s)));
            }
        }
        project.setBuildConfiguration("Prefer: " + (preferString != null ? preferString : "") + "\n" + DEFAULT_CONFIGURATION);
        return project;
    }

    private static final String DEFAULT_CONFIGURATION =
        "Macros:\n" +
        "%vendor Open-Xchange\n";
}
