#!/usr/bin/perl -w
#
# needed:
# libtext-csv-perl 
# liblog-log4perl-perl
# liblog-dispatch-perl
# libnet-ldap-perl

######### System initialization section ###

use strict;
use warnings;
use Getopt::Std;
use Log::Log4perl qw(:easy);
use FindBin qw($Bin);
use lib "$Bin/../lib";
use OX::User::Helper;
use OX::Group::Helper;
use OX::Config::Reader;
	
use Data::Dumper;

use vars qw/ %opt /;
use vars qw/ $loglevel /;

use POSIX qw(strftime);

binmode(*STDOUT,":utf8");
binmode(*STDERR,":utf8");

my $OXPath = "/opt/open-xchange/sbin/";
my $logger;

##########################################
#
#
#  Main sub
#
#
##########################################

sub init() { 

#	print(strftime("%Y%m%d%H%M%SZ", localtime)."\n");

	my $configfile;		

        my $opt_string = 'A:P:c:f:ehdabvs';

        getopts( "$opt_string", \%opt ) or usage();
        usage() if $opt{h};
        
	if ($opt{f}) 
	{ 
	    if (! -r $opt{f})
	    {
		die ("Could not read configfile\n");
	    }
	    $configfile = $opt{f};
	} else 	{
	    $configfile = "/opt/oxldapsync/etc/ldapsync.conf";
	}
	
	if (! -e  $configfile) 
	{
	    print("Error: Configfile does not exist\n");
	    usage();
	}

	my $config_hash = OX::Config::Reader::readConfig($configfile,1);
	
	my $logfile;
	my $layout;
	if ($opt{s})
	{
	    $logfile = "STDOUT";
	    $layout = '%m';
	} else {
	    $logfile = ">>".$config_hash->{logfile};
	    $layout = '%d > %m';
	}
    
	if ($opt{v})
	{
	    Log::Log4perl->easy_init( { level    => $DEBUG,
                                file     => $logfile,
                                layout   => $layout} );
	} else {
	    Log::Log4perl->easy_init( { level    => $INFO,
                                file     => $logfile,
                                layout   => $layout } );
	}

	if (!defined($config_hash->{oxpath}) || length($config_hash->{oxpath}) < 1)
	{
	    $config_hash->{oxpath} = $OXPath;
	}

        if (!$opt{c}){LOGDIE "Missing contextID\n";}
	else { $config_hash->{contextid} = $opt{c}; }
        
	if ($opt{A})
	{
	    $config_hash->{adminuser} = $opt{A};
	}

	if ($opt{P})
	{
	    $config_hash->{adminpassword} = $opt{P};
	}

	if (lc($config_hash->{updatealiases}) eq "yes" || $config_hash->{updatealiases} eq "1")
	{
	    $config_hash->{updatealiases} = 1;
	} else 
	{
	    $config_hash->{updatealiases} = 0;
	}

	require "OX/LDAPConnector/$config_hash->{ldaptype}.pm";

	my $ldapconn = "OX::LDAPConnector::".$config_hash->{ldaptype};
	my $ldapobj = $ldapconn->new($config_hash->{ldapuri}, $config_hash);
    
	my $ignore_list = undef;
	# if userignorefilter is defined, query server for list of ignored users
	if( defined($config_hash->{userignorefilter}) ) {
	    my $tmpfilter = $config_hash->{userfilter};
	    my $tmpugrp   = $config_hash->{updategroups};
	    $config_hash->{userfilter} = $config_hash->{userignorefilter};
	    $config_hash->{updategroups} = "no";
	    my $ldapignobj = $ldapconn->new($config_hash->{ldapuri}, $config_hash);
	    $ignore_list = &getIgnoreList($ldapignobj, $config_hash);
	    $config_hash->{userfilter} = $tmpfilter;
	    $config_hash->{updategroups} = $tmpugrp;
	}


	&syncUsers($ldapobj, $config_hash, $ignore_list);
	
	$ldapobj->saveLastUsertime() if (! defined($opt{n}));

	if ($config_hash->{updategroups} eq "yes")
	{
	    &syncGroups($ldapobj, $config_hash);
	}

}

##########################################
#
#
#  Build a list of users that should be ignored in this sync
#
#
##########################################

sub getIgnoreList()
{
	my $ldapobj = shift;
	my $config_hash = shift;

	if (! defined($ldapobj)) { die("sub getIgnoreList: ldapobj undefined\n"); }
	if (! defined($config_hash)) { die("sub getIgnoreList: config_hash undefined\n"); }

	$ldapobj->searchusers();

	my $userhelper = OX::User::Helper->new($config_hash->{contextid},
					       $config_hash->{adminuser},
					       $config_hash->{adminpassword},
    					       $config_hash->{oxpath},
					       $config_hash->{dontmodifyuids},
					       $config_hash->{updatealiases},
					       $config_hash->{vardirectory},
					       $config_hash->{usemodifytimestamp});

	if (! -r $config_hash->{mappingfile}) { LOGDIE ("Mapping file ".$config_hash->{mappingfile} ." could not be read!\n");}

	my $mapping_hash = $userhelper->readMapping($config_hash->{mappingfile});

	my $ignlist = undef;
	while (defined(my $entry = $ldapobj->user_pop_entry()))
	{
    	    push @$ignlist, $entry->get_value($mapping_hash->{username});
	}

	return $ignlist;
}

##########################################
#
#
#  Syncs all users
#
#
##########################################

sub syncUsers()
{
	my $ldapobj = shift;
	my $config_hash = shift;
	my $ignore_list = shift;

	if (! defined($ldapobj)) { die("sub syncUsers: ldapobj undefined\n"); }
	if (! defined($config_hash)) { die("sub syncUsers: config_hash undefined\n"); }

	$ldapobj->searchusers();

	my $userhelper = OX::User::Helper->new($config_hash->{contextid},
					       $config_hash->{adminuser},
					       $config_hash->{adminpassword},
    					       $config_hash->{oxpath},
					       $config_hash->{dontmodifyuids},
					       $config_hash->{updatealiases},
					       $config_hash->{vardirectory},
					       $config_hash->{usemodifytimestamp},
					       $config_hash->{updateemptyattributes});

	if (! -r $config_hash->{mappingfile}) { LOGDIE ("Mapping file ".$config_hash->{mappingfile} ." could not be read!\n");}

	$userhelper->readMapping($config_hash->{mappingfile});

	while (defined(my $entry = $ldapobj->user_pop_entry()))
	{
    	    $userhelper->processUser($entry);
	}


	if ($config_hash->{deleteusers} eq "yes")
	{
	    $userhelper->deleteNotTouchedUsers($ignore_list);
	}
}

##########################################
#
#
#  Syncs alle groups
#
#
##########################################

sub syncGroups()
{
	my $ldapobj = shift;
	my $config_hash = shift;

	if (! defined($ldapobj)) { die("sub syncGroups: ldapobj undefined\n"); }
	if (! defined($config_hash)) { die("sub syncGroups: config_hash undefined\n"); }

	my $userhelper = OX::User::Helper->new($config_hash->{contextid},$config_hash->{adminuser},$config_hash->{adminpassword},$config_hash->{oxpath},$config_hash->{dontmodifyuids},$config_hash->{updatealiases});
	my @oxUserList = $userhelper->getOXUserList();

	my $grouphelper = OX::Group::Helper->new($config_hash->{contextid},$config_hash->{adminuser},$config_hash->{adminpassword},$config_hash->{oxpath},\@oxUserList);
	
	my $groups = $ldapobj->getGroups();

	foreach my $groupkey (keys%$groups)
	{
	    if ($groups->{$groupkey}{'member'})
	    {
		$grouphelper->processGroup($groupkey, $groups->{$groupkey}{'displayname'}, @{$groups->{$groupkey}{'member'}});
	    }
	}

	if ($config_hash->{deletegroups} eq "yes")
	{
	    $grouphelper->deleteNotTouchedGroups();
	}
}

##########################################
#
#
#  Print help message
#
#
##########################################

sub usage()
{
    print STDERR << "EOF";

    This program evaluates LDAP directories for syncing with a given OX-Context.


    usage: $0 [-c] [-h] [-f configfile] [-A adminuser] [-P adminpass]

     -h        		: this (help) message
     -c contextID	: the contextID
     -f	filename	: config file name
     -A			: Context admin username
     -P			: Context admin password
     -n			: don't save last user search time
     -v			: Verbose mode 
     -s			: print messages to stdout
     

    example: $0 -c <contextID> -A <adminuser> -P <adminpass> -f <configfilename>

EOF
    exit;
}

init();

#print STDERR "Verbose mode ON.\n" if $opt{v};
#print STDERR "Debugging mode ON.\n" if $opt{d};
    
 
 	
	
	
    


 
 
