package com.openexchange.office.tools.osgi;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.Validate;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.slf4j.Logger;

import com.openexchange.office.tools.logging.LogFactory;
import com.openexchange.server.ServiceLookup;

//=============================================================================
public class ServiceLookupRegistry implements ServiceLookup
{
	//-------------------------------------------------------------------------
	public static final Logger LOG = LogFactory.getSlfLog (ServiceLookupRegistry.class);

	//-------------------------------------------------------------------------
	// make it private : force using of static factory method
	private ServiceLookupRegistry ()
	{}

	//-------------------------------------------------------------------------
	/** @return the singleton instance of this registry
	 */
	public synchronized static ServiceLookupRegistry get ()
	    /* no throws */
	{
		if (m_gSingleton == null)
			m_gSingleton = new ServiceLookupRegistry ();
		return m_gSingleton;
	}

	//-------------------------------------------------------------------------
	/** register a new service lookup within this registry.
	 *
	 *  @param	aLookup [IN]
	 *  		the activator where it's service lookup needs to be registered within this registry
	 */
	public synchronized void registerServiceLookup (final ActivatorBase aLookup)
	    throws Exception
	{
		Validate.notNull(aLookup, "Invalid argument 'lookup'.");

		final WeakReference< ActivatorBase >                aLookupRef       = new WeakReference< ActivatorBase > (aLookup);
		final Map< String, WeakReference< ActivatorBase > > aRegistry        = mem_Registry ();
		      String                                        sBundleContextId = aLookup.getBundleContextId();

		LOG.info("[ServiceLookupRegistry] : register lookup '"+aLookup+"' for bundle context '"+sBundleContextId+"' (direct) ...");
		aRegistry.put(sBundleContextId, aLookupRef);
		
		final List< Class< ? > > lExportedServices = aLookup.getServiceExports();
		for (final Class< ? > aExportedService : lExportedServices)
		{
			final BundleContext aBundleContext   = FrameworkUtil.getBundle(aExportedService).getBundleContext();
			                    sBundleContextId = ObjectUtils.identityToString(aBundleContext);

			LOG.info("[ServiceLookupRegistry] : register lookup '"+aLookup+"' for bundle context '"+sBundleContextId+"' (exports) ...");
			aRegistry.put(sBundleContextId, aLookupRef);
		}

		final List< Class< ? > > lMandatoryServices = aLookup.getMandatoryServiceImports();
		for (final Class< ? > aService : lMandatoryServices)
		{
			final BundleContext aBundleContext   = FrameworkUtil.getBundle(aService).getBundleContext();
			                    sBundleContextId = ObjectUtils.identityToString(aBundleContext);

			LOG.info("[ServiceLookupRegistry] : register lookup '"+aLookup+"' for bundle context '"+sBundleContextId+"' (exports) ...");
			aRegistry.put(sBundleContextId, aLookupRef);
		}
	}
	
	//-------------------------------------------------------------------------
	/** deregister an existing service lookup from this registry.
	 * 
	 *  @param	aLookup [IN]
	 *  		the activator where it's service lookup needs to be deregistered from this registry
	 */
	public synchronized void deregisterServiceLookup (final ActivatorBase aLookup)
	    throws Exception
	{
		Validate.notNull(aLookup, "Invalid argument 'lookup'.");

		final Map< String, WeakReference< ActivatorBase > > aRegistry        = mem_Registry ();
		final String                                        sBundleContextId = aLookup.getBundleContextId();
		
		LOG.info("[ServiceLookupRegistry] : deregister lookup '"+aLookup+"' for bundle context '"+sBundleContextId+"' ...");
		aRegistry.remove(sBundleContextId);
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized < S > S getService(final Class< ? extends S > aService)
	{
		return (S) impl_getService (aService, /*mandatory*/ true);
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized < S > S getOptionalService(final Class< ? extends S > aService)
	{
		return (S) impl_getService (aService, /*mandatory*/ false);
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	private < S > S impl_getService (final Class< ? extends S > aService  ,
			 						 final boolean              bMandatory)
    	/* no throws */
	{
		LOG.trace ("search for service '"+aService+"' ...");
		Object  aInstance        = null ;
		boolean bIsInOptRegistry = false;

		try
		{
			if (aInstance == null)
			{
				LOG.trace ("search for service '"+aService+"' ... (opt registry)");
				aInstance = impl_findInOptRegistry (aService);
				if (aInstance != null)
				{
					LOG.trace ("found service '"+aService+"' by opt registry");
					bIsInOptRegistry = true;
				}
			}
		}
		catch (Throwable ex)
		{
			// ignore it - very often the bundle context is bound to the activator - but not the implementation bundle ;-)
		}
		
		try
		{
			if (aInstance == null)
			{
				LOG.trace ("search for service '"+aService+"' ... (bundle context)");
				aInstance = impl_findByBundleContext (aService, bMandatory);
				if (aInstance != null)
					LOG.trace ("found service '"+aService+"' by bundle context");
			}
		}
		catch (Throwable ex)
		{
			// ignore it - very often the bundle context is bound to the activator - but not the implementation bundle ;-)
		}

		try
		{
			if (aInstance == null)
			{
				LOG.trace ("search for service '"+aService+"' ... (brute force)");
				aInstance = impl_findByBruteForce (aService, bMandatory);
				if (aInstance != null)
					LOG.trace ("found service '"+aService+"' by brute force");
			}
		}
		catch (Throwable ex)
		{
			// ignore it ...
		}

		if (aInstance != null)
		{
			if ( ! bIsInOptRegistry)
			{
				try
				{
					impl_optimizeServiceLookup (aService, aInstance);
				}
				catch (Throwable ex)
				{
					// ignore it ...
				}
			}
		}
		else
		{
			LOG.error ("did not found service '"+aService+"'");
		}
		
		return (S) aInstance;
	}
	
	//-------------------------------------------------------------------------
	private void impl_optimizeServiceLookup (final Class< ? > aService ,
											 final Object     aInstance)
	    throws Exception
	{
		final Map< String, WeakReference< Object > > aOptRegistry = mem_OptRegistry ();
		final String                                 sService     = aService.getName();
		final WeakReference< Object >                rInstance    = new WeakReference< Object > (aInstance);
		aOptRegistry.put(sService, rInstance);
	}
	
	//-------------------------------------------------------------------------
	private Object impl_findInOptRegistry (final Class< ? > aService)
	    throws Exception
	{
		final Map< String, WeakReference< Object > > aOptRegistry = mem_OptRegistry ();
		final String                                 sService     = aService.getName();
		final WeakReference< Object >                rInstance    = aOptRegistry.get(sService);
		
		if (rInstance != null)
			return rInstance.get ();
		return null;
	}

	//-------------------------------------------------------------------------
	private Object impl_findByBundleContext (final Class< ? > aService  ,
											 final boolean    bMandatory)
		throws Exception
	{
		final Bundle                                        aBundle          = FrameworkUtil.getBundle       (aService      );
		final BundleContext                                 aBundleContext   = aBundle      .getBundleContext(              );
		final String                                        sBundleContextId = ObjectUtils  .identityToString(aBundleContext);
		final Map< String, WeakReference< ActivatorBase > > aRegistry        = mem_Registry  ();
		final WeakReference< ActivatorBase >                rLookup          = aRegistry.get (sBundleContextId);
		final ActivatorBase                                 aLookup          = rLookup  .get ();
		      Object                                        aInstance        = null;
		      
	    if ( ! aLookup.hasService(aService))
	    	return null;

		if (bMandatory)
			aInstance = aLookup.getService         (aService);
		else
			aInstance = aLookup.getOptionalService (aService);

		return aInstance;
	}
	
	//-------------------------------------------------------------------------
	private Object impl_findByBruteForce (final Class< ? > aService  ,
			 							  final boolean    bMandatory)
		throws Exception
	{
		final Map< String, WeakReference< ActivatorBase > > aRegistry = mem_Registry ();
		final Iterator< WeakReference< ActivatorBase > >    rRegistry = aRegistry.values().iterator();
		
		while (rRegistry.hasNext())
		{
			try
			{
				final WeakReference< ActivatorBase > rLookup   = rRegistry.next();
				final ActivatorBase                  aLookup   = rLookup  .get ();
			          Object                         aInstance = null;
			    
			    if ( ! aLookup.hasService(aService))
			    	continue;

				if (bMandatory)
					aInstance = aLookup.getService         (aService);
				else
					aInstance = aLookup.getOptionalService (aService);
				
				if (aInstance != null)
					return aInstance;
			}
			catch (Throwable ex)
			{
				// ignore it !
			}
		}
		
		return null;
	}

	//-------------------------------------------------------------------------
	private synchronized Map< String, WeakReference< ActivatorBase > > mem_Registry ()
	    throws Exception
	{
		if (m_aRegistry == null)
			m_aRegistry = new HashMap< String, WeakReference< ActivatorBase > > ();
		return m_aRegistry;
	}

	//-------------------------------------------------------------------------
	private synchronized Map< String, WeakReference< Object > > mem_OptRegistry ()
	    throws Exception
	{
		if (m_aOptRegistry == null)
			m_aOptRegistry = new HashMap< String, WeakReference< Object > > ();
		return m_aOptRegistry;
	}

	//-------------------------------------------------------------------------
	private static ServiceLookupRegistry m_gSingleton = null;

	//-------------------------------------------------------------------------
	private Map< String, WeakReference< ActivatorBase > > m_aRegistry = null;

	//-------------------------------------------------------------------------
	private Map< String, WeakReference< Object > > m_aOptRegistry = null;
}
