/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rest;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;

import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.groupware.results.TimedResult;
import com.openexchange.office.document.TemplateFilesScanner;
import com.openexchange.office.tools.doc.ApplicationType;
import com.openexchange.office.tools.doc.RecentFileListManager;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.files.FileDescriptorHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.filename.FileNameTools;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.tools.session.ServerSession;

public class GetTemplateAndRecentListAction extends DocumentRESTAction {
    private static final Log LOG = com.openexchange.log.Log.loggerFor(GetTemplateAndRecentListAction.class);

    private final TemplateFilesScanner m_templateFilesScanner;
    private final IDBasedFileAccessFactory m_fileFactory;
    private final IDBasedFolderAccessFactory m_folderFactory;
    private final List<Field> m_fields;
    private final String[] m_props = { "filename", "last_modified", "title" };

    /**
     * Initializes a new {@link GetTemplateAndRecentListAction}.
     * @param services
     */
    public GetTemplateAndRecentListAction(ServiceLookup servicesDEPRECATED, TemplateFilesScanner templateFilesScanner) {
        super(null);
        m_templateFilesScanner = templateFilesScanner;
        m_fileFactory = ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class);
        m_folderFactory = ServiceLookupRegistry.get().getService(IDBasedFolderAccessFactory.class);
        m_fields = new ArrayList<Field>(5);
        // initialize fields array with good to know fields
        m_fields.add(Field.ID);
        m_fields.add(Field.FOLDER_ID);
        m_fields.add(Field.FILENAME);
        m_fields.add(Field.LAST_MODIFIED);
        m_fields.add(Field.TITLE);
    }

	@Override
	public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) {
        final JSONObject jsonObj = new JSONObject();
        final String type = requestData.getParameter("type").toLowerCase();
    	final ApplicationType appType = ApplicationType.stringToEnum(type);
        final JSONArray templates = new JSONArray();
        AJAXRequestResult requestResult = null;

        if ((null != session) && (appType != ApplicationType.APP_NONE)) {
        	try {
        	    ErrorCode errorCode = m_templateFilesScanner.searchForTemplates(session, type, templates);
                jsonObj.put("error", errorCode.getAsJSON());
                jsonObj.put("templates", templates);

                RecentFileListManager recentFileListManager = new RecentFileListManager(null, session);
                List<JSONObject> appRecentFiles = recentFileListManager.readRecentFileList(appType);
                if ((null != appRecentFiles) && (!appRecentFiles.isEmpty())) {
                    // now we have to check the recents list to be up to date
                    final IDBasedFileAccess fileAccess = ((null != m_fileFactory)) ? m_fileFactory.createAccess(session) : null;
                    final IDBasedFolderAccess folderAccess = ((null != m_folderFactory)) ? m_folderFactory.createAccess(session) : null;
                    if ((null != fileAccess) && (null != folderAccess)) {
                        final List<String> ids = new ArrayList<String>(appRecentFiles.size());

                        for (JSONObject fileObj : appRecentFiles) {
                            
                            final String id = fileObj.optString("id", null);
                            fileObj.put("com.openexchange.file.sanitizedFilename", FileNameTools.sanitizeFilename(fileObj.getString("filename")));
                            
                            if (null != id) {
                                ids.add(id);
                            }
                        }

                        HashMap<String, JSONObject> newMetaDataMap = null;
                        try {
                            newMetaDataMap = getMetaDataForFiles(fileAccess, ids);
                        } catch (Exception e) {
                            // Fallback, if we cannot retrieve the meta data in a
                            // fast way. Use the slow more time consuming way to
                            // files one by one.
                            newMetaDataMap = getMetaDataForFileList(fileAccess, ids);
                        }
                        if (null != newMetaDataMap) {
                            final HashMap<String, Boolean> removedIds = new HashMap<String, Boolean>();

                            boolean recentListChanged = checkRecentListWithMetaData(folderAccess, appRecentFiles, newMetaDataMap, removedIds);
                            if (recentListChanged && (removedIds.size() > 0)) {
                                // remove recent entries which were moved to trash or deleted
                                ListIterator<JSONObject> recentsIter = appRecentFiles.listIterator();
                                while (recentsIter.hasNext()) {
                                    String id = recentsIter.next().optString("id", null);
                                    if ((null != id) && (removedIds.containsKey(id))) {
                                        recentsIter.remove();
                                    }
                                }
                            }

                            // write-back adapted recent list
                            if (recentListChanged) {
                                recentFileListManager.writeRecentFileList(appType, appRecentFiles);
                                recentFileListManager.flush();
                            }
                        }
                    }
                	jsonObj.put("recents", new JSONArray(appRecentFiles));
                } else {
                	jsonObj.put("recents", new JSONArray());
                }
        	} catch (OXException e) {
                LOG.error("Exception while updating recent file list", e);
        	} catch (JSONException e) {
        		LOG.error("Exception while creating JSONObject answering AJAXRequest 'gettemplateandrecentlist'", e);
        	}

            requestResult = new AJAXRequestResult(jsonObj);
        }

        return requestResult;
	}

    /**
     * Checks the recent file list for removed, moved or updated entries using
     * the updated meta data.
     *
     * @param folderAccess
     *  A folder access instance to be used to retrieve the meta data.
     *
     * @param appRecentFiles
     *  A list of recent file entries.
     *
     * @param newMetaDataMap
     *  A map of new meta data associated by the file id.
     *
     * @param removedIds
     *  A map of ids which must be removed from the recent file list.
     *
     * @return
     *  TRUE if the recent file list has changed and must be written, otherwise
     *  FALSE and no write-back must be done.
     *
     * @throws JSONException
     */
	private boolean checkRecentListWithMetaData(IDBasedFolderAccess folderAccess, List<JSONObject> appRecentFiles, HashMap<String, JSONObject> newMetaDataMap, HashMap<String, Boolean> removedIds) throws OXException, JSONException {
        // Use new meta data to update recent files - there are some situations
	    // where we have to dig deeper to find out what happened. We want to
	    // support to still open files that have been moved to other folders -
	    // except the trash folder!!

        // Iterate through the recent file list and check
        // every file.
        boolean recentListChanged = false;
        long time1 = System.currentTimeMillis();

        for (int i = 0; i < appRecentFiles.size(); i++) {
            final JSONObject fileObj = appRecentFiles.get(i);
            final String id = fileObj.optString("id", null);
            final String folderId = fileObj.optString("folder_id", null);

            if ((null != id) && (null != folderId)) {
                JSONObject newFileMetaData = newMetaDataMap.get(id);
                if (null != newFileMetaData) {
                    String newFolderId = newFileMetaData.optString("folder_id");
                    FileStorageFolder folder = folderAccess.getFolder(newFolderId);
                    if (folderId.equals(newFolderId)) {
                        // check if the parent folder was moved to the trash
                        if (FolderHelper.isFolderTrashOrInTrash(folder)) {
                            removedIds.put(id, true);
                            recentListChanged = true;
                        } else if (FileDescriptorHelper.containsChanges(fileObj, newFileMetaData, m_props)) {
                            // no move - just take over the new meta data
                            fileObj.put("filename", newFileMetaData.optString("filename", ""));
                            fileObj.put("title", newFileMetaData.optString("title", ""));
                            Date lastModified = (Date)newFileMetaData.opt("last_modified");
                            if (null != lastModified) {
                                fileObj.put("last_modified", lastModified.getTime());
                            }
                            recentListChanged = true;
                        }
                    } else {
                        // We have a move - check if the new folder is the trash folder
                        if (FolderHelper.isFolderTrashOrInTrash(folder)) {
                            // file has been moved to the trash => remove from recent files
                            removedIds.put(id, true);
                            recentListChanged = true;
                        } else {
                            // move to another folder => take over changes
                            fileObj.put("folder_id", newFolderId);
                            fileObj.put("filename", newFileMetaData.optString("filename", ""));
                            fileObj.put("title", newFileMetaData.optString("title", ""));
                            Date lastModified = (Date)newFileMetaData.opt("last_modified");
                            if (null != lastModified) {
                                fileObj.put("last_modified", lastModified.getTime());
                            }
                            recentListChanged = true;
                        }
                    }
                } else {
                    // id not known anymore => remove from recent list
                    removedIds.put(id, true);
                    recentListChanged = true;
                }
            } else {
                // invalid properties => remove from recent list
                removedIds.put(id, true);
                recentListChanged = true;
            }
        }

        // time runtime to update the recent file list (moves/renames/deletes)
        long time2 = System.currentTimeMillis();
        LOG.debug("Check recent file list for moved/deleted entries lasts " + (time2 - time1) + "ms");

        return recentListChanged;
	}

	/**
	 * Retrieves the meta data of provided file ids.
	 *
	 * @param fileAccess
	 *  A file access instance to be used to retrieve the meta data.
	 *
	 * @param ids
	 *  A list of file ids for which meta data should be retrieved.
	 *
	 * @return
	 *  The map with key id and value JSONObject which includes the
	 *  meta data of the file ids provided. If there is no mapping for
	 *  a provided key it must be assumed that the file is not available
	 *  anymore (deleted or access rights have been modified).
	 *
	 * @throws OXException
	 * @throws JSONException
	 */
	private HashMap<String, JSONObject> getMetaDataForFiles(IDBasedFileAccess fileAccess, List<String> ids) throws OXException, JSONException {
	    HashMap<String, JSONObject> newMetaDataMap = null;
        // now check the meta data of the files using file access
        TimedResult<File> fileResult = fileAccess.getDocuments(ids, m_fields);
        if (null != fileResult) {

            newMetaDataMap = new HashMap<String, JSONObject>();
            SearchIterator<File> iter = fileResult.results();

            // create hash map for faster access to store new meta data
            while (iter.hasNext()) {
                File file = iter.next();

                final String id = file.getId();
                if (null != id) {
                    newMetaDataMap.put(id, getFileMetaDataAsJSON(file));
                }
            }
        }

        return newMetaDataMap;
	}

	/**
	 * Checks the list of file ids for their meta data using a slower access
	 * variant.
	 *
	 * @param fileAccess
	 *  The file access instance to be used to retrieve meta data.
	 *
	 * @param ids
	 *  The list of file ids where the meta data should be retrieved.
	 *
	 * @return
	 *  A mapping of id to File which provides access to the file meta data,
	 *  can be null if no meta data can be retrieved.
	 */
	private HashMap<String, JSONObject> getMetaDataForFileList(IDBasedFileAccess fileAccess, List<String> ids) {
        HashMap<String, JSONObject> newMetaDataMap = null;
        // now check the meta data of the files using file access
        // use the slower single access to filter out problematic files
        final ListIterator<String> idIter = ids.listIterator();

        while (idIter.hasNext()) {
            try {
                final String id = idIter.next();
                File fileMetaData = fileAccess.getFileMetadata(id, FileStorageFileAccess.CURRENT_VERSION);
                if (newMetaDataMap == null) {
                    newMetaDataMap = new HashMap<String, JSONObject>();
                }
                newMetaDataMap.put(id, getFileMetaDataAsJSON(fileMetaData));
            } catch (Exception e) {
                // do nothing
            }
        }
        return newMetaDataMap;
	}

    /**
     * Put the file meta data as properties into a JSONObject.
     *
     * @param file
     *  The file whose properties should be set into the JSONObject.
     *
     * @return
     *  A JSONObject which contains the file meta data as properties.
     *
     * @throws JSONException
     */
	private JSONObject getFileMetaDataAsJSON(final File file) throws JSONException {
        JSONObject newFileMetaData = new JSONObject();

        newFileMetaData.put("id", file.getId());
        newFileMetaData.put("folder_id", file.getFolderId());
        newFileMetaData.put("filename", file.getFileName());
        newFileMetaData.put("title", file.getTitle());
        newFileMetaData.put("last_modified", file.getLastModified());
        
        return newFileMetaData;
	}
}
