/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.htmldoc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.file.storage.File;

public class PresentationHtmlDocumentBuilder
{
    private final static Logger      LOG       = LoggerFactory.getLogger(PresentationHtmlDocumentBuilder.class);
    private final static Set<String> IGNOREOPS = new HashSet<String>();

    static
    {
        IGNOREOPS.add("setDocumentAttributes");
        IGNOREOPS.add("insertFontDescription");
        IGNOREOPS.add("insertStyleSheet");
        IGNOREOPS.add("insertTheme");
        IGNOREOPS.add("insertListStyle");
    }

    static public
            Logger
            getLogger()
    {
        return LOG;
    }

    static
            String
            buildHtmlDocument(
                JSONObject documentOperations,
                File file,
                TextTableLimits tableLimits)
    {

        return buildHtmlDocument(documentOperations, file.getId(), file.getFolderId(), file.getVersion(), file.getFileName(), tableLimits);
    }

    static private
            String
            buildHtmlDocument(
                JSONObject documentOperations,
                String fileId,
                String folderId,
                String fileVersion,
                String fileName,
                TextTableLimits textTableLimits)
    {
        String result = null;

        Comparator<Entry<String, Container>> slideSorter = new Comparator<Entry<String, Container>>()
        {
            @Override
            public
                    int
                    compare(
                        Entry<String, Container> o1,
                        Entry<String, Container> o2)
            {
                return Integer.compare(o1.getValue().getIndex(), o2.getValue().getIndex());
            }
        };

        try
        {

            int layoutSlideIndex = 0;
            int masterSlideIndex = 0;
            Map<String, Container> layoutSlides = new HashMap<String, Container>();
            Map<String, Container> masterSlides = new HashMap<String, Container>();
            Container pagecontent = new Container(-1, "pagecontent", null, false, false, null);

            final long time = System.currentTimeMillis();
            final JSONArray operations = documentOperations.getJSONArray("operations");
            final JSONArray reducedOperations = new JSONArray();

            for (int i = 0; i < operations.length(); i++)
            {
                JSONObject operation = operations.getJSONObject(i);
                try
                {
                    final String opName = operation.getString("name");
                    if (IGNOREOPS.contains(opName))
                    {
                        reducedOperations.put(operation);
                        continue;
                    }

                    JSONArray start = operation.optJSONArray("start");
                    JSONArray end = operation.optJSONArray("end");
                    JSONObject attrs = operation.optJSONObject("attrs");
                    String id = operation.optString("id");
                    String target = operation.optString("target");

                    INode insert = null;

                    if ("insertMasterSlide".equals(opName))
                    {
                        masterSlides.put(id, new Container(masterSlideIndex, id, target, true, true, attrs));
                        masterSlideIndex++;

                        attrs = null;
                    }
                    else if ("insertLayoutSlide".equals(opName))
                    {
                        layoutSlides.put(id, new Container(layoutSlideIndex, id, target, true, true, attrs));
                        layoutSlideIndex++;
                        attrs = null;
                    }
                    else if ("insertSlide".equals(opName))
                    {
                        long slideId = operation.getLong("start");
                        insert = new SlideParagraph("slide_" + (slideId + 1), target);

                        if (null == attrs)
                        {
                            attrs = new JSONObject();
                        }
                        target = null;
                        start = new JSONArray().put(slideId);

                    }
                    else if ("insertDrawing".equals(opName))
                    {
                        insert = new Drawing(operation.optString("type", null), fileId, folderId, fileVersion).setPresentation(true).setTarget(target);

                        if (attrs != null)
                        {
                            JSONObject newOp = new JSONObject();
                            newOp.put("name", "setAttributes");
                            newOp.put("start", start);
                            newOp.put("attrs", attrs);
                            newOp.put("target", target);
                            reducedOperations.put(newOp);
                        }
                    }
                    else if ("insertRows".equals(opName))
                    {
                        insert = new Row(operation.optBoolean("insertDefaultCells", false));
                    }
                    else if ("insertCells".equals(opName))
                    {
                        insert = new Cell();
                    }
                    else if ("insertParagraph".equals(opName))
                    {
                        insert = new Paragraph();
                    }
                    else if ("insertText".equals(opName))
                    {
                        final int textIndex = start.getInt(start.length() - 1);
                        final String text = operation.getString("text");
                        insert = new Text(text, textIndex);

                    }
                    else if ("insertField".equals(opName))
                    {
                        final int textIndex = start.getInt(start.length() - 1);
                        insert = new SimpleField(textIndex, operation.optString("representation", ""), operation.optString("type", null));
                    }
                    else if (opName.equalsIgnoreCase("insertHardbreak"))
                    {
                        final int textIndex = start.getInt(start.length() - 1);
                        final String type = operation.optString("type");
                        insert = new HardBreak(textIndex, type);
                    }
                    else if (opName.equalsIgnoreCase("insertTab"))
                    {
                        final int textIndex = start.getInt(start.length() - 1);
                        insert = new Tab(textIndex);
                    }
                    else if (opName.equalsIgnoreCase("insertRange"))
                    {
                        final int textIndex = start.getInt(start.length() - 1);
                        final String type = operation.optString("type");
                        final String pos = operation.optString("position");
                        insert = new RangeMarker(textIndex, operation.optString("id"), type, pos);
                    }
                    else if ("setAttributes".equals(opName))
                    {

                    }
                    else
                    {
                        throw new Exception("PresentationHtmlDocumentBuilder.buildHtmlDocument() " + opName + " not implemented yet");

                    }

                    if (null == end)
                    {
                        end = start;
                    }

                    Container targetContainer = null;
                    if (!StringUtils.isEmpty(target))
                    {
                        targetContainer = masterSlides.get(target);

                        if (null == targetContainer)
                        {
                            targetContainer = layoutSlides.get(target);
                        }
                    }
                    else
                    {
                        targetContainer = pagecontent;
                    }

                    if (insert != null)
                    {
                        targetContainer.insert(start, insert);
                    }
                    if (attrs != null)
                    {
                        targetContainer.setAttribute(start, end, attrs);
                    }
                }
                catch (final Exception e)
                {
                    throw new Exception("operation: " + operation, e);
                }

            }

            final StringBuilder document = new StringBuilder();

            document.append("<div class='masterslidelayer'>");
            ArrayList<Entry<String, Container>> masterSorted = new ArrayList<Entry<String, Container>>(masterSlides.entrySet());
            Collections.sort(masterSorted, slideSorter);
            for (Entry<String, Container> e : masterSorted)
            {
                e.getValue().appendContent(document);
            }
            document.append("</div>");

            document.append("<div class='layoutslidelayer'>");
            ArrayList<Entry<String, Container>> layoutSorted = new ArrayList<Entry<String, Container>>(layoutSlides.entrySet());
            Collections.sort(layoutSorted, slideSorter);
            for (Entry<String, Container> e : layoutSorted)
            {
                e.getValue().appendContent(document);
            }
            document.append("</div>");

            pagecontent.appendContent(document);

            String content = document.toString();

            JSONObject res = new JSONObject();
            res.put("mainDocument", content);

            result = res.toString();

            documentOperations.put("operations", reducedOperations);

            LOG.debug("RT connection: fastLoad optimizations used for loading document: " + ((null != folderId) ? folderId : "") + "." + ((null != fileId) ? fileId : ""));
            LOG.trace("RT connection: TIME creating the generic html document took: " + (System.currentTimeMillis() - time) + "ms");
        }
        catch (

        final Exception e)

        {
            LOG.error("RT connection: Exception while creating the generic html document string from operations. docDetails: " + fileName + " fileId: " + fileId + " folderId: " + folderId + " fileVersion: " + fileVersion, e);
        }
        return result;
    }

}
