package com.openexchange.office.hazelcast.doc;

import java.util.Collection;
import java.util.Map;

import com.openexchange.exception.OXException;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentResources;

public interface DocumentResourcesDirectory {

    /**
     * Gets all available resources for the supplied ID, or an empty map if no matching resources are known. If the ID is in general
     * form, multiple entries may be returned, each mapped to their concrete ID. If the ID denotes a concrete resource, i.e. it's
     * resource-part is set, only one entry is returned if found.
     *
     * @param id The ID to lookup the status for
     * @return The resolved resources matching the supplied ID
     * @throws OXException
     */
    DocumentResources get(DocRestoreID id) throws OXException;

    /**
     * Gets all available resources for the supplied IDs, or an empty map if no matching resources are known. For each supplied ID, if
     * it is in general form, multiple entries may be returned, each mapped to their concrete ID. If a supplied ID denotes a concrete
     * resource, i.e. it's resource-part is set, only one entry is returned for that ID if found.
     *
     * @param ids The IDs to lookup the status for
     * @return The resolved resources matching the supplied ID
     * @throws OXException
     */
    Map<DocRestoreID, DocumentResources> get(Collection<DocRestoreID> ids) throws OXException;

    /**
     * Sets or updates the presence data of a resource identified by the supplied ID.
     *
     * @param id The (concrete) resource ID to set the status for
     * @param resource The resource data to set
     * @return The previously associated resource data of the ID in case of an update, <code>null</code> if there was no value associated
     *         with the ID before
     * @throws OXException
     */
    DocumentResources set(DocRestoreID id, DocumentResources resource) throws OXException;

    /**
     * Sets or updates the presence data of a resource identified by the supplied ID if and only if no previous resource was set for this id.
     *
     * @param id The (concrete) resource ID to set the status for
     * @param resource The resource data to set
     * @return The previously associated resource of the ID in case of an update, <code>null</code> if there was no value associated
     *         with the ID before. If a non-null value is returned here, the resource directory has not saved the supplied value.
     * @throws OXException
     */
    void setIfAbsent(DocRestoreID id, DocumentResources resource) throws OXException;

    /**
     * Removes all available resources for the supplied ID. If the ID is in general form, all matching entries are removed from the
     * directory. If the ID denotes a concrete resource, i.e. it's resource-part is set, only one entry is removed if found.
     *
     * @param id The ID to remove from the directory
     * @return All previously associated resources mapped to the ID, each mapped to the it's concrete resource ID.
     * @throws OXException
     */
    DocumentResources remove(DocRestoreID id) throws OXException;

    /**
     * Removes all available resources for the supplied IDs. For each ID, if it is in general form, all matching entries are removed
     * from the directory. If an ID denotes a concrete resource, i.e. it's resource-part is set, only one entry is removed if found.
     *
     * @param ids The IDs to remove from the directory
     * @return All previously associated resource mapped to the IDs, each mapped to the it's concrete resource ID.
     * @throws OXException
     */
    Map<DocRestoreID, DocumentResources> remove(Collection<DocRestoreID> ids) throws OXException;
}
