/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.drawingml;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Logger;
import javax.xml.bind.annotation.XmlElements;
import org.docx4j.dml.CTEffectList;
import org.docx4j.dml.CTLineJoinRound;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTNoFillProperties;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTSolidColorFillProperties;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextBodyProperties;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextField;
import org.docx4j.dml.CTTextLineBreak;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.ITextCharacterProperties;
import org.docx4j.dml.STLineCap;
import org.docx4j.dml.chart.CTArea3DChart;
import org.docx4j.dml.chart.CTAreaChart;
import org.docx4j.dml.chart.CTAxDataSource;
import org.docx4j.dml.chart.CTAxPos;
import org.docx4j.dml.chart.CTBar3DChart;
import org.docx4j.dml.chart.CTBarChart;
import org.docx4j.dml.chart.CTBarDir;
import org.docx4j.dml.chart.CTBarSer;
import org.docx4j.dml.chart.CTBoolean;
import org.docx4j.dml.chart.CTBubbleChart;
import org.docx4j.dml.chart.CTBubbleSer;
import org.docx4j.dml.chart.CTCatAx;
import org.docx4j.dml.chart.CTChart;
import org.docx4j.dml.chart.CTChartLines;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTCrossBetween;
import org.docx4j.dml.chart.CTCrosses;
import org.docx4j.dml.chart.CTDLbls;
import org.docx4j.dml.chart.CTDPt;
import org.docx4j.dml.chart.CTDateAx;
import org.docx4j.dml.chart.CTDoughnutChart;
import org.docx4j.dml.chart.CTFirstSliceAng;
import org.docx4j.dml.chart.CTGapAmount;
import org.docx4j.dml.chart.CTHoleSize;
import org.docx4j.dml.chart.CTLayout;
import org.docx4j.dml.chart.CTLegend;
import org.docx4j.dml.chart.CTLegendPos;
import org.docx4j.dml.chart.CTLine3DChart;
import org.docx4j.dml.chart.CTLineChart;
import org.docx4j.dml.chart.CTLineSer;
import org.docx4j.dml.chart.CTMarker;
import org.docx4j.dml.chart.CTMarkerStyle;
import org.docx4j.dml.chart.CTNumData;
import org.docx4j.dml.chart.CTNumDataSource;
import org.docx4j.dml.chart.CTNumFmt;
import org.docx4j.dml.chart.CTNumRef;
import org.docx4j.dml.chart.CTNumVal;
import org.docx4j.dml.chart.CTOfPieChart;
import org.docx4j.dml.chart.CTOrientation;
import org.docx4j.dml.chart.CTOverlap;
import org.docx4j.dml.chart.CTPie3DChart;
import org.docx4j.dml.chart.CTPieChart;
import org.docx4j.dml.chart.CTPieSer;
import org.docx4j.dml.chart.CTPlotArea;
import org.docx4j.dml.chart.CTRadarChart;
import org.docx4j.dml.chart.CTRadarStyle;
import org.docx4j.dml.chart.CTScaling;
import org.docx4j.dml.chart.CTScatterChart;
import org.docx4j.dml.chart.CTScatterSer;
import org.docx4j.dml.chart.CTScatterStyle;
import org.docx4j.dml.chart.CTSerAx;
import org.docx4j.dml.chart.CTSerTx;
import org.docx4j.dml.chart.CTStockChart;
import org.docx4j.dml.chart.CTStrData;
import org.docx4j.dml.chart.CTStrRef;
import org.docx4j.dml.chart.CTStrVal;
import org.docx4j.dml.chart.CTStyle;
import org.docx4j.dml.chart.CTSurface3DChart;
import org.docx4j.dml.chart.CTSurfaceChart;
import org.docx4j.dml.chart.CTTickLblPos;
import org.docx4j.dml.chart.CTTickMark;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.chart.CTTx;
import org.docx4j.dml.chart.CTUnsignedInt;
import org.docx4j.dml.chart.CTValAx;
import org.docx4j.dml.chart.IAxisDescription;
import org.docx4j.dml.chart.IListSer;
import org.docx4j.dml.chart.ISerContent;
import org.docx4j.dml.chart.ObjectFactory;
import org.docx4j.dml.chart.STAxPos;
import org.docx4j.dml.chart.STCrossBetween;
import org.docx4j.dml.chart.STCrosses;
import org.docx4j.dml.chart.STGrouping;
import org.docx4j.dml.chart.STLegendPos;
import org.docx4j.dml.chart.STMarkerStyle;
import org.docx4j.dml.chart.STOrientation;
import org.docx4j.dml.chart.STRadarStyle;
import org.docx4j.dml.chart.STScatterStyle;
import org.docx4j.dml.chart.STTickLblPos;
import org.docx4j.dml.chart.STTickMark;
import org.docx4j.dml.chart.layer.NoAxIdChart;
import org.docx4j.dml.chartStyle2012.CTChartStyle;
import org.docx4j.dml.chartStyle2012.CTColorStyle;
import org.docx4j.dml.chartStyle2012.CTColorStyleVariation;
import org.docx4j.dml.chartStyle2012.CTStyleEntry;
import org.docx4j.dml.chartStyle2012.CTStyleReference;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.ThemePart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import com.openexchange.office.filter.ooxml.tools.Commons;
import com.openexchange.office.filter.ooxml.tools.Commons.Pair;

final public class DMLChartSpace {

    private static final JSONArray         EMPTYTEXTARRAY;
    private static final CTShapeProperties DEFAULTSHAPE;

    static {
        EMPTYTEXTARRAY = new JSONArray();
        EMPTYTEXTARRAY.put(" ");

        DEFAULTSHAPE = new CTShapeProperties();
        DEFAULTSHAPE.setSolidFill(new CTSolidColorFillProperties());
    }

    public static void createChartSpaceOperations(JSONArray operationsArray, List<Integer> position, ChartWrapper chartPart) throws Exception {

        final CTChartSpace chartSpace = chartPart.getChartSpace();

        // MSO2012 chartStyles can be zero...
        final CTChartStyle chartStyle = chartPart.getChartStyle();
        // final CTColorStyle chartColorStyle = chartPart.getChartColorStyle();
        final ThemePart themePart = chartPart.getThemePart();

        final CTChart chart = chartSpace.getChart();
        if (chart == null) {
            return;
        }

        final CTPlotArea plotArea = chart.getPlotArea();
        if (plotArea != null) {
            final List<IListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
            // TODO: what if there is more than one chart in this list ?
            if (!chartList.isEmpty()) {

                boolean varyColors = false;
                boolean curved = false;
                boolean pie = false;

                final JSONObject insertDrawingOperation = operationsArray.getJSONObject(operationsArray.length() - 1);

                int seriesIndex = -1;
                for (final IListSer chartSer : chartList) {

                    varyColors = varyColors || getBoolean(chartSer.getVaryColors());
                    pie = pie || chartSer instanceof CTPieChart;
                    curved = curved || isCurved(chartSer);

                    String chartType = getSeriesType(chartSer);

                    int axisXIndex = getAxIndex(plotArea, chartSer.getAxId(), "x");
                    int axisYIndex = getAxIndex(plotArea, chartSer.getAxId(), "y");

                    // we are taking only the first chart ...
                    for (final ISerContent ser : chartSer.getSer()) {
                        seriesIndex++;
                        String dataLabel = getDataLabel(ser);

                        // title
                        final JSONObject series = new JSONObject();

                        series.put("axisXIndex", axisXIndex);
                        series.put("axisYIndex", axisYIndex);

                        final CTSerTx tx = ser.getTx();
                        Object title = null;
                        if (tx != null) {
                            if (tx.getStrRef() != null) {
                                title = tx.getStrRef().getF();
                            } else {
                                title = new JSONArray();
                                ((JSONArray) title).put(tx.getV());
                            }
                        }
                        if (title != null) {
                            series.put("title", title);
                        }

                        // names
                        Object names = null;
                        final CTAxDataSource cat = ser.getCat();
                        if (cat != null) {
                            if (cat.getStrRef() != null) {
                                names = cat.getStrRef().getF();
                            } else if (cat.getStrLit() != null) {
                                final CTStrData strLit = cat.getStrLit();
                                final List<CTStrVal> strLitPtList = strLit.getPt();
                                if (strLitPtList != null && !strLitPtList.isEmpty()) {
                                    names = new JSONArray();
                                    for (final CTStrVal strLitPtEntry : strLitPtList) {
                                        if (strLitPtEntry != null) {
                                            ((JSONArray) names).put(strLitPtEntry.getV());
                                        } else {
                                            ((JSONArray) names).put((String) null);
                                        }
                                    }
                                }
                            } else if (cat.getNumRef() != null) {
                                names = cat.getNumRef().getF();
                            } else if (cat.getNumLit() != null) {
                                final CTNumData numLit = cat.getNumLit();
                                final List<CTNumVal> numLitPtList = numLit.getPt();
                                if (numLitPtList != null && !numLitPtList.isEmpty()) {
                                    JSONArray array = new JSONArray();
                                    for (final CTNumVal numLitPtEntry : numLitPtList) {
                                        if (numLitPtEntry != null) {
                                            String v = numLitPtEntry.getV();
                                            try {
                                                array.put(Double.parseDouble(v));
                                            } catch (NumberFormatException e) {
                                                array.put(v);
                                            }
                                        } else {
                                            array.put((String) null);
                                        }
                                    }
                                    names = array;
                                }
                            }
                            else if (cat.getMultiLvlStrRef() != null) {
                                names = cat.getMultiLvlStrRef().getF();
                            }
                        }
                        if (names != null) {
                            series.put("names", names);
                        }

                        // values
                        final CTNumDataSource numDataSource = ser.getVal();
                        final Object values = getData(numDataSource);

                        if (values != null) {
                            series.put("values", values);
                        }

                        if (ser instanceof CTBubbleSer) {
                            final CTBubbleSer serBB = (CTBubbleSer) ser;

                            final CTNumDataSource bubblesizes = serBB.getBubbleSize();
                            if (bubblesizes != null) {
                                final Object bubbles = getData(bubblesizes);
                                if (bubbles != null) {
                                    series.put("bubbles", bubbles);
                                }
                            }
                        }

                        series.put("type", chartType);

                        JSONArray dataPoints = new JSONArray();

                        if (ser.getDPt() != null) {
                            for (CTDPt dpt : ser.getDPt()) {
                                final CTShapeProperties shape = dpt.getSpPr();
                                final JSONObject attrs = new JSONObject();
                                DMLHelper.createJsonFromShapeProperties(attrs, shape, themePart, chartStyle.getDataPoint(), chartPart.getChartPart(), false, true);
                                dataPoints.put(attrs);
                            }
                        }

                        if (!dataPoints.isEmpty()) {
                            series.put("dataPoints", dataPoints);
                        }

                        if (dataLabel != null) {
                            series.put("dataLabel", dataLabel);
                        }

                        CTDLbls dLbl = ser.getDLbls();
                        if (dLbl != null && dLbl.getNumFmt() != null && !dLbl.getNumFmt().isSourceLinked()) {
                            series.put("format", dLbl.getNumFmt().getFormatCode());
                        }

                        final JSONObject attrs = Commons.surroundJSONObject("series", series);

                        DMLHelper.createJsonFromShapeProperties(attrs, ser.getSpPr(), themePart, chartStyle.getDataPoint(), chartPart.getChartPart(), false, true);

                        if (ser instanceof CTLineSer || ser instanceof CTScatterSer) {
                            JSONObject line = attrs.optJSONObject("line");
                            if (line != null) {
                                if (!line.has("type")) {
                                    line.put("type", "solid");
                                }
                            } else {
                                line = new JSONObject();
                                line.put("type", "solid");
                                attrs.put("line", line);
                            }
                            attrs.remove("fill");
                        } else {
                            // fix for Bug 52765 & Bug 46848 
                            if (ser.getSpPr() != null && ser.getSpPr().getLn() == null) {
                                attrs.remove("line");
                            }
                        }

                        boolean hasMarker = false;
                        final CTMarker marker = ser.getMarker();
                        if (marker != null) {
                            final CTShapeProperties shapePr = marker.getSpPr();
                            if (shapePr != null) {
                                final CTStyleEntry styleRef = chartStyle.getDataPointMarker();
                                final CTStyleReference fillRef = styleRef != null ? styleRef.getFillRef() : null;
                                DMLHelper.createJsonFromFillProperties(attrs, shapePr, themePart, fillRef, chartPart.getChartPart());
                            } else if (marker.getSymbol() == null || marker.getSymbol().getVal() == null || marker.getSymbol().getVal() != STMarkerStyle.NONE) {
                                hasMarker = true;
                            }
                        } else if (chartSer instanceof CTLineChart && getBoolean(((CTLineChart) chartSer).getMarker())) {
                            hasMarker = true;
                        } else if (chartSer instanceof CTScatterChart && ((CTScatterChart) chartSer).getScatterStyle().getVal().value().toLowerCase().contains("marker")) {
                            hasMarker = true;
                        }

                        if (hasMarker) {
                            attrs.put("fill", new JSONObject(new JSONTokener("{type:'solid'}")));
                        }

                        // TODO: chartColorStyle needs to be supported... line and fill styles should be created by following call (text, txtBody has to be added)
                        // DMLHelper.createJsonFromShapeProperties(attrs, ser.getSpPr(), theme, chartStyle!=null?chartStyle.getChartArea():null, false);
                        if (series.length() > 0) {
                            final JSONObject insertChartDataSeriesOperation = new JSONObject(4);
                            insertChartDataSeriesOperation.put("name", "insertChartDataSeries");
                            insertChartDataSeriesOperation.put("start", position);
                            insertChartDataSeriesOperation.put("series", seriesIndex);
                            insertChartDataSeriesOperation.put("attrs", attrs);
                            operationsArray.put(insertChartDataSeriesOperation);
                        }

                    }
                }
                final JSONObject attrs = insertDrawingOperation.getJSONObject("attrs");
                final JSONObject chartProperties = attrs.getJSONObject("chart");

                if (pie) {
                    chartProperties.put("rotation", ((CTPieChart) chartList.get(0)).getFirstSliceAng().getVal());
                }

                if (varyColors) {
                    chartProperties.put("varyColors", varyColors);
                }

                if (curved) {
                    chartProperties.put("curved", true);
                }

                final CTShapeProperties shapePr = chartSpace.getSpPr();
                if (shapePr != null) {
                    final CTStyleEntry styleRef = chartStyle.getDataPointMarker();
                    final CTStyleReference fillRef = styleRef != null ? styleRef.getFillRef() : null;
                    DMLHelper.createJsonFromFillProperties(attrs, shapePr, themePart, fillRef, chartPart.getChartPart());
                }
            }

            addToOperations(chartPart, operationsArray, position);

            final CTTitle title = chart.getTitle();
            final JSONObject attrs = createJsonFromProperties(title);
            if (attrs != null) {
                final JSONObject op = new JSONObject(4);
                op.put("name", "setChartTitleAttributes");
                op.put("start", position);
                op.put("axis", "main");
                op.put("attrs", attrs);
                operationsArray.put(op);
            }

            final JSONObject legendAttrs = createLegendAttrs(chart);
            if (legendAttrs != null) {
                final JSONObject op = new JSONObject(3);
                op.put("name", "setChartLegendAttributes");
                op.put("start", position);
                op.put("attrs", legendAttrs);
                operationsArray.put(op);
            }
        }
    }

    private static JSONObject createLegendAttrs(CTChart chart) throws JSONException {
        final JSONObject ljset = new JSONObject();
        final JSONObject lj = new JSONObject();
        lj.put("pos", "off");

        final CTLegend legend = chart.getLegend();
        if (legend != null) {
            //right is default, if legend exists
            lj.put("pos", "right");
            if (legend.getLegendPos() != null && legend.getLegendPos().getVal() != null) {

                final String pos;
                switch (legend.getLegendPos().getVal()) {
                    case R:
                        pos = "right";
                        break;
                    case T:
                        pos = "top";
                        break;
                    case B:
                        pos = "bottom";
                        break;
                    case TR:
                        pos = "topRight";
                        break;
                    default:
                        pos = "left";
                        break;
                }

                lj.put("pos", pos);

                final JSONObject character = DMLHelper.createJsonFromProperties(legend.getTxPr());
                if (character != null) {
                    ljset.put("character", character);
                }
            }
        }
        ljset.put("legend", lj);

        return ljset;
    }

    private static Object getData(CTNumDataSource numDataSource) throws JSONException {
        if (numDataSource == null) {
            Logger.getAnonymousLogger().warning("CTNumDataSource is null");
            return null;
        }
        if (numDataSource.getNumRef() != null) {
            return numDataSource.getNumRef().getF();
        } else if (numDataSource.getNumLit() != null) {
            final CTNumData numLit = numDataSource.getNumLit();
            if (numLit != null) {
                final List<CTNumVal> numLitPtList = numLit.getPt();
                if (numLitPtList != null && !numLitPtList.isEmpty()) {
                    final JSONArray values = new JSONArray();
                    for (final CTNumVal numLitPt : numLitPtList) {
                        if (numLitPt != null) {
                            String v = numLitPt.getV();
                            try {
                                values.put(Float.parseFloat(v));
                            } catch (NumberFormatException e) {
                                values.put(v);
                            }
                        } else {
                            values.put((String) null);
                        }
                    }
                    return values;
                }
            }
        }
        return null;
    }

    public static void applyChartSpaceProperties(String chartType, JSONObject chartProperties, ChartWrapper chart, JSONObject fill) throws Exception {
        final CTPlotArea plotArea = chart.getPlotArea();
        final CTChartSpace chartSpace = chart.getChartSpace();

        if (chartProperties == null) {
            chartProperties = new JSONObject();
        }

        String stacking = chartProperties.optString("stacking");
        if (stacking == null || stacking.isEmpty()) {
            stacking = getStackingType(chart);
        }
        if (stacking == null || stacking.isEmpty()) {
            stacking = STGrouping.STANDARD.value();
        }

        if (chartType == null && chartProperties.optBoolean("curved")) {
            final IListSer oldListSer = plotArea.getAreaChartOrArea3DChartOrLineChart().get(0);
            if (!(oldListSer instanceof CTLineChart) && !(oldListSer instanceof CTScatterChart)) {
                chartType = "line2d";
            }
        }

        if (chartType != null) {
            // indirect call of getThemePart(createIfMissing=true) (Bug 46806)
            chart.getThemePart();

            List<IListSer> allCharts = plotArea.getAreaChartOrArea3DChartOrLineChart();

            //workaround for Bug 47969
            while (allCharts.size() > 1) {
                allCharts.remove(1);
            }

            final IListSer oldListSer = allCharts.get(0);

            final IListSer newListSer = createChartType(chartType, stacking);

            final Class<? extends ISerContent> type = getNewContentType(newListSer);

            if (newListSer != null) {
                boolean noAxIdChart = newListSer instanceof NoAxIdChart;

                final List<CTUnsignedInt> axid = newListSer.getAxId();
                final List<CTUnsignedInt> oldAxid = oldListSer.getAxId();
                final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
                if (axid != null && oldAxid != null && !noAxIdChart) {
                    axid.clear();
                    axid.addAll(oldAxid);
                } else if (axid == null) {
                    if (axes != null && oldAxid != null) {
                        for (int i = axes.size() - 1; i >= 0; i--) {
                            IAxisDescription ax = axes.get(i);
                            CTUnsignedInt id = ax.getAxId();
                            for (CTUnsignedInt old : oldAxid) {
                                if (id.getVal() == old.getVal()) {
                                    axes.remove(i);
                                    break;
                                }
                            }

                        }
                    }
                } else if (axes.size() == 0 && !noAxIdChart) {
                    getAxis(plotArea, "x");
                    getAxis(plotArea, "y");
                }

                if (noAxIdChart && axes != null) {
                    axes.clear();
                }

                allCharts.set(0, newListSer);

                newListSer.setVaryColors(oldListSer.getVaryColors());

                setCurved(newListSer, chartProperties.optBoolean("curved", isCurved(oldListSer)));

                for (final ISerContent serContent : oldListSer.getSer()) {
                    final ISerContent newContent;
                    if (serContent.getClass().equals(type)) {
                        newContent = serContent;
                    } else {
                        newContent = type.newInstance();
                        newContent.setCat(serContent.getCat());
                        newContent.setExtLst(serContent.getExtLst());
                        newContent.setIdx(serContent.getIdx());
                        newContent.setOrder(serContent.getOrder());
                        newContent.setSpPr(serContent.getSpPr());
                        newContent.setTx(serContent.getTx());
                        newContent.setVal(serContent.getVal());
                        newContent.setDLbls(serContent.getDLbls());

                        if (newContent instanceof CTDoughnutChart) {
                            handleStandardDonutChart((CTDoughnutChart) newContent);
                        } else if (newContent instanceof CTPieChart) {
                            handleStandardPieChart((CTPieChart) newContent);
                        }
                    }
                    newListSer.getSer().add(newContent);
                }

                if (axes.size() > 0) {
                    ArrayList<IAxisDescription> oldList = new ArrayList<>(axes);
                    axes.clear();

                    for (int j = 0; j < oldList.size(); j++) {
                        IAxisDescription ax = oldList.get(j);
                        AxPos pos = getAxPos(oldList, j);
                        IAxisDescription nAx = getAxis(plotArea, pos.id);
                        copy(nAx, ax);
                    }
                }
            }
            refreshTitleRotations(chart);
        }

        final IListSer listSer = plotArea.getAreaChartOrArea3DChartOrLineChart().get(0);
        deleteEmptyContentLists(listSer);

        if (chartProperties.has("chartStyleId")) {
            CTStyle chartStyle = chartSpace.getStyle();
            if (chartStyle == null) {
                chartStyle = new CTStyle();
                chartSpace.setStyle(chartStyle);
            }
            chartStyle.setVal((short) chartProperties.getInt("chartStyleId"));

            /*
             * // FIXME: this should come with the series-operation
             * // delete all shapeproperties, but keep the Line-NoFill-Property
             * for (final ISerContent serContent : listSer.getSer()) {
             * CTShapeProperties sppr = serContent.getSpPr();
             * if (sppr != null ){
             * CTLineProperties ln = sppr.getLn();
             * if (null != ln && null != ln.getNoFill()) {
             * sppr = new CTShapeProperties();
             * sppr.setLn(ln);
             * serContent.setSpPr(sppr);
             * } else {
             * serContent.setSpPr(null);
             * }
             * }
             * final List<CTDPt> dpt = serContent.getDPt();
             * if (dpt != null) {
             * for (final CTDPt point : serContent.getDPt()) {
             * point.setSpPr(null);
             * }
             * }
             * }
             */
        }
        listSer.setStacking(stacking);

        if (listSer instanceof CTBarChart) {
            handleOverlap((CTBarChart) listSer, stacking);
        }

        if (chartProperties.has("curved")) {
            if (listSer instanceof CTLineChart || listSer instanceof CTScatterChart) {
                boolean curved = chartProperties.optBoolean("curved", false);
                setCurved(listSer, curved);
            }
        }
        if (chartProperties.has("varyColors")) {
            listSer.setVaryColors(getBoolean(chartProperties.optBoolean("varyColors", false)));
        }

        if (listSer instanceof CTScatterChart) {
            handleStandardScatterChart((CTScatterChart) listSer);

        }

        // reset generic axis numbers
        // for Bug 46441
        if (chartProperties.has("stacking") || chartProperties.has("type")) {
            for (IAxisDescription axis : plotArea.getValAxOrCatAxOrDateAx()) {
                if (axis instanceof CTDateAx) {
                    ((CTDateAx) axis).setMajorUnit(null);
                    ((CTDateAx) axis).setMinorUnit(null);
                    ((CTDateAx) axis).setMajorTimeUnit(null);
                    ((CTDateAx) axis).setMinorTimeUnit(null);
                } else if (axis instanceof CTValAx) {
                    ((CTValAx) axis).setMajorUnit(null);
                    ((CTValAx) axis).setMinorUnit(null);
                }

                CTScaling scaling = axis.getScaling();
                if (null != scaling) {
                    scaling.setMin(null);
                    scaling.setMax(null);
                }
            }
        }

        if (fill != null) {
            final JSONObject fillHolder = new JSONObject();
            fillHolder.put("fill", fill);

            chartSpace.setSpPr(applyShapePropertiesFromJson(plotArea.getSpPr(), fillHolder, true));

            plotArea.setSpPr(new CTShapeProperties());
            plotArea.getSpPr().setNoFill(new CTNoFillProperties());

            chartSpace.setRoundedCorners(getBoolean(false));
        }

    }

    private static void copy(IAxisDescription target, IAxisDescription source) {
        // workaround for Bug 53208, new code create its own axes ids, so copy these ids is not needed any more
        //        target.setAxId(source.getAxId());
        //        target.setAxPos(source.getAxPos());
        //        target.setDelete(source.getDelete());

        target.setMajorGridlines(source.getMajorGridlines());
        target.setMajorTickMark(source.getMajorTickMark());
        target.setSpPr(source.getSpPr());
        target.setTickLblPos(source.getTickLblPos());
        target.setTitle(source.getTitle());
        target.setTxPr(source.getTxPr());
    }

    public static CTBoolean getBoolean(boolean value) {
        final CTBoolean res = new CTBoolean();
        res.setVal(value);
        return res;
    }

    public static boolean getBoolean(CTBoolean value) {
        if (value == null) {
            return false;
        }
        return value.isVal();
    }

    public static CTUnsignedInt getInteger(int value) {
        final CTUnsignedInt res = new CTUnsignedInt();
        res.setVal(value);
        return res;
    }

    public static CTUnsignedInt getInteger(long value) {
        final CTUnsignedInt res = new CTUnsignedInt();
        res.setVal(value);
        return res;
    }

    private static void deleteEmptyContentLists(IListSer listSer) {
        for (final ISerContent serContent : listSer.getSer()) {
            if (hasEmptyContents(serContent.getCat())) {
                serContent.setCat(null);
            }
            if (hasEmptyContents(serContent.getTx())) {
                serContent.setTx(null);
            }
            if (serContent instanceof CTBubbleSer) {
                final CTBubbleSer serBub = ((CTBubbleSer) serContent);
                if (hasEmptyContents(serBub.getBubbleSize())) {
                    serBub.setBubbleSize(null);
                }
            }
        }
    }

    private static boolean hasEmptyContents(CTNumDataSource data) {
        if (data == null) {
            return true;
        }
        if (data.getNumLit() != null || data.getNumRef() != null) {
            return false;
        }
        return true;
    }

    private static boolean hasEmptyContents(CTAxDataSource data) {
        if (data == null) {
            return true;
        }
        if (data.getMultiLvlStrRef() != null || data.getNumLit() != null || data.getNumRef() != null || data.getStrLit() != null || data.getStrRef() != null) {
            return false;
        }
        return true;
    }

    private static boolean hasEmptyContents(CTSerTx data) {
        if (data == null) {
            return true;
        }
        if (data.getStrRef() != null || data.getV() != null) {
            return false;
        }
        return true;
    }

    @SuppressWarnings("unchecked")
    private static Class<? extends ISerContent> getNewContentType(IListSer listSer) throws Exception {
        final XmlElements serType = listSer.getClass().getDeclaredField("ser").getAnnotation(XmlElements.class);
        return serType.value()[0].type();
    }

    // Idx and Order are required elements, so at least we have to create this
    // two elements for our new
    // inserted series... but we will correct indexes of other series, so that
    // they are properly increasing
    private static void updateIndexAndOrder(CTPlotArea plotArea) {
        int index = 0;
        for (IListSer listSer : plotArea.getAreaChartOrArea3DChartOrLineChart()) {
            for (ISerContent serCont : listSer.getSer()) {
                serCont.setIdx(getInteger(index));
                serCont.setOrder(getInteger(index));
                index++;
            }
        }
    }

    public static void insertChartSpace(ChartWrapper chart, int series, JSONObject attrs) throws Exception {

        JSONObject attrSeries = attrs.optJSONObject("series");
        String type = opToOOXMLType(attrSeries.optString("type"));

        String oldType = opToOOXMLType(getChartType(chart));

        if (oldType != null && !oldType.equals(type)) {

            if (series > 0) {
                final IListSer newListSer = createChartType(type, null);
                chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart().add(newListSer);
            } else {
                applyChartSpaceProperties(type, null, chart, null);
            }
            if (attrSeries != null) {
                List<IListSer> allCharts = chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart();
                final IListSer listSer = allCharts.get(allCharts.size() - 1);

                if (!(listSer instanceof NoAxIdChart)) {
                    List<CTUnsignedInt> axId = listSer.getAxId();
                    axId.clear();

                    String xAxis = attrSeries.optInt("axisXIndex", 0) == 1 ? "x2" : "x";
                    String yAxis = attrSeries.optInt("axisYIndex", 0) == 1 ? "y2" : "y";

                    axId.add(getAxis(chart.getPlotArea(), xAxis).getAxId());
                    axId.add(getAxis(chart.getPlotArea(), yAxis).getAxId());
                }
            }
        }

        // "series - 1" is important to find the correct element before, to put it after it
        Pair<Integer, Pair<ISerContent, IListSer>> local = globalToLocal(chart.getPlotArea(), series - 1);

        final int seriesIndex;
        final IListSer serContentList;
        if (local.getFirst() < 0) {
            seriesIndex = 0;
            List<IListSer> allCharts = chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart();
            serContentList = allCharts.get(allCharts.size() - 1);
        } else {
            seriesIndex = local.getFirst() + 1;
            serContentList = local.getSecond().getSecond();
        }

        final ISerContent serContent = serContentList.createSer();

        serContentList.getSer().add(seriesIndex, serContent);

        disableInvertIfNegative(serContent);
        setCurved(serContent, isCurved(serContentList));

        if (!attrs.has("fill")) {
            // no fill -> no marker info -> visible marker, but we dont want that!
            attrs.put("fill", new JSONObject(new JSONTokener("{type:'none'}")));
        }

        setDataseriesAttrs(chart.getPlotArea(), serContent, attrs);

    }

    public static void setDataSeriesAttributes(ChartWrapper chart, int series, JSONObject attrs) throws JSONException, InvalidFormatException, PartUnrecognisedException {

        Pair<Integer, Pair<ISerContent, IListSer>> local = globalToLocal(chart.getPlotArea(), series);
        ISerContent serContent = local.getSecond().getFirst();
        IListSer listSer = local.getSecond().getSecond();

        disableInvertIfNegative(serContent);
        setCurved(serContent, isCurved(listSer));

        setDataseriesAttrs(chart.getPlotArea(), serContent, attrs);
    }

    public static void deleteChartSpace(ChartWrapper chart, int series) {

        Pair<Integer, Pair<ISerContent, IListSer>> local = globalToLocal(chart.getPlotArea(), series);
        ISerContent serContent = local.getSecond().getFirst();
        IListSer listSer = local.getSecond().getSecond();

        if (listSer == null) {
            // TODO: look at insertChartSpace() "if (series > 0) {"
            // seems to be already deleted
            // is broken by design
            // 
            return;
        }

        listSer.getSer().remove(serContent);

        // fix for Bug 53212 (delete all series)
        if (listSer.getSer().isEmpty() && chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart().size() > 1) {
            chart.getPlotArea().getAreaChartOrArea3DChartOrLineChart().remove(listSer);
        }

        updateIndexAndOrder(chart.getPlotArea());
    }

    private static Pair<Integer, Pair<ISerContent, IListSer>> globalToLocal(CTPlotArea plotArea, int series) {
        List<IListSer> all = plotArea.getAreaChartOrArea3DChartOrLineChart();

        int index = 0;
        for (IListSer listSer : all) {
            int localIndex = 0;
            for (ISerContent content : listSer.getSer()) {
                if (series == index) {
                    return new Pair<Integer, Pair<ISerContent, IListSer>>(localIndex, new Pair<>(content, listSer));
                }
                localIndex++;
                index++;
            }
        }
        return new Pair<Integer, Pair<ISerContent, IListSer>>(-1, new Pair<>((ISerContent) null, (IListSer) null));
    }

    private static void disableInvertIfNegative(ISerContent serContent) {
        if (serContent instanceof CTBarSer) {
            ((CTBarSer) serContent).setInvertIfNegative(getBoolean(false));
        } else if (serContent instanceof CTBubbleSer) {
            ((CTBubbleSer) serContent).setInvertIfNegative(getBoolean(false));
        }
        if (serContent.getDPt() != null) {
            for (CTDPt dpt : serContent.getDPt()) {
                dpt.setInvertIfNegative(getBoolean(false));
            }
        }
    }

    private static void setDataseriesAttrs(CTPlotArea plotArea, ISerContent serContent, JSONObject attrs) throws JSONException, InvalidFormatException, PartUnrecognisedException {

        if (attrs != null) {
            final JSONObject attrSeries = attrs.optJSONObject("series");
            if (attrSeries != null) {

                // title
                if (attrSeries.has("title")) {
                    CTSerTx serTx = null;
                    final Object title = attrSeries.get("title");
                    if (title != null) {
                        serTx = Context.getDmlChartObjectFactory().createCTSerTx();

                        if (title instanceof String) {
                            final CTStrRef strRef = serTx.getStrRef(true);
                            strRef.setF((String) title);

                            // empty cache for Net-Charts! (Bug 49821)
                            strRef.setStrCache(Context.getDmlChartObjectFactory().createCTStrData());
                        } else if (title instanceof JSONArray && !((JSONArray) title).isEmpty()) {
                            final Object o = ((JSONArray) title).get(0);
                            if (o instanceof String) {
                                serTx.setV((String) o);
                            }
                        } else {
                            Logger.getAnonymousLogger().warning("Cant handle Chart Title " + title);
                            serTx = null;
                        }
                    }
                    serContent.setTx(serTx);
                }
                // names
                if (attrSeries.has("names")) {
                    CTAxDataSource cat = null;
                    final Object names = attrSeries.get("names");
                    if (names != null) {
                        cat = Context.getDmlChartObjectFactory().createCTAxDataSource();

                        if (names instanceof String) {
                            final CTStrRef strRef = cat.getStrRef(true);
                            strRef.setF((String) names);

                            // empty cache for Net-Charts! (Bug 49821)
                            strRef.setStrCache(Context.getDmlChartObjectFactory().createCTStrData());
                        } else if (names instanceof JSONArray) {
                            final JSONArray titleArray = (JSONArray) names;
                            final CTStrData strData = cat.getStrLit(true);
                            strData.setPtCount(getInteger(titleArray.length()));
                            for (int i = 0; i < titleArray.length(); i++) {
                                final CTStrVal strVal = Context.getDmlChartObjectFactory().createCTStrVal();
                                strVal.setV(titleArray.optString(i, ""));
                                strData.getPt().add(strVal);
                            }
                        } else {
                            Logger.getAnonymousLogger().warning("Cant handle Chart Names " + names);
                            cat = null;
                        }
                    }
                    serContent.setCat(cat);
                }
                // values
                if (attrSeries.has("values")) {
                    CTNumDataSource val = null;
                    final Object values = attrSeries.get("values");
                    if (values != null) {
                        val = Context.getDmlChartObjectFactory().createCTNumDataSource();
                        insertSeriesAttribute(val, values);
                    }
                    serContent.setVal(val);
                }

                // bubbles
                if (attrSeries.has("bubbles") && serContent instanceof CTBubbleSer) {
                    CTNumDataSource bub = null;
                    final Object bubbles = attrSeries.opt("bubbles");
                    if (bubbles != null) {
                        bub = Context.getDmlChartObjectFactory().createCTNumDataSource();
                        insertSeriesAttribute(bub, bubbles);
                    }
                    ((CTBubbleSer) serContent).setBubbleSize(bub);
                }
                if (serContent instanceof CTScatterSer) {
                    handleStandardScatterSer((CTScatterSer) serContent);
                }

                if (attrSeries.has("dataLabel")) {
                    setDataLabel(serContent, attrSeries.optString("dataLabel", "none"));
                }

                if (attrSeries.has("dataPoints")) {
                    JSONArray dataPoints = attrSeries.getJSONArray("dataPoints");
                    List<CTDPt> points = serContent.getDPt();
                    if (points != null) {
                        points.clear();

                        for (int i = 0; i < dataPoints.length(); i++) {
                            JSONObject dataPoint = dataPoints.getJSONObject(i);

                            CTDPt point = Context.getDmlChartObjectFactory().createCTDPt();
                            point.setInvertIfNegative(getBoolean(false));
                            point.setIdx(getInteger(i));
                            point.setSpPr(applyShapePropertiesFromJson(point.getSpPr(), dataPoint, false));

                            points.add(point);
                        }
                    } else {
                        Logger.getAnonymousLogger().warning("no DPt! for " + serContent.getClass().getSimpleName());
                    }
                }

                if (attrSeries.has("format")) {
                    CTDLbls lbl = serContent.getDLbls();
                    if (lbl == null) {
                        lbl = newDataLabel();
                        serContent.setDLbls(lbl);
                    }
                    String format = attrSeries.getString("format");
                    if (format != null && !format.isEmpty()) {
                        CTNumFmt nf = new CTNumFmt();
                        nf.setSourceLinked(false);
                        nf.setFormatCode(format);
                        lbl.setNumFmt(nf);
                    } else {
                        lbl.setNumFmt(null);
                    }
                }
            }

            if (serContent instanceof CTLineSer || serContent instanceof CTScatterSer) {

                serContent.setSpPr(applyShapePropertiesFromJson(serContent.getSpPr(), attrs, false));

                if (attrs.has("fill")) {

                    // System.out.println("DMLChartSpace.setDataseriesAttrs() " + attrs.opt("fill"));

                    CTMarker marker = serContent.getMarker();
                    if (marker == null) {
                        marker = Context.getDmlChartObjectFactory().createCTMarker();
                        if (serContent instanceof CTLineSer) {
                            ((CTLineSer) serContent).setMarker(marker);
                        } else if (serContent instanceof CTScatterSer) {
                            ((CTScatterSer) serContent).setMarker(marker);
                        }
                    }

                    if (marker.getSpPr() == null) {
                        marker.setSpPr(new CTShapeProperties());
                        //                        marker.getSpPr().setNoFill(new CTNoFillProperties());
                        //                        marker.getSpPr().setLn(new CTLineProperties());
                        //                        marker.getSpPr().getLn().setNoFill(new CTNoFillProperties());
                    }

                    DMLHelper.applyFillPropertiesFromJson(marker.getSpPr(), attrs, null, null);

                    JSONObject fill = attrs.optJSONObject("fill");
                    if (fill.optString("type").equals("none")) {
                        marker.setSymbol(new CTMarkerStyle());
                        marker.getSymbol().setVal(STMarkerStyle.NONE);
                    }

                }

                // Bug 53085
                if (attrs.has("line")) {
                    CTShapeProperties sp = serContent.getSpPr();
                    if (sp == null) {
                        sp = new CTShapeProperties();
                        serContent.setSpPr(sp);
                    }
                    sp.setLn(DMLHelper.applyLinePropertiesFromJson(sp.getLn(), attrs.optJSONObject("line")));
                }
            } else {
                // Bug 51323 & 41988
                if (serContent instanceof CTPieSer || (attrSeries != null && !attrSeries.has("dataPoints"))) {
                    attrs.remove("fill");
                    serContent.setSpPr(null);
                } else {
                    serContent.setSpPr(applyShapePropertiesFromJson(serContent.getSpPr(), attrs, false));
                }
            }

            updateIndexAndOrder(plotArea);
        }
    }

    private static CTShapeProperties applyShapePropertiesFromJson(CTShapeProperties shape, JSONObject attrs, boolean rootShape) throws InvalidFormatException, PartUnrecognisedException, JSONException {

        // workaround for Bug 43983 & 44012
        // normal fill behavior works fine, but not for Dark Chart, then they will be invisible,
        // but without any fill info everything looks fine

        if (attrs.has("fill")) {
            JSONObject fill = attrs.optJSONObject("fill");
            if (fill != null) {
                if (fill.optString("type", "").equals("solid")) {
                    JSONObject color = fill.optJSONObject("color");
                    if (color == null || color.optString("type", "").equals("auto")) {
                        //attrs.remove("fill");
                        fill.remove("color");
                    }
                }
            }
            // fix for Bug 48209
            handlePlaceholderColor(attrs, "fill");

        }
        if (attrs.has("line")) {
            handlePlaceholderColor(attrs, "line");
        }

        return DMLHelper.applyShapePropertiesFromJson(shape, attrs, null, null, rootShape);
    }

    private static void handlePlaceholderColor(JSONObject attrs, String key) {
        JSONObject shape = attrs.optJSONObject(key);
        if (null != shape) {
            if (shape.optString("type", "").equals("none")) {
                shape.remove("color");
            } else {
                JSONObject color = shape.optJSONObject("color");
                if (null != color) {
                    String type = color.optString("type");
                    if (null != type && type.equals("scheme")) {
                        String value = color.optString("value");
                        if (null != value && value.equals("phClr")) {
                            //attrs.remove(key);
                            shape.remove("color");
                        }
                    }
                }
            }
        }
    }

    private static void insertSeriesAttribute(CTNumDataSource source, Object attribute) {
        if (attribute instanceof String) {
            final CTNumRef numRef = source.getNumRef(true);
            numRef.setF((String) attribute);

            // empty cache for Net-Charts! (Bug 49821)
            numRef.setNumCache(Context.getDmlChartObjectFactory().createCTNumData());
        } else if (attribute instanceof JSONArray) {
            final JSONArray bubblesArray = (JSONArray) attribute;
            final CTNumData numData = Context.getDmlChartObjectFactory().createCTNumData();
            source.setNumLit(numData);
            numData.setPtCount(getInteger(bubblesArray.length()));
            for (int i = 0; i < bubblesArray.length(); i++) {
                final CTNumVal numVal = Context.getDmlChartObjectFactory().createCTNumVal();
                numVal.setIdx(i);
                numVal.setV(bubblesArray.optString(i, ""));
                numData.getPt().add(numVal);
            }
        }
    }

    public static CTChartSpace createChartSpace(JSONObject chartAttrs) {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTChartSpace chartSpace = chartObjectFactory.createCTChartSpace();
        final CTChart chart = chartObjectFactory.createCTChart();
        chartSpace.setChart(chart);
        final CTPlotArea plotArea = chartObjectFactory.createCTPlotArea();
        chart.setPlotArea(plotArea);
        final CTLegend legend = chartObjectFactory.createCTLegend();
        chart.setLegend(legend);
        final CTLegendPos legendPos = chartObjectFactory.createCTLegendPos();
        legend.setLegendPos(legendPos);
        legendPos.setVal(STLegendPos.R);
        chart.setPlotVisOnly(getBoolean(true));
        final IListSer listSer = createChartType("bar2d", chartAttrs.optString("stacking"));
        plotArea.getAreaChartOrArea3DChartOrLineChart().add(listSer);

        getAxis(plotArea, "x");
        getAxis(plotArea, "y");

        return chartSpace;
    }

    private static CTCatAx createCatAx() {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTCatAx catAx = chartObjectFactory.createCTCatAx();
        final CTScaling scaling = chartObjectFactory.createCTScaling();
        final CTOrientation orientation = chartObjectFactory.createCTOrientation();
        orientation.setVal(STOrientation.MIN_MAX);
        scaling.setOrientation(orientation);
        catAx.setScaling(scaling);
        catAx.setNumFmt(null);
        catAx.setSpPr(getHiddenShapeProperties());

        catAx.setTickLblPos(new CTTickLblPos());
        catAx.getTickLblPos().setVal(STTickLblPos.NONE);

        return catAx;
    }

    private static CTValAx createValAx() {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTValAx valAx = chartObjectFactory.createCTValAx();
        final CTScaling scaling = chartObjectFactory.createCTScaling();
        final CTOrientation orientation = chartObjectFactory.createCTOrientation();
        orientation.setVal(STOrientation.MIN_MAX);
        scaling.setOrientation(orientation);
        valAx.setScaling(scaling);
        valAx.setNumFmt(null);
        valAx.setSpPr(getHiddenShapeProperties());

        valAx.setTickLblPos(new CTTickLblPos());
        valAx.getTickLblPos().setVal(STTickLblPos.NONE);

        return valAx;
    }

    private static CTSerAx createSerAx() {
        final org.docx4j.dml.chart.ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        final CTSerAx serAx = chartObjectFactory.createCTSerAx();
        final CTScaling scaling = chartObjectFactory.createCTScaling();
        final CTOrientation orientation = chartObjectFactory.createCTOrientation();
        orientation.setVal(STOrientation.MIN_MAX);
        scaling.setOrientation(orientation);
        serAx.setScaling(scaling);
        serAx.setNumFmt(null);
        serAx.setSpPr(getHiddenShapeProperties());
        serAx.setTickLblPos(new CTTickLblPos());
        serAx.getTickLblPos().setVal(STTickLblPos.NONE);

        return serAx;
    }

    private static void createAxId(List<CTUnsignedInt> axId) {
        axId.add(getInteger(5000));
        axId.add(getInteger(5001));
    }

    // TODO: creator for pie3d needs to be implemented...

    private static String opToOOXMLType(String chartType) {
        if (!chartType.endsWith("2d") && !chartType.endsWith("3d")) {
            chartType = chartType + "2d";
        }
        return chartType;
    }

    private static IListSer createChartType(String type, String stacking) {

        IListSer listSer = null;
        final ObjectFactory chartObjectFactory = Context.getDmlChartObjectFactory();
        String chartType = type != null ? type : "bar2d";

        chartType = opToOOXMLType(chartType);

        if (chartType.equals("bar3d")) {
            listSer = createBarChartType(chartType, stacking, chartObjectFactory);
        } else if (chartType.equals("column2d")) {
            listSer = createBarChartType(chartType, stacking, chartObjectFactory);
        } else if (chartType.equals("column3d")) {
            listSer = createBarChartType(chartType, stacking, chartObjectFactory);
        } else if (chartType.equals("line2d")) {
            listSer = chartObjectFactory.createCTLineChart();
        } else if (chartType.equals("line3d")) {
            listSer = chartObjectFactory.createCTLine3DChart();
        } else if (chartType.equals("area2d")) {
            listSer = chartObjectFactory.createCTAreaChart();
        } else if (chartType.equals("area3d")) {
            listSer = chartObjectFactory.createCTArea3DChart();
        } else if (chartType.equals("radar2d")) {
            CTRadarChart radar = chartObjectFactory.createCTRadarChart();
            radar.setRadarStyle(new CTRadarStyle());
            radar.getRadarStyle().setVal(STRadarStyle.STANDARD);
            listSer = radar;
        } else if (chartType.equals("scatter2d")) {
            final CTScatterChart scatter = chartObjectFactory.createCTScatterChart();
            handleStandardScatterChart(scatter);
            listSer = scatter;
        } else if (chartType.equals("bubble2d")) {
            listSer = chartObjectFactory.createCTBubbleChart();
        } else if (chartType.equals("pie2d")) {
            final CTPieChart pie = chartObjectFactory.createCTPieChart();
            handleStandardPieChart(pie);
            listSer = pie;
        } else if (chartType.equals("donut2d")) {
            final CTDoughnutChart donut = chartObjectFactory.createCTDoughnutChart();
            handleStandardDonutChart(donut);
            listSer = donut;
        } else { // bar2d -> default
            listSer = createBarChartType(chartType, stacking, chartObjectFactory);
        }
        final List<CTUnsignedInt> axid = listSer.getAxId();
        if (axid != null) {
            createAxId(listSer.getAxId());
        }
        listSer.setStacking("standard");

        return listSer;
    }

    private static void handleStandardScatterSer(CTScatterSer serContent) {
        CTShapeProperties shape = serContent.getSpPr();
        if (null == shape) {
            shape = new CTShapeProperties();
            serContent.setSpPr(shape);
        }
        CTLineProperties line = shape.getLn();
        if (null == line) {
            line = new CTLineProperties();
            shape.setLn(line);
            line.setW(19050);
            line.setCap(STLineCap.RND);
            line.setRound(new CTLineJoinRound());
            shape.getEffectLst();
        }

    }

    private static void handleStandardScatterChart(CTScatterChart scatter) {
        if (scatter.getScatterStyle() == null || scatter.getScatterStyle().getVal() == null) {
            final CTScatterStyle style = new CTScatterStyle();
            style.setVal(STScatterStyle.LINE_MARKER);
            scatter.setScatterStyle(style);
        }

        for (final ISerContent cont : scatter.getSer()) {
            handleStandardScatterSer((CTScatterSer) cont);
        }
    }

    private static void handleStandardPieChart(CTPieChart pie) {
        pie.setFirstSliceAng(new CTFirstSliceAng());
    }

    private static void handleStandardDonutChart(CTDoughnutChart donut) {
        final CTHoleSize holeSize = new CTHoleSize();
        holeSize.setVal((short) 60);
        donut.setHoleSize(holeSize);
    }

    private static IListSer createBarChartType(String chartType, String stacking, ObjectFactory chartObjectFactory) {
        IListSer listSer = null;
        String barDir = null;
        if (chartType.equals("bar3d")) {
            listSer = chartObjectFactory.createCTBar3DChart();
            barDir = "bar";
        } else if (chartType.equals("column2d")) {
            listSer = chartObjectFactory.createCTBarChart();
            barDir = "col";
        } else if (chartType.equals("column3d")) {
            listSer = chartObjectFactory.createCTBar3DChart();
            barDir = "col";
        } else {
            listSer = chartObjectFactory.createCTBarChart();
            barDir = "bar";
        }
        listSer.setBarDirection(barDir);

        if (listSer instanceof CTBarChart) {
            handleOverlap((CTBarChart) listSer, stacking);
        }

        return listSer;
    }

    private static void handleOverlap(CTBarChart barSer, String stacking) {
        final CTGapAmount gap = new CTGapAmount();
        gap.setVal(150);
        final CTOverlap overlap;
        if (stacking == null || stacking.equals("standard") || stacking.equals("clustered")) {
            overlap = null;
        } else {
            overlap = new CTOverlap();
            overlap.setVal((byte) 100);
        }
        barSer.setGapWidth(gap);
        barSer.setOverlap(overlap);

    }

    public static String getChartType(ChartWrapper chartPart) {
        HashMap<String, Object> chartMap = new HashMap<>();
        getChartType(chartPart, chartMap);
        return (String) chartMap.get("chartType");
    }

    public static String getStackingType(ChartWrapper chartPart) {
        HashMap<String, Object> chartMap = new HashMap<>();
        getChartType(chartPart, chartMap);
        return (String) chartMap.get("chartGroup");
    }

    public static String getSeriesType(IListSer listSer) {
        String chartType = null;
        if (listSer instanceof CTBubbleChart) {
            chartType = "bubble2d";
        } else if (listSer instanceof CTRadarChart) {
            chartType = "radar2d";
        } else if (listSer instanceof CTSurface3DChart) {
            chartType = "surface3d";
        } else if (listSer instanceof CTArea3DChart) {
            chartType = "area3d";
        } else if (listSer instanceof CTOfPieChart) {
            chartType = "ofPie2d";
        } else if (listSer instanceof CTAreaChart) {
            chartType = "area2d";
        } else if (listSer instanceof CTBarChart) {
            chartType = handleBarDir(listSer);
        } else if (listSer instanceof CTSurfaceChart) {
            chartType = "surface";
        } else if (listSer instanceof CTLine3DChart) {
            chartType = "line3d";
        } else if (listSer instanceof CTDoughnutChart) {
            chartType = "donut2d";
        } else if (listSer instanceof CTLineChart) {
            chartType = "line2d";
        } else if (listSer instanceof CTScatterChart) {
            chartType = "scatter2d";
        } else if (listSer instanceof CTBar3DChart) {
            chartType = handleBarDir(listSer);
        } else if (listSer instanceof CTPieChart) {
            chartType = "pie2d";
        } else if (listSer instanceof CTPie3DChart) {
            chartType = "pie3d";
        } else if (listSer instanceof CTStockChart) {
            chartType = "bar2d";
        }
        return chartType;
    }

    public static void getChartType(ChartWrapper chart, HashMap<String, Object> map) {
        final CTPlotArea plotArea = chart.getPlotArea();
        if (plotArea == null) {
            return;
        }
        final List<IListSer> chartList = plotArea.getAreaChartOrArea3DChartOrLineChart();
        if (chartList.isEmpty()) {
            return;
        }

        final IListSer listSer = chartList.get(0);
        String chartType = getSeriesType(listSer);
        if (chartType != null) {
            map.put("chartType", chartType);
        }
        final String chartGroup = listSer.getStacking();
        if (chartGroup != null) {
            map.put("chartGroup", chartGroup);
        }
    }

    private static String handleBarDir(IListSer listSer) {
        final CTBarDir dir;
        final String view;
        if (listSer instanceof CTBar3DChart) {
            view = "3d";
            dir = ((CTBar3DChart) listSer).getBarDir();
        } else {
            view = "2d";
            dir = ((CTBarChart) listSer).getBarDir();
        }
        final String dirString;
        switch (dir.getVal()) {
            case BAR:
                dirString = "bar";
                break;
            default:
                dirString = "column";
                break;
        }
        return dirString + view;
    }

    // workaround for Bug 43959
    public static CTTextBody setTextBodyFromAttrs(CTTextBody textBody, JSONObject attrs) throws JSONException {
        JSONObject character = attrs.optJSONObject("character");
        if (character == null || character.length() == 0) {
            //nothing
        } else {
            if (textBody == null) {
                textBody = new CTTextBody();
                DMLHelper.handleStandardTypes(textBody);
            }
            setCharacterFromAttrs(textBody, attrs);
        }
        return textBody;
    }

    public static void setLegendFromAttrs(ChartWrapper chart, JSONObject attrs) throws JSONException {
        final CTChart ctChart = chart.getChartSpace().getChart();
        final String pos = attrs.getJSONObject("legend").optString("pos", "off");
        if (pos.equals("off")) {
            ctChart.setLegend(null);
        } else {
            CTLegend legend = ctChart.getLegend();
            if (legend == null) {
                legend = new CTLegend();
                ctChart.setLegend(legend);
            }

            legend.setTxPr(setTextBodyFromAttrs(legend.getTxPr(), attrs));

            legend.setOverlay(getBoolean(false));

            CTLegendPos legendPos = legend.getLegendPos();
            if (legendPos == null) {
                legendPos = new CTLegendPos();
                legend.setLegendPos(legendPos);
            }

            final STLegendPos stpos;
            if (pos.equals("left")) {
                stpos = STLegendPos.L;
            } else if (pos.equals("top")) {
                stpos = STLegendPos.T;
            } else if (pos.equals("right")) {
                stpos = STLegendPos.R;
            } else if (pos.equals("bottom")) {
                stpos = STLegendPos.B;
            } else if (pos.equals("topRight")) {
                stpos = STLegendPos.TR;
            } else {
                Logger.getAnonymousLogger().info("No STLegendPost-enum found for " + pos);
                stpos = STLegendPos.L;
            }
            legendPos.setVal(stpos);
        }
    }

    private static boolean isCurved(IListSer chartSer) {
        if (chartSer instanceof CTScatterChart) {
            CTScatterStyle style = ((CTScatterChart) chartSer).getScatterStyle();
            if (style != null && (style.getVal() == STScatterStyle.SMOOTH || style.getVal() == STScatterStyle.SMOOTH_MARKER)) {
                return true;
            }
        } else if (chartSer instanceof CTLineChart) {
            CTBoolean smooth = ((CTLineChart) chartSer).getSmooth();
            if (getBoolean(smooth)) {
                return true;
            }
        }
        for (ISerContent serContent : chartSer.getSer()) {
            if (isCurved(serContent)) {
                return true;
            }
        }
        return false;
    }

    private static boolean isCurved(ISerContent serContent) {
        if (serContent instanceof CTLineSer) {
            final CTBoolean smooth = ((CTLineSer) serContent).getSmooth();
            //no smooth in line chart means it is activated!!!
            if (getBoolean(smooth)) {
                return true;
            }
        } else if (serContent instanceof CTScatterSer) {
            final CTBoolean smooth = ((CTScatterSer) serContent).getSmooth();
            //no smooth in line chart means it is activated!!!
            if (getBoolean(smooth)) {
                return true;
            }
        }
        return false;
    }

    private static void setCurved(ISerContent serContent, boolean curved) {
        if (serContent instanceof CTLineSer) {
            ((CTLineSer) serContent).setSmooth(getBoolean(curved));
        } else if (serContent instanceof CTScatterSer) {
            ((CTScatterSer) serContent).setSmooth(getBoolean(curved));
        } else if (curved) {
            // System.err.println("you try to set on wrong type " + serContent.getClass().getSimpleName());
        }

    }

    private static void setCurved(IListSer listSer, boolean curved) {
        if (listSer instanceof CTLineChart) {
            ((CTLineChart) listSer).setSmooth(getBoolean(curved));
        } else if (listSer instanceof CTScatterChart) {
            final CTScatterStyle style = new CTScatterStyle();
            if (curved) {
                style.setVal(STScatterStyle.SMOOTH_MARKER);
            } else {
                style.setVal(STScatterStyle.LINE_MARKER);
            }
            ((CTScatterChart) listSer).setScatterStyle(style);
        } else if (curved) {
            // System.err.println("you try to set on wrong type " + listSer.getClass().getSimpleName());
        }
        for (final ISerContent serContent : listSer.getSer()) {
            setCurved(serContent, curved);
        }
    }

    private static String getDataLabel(ISerContent serContent) {
        CTDLbls lbl = serContent.getDLbls();
        if (lbl != null) {
            StringBuilder res = new StringBuilder(0);
            addIfTrue(res, lbl.getShowBubbleSize(), "size");
            addIfTrue(res, lbl.getShowCatName(), "cat");
            addIfTrue(res, lbl.getShowLegendKey(), "legend");
            addIfTrue(res, lbl.getShowPercent(), "percent");
            addIfTrue(res, lbl.getShowSerName(), "series");
            addIfTrue(res, lbl.getShowVal(), "value");
            return res.toString();
        }
        return "";
    }

    private static void addIfTrue(StringBuilder res, CTBoolean b, String name) {
        if (getBoolean(b)) {
            if (res.length() > 0) {
                res.append(";");
            }
            res.append(name);
        }
    }

    private static CTBoolean setIfContains(String source, String search) {
        return getBoolean(source.contains(search));
    }

    private static void setDataLabel(ISerContent serContent, String label) {
        CTDLbls lbl = serContent.getDLbls();
        if (lbl == null) {
            lbl = newDataLabel();
            serContent.setDLbls(lbl);
        }
        lbl.setShowBubbleSize(setIfContains(label, "size"));
        lbl.setShowCatName(setIfContains(label, "cat"));
        lbl.setShowLegendKey(setIfContains(label, "legend"));
        lbl.setShowPercent(setIfContains(label, "percent"));
        lbl.setShowSerName(setIfContains(label, "series"));
        lbl.setShowVal(setIfContains(label, "value"));
    }

    private static CTDLbls newDataLabel() {
        final CTDLbls lbl = new CTDLbls();
        lbl.setShowLegendKey(getBoolean(false));
        lbl.setShowCatName(getBoolean(false));
        lbl.setShowSerName(getBoolean(false));
        lbl.setShowPercent(getBoolean(false));
        lbl.setShowBubbleSize(getBoolean(false));
        lbl.setShowLeaderLines(getBoolean(false));
        lbl.setShowVal(getBoolean(false));

        CTShapeProperties spr = lbl.getSpPr();
        if (spr == null) {
            spr = new CTShapeProperties();
            lbl.setSpPr(spr);
            spr.setNoFill(new CTNoFillProperties());
            final CTLineProperties ln = new CTLineProperties();
            ln.setNoFill(new CTNoFillProperties());
            spr.setLn(ln);

            spr.setEffectLst(new CTEffectList());
        }
        CTTextBody txPr = lbl.getTxPr();
        if (txPr == null) {
            txPr = new CTTextBody();
            lbl.setTxPr(txPr);
            DMLHelper.handleStandardTypes(txPr);
        }

        final CTTextParagraph p = new CTTextParagraph();
        final CTTextParagraphProperties prop = p.getPPr(true);
        final CTTextCharacterProperties def = prop.getDefRPr(true);

        def.setSz(1000);

        final List<Object> paras = txPr.getContent();
        paras.clear();

        paras.add(p);

        return lbl;
    }

    public static JSONObject createJsonFromChartColor(CTColorStyle color) throws JSONException {

        JSONObject jsonChartColor = null;
        if (color != null) {
            jsonChartColor = new JSONObject();
            jsonChartColor.put("id", color.getId());
            jsonChartColor.put("meth", color.getMeth());
            final List<Object> schemeClrList = color.getEGColorChoice();
            ;
            if (schemeClrList != null) {
                final JSONArray jsonArray = new JSONArray();
                for (int i = 0; i < schemeClrList.size(); i++) {
                    final ColorChoiceListEntry listEntry = new ColorChoiceListEntry(schemeClrList, i);
                    if (listEntry.getSchemeClr() != null) {
                        final JSONObject jsonColor = DMLHelper.createJsonColorFromSolidColorFillProperties(listEntry);
                        if (jsonColor != null) {
                            jsonArray.put(jsonColor);
                        }
                    }
                }
                jsonChartColor.put("schemeClr", jsonArray);
            }
            final List<CTColorStyleVariation> variationList = color.getVariation();
            if (variationList != null) {

                /*
                 * TODO: variationList needs to be implemented...
                 * 
                 * final JSONArray jsonArray = new JSONArray();
                 * for(final CTColorStyleVariation schemeClr:variationList) {
                 * final JSONObject resColor = new JSONObject();
                 * createJsonFromSchemeClr(resColor, schemeClr);
                 * if (resColor != null){
                 * jsonArray.put(resColor);
                 * }
                 * }
                 * if (!jsonArray.isEmpty()){
                 * jsonChartColor.put("variation", jsonArray);
                 * }
                 */
            }
        }
        return jsonChartColor != null && !jsonChartColor.isEmpty() ? jsonChartColor : null;
    }

    public static void refreshTitleRotations(ChartWrapper chart) {
        CTPlotArea plotArea = chart.getPlotArea();
        final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
        final IListSer ser = plotArea.getAreaChartOrArea3DChartOrLineChart().get(0);
        final String dir = ser.getBarDirection();
        for (int i = 0; i < axes.size(); i++) {
            boolean rotate = false;
            if (i == 1) {
                rotate = true;
            }
            if (dir == null || dir.equals("col")) {
                // normal
            } else {
                rotate = !rotate;
            }
            int rot = 0;
            if (rotate) {
                rot = -5400000;
            }
            final IAxisDescription axis = axes.get(i);
            final CTTitle title = axis.getTitle();

            if (title != null) {
                if (title.getTx() != null && title.getTx().getRich() != null && title.getTx().getRich().getBodyPr() != null) {
                    final CTTx tx = title.getTx();
                    final CTTextBody rich = tx.getRich();
                    final CTTextBodyProperties body = rich.getBodyPr();
                    body.setRot(rot);

                }

                final CTTextBody txPr = title.getTxPr();
                if (null != txPr && null != txPr.getBodyPr()) {
                    final CTTextBodyProperties body2 = txPr.getBodyPr();
                    body2.setRot(rot);
                }
            }

            final CTTextBody lbl = axis.getTxPr();
            if (lbl != null && lbl.getBodyPr() != null) {
                final CTTextBodyProperties body = lbl.getBodyPr();
                body.setRot(0);
            }
        }
    }

    static final void getAxeTitleOperation(IAxisDescription axe, JSONArray operations, String axisParam, List<Integer> position) throws JSONException {
        final CTTitle title = axe.getTitle();

        final JSONObject attrs = createJsonFromProperties(title);
        if (attrs != null) {

            final JSONObject op = new JSONObject();
            op.put("name", "setChartTitleAttributes");
            op.put("start", position);
            op.put("axis", axisParam);
            op.put("attrs", attrs);
            operations.put(op);
        }

    }

    public static void setChartTitleAttributes(ChartWrapper chart, String axis, JSONObject attrs) throws JSONException {
        final IAxisDescription ax = getAxis(chart.getPlotArea(), axis);
        if (ax == null) {
            return;
        }

        CTTitle title = ax.getTitle();
        if (title == null) {
            title = new CTTitle();
            ax.setTitle(title);
        }
        setTitleFromAttrs(title, attrs);
        refreshTitleRotations(chart);
    }

    static final void getGridOperation(ChartWrapper chart, IAxisDescription axe, JSONArray operations, String axisParam, List<Integer> position) throws Exception {

        final CTChartLines grid = axe.getMajorGridlines();
        if (grid != null) {
            final JSONObject gridOp = new JSONObject();
            final CTShapeProperties shape = grid.getSpPr();
            if (shape != null) {
                DMLHelper.createJsonFromShapeProperties(gridOp, shape, chart.getThemePart(), chart.getChartStyle().getGridlineMajor(), chart.getChartPart(), false, true);
            }
            if (!gridOp.has("line")) {
                //FIXME: nessecary?
                gridOp.put("line", makeStandardShapeType());
            }
            final JSONObject op = new JSONObject();
            op.put("name", "setChartGridlineAttributes");
            op.put("start", position);
            op.put("axis", axisParam);
            op.put("attrs", gridOp);
            operations.put(op);
        }
    }

    static IAxisDescription getAxis(CTPlotArea plotArea, String axis) {
        if (plotArea.getAreaChartOrArea3DChartOrLineChart().get(0) instanceof NoAxIdChart) {
            return null;
        }

        List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();

        AxPos pos = AxPos.valueOf(axis.toUpperCase());

        IAxisDescription res = null;
        for (IAxisDescription other : axes) {
            STAxPos otherPos = other.getAxPos().getVal();
            if (pos.pos == otherPos) {
                res = other;
                break;
            }
        }

        if (res == null) {
            IAxisDescription opp = getAxByPos(plotArea, pos.getCross().pos);

            final long index = pos.pos.ordinal() + 5000;
            final long crossIndex;
            if (opp != null) {
                crossIndex = opp.getAxId().getVal();
            } else {
                crossIndex = pos.getCross().ordinal() + 5000;
            }

            String chartType = getSeriesType(plotArea.getAreaChartOrArea3DChartOrLineChart().get(0));

            STCrosses crossMinMax = STCrosses.AUTO_ZERO;
            if (chartType.contains("bubble") || chartType.contains("scatter")) {
                crossMinMax = STCrosses.MIN;
            }
            if (pos == AxPos.Y2 || pos == AxPos.X2) {
                crossMinMax = STCrosses.MAX;
            }

            // System.out.println("DMLChartSpace.getAxis() create axis " + axis + " pos " + pos + " index " + index + " crossIndex " + crossIndex + " chartType " + chartType);

            if (pos == AxPos.Z) {
                final CTSerAx ser = createSerAx();
                ser.setMinorGridlines(getHiddenChartLines());
                ser.setMinorTickMark(getHiddenTickMark());
                ser.setNumFmt(null);

                ser.setCrosses(new CTCrosses());
                ser.getCrosses().setVal(crossMinMax);

                res = ser;
            } else if (pos.type.equals("x") && !chartType.contains("bubble") && !chartType.contains("scatter")) {
                final CTCatAx cat = createCatAx();
                cat.setMinorGridlines(getHiddenChartLines());
                cat.setMinorTickMark(getHiddenTickMark());
                cat.setNumFmt(null);

                cat.setCrosses(new CTCrosses());
                cat.getCrosses().setVal(crossMinMax);

                res = cat;
            } else {
                final CTValAx val = createValAx();
                val.setMinorGridlines(getHiddenChartLines());
                val.setMinorTickMark(getHiddenTickMark());
                val.setNumFmt(null);

                val.setCrosses(new CTCrosses());
                val.getCrosses().setVal(crossMinMax);
                val.setCrossBetween(new CTCrossBetween());
                val.getCrossBetween().setVal(STCrossBetween.BETWEEN);

                res = val;
            }
            res.setCrossAx(getInteger(crossIndex));
            res.setAxPos(new CTAxPos());

            res.getAxPos().setVal(pos.pos);
            res.setAxId(getInteger(index));

            if (pos.index != 0) {
                res.setNumFmt(new CTNumFmt());
                res.getNumFmt().setFormatCode("general");
                res.getNumFmt().setSourceLinked(true);
            }
            /*
             * if (pos.index == 0) {
             * res.setCrossAx(value););
             * TODO: task
             * }
             */

            axes.add(res);
        }

        try {
            // workaround for Bug 46915
            if (plotArea.getAreaChartOrArea3DChartOrLineChart().get(0) instanceof NoAxIdChart) {
                return null;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        // fix for Bug 49940
        if (getBoolean(res.getDelete())) {
            hideAxis(res);
        } else {
            // workaround for Bug 49959
            res.setDelete(getBoolean(false));
        }
        return res;
    }

    static void hideAxis(IAxisDescription axis) {
        final CTBoolean delete = getBoolean(false);
        axis.setDelete(delete);

        axis.setSpPr(getHiddenShapeProperties());

        axis.setMajorGridlines(getHiddenChartLines());
        axis.setTickLblPos(getHiddenLblPos());

        axis.setTitle(null);
        axis.setTxPr(null);

    }

    public static JSONObject createJsonFromProperties(CTTitle title) throws JSONException {
        if (title == null) {
            return null;
        }
        final CTTx tx = title.getTx();
        if (tx == null) {
            return null;
        }

        final CTTextBody rich = tx.getRich();
        final CTStrRef ref = tx.getStrRef();

        if (null == ref && null == rich) {
            return null;
        }

        CTTextCharacterProperties characterProb = null;

        final JSONObject attrs = new JSONObject();
        final JSONObject textObj = new JSONObject();
        attrs.put("text", textObj);

        if (rich != null) {

            final StringBuffer buf = new StringBuffer();

            final List<Object> paras = rich.getContent();
            for (int i = 0; i < paras.size(); i++) {
                final CTTextParagraph p = (CTTextParagraph) paras.get(i);
                final List<Object> textes = p.getContent();
                for (final Object text : textes) {
                    if (text instanceof CTRegularTextRun) {
                        addTo((CTRegularTextRun) text, buf);
                    } else if (text instanceof CTTextField) {
                        addTo((CTTextField) text, buf);
                    } else if (text instanceof CTTextLineBreak) {
                        addTo((CTTextLineBreak) text, buf);
                    }

                    if (characterProb == null) {
                        characterProb = ((ITextCharacterProperties) text).getRPr(false);
                    }

                }
                if (i < paras.size() - 1) {
                    buf.append("\n");
                }
            }

            final JSONArray link = new JSONArray();
            link.put(buf.toString());
            textObj.put("link", link);
        }

        if (ref != null) {
            textObj.put("link", ref.getF());
        }

        if (characterProb != null) {
            DMLHelper.createJsonFromTextCharacterProperties(attrs, characterProb, null);
        }
        return attrs;
    }

    private static JSONArray attrHasTitle(JSONObject attrs) {
        final JSONObject text = attrs.optJSONObject("text");
        if (text == null) {
            return null;
        }

        final Object olink = text.opt("link");

        if (olink == null || !(olink instanceof JSONArray)) {

            if (olink == null || olink.equals(JSONObject.NULL)) {
                return null;
            }
            Logger.getAnonymousLogger().warning("title text has no correct content " + text.opt("link"));
            return null;
        }

        final JSONArray links = (JSONArray) olink;
        if (links.length() > 0) {
            return links;
        }
        return null;
    }

    private static String attrHasRef(JSONObject attrs) {
        final JSONObject text = attrs.optJSONObject("text");
        if (text == null) {
            return null;
        }

        final Object olink = text.opt("link");

        if (olink instanceof String) {
            return (String) olink;
        }

        return null;
    }

    public static void setTitleFromAttrs(CTTitle title, JSONObject attrs) throws JSONException {
        handleStandardTypes(title);

        JSONArray links = attrHasTitle(attrs);
        String ref = attrHasRef(attrs);
        if (links == null && ref == null) {

            title.setTx(null);
            title.setTxPr(setTextBodyFromAttrs(title.getTxPr(), attrs));
        } else {

            final CTTx tx = title.getTx();

            final CTTextBody rich = tx.getRich();
            final CTTextBodyProperties body = rich.getBodyPr();
            body.setRot(0);

            final List<Object> paras = rich.getContent();
            paras.clear();

            //delete cell ref, fix for Bug 47761
            tx.setStrRef(null);
            if (ref != null && ref.length() > 0) {

                CTStrRef strRef = new CTStrRef();
                strRef.setF(ref);
                tx.setStrRef(strRef);
                tx.setRich(null);
                return;
            }

            title.setTxPr(null);

            if (links != null && links.length() > 0) {
                for (int i = 0; i < links.length(); i++) {
                    final CTTextParagraph p = Context.getDmlObjectFactory().createCTTextParagraph();
                    setCharacterFromAttrs(p, attrs);
                    paras.add(p);
                    final List<Object> textHolder = p.getContent();
                    textHolder.clear();

                    final String link = links.getString(i);
                    final String[] textes = link.split("\n");
                    for (int j = 0; j < textes.length; j++) {
                        final CTRegularTextRun reg = Context.getDmlObjectFactory().createCTRegularTextRun();
                        reg.setRPr(setCharacterFromAttrs(reg.getRPr(false), attrs));
                        reg.setT(textes[j]);
                        textHolder.add(reg);
                        if (j < textes.length - 1) {
                            final CTTextLineBreak lb = Context.getDmlObjectFactory().createCTTextLineBreak();
                            lb.setRPr(setCharacterFromAttrs(lb.getRPr(false), attrs));
                            textHolder.add(lb);
                        }
                    }
                }
            }
        }
    }

    static void handleStandardTypes(CTTitle title) {
        CTTx tx = title.getTx();
        if (tx == null) {
            tx = new CTTx();
            title.setTx(tx);
        }
        CTTextBody rich = tx.getRich();
        if (rich == null) {
            rich = new CTTextBody();
            tx.setRich(rich);
        }
        DMLHelper.handleStandardTypes(rich);

        CTLayout layout = title.getLayout();
        if (layout == null) {
            layout = new CTLayout();
            title.setLayout(layout);
        }
        title.setOverlay(getBoolean(false));

        CTShapeProperties spr = title.getSpPr();
        if (spr == null) {
            spr = getHiddenShapeProperties();
        }

        final List<Object> ps = rich.getContent();
        ps.clear();

        final CTTextParagraph p = new CTTextParagraph();
        p.getPPr(true);
        ps.add(p);

    }

    public static void addToOperations(ChartWrapper chart, JSONArray operations, List<Integer> position) throws Exception {
        CTPlotArea plotArea = chart.getPlotArea();
        final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
        if (axes != null && axes.size() > 0) {
            for (int i = 0; i < axes.size(); i++) {
                final IAxisDescription axe = axes.get(i);
                AxPos axPos = getAxPos(axes, i);

                String axisParam = axPos.id;
                getAxeOperation(chart, axe, operations, axisParam, position);
                getGridOperation(chart, axe, operations, axisParam, position);
            }
        }
    }

    private static int getAxIndex(CTPlotArea plotArea, List<CTUnsignedInt> axIds, String axType) {
        if (axIds == null) {
            return 0;
        }
        for (CTUnsignedInt axId : axIds) {
            AxPos axPos = getAxPos(plotArea, axId);
            if (axPos != null && axPos.type.equals(axType)) {
                return axPos.index;
            }
        }
        return 0;
    }

    private static AxPos getAxPos(CTPlotArea plotArea, CTUnsignedInt axId) {
        final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
        if (axes != null && axes.size() > 0) {
            for (int i = 0; i < axes.size(); i++) {
                final IAxisDescription axe = axes.get(i);
                if (axe.getAxId().getVal() == axId.getVal()) {
                    return getAxPos(axes, i);
                }
            }
        }
        return null;
    }

    private static IAxisDescription getAxByPos(CTPlotArea plotArea, STAxPos pos) {

        final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
        if (axes != null && axes.size() > 0) {
            for (int i = 0; i < axes.size(); i++) {
                final IAxisDescription axe = axes.get(i);
                if (axe.getAxPos().getVal() == pos) {
                    return axe;
                }
            }
        }
        return null;
    }

    public static AxPos getAxPos(List<IAxisDescription> axes, int i) {
        final IAxisDescription axe = axes.get(i);

        AxPos axisParam = null;
        if (axe.getAxPos() != null) {

            STAxPos axPos = axe.getAxPos().getVal();

            switch (axPos) {
                case B:
                    axisParam = AxPos.X;
                    break;
                case L:
                    axisParam = AxPos.Y;
                    break;
                case T:
                    axisParam = AxPos.X2;
                    break;
                case R:
                    axisParam = AxPos.Y2;
                    break;
            }
        }
        if (axisParam == null) {
            if (i == 0) {
                axisParam = AxPos.X;
            } else if (i == 1) {
                axisParam = AxPos.Y;
            } else {
                axisParam = AxPos.Z;
            }
        }

        return axisParam;
    }

    static final void getAxeOperation(ChartWrapper chart, IAxisDescription axe, JSONArray operations, String axisParam, List<Integer> position) throws Exception {

        final JSONObject axis = new JSONObject();

        /*
         * disabled for Bug 46441, if we want to support scaling,w e also must support major & minor units!
         * final CTScaling scaling = axe.getScaling();
         * if(scaling!=null) {
         * if (scaling.getMin()!=null) {
         * axis.put("min", scaling.getMin().getVal());
         * }
         * if(scaling.getMax()!=null) {
         * axis.put("max", scaling.getMax().getVal());
         * }
         * }
         */
        JSONObject character = null;
        final JSONObject axisOp = new JSONObject();
        final CTBoolean del = axe.getDelete();
        if (getBoolean(del)) {
            final JSONObject line = new JSONObject();
            line.put("type", "none");
            axisOp.put("line", line);
        } else {
            final CTShapeProperties shape = axe.getSpPr();
            if (shape != null) {
                DMLHelper.createJsonFromShapeProperties(axisOp, shape, chart.getThemePart(), chart.getChartStyle().getSeriesAxis(), chart.getChartPart(), false, true);
            }
            if (!axisOp.has("line")) {
                //FIXME: nessecary?
                axisOp.put("line", makeStandardShapeType());
            }

            final CTTickLblPos label = axe.getTickLblPos();
            if (label != null && label.getVal() != STTickLblPos.NONE) {
                axis.put("label", true);
                character = DMLHelper.createJsonFromProperties(axe.getTxPr());
            }

            getAxeTitleOperation(axe, operations, axisParam, position);
        }

        if (axis.length() > 0) {
            axisOp.put("axis", axis);
        }
        if (character != null) {
            axisOp.put("character", character);
        }

        if (axisOp.length() > 0) {
            final JSONObject op = new JSONObject();
            op.put("name", "setChartAxisAttributes");
            op.put("start", position);
            op.put("axis", axisParam);
            op.put("attrs", axisOp);
            operations.put(op);
        }
    }

    public static void setAxisAttributes(ChartWrapper chart, String axis, JSONObject attrs) throws JSONException, InvalidFormatException, PartUnrecognisedException {

        final IAxisDescription ax = getAxis(chart.getPlotArea(), axis);
        if (ax == null) {
            return;
        }
        final CTShapeProperties shape = DMLHelper.applyShapePropertiesFromJson(ax.getSpPr(), attrs, null, null, true);
        ax.setSpPr(shape);
        CTTickMark tickmark = ax.getMajorTickMark();
        if (tickmark == null) {
            tickmark = new CTTickMark();
            ax.setMajorTickMark(tickmark);
        }
        if (shape.getLn() != null && shape.getLn().getNoFill() != null) {
            tickmark.setVal(STTickMark.NONE);
        } else {
            tickmark.setVal(STTickMark.OUT);
        }

        final JSONObject a = attrs.optJSONObject("axis");
        if (a != null) {
            CTTickLblPos lblPos = ax.getTickLblPos();
            if (lblPos == null) {
                lblPos = new CTTickLblPos();
                ax.setTickLblPos(lblPos);
            }
            final boolean label = a.optBoolean("label", false);

            if (label) {
                // fix for Bug 53425
                lblPos.setVal(STTickLblPos.LOW);

                ax.setTxPr(setTextBodyFromAttrs(ax.getTxPr(), attrs));
            } else {
                lblPos.setVal(STTickLblPos.NONE);
            }
        }
    }

    public static void setChartGridlineAttributes(ChartWrapper chart, String axis, JSONObject attrs) throws JSONException, InvalidFormatException, PartUnrecognisedException {

        final IAxisDescription ax = getAxis(chart.getPlotArea(), axis);
        if (ax == null) {
            return;
        }

        CTChartLines grids = ax.getMajorGridlines();
        if (grids == null) {
            grids = new CTChartLines();
            ax.setMajorGridlines(grids);
        }
        grids.setSpPr(DMLHelper.applyShapePropertiesFromJson(grids.getSpPr(), attrs, null, null, true));
    }

    static CTChartLines getHiddenChartLines() {
        final CTChartLines line = new CTChartLines();
        line.setSpPr(getHiddenShapeProperties());
        return line;
    }

    static CTTickLblPos getHiddenLblPos() {
        final CTTickLblPos lbl = new CTTickLblPos();
        lbl.setVal(STTickLblPos.NONE);
        return lbl;
    }

    static CTTickMark getHiddenTickMark() {
        final CTTickMark mark = new CTTickMark();
        mark.setVal(STTickMark.NONE);
        return mark;
    }

    public static void setCharacterFromAttrs(CTTextBody txPr, JSONObject attrs) throws JSONException {
        final List<Object> ps = txPr.getContent();
        ps.clear();
        final CTTextParagraph p = Context.getDmlObjectFactory().createCTTextParagraph();
        ps.add(p);
        setCharacterFromAttrs(p, attrs);
    }

    public static void setCharacterFromAttrs(CTTextParagraph p, JSONObject attrs) throws JSONException {

        final JSONObject character = attrs.optJSONObject("character");

        if (character != null) {
            final CTTextParagraphProperties prop = p.getPPr(true);
            prop.setDefRPr(setCharacterFromAttrs(prop.getDefRPr(false), attrs));
        }
    }

    public static CTTextCharacterProperties setCharacterFromAttrs(CTTextCharacterProperties t, JSONObject attrs) throws JSONException {
        final JSONObject character = attrs.optJSONObject("character");

        if (character != null) {
            if (t == null) {
                t = Context.getDmlObjectFactory().createCTTextCharacterProperties();
            }
            DMLHelper.applyTextCharacterPropertiesFromJson(t, attrs, null);

        }
        return t;
    }

    public static CTShapeProperties getHiddenShapeProperties() {
        final CTShapeProperties shape = new CTShapeProperties();
        shape.setNoFill(new CTNoFillProperties());
        final CTLineProperties line = new CTLineProperties();
        line.setNoFill(new CTNoFillProperties());
        shape.setLn(line);
        return shape;
    }

    //FIXME: nessecary?
    static JSONObject makeStandardShapeType() throws JSONException {
        final JSONObject shape = new JSONObject();
        shape.put("type", "solid");
        final JSONObject color = new JSONObject();
        color.put("type", "auto");
        shape.put("color", color);
        return shape;
    }

    public static void addTo(CTRegularTextRun textRun, StringBuffer stringBuffer) {
        stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextField textRun, StringBuffer stringBuffer) {
        stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextLineBreak textRun, StringBuffer stringBuffer) {
        stringBuffer.append("\n");
    }

    private static enum AxPos {
        X(0, STAxPos.B, "Y"), Y(0, STAxPos.L, "X"), X2(1, STAxPos.T, "Y2"), Y2(1, STAxPos.R, "X2"), Z(0, null, "X");

        private final int    index;
        private final String id;
        private final STAxPos pos;
        private final String cross;
        private final String  type;

        private AxPos(int index, STAxPos pos, String cross) {
            this.index = index;
            this.pos = pos;
            this.cross = cross;
            this.id = this.name().toLowerCase();
            this.type = this.id.substring(0, 1);
        }

        public AxPos getCross() {
            return AxPos.valueOf(this.cross);
        }
    }
}
