/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom.chart;

import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import org.xml.sax.Attributes;
import org.xml.sax.XMLReader;
import com.openexchange.office.filter.core.chart.AttributeSet;
import com.openexchange.office.filter.core.chart.ChartSeries;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DOMBuilder;
import com.openexchange.office.filter.odf.ElementNS;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.UnknownContentHandler;
import com.openexchange.office.filter.odf.styles.AutomaticStylesHandler;
import com.openexchange.office.filter.odf.styles.FontFaceDeclsHandler;

public class ChartContentHandler extends DOMBuilder {

    private ChartContent content;
    private String chartType;

    public ChartContentHandler(ChartContent fDom, XMLReader reader) {
        super(fDom, reader, fDom);
        content = fDom;
    }

    ChartContent getChart() {
        return content;
    }

    private String getChartType(String type) {
        if (StringUtils.isNotBlank(type)) {

            type = type.substring(6);

            if ("circle".equals(type)) {
                type = "pie";
                if ("chart:ring".equals(content.getChartArea().getAttribute("chart:class"))) {
                    type = "donut";
                }
            } else if ("bar".equals(type) && !content.getPlotAreaStyle().getChartProperties().getBoolean("chart:vertical", false)) {
                type = "column";
            }
            return type;
        }
        return null;
    }

    @Override
    public SaxContextHandler startElement(Attributes attributes, String uri, String localName, String qName) {
        if (OdfOperationDoc.isLowMemoryAbort()) {
            throw new RuntimeException();
        }

        AttributesImpl attrs = new AttributesImpl(attributes);
        String styleName = attrs.getValue(ChartContent.ODF_STYLE);

        switch (qName) {
            case "chart:series":

                String type = getChartType(attrs.getValue("chart:class"));
                if (StringUtils.isBlank(type)) {
                    type = chartType;
                }

                if (chartType == null) {
                    chartType = type;
                }

                String values = ChartContent.getFormula(attrs.getValue("chart:" + ChartContent.ODF_VALUES));
                String title = ChartContent.getFormula(attrs.getValue("chart:" + ChartContent.ODF_TITLE));

                int axisYIndex = 0;
                if (attrs.containsKey("chart:attached-axis") && "secondary-y".equals(attrs.getValue("chart:attached-axis"))) {
                    axisYIndex = 1;
                }

                ElementNS serieEl = content.addDataSeries(type, values, title, styleName, 0, axisYIndex);
                serieEl.addAttributes(attributes);

                if (StringUtils.isNotEmpty(styleName)) {
                    String symbolType = content.getStyleChart(styleName).getChartProperties().getAttribute("chart:symbol-type");
                    if (StringUtils.equals(symbolType, "none")) {
                        ChartSeries series = (ChartSeries) serieEl.getUserData(ChartContent.USERDATA);
                        if (series.isLine() || series.isScatter()) {
                            series.setHiddenMarker(true);
                        }
                    }
                }
                return new ChartSeriesHandler(this, serieEl);

            case "office:font-face-decls":
                return new FontFaceDeclsHandler(this, content.getDocument().getStyleManager(), true);
            case "office:automatic-styles":
                return new AutomaticStylesHandler(this, content.getDocument().getStyleManager(), true);

            case "chart:chart":

                JSONObject bgStyle = content.getStyleAttrs(styleName);
                if (!bgStyle.hasAndNotNull("fill")) {
                    //TODO HACK
                    JSONObject fill = new JSONObject();
                    fill.putSafe("type", "solid");

                    JSONObject color = new JSONObject();
                    color.putSafe("value", "ffffff");
                    color.putSafe("type", "rgb");
                    fill.putSafe("color", color);

                    bgStyle.putSafe("fill", fill);
                }
                content.getChart().setAttributes(bgStyle);
                super.startElement(attributes, uri, localName, qName);
                content.setChartArea((ElementNS) getCurrentNode());

                return this;

            case "chart:wall":

                ElementNS el = new ElementNS(content, attributes, uri, qName);
                content.getPlotArea().appendChild(el);

                return this;

            case "chart:plot-area":
                ElementNS plotArea = new ElementNS(content, attributes, uri, qName);
                content.setPlotArea(plotArea);
                content.getChartArea().appendChild(plotArea);

                content.getChart().setAttributes(content.getStyleAttrs(styleName));

                String chartStyleId = attrs.getValue("chart:" + ChartContent.CHARTSTYLEID);
                if (chartStyleId != null) {
                    try {
                        content.getChart().setChartStyleId(Integer.parseInt(chartStyleId));
                    } catch (NumberFormatException e) {
                        //ignore broken param
                    }
                }

                chartType = getChartType(content.getChartArea().getAttribute("chart:class"));

                return this;

            case "chart:axis":

                String axName = attrs.getValue("chart:name");
                String axId = axName.substring(axName.length() - 1, axName.length());
                if ("secondary-y".equals(axName)) {
                    axId = "y2";
                }

                ElementNS axisEl = content.addAxis(axId, styleName);
                axisEl.addAttributes(attributes);

                return new ChartAxisHandler(this, axisEl);

            case "chart:title":
                ElementNS titleEl = content.addMainTitle(styleName);

                AttributeSet mainTitle = content.getChart().getTitle("main");
                mainTitle.setAttributes(content.getStyleAttrs(styleName));
                return new ChartTitleHandler(this, "main", titleEl);

            case "chart:legend":

                ElementNS legendEl = content.addLegend(styleName);

                String legendPos = attrs.getValue("chart:" + ChartContent.ODF_LEGEND);
                JSONObject legend = new JSONObject();
                legend.putSafe("pos", ChartContent.LEGENDPOS_ODF_TO_OP.get(legendPos));

                JSONObject a = content.getStyleAttrs(styleName);
                a.putSafe("legend", legend);
                content.setLegendAttributes(a);

                return this;

            case "chartooo:coordinate-region":
            case "chart:floor":
                ElementNS ele = new ElementNS(content, attributes, uri, qName);
                content.getPlotArea().appendChild(ele);
                return new UnknownContentHandler(this, ele);
            default:
                if (qName.startsWith("table")) {
                    ElementNS ele2 = new ElementNS(content, attributes, uri, qName);
                    content.getChartArea().appendChild(ele2);
                    return new UnknownContentHandler(this, ele2);
                }
                return super.startElement(attributes, uri, localName, qName);
        }

    }

    @Override
    public void endElement(String localName, String qName) {
        if (qName.equals("office:automatic-styles") || qName.equals("office:font-face-decls")) {
            // do nothing since we do not want this element to be inserted into the dom,
            // this is done elsewhere
            return;
        }
        super.endElement(localName, qName);
    }

}
