/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.ods.dom;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableSet;
import java.util.TreeSet;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.FilterException.ErrorCode;
import com.openexchange.office.filter.odf.ConfigItemMapEntry;
import com.openexchange.office.filter.odf.ConfigItemMapNamed;
import com.openexchange.office.filter.odf.IParagraph;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.Settings;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.Component.Type;
import com.openexchange.office.filter.odf.components.TextFieldComponent;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawObject;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.styles.DocumentStyles;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.ods.dom.SmlUtils.CellRef;
import com.openexchange.office.filter.ods.dom.SmlUtils.CellRefRange;
import com.openexchange.office.filter.ods.dom.chart.ChartContent;
import com.openexchange.office.filter.ods.dom.components.DrawingComponent;
import com.openexchange.office.filter.odt.dom.TextField;

public class JsonOperationConsumer {

    private final OdfOperationDoc opsDoc;
    private final OdfSpreadsheetDocument doc;
    private final DocumentStyles styles;
    private final SpreadsheetContent content;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationConsumer(OdfOperationDoc opsDoc)
        throws SAXException {

        this.opsDoc = opsDoc;
        doc = (OdfSpreadsheetDocument)opsDoc.getDocument();
        styles = doc.getStylesDom();
        content = (SpreadsheetContent)doc.getContentDom();
        styleManager = doc.getStyleManager();
        settings = doc.getSettingsDom();
    }

    public int applyOperations(JSONArray operations)
        throws Exception {

        for (int i = 0; i < operations.length(); i++) {
            OdfOperationDoc.setSuccessfulAppliedOperations(i);
            final JSONObject op = (JSONObject) operations.get(i);
            final String opName = op.getString("name");
            switch(opName)
            {
                case "changeCells": {
                    changeCells(op.getInt("sheet"), op.getJSONArray("start"), op.getJSONArray("contents"));
                    break;
                }
                case "mergeCells": {
                    mergeCells(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("keepContent", false), op.optString("type", "merge"));
                    break;
                }
                case "insertRows": {
                    insertRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "deleteRows": {
                    deleteRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "insertColumns": {
                    insertColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "deleteColumns": {
                    deleteColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "insertSheet": {
                    insertSheet(op.getInt("sheet"), op.getString("sheetName"), op.optJSONObject("attrs"));
                    break;
                }
                case "moveSheet": {
                    content.moveSheet(op.getInt("sheet"), op.getInt("to"));
                    break;
                }
                case "deleteSheet": {
                    content.deleteSheet(op.getInt("sheet"));
                    break;
                }
                case "setSheetName": {
                    setSheetName(op.getInt("sheet"), op.getString("sheetName"));
                    break;
                }
                case "changeColumns": {
                    changeColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.optJSONObject("attrs"), op.optString("s", null));
                    break;
                }
                case "changeRows": {
                    changeRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.optJSONObject("attrs"), op.optString("s", null));
                    break;
                }
                case "insertAutoStyle": {
                    styleManager.insertAutoStyle(op.getString("type"), op.getString("styleId"), op.getJSONObject("attrs"), op.optBoolean("default"));
                    break;
                }
                case "deleteAutoStyle": {
                    styleManager.deleteAutoStyle(op.getString("type"), op.getString("styleId"));
                    break;
                }
                case "changeAutoStyle": {
                    styleManager.changeAutoStyle(op.getString("type"), op.getString("styleId"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertStyleSheet": {
                    styleManager.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.optString("styleName"), op.optJSONObject("attrs"), op.optString("parent"), op.optBoolean("default"), op.optBoolean("hidden"));
                    break;
                }
                case "insertDrawing": {
                    insertDrawing(op.getJSONArray("start"), op.getString("type"), op.optJSONObject("attrs"));
                    break;
                }
                case "deleteDrawing": {
                    deleteDrawing(op.getJSONArray("start"));
                    break;
                }
                case "moveDrawing": {
                    moveDrawing(op.getJSONArray("start"), op.getJSONArray("to"));
                    break;
                }
                case "setDrawingAttributes": {
                    applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertName": {
                    changeOrInsertName(op.optInt("sheet", -1), op.getString("label"), op.getString("formula"), null, op.optBoolean("isExpr", false), op.optBoolean("hidden", false), op.getString("ref"), true);
                    break;
                }
                case "changeName": {
                    changeOrInsertName(op.optInt("sheet", -1), op.getString("label"), op.optString("formula", null), op.optString("newLabel", null), op.optBoolean("isExpr", false), op.optBoolean("hidden", false), op.optString("ref", null), false);
                    break;
                }
                case "deleteName": {
                    deleteName(op.optInt("sheet", -1), op.getString("label"));
                    break;
                }
                case "insertTable": {
                    insertTable(op.getInt("sheet"), op.optString("table", null), op.getJSONArray("start"), op.getJSONArray("end"), op.optJSONObject("attrs"));
                    break;
                }
                case "changeTable": {
                    insertTable(op.getInt("sheet"), op.optString("table", null), op.optJSONArray("start"), op.optJSONArray("end"), op.optJSONObject("attrs"));
                    break;
                }
                case "deleteTable": {
                    deleteTable(op.getInt("sheet"), op.optString("table", null));
                    break;
                }
                case "changeTableColumn": {
                    changeTableColumn(op.getInt("sheet"), op.optString("table", null), op.getInt("col"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertHyperlink": {
                    insertHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.getString("url"));
                    break;
                }
                case "deleteHyperlink": {
                    deleteHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"));
                    break;
                }
                case "setDocumentAttributes": {
                    setDocumentAttributes(op.getJSONObject("attrs"));
                    break;
                }
                case "insertCFRule": {
                    changeOrinsertCondFormatRule(true, op.getInt("sheet"), op.getString("id"), op.getString("ranges"), op.optString("type", "formula"),
                        op.has("value1") ? op.get("value1") : "", op.optString("value2", ""), op.optInt("priority"), op.optBoolean("stop", false),
                        op.optJSONObject("dataBar"), op.optJSONObject("iconSet"), op.optJSONArray("colorScale"),
                        op.has("attrs") ? op.getJSONObject("attrs") : new JSONObject());
                    break;
                }
                case "changeCFRule": {
                    changeOrinsertCondFormatRule(false, op.getInt("sheet"), op.getString("id"), op.optString("ranges", null), op.optString("type", null),
                        op.opt("value1"), op.optString("value2", null), op.optInt("priority"), op.optBoolean("stop"),
                        op.optJSONObject("dataBar"), op.optJSONObject("iconSet"), op.optJSONArray("colorScale"),
                        op.has("attrs") ? op.getJSONObject("attrs") : new JSONObject());
                    break;
                }
                case "deleteCFRule": {
                    deleteCondFormatRule(op.getInt("sheet"), op.getString("id"));
                    break;
                }
                case "setSheetAttributes": {
                    setSheetAttributes(op.getInt("sheet"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertChartDataSeries":
                    getChart(op.getJSONArray("start")).insertDataSeries(op.getInt("series"), op.getJSONObject("attrs"));
                    break;
                case "setChartDataSeriesAttributes":
                    getChart(op.getJSONArray("start")).setDataSeriesAttributes(op.getInt("series"), op.getJSONObject("attrs"));
                    break;
                case "deleteChartDataSeries":
                    getChart(op.getJSONArray("start")).deleteDataSeries(op.getInt("series"));
                    break;
                case "setChartAxisAttributes":
                    getChart(op.getJSONArray("start")).setAxisAttributes(op.getString("axis"), op.getJSONObject("attrs"));
                    break;
                case "setChartGridlineAttributes":
                    getChart(op.getJSONArray("start")).setGridlineAttributes(op.getString("axis"), op.getJSONObject("attrs"));
                    break;
                case "setChartTitleAttributes":
                    getChart(op.getJSONArray("start")).setTitleAttributes(op.getString("axis"), op.getJSONObject("attrs"));
                    break;
                case "setChartLegendAttributes":
                    getChart(op.getJSONArray("start")).setLegendAttributes(op.getJSONObject("attrs"));
                    break;
                case "insertParagraph":
                    insertParagraph(op.getJSONArray("start"), op.optString("target", ""), op.optJSONObject("attrs"));
                    break;
                case "splitParagraph":
                    splitParagraph(op.getJSONArray("start"), op.optString("target", ""));
                    break;
                case "mergeParagraph":
                    mergeParagraph(op.getJSONArray("start"), op.optString("target", ""));
                    break;
                case "insertText" : {
                    insertText(op.getJSONArray("start"), op.optString("target", ""), op.optJSONObject("attrs"), op.getString("text").replaceAll("\\p{Cc}", " "));
                    break;
                }
                case "move":
                    move(op.getJSONArray("start"), op.optString("target", ""), op.optJSONArray("end"), op.getJSONArray("to"));
                    break;
                case "delete":
                    delete(op.getJSONArray("start"), op.optString("target", ""), op.optJSONArray("end"));
                    break;
                case "setAttributes":
                    setAttributes(op.getJSONObject("attrs"), op.optString("target", ""), op.getJSONArray("start"), op.optJSONArray("end"));
                    break;
                case "insertHardBreak":
                    insertHardBreak(op.getJSONArray("start"), op.optString("target", ""), /* op.optString("type"), */ op.optJSONObject("attrs"));
                    break;
                case "insertTab":
                    insertTab(op.getJSONArray("start"), op.optString("target", ""), op.optJSONObject("attrs"));
                    break;
                case "insertField":
                    insertField(op.getJSONArray("start"), op.optString("target", ""), op.getString("type"), op.getString("type"), op.optJSONObject("attrs"));
                    break;
                case "updateField":
                    updateField(op.getJSONArray("start"), op.optString("target", ""), op.optString("type"), op.optString("type"), op.optJSONObject("attrs"));
                    break;
                case "noOp": {
                    break;
                }
                case "createError": {
                    throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                }
            }
/*
            else if (opName.equals("copySheet"))
                applyOperationHelper.copySheet(op.getInt("sheet"), op.getInt("to"), op.getString("sheetName"));
            else if (opName.equals("setDrawingAttributes"))
                applyOperationHelper.applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
            else if (opName.equals("insertValidation"))
                ValidationUtils.insertValidation(this, op);
            else if (opName.equals("changeValidation"))
                ValidationUtils.changeValidation(this, op);
            else if (opName.equals("deleteValidation"))
                ValidationUtils.deleteValidation(this, op.getInt("sheet"), op.getInt("index"));
            else
                logMessage("warn", "Ignoring unsupported operation: " + opName);
*/
        }
        OdfOperationDoc.setSuccessfulAppliedOperations(operations.length());
        return 1;
    }

    private ChartContent getChart(JSONArray start) {
        final DrawingComponent drawingComponent = (DrawingComponent)Component.getComponent(content.getRootComponent(null), start, start.length());
        final Shape shape = drawingComponent.getShape();
        return ((DrawObject)((DrawFrame)shape).getDrawing()).getChart();
    }

    private void changeOrinsertCondFormatRule(boolean insert, int sheetIndex, String id, String ranges, String type, Object value1, String value2, Integer priority, boolean stop, JSONObject dataBar, JSONObject iconSet, JSONArray colorScale, JSONObject attrs)
        throws JSONException {

        final Map<String, Condition> conditions = content.getSheets().get(sheetIndex).getConditionalFormats(true).getConditionalFormatList();
        Condition condition;
        if(insert) {
            switch (type) {
                case "dataBar":
                    condition = new EntryHolderCondition(id, new Ranges(content, ranges), "data-bar");
                    break;
                case "colorScale":
                    condition = new EntryHolderCondition(id, new Ranges(content, ranges), "color-scale");
                    break;
                case "iconSet":
                    condition = new EntryHolderCondition(id, new Ranges(content, ranges), "icon-set");
                    break;
                default:
                    condition = new Condition(id, new Ranges(content, ranges));
                    break;
            }
            conditions.put(id, condition);
        }
        else {
            condition = conditions.get(id);
            if(ranges!=null) {
                condition.setRanges(new Ranges(content, ranges));
            }
        }
        condition.applyCondFormatRuleOperation(doc, type, value1, value2, priority, stop, dataBar, iconSet, colorScale, attrs);
    }

    private void deleteCondFormatRule(int sheetIndex, String id) {

        content.getSheets().get(sheetIndex).getConditionalFormats(false).getConditionalFormatList().remove(id);
    }

    private void setDocumentAttributes(JSONObject attrs) {

        final JSONObject documentAttrs = attrs.optJSONObject("document");
        if(documentAttrs!=null) {
            final Object activeSheet = documentAttrs.opt("activeSheet");
            if(activeSheet instanceof Integer) {
                final String sheetName = content.getSheets().get(((Integer)activeSheet).intValue()).getSheetName();
                if(sheetName!=null) {
                    settings.setActiveSheet(sheetName);
                }
            }
        }
    }

    private void insertHyperlink(int sheet, JSONArray start, JSONArray optEnd, String url)
        throws JSONException {

        if(optEnd==null) {
            optEnd = start;
        }
        final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<Hyperlink> hyperlinkList = content.getSheets().get(sheet).getHyperlinks();
        for(int i=hyperlinkList.size()-1;i>=0;i--) {
            final Hyperlink hyperlink = hyperlinkList.get(i);
            if((hyperlink.getCellRefRange()==null)||cellRefRange.contains(hyperlink.getCellRefRange())) {
                hyperlinkList.remove(i);
            }
        }
        hyperlinkList.add(new Hyperlink(cellRefRange, url));
    }

    private void deleteHyperlink(int sheet, JSONArray start, JSONArray optEnd)
        throws JSONException {

        if(optEnd==null) {
            optEnd = start;
        }
        final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<Hyperlink> hyperlinkList = content.getSheets().get(sheet).getHyperlinks();
        for(int i=hyperlinkList.size()-1;i>=0;i--) {
            final Hyperlink hyperlink = hyperlinkList.get(i);
            if((hyperlink.getCellRefRange()==null)||cellRefRange.intersects(hyperlink.getCellRefRange())) {
                hyperlinkList.remove(i);
            }
        }
    }

    public Component insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws UnsupportedOperationException, JSONException, SAXException {

        Component.Type childComponentType = Component.Type.AC_IMAGE;
        if(type.equals("shape")||type.equals("connector")) {
            if(attrs==null||!attrs.has("geometry")) {
                childComponentType = Component.Type.AC_FRAME;
            }
            else {
                childComponentType = Component.Type.AC_SHAPE;
            }
        }
        else if(type.equals("chart")) {
            childComponentType = Component.Type.AC_CHART;
        }
        else if(type.equals("group")) {
            childComponentType = Component.Type.AC_GROUP;
        }
        else if(type.equals("image")) {
            childComponentType = Component.Type.AC_IMAGE;
        }
        return Component.getComponent(content.getRootComponent(null), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, childComponentType);
    }

    public void moveDrawing(JSONArray start, JSONArray to) throws JSONException {
        final Sheet sheet = content.getSheets().get(start.getInt(0));
        sheet.getDrawings().moveDrawing(start.getInt(1), to.getInt(1));
    }

    public void deleteDrawing(JSONArray start)
        throws JSONException {

        final Sheet sheet = content.getSheets().get(start.getInt(0));
        sheet.getDrawings().deleteDrawing(start.getInt(1), opsDoc);
    }

    private void changeOrInsertName(int sheet, String label, String formula, String newLabel, boolean expr, boolean hidden, String ref, boolean insert) {

        final NamedExpressions namedExpressions = sheet==-1?content.getNamedExpressions(true):content.getSheets().get(sheet).getNamedExpressions(true);
        NamedExpression namedExpression;
        if(insert) {
            namedExpression = new NamedExpression(label);
        }
        else {
            namedExpression = namedExpressions.getExpressionList().remove(label);
            if(newLabel!=null) {
                namedExpression.setName(newLabel);
            }
        }
        if(formula!=null) {
            if(expr) {
                namedExpression.setExpression(formula);
            }
            else {
                namedExpression.setCellRangeAddress(getAddressWithoutBraces(formula));
            }
        }
        if(ref!=null) {
            namedExpression.setBaseCellAddress(getAddressWithoutBraces(ref));
        }
        namedExpressions.getExpressionList().put(namedExpression.getName(), namedExpression);
    }

    private String getAddressWithoutBraces(String in) {
        if(in==null||in.length()<2) {
            return in;
        }
        if(in.charAt(0)=='['&&in.charAt(in.length()-1)==']') {
            return in.substring(1, in.length()-1);
        }
        return in;
    }

    private void deleteName(int sheet, String label) {

        final NamedExpressions namedExpressions = sheet==-1?content.getNamedExpressions(false):content.getSheets().get(sheet).getNamedExpressions(false);
        namedExpressions.getExpressionList().remove(label);
    }

    private void insertTable(int sheetIndex, String table, JSONArray start, JSONArray end, JSONObject attrs)
        throws JSONException {

        final DatabaseRange databaseRange = content.getDatabaseRanges(true).getDatabaseRange(content, sheetIndex, table, true);
        databaseRange.setDisplayFilterButtons(true);
        final Range sheetRange = databaseRange.getRange();
        final CellRefRange cellRefRange = sheetRange.getCellRefRange();
        if(start!=null) {
            final CellRef startRef = cellRefRange.getStart();
            startRef.setColumn(start.getInt(0));
            startRef.setRow(start.getInt(1));
        }
        if(end!=null) {
            final CellRef endRef = cellRefRange.getEnd();
            endRef.setColumn(end.getInt(0));
            endRef.setRow(end.getInt(1));
        }
        sheetRange.setSheetIndex(sheetIndex);
        if(attrs!=null) {
            final JSONObject tableAttrs = attrs.optJSONObject("table");
            if(tableAttrs!=null) {
                Object containsHeader = tableAttrs.opt("headerRow");
                if(containsHeader instanceof Boolean) {
                    databaseRange.setContainsHeader((Boolean)containsHeader);
                }
                Object hideButtons = tableAttrs.opt("hideButtons");
                if(hideButtons!=null) {
                    if(hideButtons==JSONObject.NULL) {
                        databaseRange.setDisplayFilterButtons(true);
                    }
                    else if(hideButtons instanceof Boolean) {
                        databaseRange.setDisplayFilterButtons(!((Boolean)hideButtons).booleanValue());
                    }
                }
            }
        }
    }

    private void deleteTable(int sheet, String table) {
        content.getDatabaseRanges(false).deleteTable(content, sheet, table);
    }

    private void changeTableColumn(int sheet, String table, int col, JSONObject attrs) {
        content.getDatabaseRanges(false).getDatabaseRange(content, sheet, table, false).changeTableColumn(content, col, attrs);
    }

    public void applyDrawingAttributes(JSONArray start, JSONObject attrs)
        throws JSONException {

        final int index = start.getInt(1);
        final Sheet sheet = content.getSheets().get(start.getInt(0));
        sheet.getDrawings().getDrawing(index).applyAttrsFromJSON(opsDoc, attrs, true);
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
            throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        final Sheet sheet = content.getSheets().get(sheetIndex);

        // first we will remove each mergeCell that is covering our new mergeCellRange
        final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(new SmlUtils.CellRef(start.getInt(0), start.getInt(1)), new SmlUtils.CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<MergeCell> mergeCellList = sheet.getMergeCells();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            final MergeCell mergeCell = mergeCellList.get(i);
            if(cellRefRange.intersects(mergeCell.getCellRefRange())) {
                mergeCellList.remove(i);
            }
        }
        if(type.equals("merge")) {
            addMergeCellRange(sheet, cellRefRange);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                addMergeCellRange(sheet, new CellRefRange(new CellRef(start.getInt(0), row), new CellRef(optEnd.getInt(0), row)));
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                addMergeCellRange(sheet, new CellRefRange(new CellRef(column, start.getInt(1)), new CellRef(column, optEnd.getInt(1))));
            }
        }
    }

    private void addMergeCellRange(Sheet sheet, CellRefRange mergeCellRange) {
        final List<MergeCell> mergeCellList = sheet.getMergeCells();
        mergeCellList.add(new MergeCell(mergeCellRange));
        // creating the last row (with ceiling cut)
        sheet.getRow(mergeCellRange.getEnd().getRow(), true, false, true);
        // creating the first row without repeatings (needed for covered cells, they are not allowed to be repeated)
        sheet.getRow(mergeCellRange.getStart().getRow(), true, true, true);

        final NavigableSet<Row> rowSelection = sheet.getRows().subSet(new Row(mergeCellRange.getStart().getRow()), true, new Row(mergeCellRange.getEnd().getRow()), true);
        final Iterator<Row> rowIter = rowSelection.iterator();
        boolean first = false;
        while(rowIter.hasNext()) {
            final Row rowEntry = rowIter.next();
            rowEntry.getCell(mergeCellRange.getEnd().getColumn(), true, false, true);
            rowEntry.getCell(mergeCellRange.getStart().getColumn(), true, true, first);
            first = false;
        }
    }

    public void insertRows(int sheetIndex, int start, int optEnd)
        throws Exception {

        final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).insertRows(sheetIndex, start, count);
    }

    public void deleteRows(int sheetIndex, int start, int optEnd)
        throws Exception {

        final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).deleteRows(sheetIndex, start, count);
    }

    public void insertColumns(int sheetIndex, int start, int optEnd)
        throws Exception {

        final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).insertColumns(sheetIndex, start, count);
    }

    public void deleteColumns(int sheetIndex, int start, int optEnd)
        throws Exception {

        final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).deleteColumns(sheetIndex, start, count);
    }

    private int getColumnRowCount(int start, int optEnd) {
        if(start<0) {
            throw new RuntimeException("ods::deleteColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("ods::deleteColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        return count;
    }

    public void changeCells(int sheetIndex, JSONArray start, JSONArray contents)
        throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("ods::JsonOperationConsumer::changeCells: size of start parameter != 2");

        int rowIndex = start.getInt(1);

        final Sheet sheet = content.getSheets().get(sheetIndex);
        for(int r = 0; r < contents.length(); r++) {

            int columnIndex = start.getInt(0);

            final JSONObject rowData = contents.getJSONObject(r);
            final int rowRepeat = rowData.optInt("r", 1);
            final Object cellContentObj = rowData.opt("c");
            JSONArray cellContent = null;
            if(cellContentObj!=null) {
                if(cellContentObj instanceof JSONArray) {
                    cellContent = (JSONArray)cellContentObj;
                }
                else {
                    cellContent = contents.getJSONObject((Integer)cellContentObj).optJSONArray("c");
                }
            }
            if(cellContent!=null) {

                // creating row entries...
                sheet.getRow((rowIndex + rowRepeat) - 1, true, false, true);
                sheet.getRow(rowIndex, true, true, false);

                for(int c = 0; c < cellContent.length(); c++) {

                    final JSONObject cellData = cellContent.getJSONObject(c);
                    final int cellRepeat = cellData.optInt("r", 1);

                    if(cellData.optBoolean("u")) {
                        sheet.clearCellRange(rowIndex, (rowIndex + rowRepeat) - 1, columnIndex, (columnIndex + cellRepeat) - 1);
                    }
                    else {

                        final String s = cellData.optString("s", null);
                        final Object f = cellData.opt("f");
                        final Object v = cellData.opt("v");
                        final String e = cellData.optString("e", null);
                        final String m = cellData.optString("mr", null);

                        // oo / lo does not like repetitions with formulas :-(
                        if((cellRepeat > 1 || rowRepeat > 1) && f instanceof String) {
                            for(int j=0; j < rowRepeat; j++) {
                                final Row row = sheet.getRow(rowIndex + j, true, true, true);
                                for(int i = 0; i < cellRepeat; i++) {
                                    changeCell(row.getCell(columnIndex + i, true, true, true), s, f, v, e, m);
                                }
                            }
                        }
                        final NavigableSet<Row> rowSelection = sheet.getRows().subSet(new Row(rowIndex), true, new Row(rowIndex+rowRepeat-1), true);
                        final Iterator<Row> rowIter = rowSelection.iterator();
                        while(rowIter.hasNext()) {
                            final Row row = rowIter.next();
                            // creating cell entries....
                            row.getCell((columnIndex + cellRepeat) - 1, true, false, true);
                            row.getCell(columnIndex, true, true, false);

                            final NavigableSet<Cell> cellSelection = row.getCells().subSet(new Cell(columnIndex), true, new Cell(columnIndex+cellRepeat-1), true);
                            final Iterator<Cell> cellIter = cellSelection.iterator();
                            while(cellIter.hasNext()) {
                                changeCell(cellIter.next(), s, f, v, e, m);
                            }
                        }
                    }
                    columnIndex += cellRepeat;
                }
            }
            rowIndex += rowRepeat;
        }
    }

    private void changeCell(Cell cell, String s, Object f, Object v, String e, String m) {
        if(s!=null) {
            cell.setCellStyle(s);
        }
        if(e!=null) {
            cell.setCellContent(new Cell.ErrorCode(e));
        }
        else if(v==JSONObject.NULL) {
            cell.setCellContent(null);
        }
        else if(v!=null) {
            cell.setCellContent(v);
        }

        if(f instanceof String) {
            cell.setCellFormula((String)f);
        }
        else if(f==JSONObject.NULL) {
            cell.setCellFormula(null);
        }
        if(m!=null) {
            if(m==JSONObject.NULL) {
                final CellAttributesEnhanced enhancedCellAttributes = cell.getCellAttributesEnhanced(false);
                if(enhancedCellAttributes!=null) {
                    enhancedCellAttributes.setNumberMatrixColumnsSpanned(null);
                    enhancedCellAttributes.setNumberMatrixRowsSpanned(null);
                }
            }
            else {
                final CellAttributesEnhanced enhancedCellAttributes = cell.getCellAttributesEnhanced(true);
                final CellRefRange range = SmlUtils.createCellRefRange(m);
                enhancedCellAttributes.setNumberMatrixColumnsSpanned(Integer.toString(range.getColumns()));
                enhancedCellAttributes.setNumberMatrixRowsSpanned(Integer.toString(range.getRows()));
            }
        }
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
        throws JSONException {

        content.insertSheet(sheetIndex, sheetName);
        if(attrs!=null) {
            setSheetAttributes(sheetIndex, attrs);
        }
    }

    public void setSheetName(int sheetIndex, String sheetName) {
        final Sheet _sheet = content.getSheets().get(sheetIndex);

        final String oldSheetName = _sheet.getSheetName();
        final String encodedOldName = encodeSheetName(oldSheetName);
        final String encodedNewName = encodeSheetName(sheetName);

        if(encodedOldName.length()<=0&&encodedNewName.length()<=0) {
            return;
        }
        _sheet.setSheetName(sheetName);

        // taking care of viewSettings
        settings.setActiveSheet(updateSheetName(settings.getActiveSheet(), encodedOldName, encodedNewName));
        final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(false);
        if(globalViewSettings!=null) {
            final ConfigItemMapNamed viewTables = settings.getViewTables(globalViewSettings, false);
            if(viewTables!=null) {
                final HashMap<String, ConfigItemMapEntry> viewTableMap = viewTables.getItems();
                final ConfigItemMapEntry oldViewSettings = viewTableMap.remove(encodedOldName);
                if(oldViewSettings!=null) {
                    oldViewSettings.setName(encodedNewName);
                    viewTableMap.put(encodedNewName, oldViewSettings);
                }
            }
        }
    }

    public static int getNameOffset(String content, int startOffset) {

        boolean stringLiteral = false;
        boolean complexString = false;
        for(int i=startOffset; i<content.length();i++) {
            final char n = content.charAt(i);
            if(complexString) {
                if(n=='\'') {
                    if(i+1<content.length()&&content.charAt(i+1)=='\'') {
                        i++;
                    }
                    else {
                        complexString = false;
                    }
                }
            }
            else if(stringLiteral) {
                if(n=='"') {
                    if(i+1<content.length()&&content.charAt(i+1)=='"') {
                        i++;
                    }
                    else {
                        stringLiteral = false;
                    }
                }
            }
            else if(n=='\'') {
                complexString = true;
            }
            else if(n=='"') {
                stringLiteral = true;
            }
            else if(n=='[') {
                return i;
            }
        }
        return -1;
    }

    public static String updateSheetName(String content, String oldName, String newName) {
        String newContent = content;
        if(content!=null&&content.length()>oldName.length()) {
            newContent = content.replaceAll(oldName, newName);
        }
        return newContent;
    }

    private static String simpleCharPattern ="^[\\w.\\xa1-\\u2027\\u202a-\\uffff]+$";

    public static String updateFormulaSheetName(String content, String oldName, String newName) {
        String newContent = content;
        List<Integer> replaceOffsets = null;
        if(content!=null&&content.length()>oldName.length()) {
            for(int startOffset = 0; startOffset<content.length();) {
                final int nameOffset = getNameOffset(content, startOffset);
                if(nameOffset<0) {
                    break;
                }
                final int possibleLength = nameOffset - startOffset;
                if(possibleLength>=oldName.length()) {
                    boolean replace = false;
                    final boolean isComplex = content.charAt(nameOffset-1)=='\'';
                    final String subString = content.substring(nameOffset-oldName.length(), nameOffset);

                    final int mOffset = nameOffset - oldName.length();
                    if(isComplex&&oldName.charAt(0)=='\'') {
                        if(subString.equals(oldName)) {
                            replace = true;
                        }
                    }
                    else if(oldName.charAt(0)!='\'') {
                        if(subString.equals(oldName)) {
                            if(mOffset>startOffset) {
                                final String prev = content.substring(mOffset-1, mOffset);
                                if(!prev.matches(simpleCharPattern)) {
                                    replace = true;
                                }
                            }
                            else {
                                replace = true;
                            }
                        }
                    }
                    if(replace) {
                        if(replaceOffsets==null) {
                            replaceOffsets = new ArrayList<Integer>();
                        }
                        replaceOffsets.add(mOffset);
                    }
                }
                startOffset = nameOffset + 1;
            }
        }
        if(replaceOffsets!=null) {
            final StringBuffer buffer = new StringBuffer(content);
            for(int i=replaceOffsets.size()-1;i>=0;i--) {
                final int offset = replaceOffsets.get(i);
                buffer.replace(offset, offset + oldName.length(), newName);
            }
            newContent = buffer.toString();
        }
        return newContent;
    }

    public static String encodeSheetName(String sheetName) {
        final StringBuffer encodedName = new StringBuffer(sheetName.length());
        for(int i=0; i<sheetName.length();i++) {
            final char c = sheetName.charAt(i);
            encodedName.append(c);
            if(c=='\'') {
                encodedName.append(c);
            }
        }
        if(!sheetName.matches(simpleCharPattern)) {
            encodedName.insert(0, '\'');
            encodedName.append('\'');
        }
        return encodedName.toString();
    }

    public void changeColumns(int sheetIndex, int start, int optEnd, JSONObject optAttrs, String s)
        throws JSONException {

        if(s==null&&optAttrs==null) {
            return;
        }
        if(optEnd==-1) {
            optEnd = start;
        }
        final Sheet sheet = content.getSheets().get(sheetIndex);
        // ... splitting up corresponding columns (repeated)..
        sheet.getColumn(optEnd, true, false, true);
        sheet.getColumn(start, true, true, false);

        final JSONObject columnAttrs = optAttrs!=null ? optAttrs.optJSONObject("column") : null;
        final NavigableSet<Column> columns = sheet.getColumns().subSet(new Column(start), true, new Column(optEnd), true);
        for(Column column:columns) {
            if(s!=null) {
                column.setDefaultCellStyle(s);
            }

            // set the column attributes (attribute family "column")
            if(columnAttrs!=null) {
                column.setStyleName(styleManager.applyAttributes("table-column", column.getStyleName(), true, optAttrs));
                final Object visible = columnAttrs.opt("visible");
                if(visible!=null) {
                    if(visible instanceof Boolean) {
                        column.setVisibility(((Boolean)visible).booleanValue()?Visibility.VISIBLE:Visibility.COLLAPSE);
                    }
                    else {
                        column.setVisibility(null);
                    }
                }
            }

            // set the grouping attributes (attribute family "group")
            if (optAttrs!=null) {
                column.changeGroupAttributes(optAttrs);
            }
        }
        // creating max repeated rows ...
        if(s!=null) {
            sheet.getRow(Sheet.getMaxRowCount()-1, true, false, false);
            final Iterator<Row> rowIter = sheet.getRows().iterator();
            while(rowIter.hasNext()) {
                final Row row = rowIter.next();
                // splitting up repeated cells and allow to create a proper iterator over the NavigableSet
                row.getCell(optEnd, true, false, true);
                row.getCell(start, true, true, false);
                final NavigableSet<Cell> cellSelection = row.getCells().subSet(new Cell(start), true, new Cell(optEnd), true);
                final Iterator<Cell> cellIter = cellSelection.iterator();
                while(cellIter.hasNext()) {
                    cellIter.next().setCellStyle(s);
                }
            }
        }
    }

    public void changeRows(int sheetIndex, int start, int optEnd, JSONObject optAttrs, String s)
        throws JSONException {

        if(optEnd==-1) {
            optEnd = start;
        }
        final Sheet sheet = content.getSheets().get(sheetIndex);
        // splitting up repeated rows and allow to create a proper iterator over the NavigableSet
        sheet.getRow(optEnd, true, false, true);
        sheet.getRow(start, true, true, false);

        final NavigableSet<Row> rows = sheet.getRows().subSet(new Row(start), true, new Row(optEnd), true);
        for(Row row:rows) {

            // set the row attributes (attribute family "row")
            final JSONObject rowAttrs = optAttrs!=null ? optAttrs.optJSONObject("row") : null;
            if(rowAttrs!=null) {
                row.setStyleName(styleManager.applyAttributes("table-row", row.getStyleName(), true, optAttrs));

                Object visibility = rowAttrs.opt("visible");
                Object filtered = rowAttrs.opt("filtered");
                if(visibility!=null||filtered!=null) {
                    if(filtered instanceof Boolean) {
                        if(((Boolean)filtered).booleanValue()) {
                            row.setVisibility(Visibility.FILTER);
                        }
                        else if(visibility instanceof Boolean) {
                            row.setVisibility(((Boolean)visibility).booleanValue() ? Visibility.VISIBLE : Visibility.COLLAPSE);
                        }
                        else {
                            row.setVisibility(Visibility.VISIBLE);
                        }
                    }
                    else if(visibility instanceof Boolean) {
                        if(((Boolean)visibility).booleanValue()) {
                            row.setVisibility(Visibility.VISIBLE);
                        }
                        else {
                            row.setVisibility(Visibility.COLLAPSE);
                        }
                    }
                    else {
                        row.setVisibility(null);
                    }
                }
            }

            // set the grouping attributes (attribute family "group")
            if (optAttrs!=null) {
                row.changeGroupAttributes(optAttrs);
            }
            if(s!=null) {

                row.setDefaultCellStyle(s);

                // creating row entries...
                row.getCell(sheet.getMaxColCount()-1, true, false, false);

                final TreeSet<Cell> cells = row.getCells();
                final Iterator<Cell> cellIter = cells.iterator();

                while(cellIter.hasNext()) {
                    cellIter.next().setCellStyle(s);
                }
            }
        }
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs)
        throws JSONException {

        final JSONObject sheetProperties = attrs.optJSONObject("sheet");
        if(sheetProperties!=null) {
            final Sheet sheet = content.getSheets().get(sheetIndex);
            final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(true);
            final ConfigItemMapEntry sheetViewSettings = settings.getViewSettings(globalViewSettings, sheet.getSheetName(), true);

            JSONArray ranges = null;
            int activeIndex = 0;

            for(Entry<String, Object> sheetPropEntry:sheetProperties.entrySet()) {
                final Object o = sheetPropEntry.getValue();
                switch(sheetPropEntry.getKey()) {
                    case "visible" : {
                        final JSONObject a = new JSONObject();
                        final JSONObject t = new JSONObject();
                        a.put("table", t);
                        t.put("visible", o);
                        sheet.setStyleName(styleManager.applyAttributes("table", sheet.getStyleName(), true, a));
                        break;
                    }
                    case "selectedRanges" : {
                        ranges = (JSONArray)o;
                        break;
                    }
                    case "activeIndex" : {
                        activeIndex = (Integer)o;
                        break;
                    }
                    case "zoom" : {
                        sheetViewSettings.addConfigItem("ZoomValue", Double.valueOf(((Number)o).doubleValue() * 100).intValue());
                        break;
                    }
                    case "scrollLeft" : {
                        sheetViewSettings.addConfigItem("PositionLeft", (Integer)o);
                        break;
                    }
                    case "scrollTop" : {
                        sheetViewSettings.addConfigItem("PositionBottom", (Integer)o);
                        break;
                    }
                    case "scrollBottom" : {
                        sheetViewSettings.addConfigItem("PositionTop", (Integer)o);
                        break;
                    }
                    case "scrollRight" : {
                        sheetViewSettings.addConfigItem("PositionRight", (Integer)o);
                        break;
                    }
                    case "showGrid" : {
                        sheetViewSettings.addConfigItem("ShowGrid", (Boolean)o);
                        break;
                    }
                    case "locked" : {
                        if(o instanceof Boolean) {
                            sheet.setAttributeNS(Namespaces.TABLE, "table:protected", ((Boolean)o).toString());
                        }
                        else if(o==JSONObject.NULL) {
                            sheet.removeAttributeNS(Namespaces.TABLE, "protected");
                        }
                        break;
                    }
                }
            }
            if(ranges!=null&&ranges.length()>=(activeIndex+1)) {
                final JSONObject range = ranges.getJSONObject(activeIndex);
                final JSONArray start = range.getJSONArray("start");
                sheetViewSettings.addConfigItem("CursorPositionX", start.getInt(0));
                sheetViewSettings.addConfigItem("CursorPositionY", start.getInt(1));
            }
        }
    }

    public void delete(JSONArray start, String target, JSONArray end)
        throws JSONException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        final Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
        component.splitStart(startComponent);
        component.delete(opsDoc, (endComponent-startComponent)+1);
    }

    public void setAttributes(JSONObject attrs, String target, JSONArray start, JSONArray end)
        throws JSONException, SAXException {

        if(attrs==null) {
            return;
        }
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
        component.splitStart(startIndex);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
            if(component.getNextComponentNumber()>=endIndex+1) {
                component.splitEnd(endIndex);
            }
            component.applyAttrsFromJSON(opsDoc, attrs);
            component = component.getNextComponent();
        }
    }

    public void insertParagraph(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
    }

    public void splitParagraph(JSONArray start, String target)
        throws JSONException {

        ((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .splitParagraph(opsDoc, start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start, String target) {

        ((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()))
            .mergeParagraph();
    }

    public void move(JSONArray start, String target, JSONArray end, JSONArray to)
        throws JSONException {

        Component.move(content.getRootComponent(target), start, end, to);
    }

    public void insertText(JSONArray start, String target, JSONObject attrs, String text)
        throws IndexOutOfBoundsException, JSONException, SAXException {

        if(text.length()>0) {
            final Component component = Component.getComponent(content.getRootComponent(target), start, start.length()-1);
            ((IParagraph)component).insertText(opsDoc, start.getInt(start.length()-1), text, attrs);
        }
    }

    public void insertTab(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.TAB);
    }

    public void insertHardBreak(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.HARDBREAK);
    }

    public void insertField(JSONArray start, String target, String type, String representation, JSONObject attrs)
        throws JSONException, UnsupportedOperationException, SAXException {

        final TextField textField = (TextField)Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Type.FIELD).getObject();

        textField.setType(type);
        textField.setRepresentation(representation);
    }

    public void updateField(JSONArray start, String target, String type, String representation, JSONObject attrs)
        throws UnsupportedOperationException {

        final TextFieldComponent tfComponent = (TextFieldComponent)Component.getComponent(content.getRootComponent(target), start, start.length());
        if(type!=null) {
            ((TextField)tfComponent.getObject()).setType(type);
        }
        if(representation!=null) {
            ((TextField)tfComponent.getObject()).setRepresentation(representation);
        }
    }
}