/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ods.dom;

import java.util.Iterator;
import java.util.List;
import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.ElementNS;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.properties.MapProperties;
import com.openexchange.office.filter.odf.styles.NumberCurrencyStyle;
import com.openexchange.office.filter.odf.styles.NumberDateStyle;
import com.openexchange.office.filter.odf.styles.NumberPercentageStyle;
import com.openexchange.office.filter.odf.styles.NumberTimeStyle;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleTableCell;
import com.openexchange.office.filter.ods.dom.SmlUtils.CellRef;

public class Cell implements Comparable<Cell> {

	public static class ErrorCode {

		final String errorCode;

		public ErrorCode(String errorCode) {
			this.errorCode = errorCode;
		}
		public String getError() {
			return errorCode;
		}
	}

	int column;
	private String cellStyle = null;
	private int repeated = 1;

	// the cell content is stored in one of the two following variables
	private Object content = null;				// immutable objects only, otherwise the clone needs to be changed
	private String formula = null;
	private ElementNS element = null;
	private CellAttributesEnhanced cellAttributesEnhanced = null;

	Cell(int c) {
		column = c;
	}

	public int getColumn() {
		return column;
	}

	public void setColumn(int c) {
		column = c;
	}

	public String getCellStyle() {
    	return cellStyle;
    }

    public void setCellStyle(String val) {
    	cellStyle = val;
    }

    public int getRepeated() {
    	return repeated;
    }

    public void setRepeated(int val) {
    	repeated = val;
    }

    public CellAttributesEnhanced getCellAttributesEnhanced(boolean createIfMissing) {
    	if(createIfMissing&&cellAttributesEnhanced==null) {
    		cellAttributesEnhanced = new CellAttributesEnhanced();
    	}
    	return cellAttributesEnhanced;
    }

    public ElementNSImpl createNode(OdfFileDom fileDom) {

    	if(element==null) {
			element = new ElementNS(fileDom, Namespaces.TABLE, "table:table-cell");
    	}
    	return element;
    }

    public ElementNSImpl getNode() {
    	return element;
    }

    // mergeCell can be null if cell is not merged / covered
    public void writeObject(SerializationHandler output, Sheet sheet, Row row, MergeCell mergeCell)
        throws SAXException {

        boolean covered = false;
        boolean spannedAttribute = false;

        if(mergeCell!=null) {
            final CellRef mergeStart = mergeCell.getCellRefRange().getStart();
            if(mergeStart.getColumn()==getColumn()&&row.getRow()==mergeStart.getRow()) {
                // only the top left cell of the merge range gets the spanned row/column attributes
                spannedAttribute = true;
            }
            else {
                covered = true;
            }
        }
        if(covered) {
            SaxContextHandler.startElement(output, Namespaces.TABLE, "covered-table-cell", "table:covered-table-cell");
        }
        else {
            SaxContextHandler.startElement(output, Namespaces.TABLE, "table-cell", "table:table-cell");
        }
        if(getRepeated()>1) {
            SaxContextHandler.addAttribute(output, Namespaces.TABLE, "number-columns-repeated", "table:number-columns-repeated", Integer.toString(getRepeated()));
        }
        if(getCellStyle()!=null) {
            SaxContextHandler.addAttribute(output, Namespaces.TABLE, "style-name", "table:style-name", getCellStyle());
        }
        if(spannedAttribute) {
            final int columns = mergeCell.getCellRefRange().getEnd().getColumn() - mergeCell.getCellRefRange().getStart().getColumn();
            final int rows = mergeCell.getCellRefRange().getEnd().getRow() - mergeCell.getCellRefRange().getStart().getRow();
            if(columns>0||rows>0) {
                SaxContextHandler.addAttribute(output, Namespaces.TABLE, "number-columns-spanned", "table:number-columns-spanned", Integer.toString(columns+1));
                SaxContextHandler.addAttribute(output, Namespaces.TABLE, "number-rows-spanned", "table:number-rows-spanned", Integer.toString(rows+1));
            }
        }

        // writing cell content
        if(cellAttributesEnhanced!=null) {
            cellAttributesEnhanced.writeAttributes(output);
        }
        if(content!=null) {
            if(formula!=null) {
                SaxContextHandler.addAttribute(output, Namespaces.TABLE, "formula", "table:formula", "of:=".concat(formula));
            }
            if(content instanceof String) {
                final String contentString = (String)content;
                if(!contentString.isEmpty()) {
                    SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "value-type", "office:value-type", "string");
                    SaxContextHandler.addAttribute(output, Namespaces.CALCEXT, "value-type", "calcext:value-type", "string");
                    writeContentString(output, sheet, row, contentString);
                }
            }
            else if(content instanceof Boolean) {
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "value-type", "office:value-type", "boolean");
                SaxContextHandler.addAttribute(output, Namespaces.CALCEXT, "value-type", "calcext:value-type", "boolean");
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "boolean-value", "office:boolean-value", ((Boolean)content)?"true":"false");
            }
            else if(content instanceof ErrorCode) {
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "value-type", "office:value-type", "string");
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "string-value", "office:string-value", "");
                SaxContextHandler.addAttribute(output, Namespaces.CALCEXT, "value-type", "calcext:value-type", "error");
                final String errorCode = ((ErrorCode)content).getError();
                if(errorCode!=null) {
                    writeContentString(output, sheet, row, errorCode);
                }
            }
            else if(content instanceof Number) {
                String valueType = "float";
                String styleName = getCellStyle();
                if(styleName==null) {
                    styleName = row.getDefaultCellStyle();
                    if(styleName==null) {
                        final Column col = sheet.getColumn(getColumn(), false, false, false);
                        if(col!=null) {
                            styleName = col.getDefaultCellStyle();
                        }
                    }
                }
                if(styleName!=null&&!styleName.isEmpty()) {
                    final StyleManager styleManager = ((OdfFileDom)sheet.getOwnerDocument()).getDocument().getStyleManager();
                    final StyleBase styleBase = styleManager.getStyle(styleName, "table-cell", true);
                    if(styleBase instanceof StyleTableCell) {
                        final StyleTableCell styleTableCell = (StyleTableCell)styleBase;
                        final String dataStyleName = styleTableCell.getAttribute("style:data-style-name");
                        if(dataStyleName!=null&&!dataStyleName.isEmpty()) {
                            StyleBase dataStyleBase = styleManager.getStyle(dataStyleName, "data-style", true);
                            if(dataStyleBase!=null) {
                                final Iterator<MapProperties> mapPropertyIter = dataStyleBase.getMapStyleList().iterator();
                                while(mapPropertyIter.hasNext()) {
                                    final MapProperties mapProperties = mapPropertyIter.next();
                                    final String mapping = mapProperties.getMapping(styleManager, true, "table-cell");
                                    if(mapping!=null) {
                                        final String applyStyleName = mapProperties.getAttribute("style:apply-style-name");
                                        if(applyStyleName!=null) {
                                            final StyleBase conditionalDataStyleBase = styleManager.getStyle(applyStyleName, "data-style", true);
                                            if(conditionalDataStyleBase!=null) {
                                                dataStyleBase = conditionalDataStyleBase;
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                            if(dataStyleBase instanceof NumberCurrencyStyle) {
                                final String currencyCode = ((NumberCurrencyStyle)dataStyleBase).getCurrencyCode();
                                if(currencyCode!=null) {
                                    SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "currency", "office:currency", currencyCode);
                                }
                                valueType = "currency";
                            }
                            else if(dataStyleBase instanceof NumberPercentageStyle) {
                                valueType = "percentage";
                            }
                            else if(dataStyleBase instanceof NumberDateStyle) {
                                // TODO: date-value instead of office-value should be created
                                valueType = "date";
                            }
                            else if(dataStyleBase instanceof NumberTimeStyle) {
//                                      SaxContextHandler.addOdfAttribute(output, OfficeTimeValueAttribute.ATTRIBUTE_NAME, null, MapHelper.doubleToTime(((Number)content).doubleValue()));
//                                      officeValueAttributeRequired = false;
                                valueType = "time";
                            }
                        }
                    }
                }
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "value", "office:value", ((Number)content).toString());
                SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "value-type", "office:value-type", valueType);
                SaxContextHandler.addAttribute(output, Namespaces.CALCEXT, "value-type", "calcext:value-type", valueType);
            }
        }
        final List<Drawing> anchoredDrawings = sheet.getDrawings().getAnchoredDrawings(new DrawingAnchor(this.getColumn(), row.getRow()));
        if(anchoredDrawings!=null) {
            for(Drawing drawing:anchoredDrawings) {
                drawing.writeObject(output);
            }
        }
        if(element!=null) {
            SaxContextHandler.serializeChildElements(output, element);
        }
        if(covered) {
            SaxContextHandler.endElement(output, Namespaces.TABLE, "covered-table-cell", "table:covered-table-cell");
        }
        else {
            SaxContextHandler.endElement(output, Namespaces.TABLE, "table-cell", "table:table-cell");
        }
    }

	private void writeContentString(SerializationHandler output, Sheet sheet, Row row, String contentString)
		throws SAXException {

		final String[] paragraphs = contentString.split("\r\n|\n|\r", -1);
    	for(int i=0; i<paragraphs.length; i++) {
        	// simply writing text ... what to do if the content is a error an starts with '#'
        	SaxContextHandler.startElement(output, Namespaces.TEXT, "p", "text:p");
        	final Hyperlink hyperlink = i==0 ? sheet.getHyperlink(column, row.getRow()) : null;
        	if(hyperlink!=null) {
        		SaxContextHandler.startElement(output, Namespaces.TEXT, "a", "text:a");
        		SaxContextHandler.addAttribute(output, Namespaces.XLINK, "type", "xlink:type", "simple");
        		SaxContextHandler.addAttribute(output, Namespaces.XLINK, "href", "xlink:href", hyperlink.getUrl());
        	}
        	if(!paragraphs[i].isEmpty()) {
        	    output.characters(paragraphs[i]);
        	}
        	if(hyperlink!=null) {
        		SaxContextHandler.endElement(output, Namespaces.TEXT, "a", "text:a");
        	}
        	SaxContextHandler.endElement(output, Namespaces.TEXT, "p", "text:p");
    	}
	}

	public Object getCellContent() {
		return content;
    }

    public void setCellContent(Object value) {
    	content = value;
    }

    public String getCellFormula() {
    	return formula;
    }

    public void setCellFormula(String formula) {
    	this.formula = formula;
    }

    public Cell clone(boolean cloneContent) {
    	final Cell clone = new Cell(column);
    	clone.setRepeated(getRepeated());
    	clone.setCellStyle(getCellStyle());
    	if(cloneContent) {
	    	if(element!=null) {
	    		clone.element = element.cloneNode(true);
	    	}
	    	else {
	    		if(content!=null) {
	    			clone.content = content;
	    		}
	    		clone.formula = formula;
	    	}
    	}
    	return clone;
    }

	@Override
	public int compareTo(Cell o) {
		return column - o.getColumn();
	}
}
