/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odp.dom;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfPresentationDocument;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.IAttributesAccessor;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.Settings;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.TextComponent;
import com.openexchange.office.filter.odf.components.TextFieldComponent;
import com.openexchange.office.filter.odf.components.TextLineBreakComponent;
import com.openexchange.office.filter.odf.components.TextSpan_Base;
import com.openexchange.office.filter.odf.components.TextTabComponent;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.properties.TextProperties;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleGraphic;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StylePresentation;
import com.openexchange.office.filter.odf.styles.TableTemplate;
import com.openexchange.office.filter.odp.dom.components.CellComponent;
import com.openexchange.office.filter.odp.dom.components.FrameComponent;
import com.openexchange.office.filter.odp.dom.components.ParagraphComponent;
import com.openexchange.office.filter.odp.dom.components.RowComponent;
import com.openexchange.office.filter.odp.dom.components.ShapeComponent;
import com.openexchange.office.filter.odp.dom.components.ShapeGroupComponent;
import com.openexchange.office.filter.odp.dom.components.SlideComponent;
import com.openexchange.office.filter.odt.dom.Text;
import com.openexchange.office.filter.odt.dom.TextField;

public class JsonOperationProducer {

    final JSONArray operationQueue;

    private final OdfOperationDoc opsDoc;
    private final OdfPresentationDocument doc;
    private final PresentationStyles styles;
    private final PresentationContent content;
    private final Presentation presentation;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationProducer(OdfOperationDoc opsDoc)
    	throws SAXException {

    	operationQueue = new JSONArray();
    	this.opsDoc = opsDoc;
        doc = (OdfPresentationDocument)opsDoc.getDocument();
        styles = (PresentationStyles)doc.getStylesDom();
        content = (PresentationContent)doc.getContentDom();
        presentation = content.getPresentation();
        settings = doc.getSettingsDom();
        styleManager = doc.getStyleManager();
        styleManager.setTabsRelativeToIndent(settings.hasTabsRelativeToIndent());
    }

    public JSONObject getDocumentOperations()
    	throws JSONException, SAXException {

    	createDocumentAttributes();
    	createTableTemplateStyleOperations();
    	createMasterSlides();
    	createSlides();

    	final JSONObject operations = new JSONObject(1);
        operations.put("operations", operationQueue);
        return operations;
    }

    public void createDocumentAttributes() throws JSONException {
        final OpAttrs attrs = new OpAttrs();
        final Iterator<Object> masterStyleContentIter = styleManager.getMasterStyles().getContent().iterator();
        MasterPage masterPage = null;
        while(masterStyleContentIter.hasNext()) {
            final Object o = masterStyleContentIter.next();
            if(o instanceof MasterPage) {
                masterPage = (MasterPage)o;
                break;
            }
        }
        if(masterPage!=null) {
            final StyleBase pageLayout = styleManager.getStyle(masterPage.getPageLayoutName(), "page-layout", false);
            if(pageLayout!=null) {
                pageLayout.createAttrs(styleManager, attrs);
            }
        }
        final StyleBase graphicStyle = styleManager.getStyle("standard", "graphic", false);
        if(graphicStyle!=null) {
            OpAttrs listStyles = new OpAttrs();
            styleManager.createPresentationAttributes(graphicStyle, "defaultTextListStyles", 10, false, listStyles);
            listStyles = listStyles.getMap("listStyles", false);
            if(listStyles!=null) {
                final OpAttrs defaultTextListStyles = listStyles.getMap("defaultTextListStyles", false);
                if(defaultTextListStyles!=null) {
                    attrs.put("defaultTextListStyles", defaultTextListStyles);
                }
            }
        }
        final StyleBase defaultGraphicStyle = styleManager.getStyle("_default", "graphic", false);
        if(defaultGraphicStyle!=null) {
            final TextProperties textProperties = ((StyleGraphic)defaultGraphicStyle).getTextProperties();
            final String language = getLanguage(textProperties.getAttribute("fo:language"), textProperties.getAttribute("fo:country"));
            if(language!=null) {
                attrs.getMap("character", true).put("language", language);
            }
            final String languageAsian = getLanguage(textProperties.getAttribute("style:language-asian"), textProperties.getAttribute("style:country-asian"));
            if(languageAsian!=null) {
                attrs.getMap("character", true).put("languageEa", languageAsian);
            }
            final String languageComplex = getLanguage(textProperties.getAttribute("style:language-complex"), textProperties.getAttribute("style:country-complex"));
            if(languageComplex!=null) {
                attrs.getMap("character", true).put("languageBidi", languageComplex);
            }
        }
        if(!attrs.isEmpty()) {
            final JSONObject jsonInsertSlideOperation = new JSONObject(3);
            jsonInsertSlideOperation.put("name", "setDocumentAttributes");
            jsonInsertSlideOperation.put("attrs", attrs);
            operationQueue.put(jsonInsertSlideOperation);
        }
    }

    private String getLanguage(String language, String country) {
        String r = null;
        if(language!=null) {
            r = country!=null ? language + "-" + country : language;
        }
        return r;
    }

    public void createTableTemplateStyleOperations()
        throws JSONException {

        final HashMap<String, DLNode<Object>> familyStyles = styleManager.getFamilyStyles("table-template");
        if(familyStyles!=null) {
            final Iterator<Entry<String, DLNode<Object>>> tableTemplateIter = familyStyles.entrySet().iterator();
            while(tableTemplateIter.hasNext()) {
                final Entry<String, DLNode<Object>> entry = tableTemplateIter.next();
                if(entry.getValue().getObject() instanceof TableTemplate) {
                    final TableTemplate tableTemplate = (TableTemplate)entry.getValue().getObject();
                    final JSONObject insertStylesheetObject = new JSONObject();
                    insertStylesheetObject.put("name", "insertStyleSheet");
                    insertStylesheetObject.put("type", "table");
                    insertStylesheetObject.put("styleId", entry.getKey());
                    insertStylesheetObject.put("styleName", entry.getKey());
                    final JSONObject attrs = new JSONObject();
                    final Iterator<Entry<String, AttributesImpl>> tableStyleEntryIter = tableTemplate.getTableTemplates().entrySet().iterator();
                    while(tableStyleEntryIter.hasNext()) {
                        String conditionalStyle = null;
                        final Entry<String, AttributesImpl> tableStyleEntry = tableStyleEntryIter.next();
                        switch(tableStyleEntry.getKey()) {
                            case "first-column": {
                                conditionalStyle = "firstCol";
                                break;
                            }
                            case "last-column": {
                                conditionalStyle = "lastCol";
                                break;
                            }
                            case "first-row": {
                                conditionalStyle = "firstRow";
                                break;
                            }
                            case "last-row": {
                                conditionalStyle = "lastRow";
                                break;
                            }
                            case "even-rows": {
                                conditionalStyle = "band2Hor";
                                break;
                            }
                            case "odd-rows": {
                                conditionalStyle = "band1Hor";
                                break;
                            }
                            case "even-columns": {
                                conditionalStyle = "band2Vert";
                                break;
                            }
                            case "odd-columns": {
                                conditionalStyle = "band1Vert";
                                break;
                            }
                            case "body": {
                                conditionalStyle = "wholeTable";
                                break;
                            }
                        }
                        if(conditionalStyle!=null) {
                            final String tableStyleName = tableStyleEntry.getValue().getValue("table:style-name");
                            if(tableStyleName!=null&&!tableStyleName.isEmpty()) {
                                final OpAttrs cellAttrs = new OpAttrs();
                                styleManager.createPresentationTemplateStyle(cellAttrs, tableStyleName);
                                if(!cellAttrs.isEmpty()) {
                                    attrs.put(conditionalStyle, cellAttrs);
                                }
                            }
                        }
                    }
                    if(!attrs.isEmpty()) {
                        insertStylesheetObject.put("attrs", attrs);
                    }
                    insertStylesheetObject.put("hidden", false);
                    operationQueue.put(insertStylesheetObject);
                }
            }
        }
        
    }

    public void createMasterSlides() throws JSONException, SAXException {
        final Iterator<Object> masterStyleContentIter = styleManager.getMasterStyles().getContent().iterator();
        while(masterStyleContentIter.hasNext()) {
            final Object o = masterStyleContentIter.next();
            if(o instanceof MasterPage) {
                final String masterPageId = ((MasterPage)o).getName();
                final SlideComponent slideComponent = new SlideComponent(null, new DLNode<Object>(o), 0, false);
                final JSONObject jsonInsertMasterSlideOperation = new JSONObject(3);
                jsonInsertMasterSlideOperation.put("name", "insertMasterSlide");
                jsonInsertMasterSlideOperation.put("id", masterPageId);
                final OpAttrs attrs = new OpAttrs();
                createMasterStyles(slideComponent, attrs);
                slideComponent.createAttrs(opsDoc, attrs);
                if(!attrs.isEmpty()) {
                    jsonInsertMasterSlideOperation.put("attrs", attrs);
                }
                operationQueue.put(jsonInsertMasterSlideOperation);
                createSlideOperations(slideComponent, 0, masterPageId);
            }
        }
    }

    public void createSlides() throws JSONException, SAXException {
        Component slideComponent = content.getRootComponent(null).getNextChildComponent(null, null);
        while(slideComponent instanceof SlideComponent) {
            final JSONObject jsonInsertSlideOperation = new JSONObject(4);
            jsonInsertSlideOperation.put("name", "insertSlide");
            jsonInsertSlideOperation.put("start", slideComponent.getComponentNumber());
            final String masterPageName = ((DrawingPage)slideComponent.getObject()).getMasterPageName();
            jsonInsertSlideOperation.put("target", masterPageName);
            final OpAttrs attrs = new OpAttrs();
            slideComponent.createAttrs(opsDoc, attrs);
            if(!attrs.isEmpty()) {
                jsonInsertSlideOperation.put("attrs", attrs);
            }
            operationQueue.put(jsonInsertSlideOperation);
            createSlideOperations((SlideComponent)slideComponent, slideComponent.getComponentNumber(), null);
            slideComponent = slideComponent.getNextComponent();
        }
    }

    public void createSlideOperations(SlideComponent slideComponent, int slide, String target)
        throws SAXException, JSONException {

        final ArrayList<Integer> position = new ArrayList<Integer>();
        position.add(slide);
        Component component = slideComponent.getNextChildComponent(null, null);
        while(component!=null) {
            final List<Integer> childPosition = new ArrayList<Integer>(position);
            childPosition.add(component.getComponentNumber());
            if(component instanceof ShapeComponent) {
                createShapeOperations((ShapeComponent)component, childPosition, target);
            }
            else if(component instanceof ShapeGroupComponent) {
                createShapeGroupOperations((ShapeGroupComponent)component, childPosition, target);
            }
            else if(component instanceof FrameComponent) {
                createFrameOperations((FrameComponent)component, childPosition, target);
            }
            component = component.getNextComponent();
        }
    }

    public void createFrameOperations(FrameComponent frameComponent, List<Integer> position, String target)
        throws JSONException {

        final OpAttrs attrs = new OpAttrs();
        frameComponent.createAttrs(opsDoc, attrs);
        addInsertDrawingOperation(position, ((IDrawingType)frameComponent).getType(), attrs, target);
        Component child = frameComponent.getNextChildComponent(null, null);
        while(child!=null) {
            final List<Integer> childPosition = new ArrayList<Integer>(position);
            childPosition.add(child.getComponentNumber());
            if(child instanceof ParagraphComponent) {
                createParagraphOperations((ParagraphComponent)child, childPosition, target);
            }
            else if(child instanceof ShapeComponent) {
                createShapeOperations((ShapeComponent)child, childPosition, target);
            }
            else if(child instanceof RowComponent) {
                createRowOperations((RowComponent)child, childPosition, target);
            }
            child = child.getNextComponent();
        }
    }

    public void createRowOperations(RowComponent rowComponent, List<Integer> rowPosition, String target)
        throws JSONException {

        OpAttrs attrs = new OpAttrs();
        rowComponent.createAttrs(opsDoc, attrs);
        addInsertRowsOperation(rowPosition, 1, target, attrs);
        Component component= rowComponent.getNextChildComponent(null, null);
        while(component!=null) {
            if(component instanceof CellComponent) {
                final List<Integer> cellPosition = new ArrayList<Integer>(rowPosition);
                cellPosition.add(component.getComponentNumber());
                createCellOperations((CellComponent)component, cellPosition, target);
            }
            component = component.getNextComponent();
        }
    }

    public void createCellOperations(CellComponent cellComponent, List<Integer> cellPosition, String target)
        throws JSONException {

        final OpAttrs attrs = new OpAttrs();
        cellComponent.createAttrs(opsDoc, attrs);
        addInsertCellsOperation(cellPosition, target, attrs);

        Component component = cellComponent.getNextChildComponent(null, null);
        while (component!=null) {
            if(component instanceof ParagraphComponent) {
                final List<Integer> paragraphPosition = new ArrayList<Integer>(cellPosition);
                paragraphPosition.add(component.getComponentNumber());
                createParagraphOperations((ParagraphComponent)component, paragraphPosition, target);
            }
            component = component.getNextComponent();
        }
    }

    public void createShapeOperations(ShapeComponent shapeComponent, List<Integer> position, String target)
        throws JSONException {

        final OpAttrs attrs = new OpAttrs();
        shapeComponent.createAttrs(opsDoc, attrs);
        addInsertDrawingOperation(position, ((IDrawingType)shapeComponent).getType(), attrs, target);
        Component child = shapeComponent.getNextChildComponent(null, null);
        while(child!=null) {
            final List<Integer> childPosition = new ArrayList<Integer>(position);
            childPosition.add(child.getComponentNumber());
            if(child instanceof ParagraphComponent) {
                createParagraphOperations((ParagraphComponent)child, childPosition, target);
            }
            child = child.getNextComponent();
        }
    }

    public void createShapeGroupOperations(ShapeGroupComponent shapeComponent, List<Integer> position, String target)
        throws JSONException, SAXException {

        final OpAttrs attrs = new OpAttrs();
        shapeComponent.createAttrs(opsDoc, attrs);
        addInsertDrawingOperation(position, ((IDrawingType)shapeComponent).getType(), attrs, target);
        Component child = shapeComponent.getNextChildComponent(null, null);
        while(child!=null) {
            final List<Integer> childPosition = new ArrayList<Integer>(position);
            childPosition.add(child.getComponentNumber());
            if(child instanceof ShapeComponent) {
                createShapeOperations((ShapeComponent)child, childPosition, target);
            }
            else if(child instanceof ShapeGroupComponent) {
                createShapeGroupOperations((ShapeGroupComponent)child, childPosition, target);
            }
            else if(child instanceof FrameComponent) {
                createFrameOperations((FrameComponent)child, childPosition, target);
            }
            child = child.getNextComponent();
        }
    }

    public void createParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> paragraphPosition, String target)
        throws JSONException {

        final JSONObject jsonInsertParagraphOperation = new JSONObject(4);
        jsonInsertParagraphOperation.put("name", "insertParagraph");
        jsonInsertParagraphOperation.put("start", paragraphPosition);
        if(target!=null) {
            jsonInsertParagraphOperation.put("target", target);
        }
        final OpAttrs attrs = new OpAttrs();
        paragraphComponent.createAttrs(opsDoc, attrs);
        if(!attrs.isEmpty()) {
            jsonInsertParagraphOperation.put("attrs", attrs);
        }
        operationQueue.put(jsonInsertParagraphOperation);
        Component paragraphChild = paragraphComponent.getNextChildComponent(null, null);
        while(paragraphChild!=null) {
            final List<Integer> textPosition = new ArrayList<Integer>(paragraphPosition);
            textPosition.add(paragraphChild.getComponentNumber());
            if(paragraphChild instanceof TextComponent) {
                createTextInsertOperation(textPosition, target, ((Text)((TextComponent)paragraphChild).getObject()).getText());
            }
            else if(paragraphChild instanceof TextTabComponent) {
                addInsertTabOperation(textPosition, target);
            }
            else if(paragraphChild instanceof TextFieldComponent) {
                addInsertFieldOperation(textPosition, (TextField)paragraphChild.getObject(), target);
            }
            else if(paragraphChild instanceof TextLineBreakComponent) {
                addInsertHardBreakOperation(textPosition, target);
            }
            paragraphChild = paragraphChild.getNextComponent();
        }
        paragraphChild = paragraphComponent.getNextChildComponent(null, null);
        while(paragraphChild!=null) {
            createTextAttributesOperation((TextSpan_Base)paragraphChild, paragraphPosition, target);
            paragraphChild = paragraphChild.getNextComponent();
        }
    }

    public void createTextInsertOperation(List<Integer> position, String target, String text)
        throws JSONException {

        final JSONObject jsonInsertTextOperation = new JSONObject(4);
        jsonInsertTextOperation.put("name", "insertText");
        jsonInsertTextOperation.put("start", position);
        if(target!=null) {
            jsonInsertTextOperation.put("target", target);
        }
        jsonInsertTextOperation.put("text", text);
        operationQueue.put(jsonInsertTextOperation);
    }

    public void addInsertDrawingOperation(List<Integer> start, DrawingType type, OpAttrs attrs, String target)
        throws JSONException {

        final JSONObject insertDrawingObject = new JSONObject(5);
        insertDrawingObject.put("name", "insertDrawing");
        insertDrawingObject.put("start", start);
        insertDrawingObject.put("type", type.toString());
        if(!attrs.isEmpty()) {
            insertDrawingObject.put("attrs", attrs);
        }
        if(target!=null) {
            insertDrawingObject.put("target", target);
        }
        operationQueue.put(insertDrawingObject);
    }

    public void addInsertCellsOperation(final List<Integer> start, String target, final OpAttrs attrs)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject(3);
        insertCellsObject.put("name", "insertCells");
        insertCellsObject.put("start", start);
        if(target!=null) {
            insertCellsObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            insertCellsObject.put("attrs", attrs);
        }
        operationQueue.put(insertCellsObject);
    }

    public void addInsertRowsOperation(final List<Integer> start, int count, String target, final OpAttrs attrs)
        throws JSONException {

        final JSONObject insertRowObject = new JSONObject(4);
        insertRowObject.put("name", "insertRows");
        insertRowObject.put("start", start);
        if(count!=1) {
            insertRowObject.put("count", count);
        }
        if(target!=null) {
            insertRowObject.put("target", target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
            insertRowObject.put("attrs", attrs);
        }
        operationQueue.put(insertRowObject);
    }

    public void addInsertTabOperation(final List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertTabObject = new JSONObject(3);
        insertTabObject.put("name", "insertTab");
        insertTabObject.put("start", start);
        if(target!=null) {
            insertTabObject.put("target", target);
        }
        operationQueue.put(insertTabObject);
    }

    public void addInsertFieldOperation(List<Integer> start, TextField field, String target)
        throws JSONException {

        final JSONObject insertFieldObject = new JSONObject(5);
        insertFieldObject.put("name", "insertField");
        insertFieldObject.put("start", start);
        insertFieldObject.put("type", field.getType());
        insertFieldObject.put("representation", field.getRepresentation());
        if(target!=null) {
            insertFieldObject.put("target", target);
        }
        operationQueue.put(insertFieldObject);
    }

    public void addInsertHardBreakOperation(List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertHardBreakObject = new JSONObject(3);
        insertHardBreakObject.put("name", "insertHardBreak");
        insertHardBreakObject.put("start", start);
        if(target!=null) {
            insertHardBreakObject.put("target", target);
        }
        operationQueue.put(insertHardBreakObject);
    }

    public void createTextAttributesOperation(TextSpan_Base textSpan_Base, List<Integer> parentPosition, String target)
        throws JSONException {

        final OpAttrs attrs = new OpAttrs();
        textSpan_Base.createAttrs(opsDoc, attrs);
        if(!attrs.isEmpty()) {
            final JSONObject jsonSetAttributesOperation = new JSONObject(5);
            jsonSetAttributesOperation.put("name", "setAttributes");

            final List<Integer> start = new ArrayList<Integer>(parentPosition);
            start.add(textSpan_Base.getComponentNumber());
            jsonSetAttributesOperation.put("start", start);

            int startComponent = textSpan_Base.getComponentNumber();
            int endComponent   = textSpan_Base.getNextComponentNumber()-1;
            if(endComponent!=startComponent) {
                final List<Integer> end = new ArrayList<Integer>(parentPosition);
                end.add(endComponent);
                jsonSetAttributesOperation.put("end", end);
            }
            if(target!=null) {
                jsonSetAttributesOperation.put("target", target);
            }
            jsonSetAttributesOperation.put("attrs", attrs);
            operationQueue.put(jsonSetAttributesOperation);
        }
    }

    public void createMasterStyles(SlideComponent slideComponent, OpAttrs attrs) {
        final StyleBase graphicStyle = styleManager.getStyle("standard", "graphic", false);
        if(graphicStyle!=null) {
            styleManager.createPresentationAttributes(graphicStyle, "other", 10, false, attrs);
        }
        Component component = slideComponent.getNextChildComponent(null, null);
        while(component!=null) {
            if(component.getObject() instanceof IAttributesAccessor) {
                final String presentationClass = ((IAttributesAccessor)component.getObject()).getAttributes().getValue("presentation:class");
                if(presentationClass!=null) {
                    final String presentationStyleName = ((IAttributesAccessor)component.getObject()).getAttributes().getValue("presentation:style-name");
                    if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
                        final StyleBase styleBase = styleManager.getStyle(presentationStyleName, "presentation", false);
                        if(styleBase instanceof StylePresentation) {
                            final StylePresentation stylePresentation = (StylePresentation)styleBase;
                            if(presentationClass.equals("title")) {
                                styleManager.createPresentationAttributes(stylePresentation, "title", 1, false, attrs);
                            }
                            else if(presentationClass.equals("outline")) {
                                styleManager.createPresentationAttributes(stylePresentation, "body", 9, false, attrs);
                            }
                        }
                    }
                }
            }
            component = component.getNextComponent();
        }
    }
}
