/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odp.dom;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfPresentationDocument;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.FilterException.ErrorCode;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.IParagraph;
import com.openexchange.office.filter.odf.ITable;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.Settings;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.Component.Type;
import com.openexchange.office.filter.odf.components.TextFieldComponent;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.styles.DocumentStyles;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StylePresentation;
import com.openexchange.office.filter.odf.styles.TextListStyle;
import com.openexchange.office.filter.odp.dom.components.ParagraphComponent;
import com.openexchange.office.filter.odp.dom.components.RowComponent;
import com.openexchange.office.filter.odp.dom.components.ShapeGroupComponent;
import com.openexchange.office.filter.odp.dom.components.SlideComponent;
import com.openexchange.office.filter.odt.dom.Paragraph;
import com.openexchange.office.filter.odt.dom.TextField;
import com.openexchange.office.filter.odt.dom.TextList;
import com.openexchange.office.filter.odt.dom.TextListItem;

public class JsonOperationConsumer {

	private final OdfOperationDoc opsDoc;
    private final OdfPresentationDocument doc;
    private final DocumentStyles styles;
    private final PresentationContent content;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationConsumer(OdfOperationDoc opsDoc)
		throws SAXException {

		this.opsDoc = opsDoc;
		doc = (OdfPresentationDocument)opsDoc.getDocument();
		styles = (DocumentStyles)doc.getStylesDom();
		content = (PresentationContent)doc.getContentDom();
		styleManager = doc.getStyleManager();
	    settings = doc.getSettingsDom();
	}

    public int applyOperations(JSONArray operations)
    	throws Exception {

    	for (int i = 0; i < operations.length(); i++) {
        	OdfOperationDoc.setSuccessfulAppliedOperations(i);
            final JSONObject op = (JSONObject) operations.get(i);
            final String target = op.optString("target", "");
            final String opName = op.getString("name");
            switch(opName)
            {
                case "insertParagraph": {
                    insertParagraph(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                    break;
                }
                case "splitParagraph": {
                    splitParagraph(op.getJSONArray("start"), target);
                    break;
                }
                case "mergeParagraph": {
                    mergeParagraph(op.getJSONArray("start"), target);
                    break;
                }
            	case "insertText" : {
                    insertText(op.getJSONArray("start"), target, op.optJSONObject("attrs"), op.getString("text").replaceAll("\\p{Cc}", " "));
                    break;
            	}
                case "delete" : {
                    delete(op.getJSONArray("start"), target, op.optJSONArray("end"));
                    break;
                }
                case "setAttributes": {
                    final JSONArray start = op.getJSONArray("start");
                    final JSONObject attrs = op.getJSONObject("attrs");
                    if(start.length()>=3&&start.length()<=3) {
                        if(target!=null&&!target.isEmpty()) {
                            final INodeAccessor nodeAccessor = doc.getTargetNodes().get(target);
                            if(nodeAccessor!=null) {
                                final Object masterPage = nodeAccessor.getContent().get(start.getInt(0));
                                if(masterPage instanceof MasterPage) {
                                    final Object drawFrame = ((MasterPage)masterPage).getContent().get(start.getInt(1));
                                    if(drawFrame instanceof DrawFrame) {
                                        final String presentationClass = ((DrawFrame)drawFrame).getAttributes().getValue("presentation:class");
                                        if(presentationClass!=null) {
                                            if(presentationClass.equals("title")||presentationClass.equals("outline")) {
                                                final Paragraph p = (Paragraph)((DrawFrame)drawFrame).getContent().get(start.getInt(2));
                                                final int listLevel = p.getListLevel() + 1;
                                                String presentationStyleName = ((DrawFrame)drawFrame).getAttributes().getValue("presentation:style-name");
                                                if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
                                                    StylePresentation presentationStyle = (StylePresentation)styleManager.getStyle(presentationStyleName, "presentation", false);
                                                    if(presentationStyle!=null) {
                                                        final TextListStyle listStyle = presentationStyle.getGraphicProperties().getTextListStyle();
                                                        String nextPresentationStyleName = null;
                                                        if(presentationClass.equals("outline")&&presentationStyleName.charAt(presentationStyleName.length()-1)=='1') {
                                                            if(listLevel>1&&listLevel<=9) {
                                                                presentationStyleName = presentationStyleName.substring(0, presentationStyleName.length()-1) + Integer.toString(listLevel);
                                                                presentationStyle = (StylePresentation)styleManager.getStyle(presentationStyleName, "presentation", false);
                                                            }
                                                            if(listLevel<9) {
                                                                nextPresentationStyleName = presentationStyleName.substring(0, presentationStyleName.length()-1) + Integer.toString(listLevel+1);
                                                            }
                                                        }

                                                        final JSONObject paraAttrs = attrs.optJSONObject("paragraph");
                                                        final JSONObject charAttrs = attrs.optJSONObject("character");

                                                        if(nextPresentationStyleName!=null) {
                                                            final StylePresentation nextPresentationStyle = (StylePresentation)styleManager.getStyle(nextPresentationStyleName, "presentation", false);
                                                            if(nextPresentationStyle!=null) {
                                                                // fixing style that is child of current level
                                                                final OpAttrs parentListLevelAttrs = new OpAttrs();
                                                                styleManager.createPresentationAttributes(presentationStyle, "outline", listLevel, false, parentListLevelAttrs);
                                                                final OpAttrs childAttrs = new OpAttrs();
                                                                nextPresentationStyle.createAttrs(styleManager, childAttrs);
    
                                                                final JSONObject newChildAttrs = new JSONObject();
                                                                if(paraAttrs!=null&&!paraAttrs.isEmpty()) {
                                                                    final HashSet<String> attrsThatNeedsDefault = new HashSet<String>();
                                                                    this.mergeChildOutlineAttributes(attrsThatNeedsDefault, parentListLevelAttrs, childAttrs, newChildAttrs, listLevel, paraAttrs, "paragraph");
                                                                    if(!attrsThatNeedsDefault.isEmpty()) {
                                                                        nextPresentationStyle.getParagraphProperties().applyHardDefaultAttributes(attrsThatNeedsDefault);
                                                                    }
                                                                }
                                                                if(charAttrs!=null&&!charAttrs.isEmpty()) {
                                                                    final HashSet<String> attrsThatNeedsDefault = new HashSet<String>();
                                                                    this.mergeChildOutlineAttributes(attrsThatNeedsDefault, parentListLevelAttrs, childAttrs, newChildAttrs, listLevel, charAttrs, "character");
                                                                    if(!attrsThatNeedsDefault.isEmpty()) {
                                                                        nextPresentationStyle.getTextProperties().applyHardDefaultAttributes(attrsThatNeedsDefault);
                                                                    }
                                                                }
                                                                if(!newChildAttrs.isEmpty()) {
                                                                    nextPresentationStyle.applyAttrs(styleManager, newChildAttrs);
                                                                }
                                                            }
                                                        }
                                                        presentationStyle.applyAttrs(styleManager, attrs);
                                                        if(listStyle!=null) {
                                                            listStyle.applyPresentationListStyleAttributes(styleManager, attrs, listLevel);
                                                        }
                                                    }
                                                }
                                                break;
                                            }   
                                        }
                                    }
                                }
                            }
                        }
                    }
                    setAttributes(attrs, target, start, op.optJSONArray("end"));
                    break;
                }
                case "insertSlide": {
                    insertSlide(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                    break;
                }
                case "insertMasterSlide": {
                    insertMasterSlide(op.getString("id"), op.optInt("start", -1), op.optJSONObject("attrs"));
                    break;
                }
                case "insertDrawing": {
                    insertDrawing(op.getJSONArray("start"), target, op.getString("type"), op.optJSONObject("attrs"));
                    break;
                }
                case "setDocumentAttributes": {
                    setDocumentAttributes(op.getJSONObject("attrs"));
                    break;
                }
                case "insertHardBreak": {
                    insertHardBreak(op.getJSONArray("start"), target, /* op.optString("type"), */ op.optJSONObject("attrs"));
                    break;
                }
                case "moveSlide": {
                    moveSlide(op.getJSONArray("start"), op.getJSONArray("end"));
                    break;
                }
                case "insertTab": {
                    insertTab(op.getJSONArray("start"), target, op.optJSONObject("attrs"));
                    break;
                }
                case "insertField": {
                    insertField(op.getJSONArray("start"), target, op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
                    break;
                }
                case "updateField": {
                    updateField(op.getJSONArray("start"), target, op.optString("type"), op.optString("representation"), op.optJSONObject("attrs"));
                    break;
                }
                case "group": {
                    group(op.getJSONArray("start"), target, op.getJSONArray("drawings"), op.getJSONObject("attrs"), op.optJSONArray("childAttrs"));
                    break;
                }
                case "ungroup": {
                    ungroup(op.getJSONArray("start"), target, op.optJSONArray("drawings"));
                    break;
                }
                case "move": {
                    move(op.getJSONArray("start"), target, op.optJSONArray("end"), op.getJSONArray("to"));
                    break;
                }
                case "insertRows": {
                    insertRows(op.getJSONArray("start"), target, op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
                    break;
                }
                case "insertCells": {
                    insertCells(op.getJSONArray("start"), target, op.optInt("count", 1), op.optJSONObject("attrs"));
                    break;
                }
                case "insertColumn": {
                    insertColumn(op.getJSONArray("start"), target, op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
                    break;
                }
                case "deleteColumns": {
                    deleteColumns(op.getJSONArray("start"), target, op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
                    break;
                }
                case "changeMaster": {
                    changeMaster(op.getJSONArray("start"), target);
                    break;
                }
                case "insertStyleSheet": {
                    changeOrInsertStyleSheet(true, op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent", null), op.optBoolean("hidden", false), op.optInt("uiPriority", 0), op.optBoolean("default", false));
                    break;
                }
                case "changeStyleSheet": {
                    changeOrInsertStyleSheet(false, op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.optJSONObject("attrs"), op.optString("parent", null), op.optBoolean("hidden", false), op.optInt("uiPriority", 0), op.optBoolean("default", false));
                    break;
                }
                case "deleteStylesheet": {
                    deleteStyleSheet(op.getString("type"), op.getString("styleId"));
                    break;
                }
                case "noOp": {
                    break;
                }
                case "createError": {
                    throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                }
            }
        }
        OdfOperationDoc.setSuccessfulAppliedOperations(operations.length());
        return 1;
    }

    private void mergeChildOutlineAttributes(HashSet<String> attrsNeedsDefault, OpAttrs parentListLevelAttrs, OpAttrs _childAttrs, JSONObject _newChildAttrs, int listLevel, JSONObject attrs,  String type)
        throws JSONException {

        final OpAttrs parentAttrs = parentListLevelAttrs.getMap("listStyles", true).getMap("outline",  true).getMap("l" + Integer.toString(listLevel), true).getMap(type, true);
        final OpAttrs childAttrs = _childAttrs.getMap(type, true);
        final JSONObject newChildAttrs = new JSONObject();

        final Iterator<String> paraKeyIter = attrs.keys();
        while(paraKeyIter.hasNext()) {
            final String paraKey = paraKeyIter.next();
            if(!childAttrs.containsKey(paraKey)) {
                if(parentAttrs.containsKey(paraKey)) {
                    final Object o = parentAttrs.get(paraKey);
                    if(o instanceof Map) {
                        final HashMap<String, Object> destAttr = new HashMap<String, Object>();
                        StyleManager.deepCopy((Map)o, destAttr);
                        newChildAttrs.put(paraKey, destAttr);
                    }
                    else {
                        newChildAttrs.put(paraKey, o);
                    }
                }
                else {
                    attrsNeedsDefault.add(paraKey);
                }
            }
        }
        if(!newChildAttrs.isEmpty()) {
            _newChildAttrs.put(type, newChildAttrs);
        }
    }

    public void insertParagraph(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        boolean createDefaultTextListItem = true;
        if(attrs!=null) {
            final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");
            if(paragraphAttrs!=null) {
                final JSONObject bullet = paragraphAttrs.optJSONObject("bullet");
                if(bullet!=null) {
                    createDefaultTextListItem = !"none".equals(bullet.opt("type"));
                }
            }
        }

        if(createDefaultTextListItem) {
            final ParagraphComponent paragraphComponent = (ParagraphComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-1)
                .insertChildComponent(opsDoc, start.getInt(start.length()-1), null, Component.Type.PARAGRAPH);

            // now apply the correct listItem
            final SlideComponent slideComponent = (SlideComponent)Component.getComponent(content.getRootComponent(target), start, 1);
            final TextListStyle defaultTextListStyle = doc.getDefaultTextListStyle(paragraphComponent.getPresentationType(), slideComponent.getPage(), false);
            final TextList textList = new TextList(null);
            textList.setStyleName(defaultTextListStyle.getName());
            paragraphComponent.getParagraph().setTextListItem(new TextListItem(textList));

            if(attrs!=null) {
                paragraphComponent.applyAttrsFromJSON(opsDoc, attrs);
            }
        }
        else {  // no list item needed ... insert paragraph child in one step
            Component.getComponent(content.getRootComponent(target), start, start.length()-1)
                .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
        }
    }

    public void splitParagraph(JSONArray start, String target)
        throws JSONException {

        ((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .splitParagraph(opsDoc, start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start, String target) {

        ((IParagraph)Component.getComponent(content.getRootComponent(target), start, start.length()))
            .mergeParagraph();
    }

    public void insertText(JSONArray start, String target, JSONObject attrs, String text)
        throws IndexOutOfBoundsException, JSONException, SAXException {

        if(text.length()>0) {
            final Component component = Component.getComponent(content.getRootComponent(target), start, start.length()-1);
            ((IParagraph)component).insertText(opsDoc, start.getInt(start.length()-1), text, attrs);
        }
    }

    public void insertTab(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.TAB);
    }

    public void insertHardBreak(JSONArray start, String target, JSONObject attrs)
        throws JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.HARDBREAK);
    }

    public void delete(JSONArray start, String target, JSONArray end)
        throws JSONException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        final Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
        component.splitStart(startComponent);
        component.delete(opsDoc, (endComponent-startComponent)+1);
    }

    public void setAttributes(JSONObject attrs, String target, JSONArray start, JSONArray end)
        throws JSONException, SAXException {

        if(attrs==null) {
            return;
        }
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(content.getRootComponent(target), start, start.length());
        component.splitStart(startIndex);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
            if(component.getNextComponentNumber()>=endIndex+1) {
                component.splitEnd(endIndex);
            }
            component.applyAttrsFromJSON(opsDoc, attrs);
            component = component.getNextComponent();
        }
    }

    public Component insertDrawing(JSONArray start, String target, String type, JSONObject attrs)
        throws UnsupportedOperationException, JSONException, SAXException {

        Component.Type childComponentType = Component.Type.AC_IMAGE;
        if(type.equals("shape")||type.equals("connector")) {
            if(attrs==null||!attrs.has("geometry")) {
                childComponentType = Component.Type.AC_FRAME;
            }
            else {
                childComponentType = Component.Type.AC_SHAPE;
            }
        }
        else if(type.equals("group")) {
            childComponentType = Component.Type.AC_GROUP;
        }
        else if(type.equals("image")) {
            childComponentType = Component.Type.AC_IMAGE;
        }
        else if(type.equals("table")) {
            childComponentType = Component.Type.TABLE;
        }
        return Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, childComponentType);
    }

    public void insertTable(JSONArray start, String target, JSONObject attrs)
        throws UnsupportedOperationException, JSONException, SAXException {

        Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Component.Type.TABLE);
    }

    public void splitTable(JSONArray start, String target)
        throws JSONException, UnsupportedOperationException, SAXException {

        final ITable splitTableComponent = (ITable)Component.getComponent(content.getRootComponent(target), start, start.length()-2)
            .insertChildComponent(opsDoc, start.getInt(start.length()-2) + 1, null, Component.Type.TABLE);

        ((ITable)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .splitTable(opsDoc, start.getInt(start.length()-1), splitTableComponent);
    }

    public void mergeTable(JSONArray start, String target) {
        ((ITable)Component.getComponent(content.getRootComponent(target), start, start.length())).mergeTable(opsDoc);
    }
    
    public void insertRows(JSONArray start, String target, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
        throws JSONException, SAXException {

        ((ITable)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .insertRows(opsDoc, start.getInt(start.length()-1), count, insertDefaultCells, referenceRow, attrs);
    }

    public void insertCells(JSONArray start, String target, int count, JSONObject attrs)
        throws JSONException, SAXException {

        ((RowComponent)Component.getComponent(content.getRootComponent(target), start, start.length()-1))
            .insertCells(opsDoc, start.getInt(start.length()-1), count, attrs);
    }

    public void insertColumn(JSONArray start, String target, JSONArray tableGrid, int gridPosition, String insertMode)
        throws JSONException {

        ((ITable)Component.getComponent(content.getRootComponent(target), start, start.length()))
            .insertColumn(opsDoc, tableGrid, gridPosition, insertMode);
    }

    public void deleteColumns(JSONArray position, String target, int gridStart, int gridEnd) {

        ((ITable)Component.getComponent(content.getRootComponent(target), position, position.length()))
            .deleteColumns(opsDoc, gridStart, gridEnd);
    }

    public void insertSlide(JSONArray position, String target, JSONObject attrs) throws JSONException, SAXException {
    	final DrawingPage drawingPage = new DrawingPage(new AttributesImpl());
    	drawingPage.setMasterPageName(target);
    	final int slideNumber = position.getInt(0);
    	content.getPresentation().getContent().add(slideNumber, drawingPage);
    	if(attrs!=null&&!attrs.isEmpty()) {
            final DLNode<Object> drawingPageNode = new DLNode<Object>(drawingPage);
    	    final SlideComponent slideComponent = new SlideComponent(content.getRootComponent(target), drawingPageNode, slideNumber, true);
    	    slideComponent.applyAttrsFromJSON(opsDoc, attrs);
    	}
    }

    public void insertMasterSlide(String id, int slideNumber, JSONObject attrs) throws JSONException, SAXException {
        final MasterPage masterPage = new MasterPage(new AttributesImpl());
        masterPage.setName(id);
        final DLNode<Object> masterPageNode = new DLNode<Object>(masterPage);
        final DLList<Object> masterStylesContent = styleManager.getMasterStyles().getContent();
        final Iterator<DLNode<Object>> masterStylesContentIter = masterStylesContent.getNodeIterator();
        int f = 0;
        DLNode<Object> refNode = null;
        while(masterStylesContentIter.hasNext()) {
            final DLNode<Object> styleContentEntry = masterStylesContentIter.next();
            if(styleContentEntry.getObject() instanceof MasterPage) {
                if(f==slideNumber) {
                    refNode = styleContentEntry;
                    break;
                }
                f++;
            }
        }
        masterStylesContent.addNode(refNode, masterPageNode, true);
        styles.getTargetNodes().put(id, new MasterPageWrapper(id, masterPage));
        if(attrs!=null&&!attrs.isEmpty()) {
            final SlideComponent slideComponent = new SlideComponent(content.getRootComponent(id), masterPageNode, slideNumber, false);
            slideComponent.applyAttrsFromJSON(opsDoc, attrs);
        }
        styleManager.insertStyleSheet("presentation", id + "-background", null, attrs, null, false, false);
    }

    public void moveSlide(JSONArray start, JSONArray end) throws JSONException {
        final DLList<Object> slideList = content.getPresentation().getContent();
        final Object o = slideList.remove(start.getInt(0));
        slideList.add(end.getInt(0), o);
    }

    public void setDocumentAttributes(JSONObject attrs)
        throws JSONException {

        final Iterator<Object> masterStyleContentIter = styleManager.getMasterStyles().getContent().iterator();
        MasterPage masterPage = null;
        while(masterStyleContentIter.hasNext()) {
            final Object o = masterStyleContentIter.next();
            if(o instanceof MasterPage) {
                masterPage = (MasterPage)o;
                break;
            }
        }
        if(masterPage!=null) {
            final StyleBase pageLayout = styleManager.getStyle(masterPage.getPageLayoutName(), "page-layout", false);
            if(pageLayout!=null) {
                pageLayout.applyAttrs(styleManager, attrs);
            }
        }
    }

    public void insertField(JSONArray start, String target, String type, String representation, JSONObject attrs)
        throws JSONException, UnsupportedOperationException, SAXException {

        final TextField textField = (TextField)Component.getComponent(content.getRootComponent(target), start, start.length()-1)
            .insertChildComponent(opsDoc, start.getInt(start.length()-1), attrs, Type.FIELD).getObject();

        textField.setType(type);
        textField.setRepresentation(representation);
    }

    public void updateField(JSONArray start, String target, String type, String representation, JSONObject attrs)
        throws UnsupportedOperationException {

        final TextFieldComponent tfComponent = (TextFieldComponent)Component.getComponent(content.getRootComponent(target), start, start.length());
        if(type!=null) {
            ((TextField)tfComponent.getObject()).setType(type);
        }
        if(representation!=null) {
            ((TextField)tfComponent.getObject()).setRepresentation(representation);
        }
    }

    public void group(JSONArray start, String target, JSONArray drawings, JSONObject attrs, JSONArray childAttrs)
        throws JSONException, SAXException {

        final List<Object> drawingChildObjects = new ArrayList<Object>();
        for(int i = drawings.length()-1; i >= 0; i--) {
            final Component child = Component.getComponent(content.getRootComponent(target), start, start.length()-1).getChildComponent(drawings.getInt(i));
            drawingChildObjects.add(0, child.getObject());
            child.splitStart(child.getComponentNumber());
            child.delete(opsDoc, 1);
        }
        final Component groupComponent = insertDrawing(start, target, "group", attrs);
        final GroupShape groupShape = (GroupShape)groupComponent.getObject();
        groupShape.getContent().addAll(drawingChildObjects);
        if(attrs!=null) {
            // oha
        }
    }

    public void ungroup(JSONArray start, String target, JSONArray drawings)
        throws JSONException {

        final int groupPosition = start.getInt(start.length()-1);
        final Component parentComponent = Component.getComponent(content.getRootComponent(target), start, start.length()-1);
        final DLList<Object> parentContent = ((INodeAccessor)parentComponent.getObject()).getContent();
        final ShapeGroupComponent groupComponent = (ShapeGroupComponent)parentComponent.getChildComponent(groupPosition);
        final GroupShape group = (GroupShape)groupComponent.getObject();
        group.finalizeGroup();
        final DLList<Object> groupContent = group.getContent();

        groupComponent.splitStart(groupComponent.getComponentNumber());
        groupComponent.delete(opsDoc, 1);

        Component referenceComponent = parentComponent.getChildComponent(groupPosition);
        DLNode<Object> referenceNode = referenceComponent!=null ? referenceComponent.getNode() : null;

        for(int i = 0; i < groupContent.size(); i++) {
            if(drawings!=null&&i<drawings.length()) {
                referenceComponent = parentComponent.getChildComponent(drawings.getInt(i));
                referenceNode = referenceComponent!=null ? referenceComponent.getNode() : null;
            }
            final Object newChild = groupContent.get(i);
            final DLNode<Object> newChildNode = new DLNode<Object>(newChild);
            parentContent.addNode(referenceNode, newChildNode, true);
        }
    }

    public void move(JSONArray start, String target, JSONArray end, JSONArray to)
        throws JSONException {

        Component.move(content.getRootComponent(target), start, end, to);
    }

    public void changeMaster(JSONArray start, String target) throws JSONException {
        ((DrawingPage)content.getPresentation().getContent().get(start.getInt(0))).setMasterPageName(target);
    }

    public void changeOrInsertStyleSheet(boolean insert, String type, String styleId, String styleName, JSONObject attrs, String parent, boolean hidden,  @SuppressWarnings("unused") int uiPriority, boolean defaultStyle)
        throws JSONException {

        if(type.equals("table")) {
            if(insert) {
                styleManager.insertStyleSheet("table-template", styleId, styleName, attrs, parent, defaultStyle, hidden);
            }
            else {
                final StyleBase templateStyle = styleManager.getStyle(styleId, "table-template", false);
                if(templateStyle!=null) {
                    templateStyle.applyAttrs(styleManager, attrs);
                }
            }
        }
    }

    public void deleteStyleSheet(String type, String id) {
        if(type.equals("table")) {
            styleManager.removeStyle("table-template", id, false, false);
        }
    }
}
