/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.styles;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.draw.DrawImage;

final public class DrawFillImage extends StyleBase {

    public DrawFillImage(String name) {
        super(null, name, false, false);
    }

    public DrawFillImage(String name, AttributesImpl attributesImpl) {
        super(name, attributesImpl, false, false, false);
    }

    @Override
    public String getFamily() {
        return "fill-image";
    }

    @Override
    public String getQName() {
        return "draw:fill-image";
    }

    @Override
    public String getLocalName() {
        return "fill-image";
    }

    @Override
    public String getNamespace() {
        return Namespaces.DRAW;
    }

    @Override
    protected void writeNameAttribute(SerializationHandler output) throws SAXException {
        output.addAttribute(Namespaces.DRAW, "name", "draw:name", "", getName());
    }

    @Override
    public void writeObject(SerializationHandler output)
        throws SAXException {

        SaxContextHandler.startElement(output, getNamespace(), getLocalName(), getQName());
        writeAttributes(output);
        SaxContextHandler.endElement(output, getNamespace(), getLocalName(), getQName());
    }

    @Override
    public void mergeAttrs(StyleBase style) {
        // TODO Auto-generated method stub
    }

    @Override
    public void applyAttrs(StyleManager styleManager, JSONObject bitmapAttrs) {
        final AttributesImpl drawFillImageAttributes = getAttributes();
        final String imageUrl = bitmapAttrs.optString("imageUrl", null);
        if(imageUrl!=null) {
            DrawImage.transferResource(imageUrl, styleManager);
            drawFillImageAttributes.setValue(Namespaces.XLINK, "href", "xlink:href", imageUrl);
            drawFillImageAttributes.setValue(Namespaces.XLINK, "type", "xlink:type", "simple");
            drawFillImageAttributes.setValue(Namespaces.XLINK, "show", "xlink:show", "embed");
            drawFillImageAttributes.setValue(Namespaces.XLINK, "actuate", "xlink:actuate", "onLoad");
        }
    }

    @Override
    public void createAttrs(StyleManager styleManager, OpAttrs attrs) {
        final String hRef = attributes.getValue("xlink:href");
        if(hRef!=null) {
            final OpAttrs fillProps = attrs.getMap("fill", true);
            final OpAttrs bitmapProps = fillProps.getMap("bitmap", true);
            bitmapProps.put("imageUrl", hRef);
        }
        final String id = attributes.getValue("xml:id");
        if(id!=null) {
            attrs.getMap("drawing", true).put("imageXmlId", id);
        }
    }

    @Override
    protected int _hashCode() {
        return 0;
    }

    @Override
    protected boolean _equals(StyleBase e) {
        return true;
    }

    @Override
    public DrawFillImage clone() {
        final DrawFillImage clone = (DrawFillImage)_clone();
        return clone;
    }

    public static String getFillImage(StyleManager styleManager, String currentFillImage, JSONObject bitmapAttrs) {
        DrawFillImage drawFillImage = null;
        if(currentFillImage!=null) {
            final StyleBase current = styleManager.getStyle(currentFillImage, "fill-image", false);
            if(current!=null) {
                drawFillImage = (DrawFillImage)current.clone();
            }
        }
        if(drawFillImage==null) {
            drawFillImage = new DrawFillImage(styleManager.getUniqueStyleName("fill-image", false));
        }
        drawFillImage.applyAttrs(styleManager, bitmapAttrs);
        final String existingStyleId = styleManager.getExistingStyleIdForStyleBase(drawFillImage);
        if(existingStyleId!=null) {
            return existingStyleId;
        }
        styleManager.addStyle(drawFillImage);
        return drawFillImage.getName();
    }

    public static void applyTilingAttrs(JSONObject sourceBitmapAttributes, AttributesImpl attrs) {
        final JSONObject tilingAttrs = sourceBitmapAttributes.optJSONObject("tiling");
        final JSONObject stretchingAttrs = sourceBitmapAttributes.optJSONObject("stretching");
        final JSONObject stretchingAlignedAttrs = sourceBitmapAttributes.optJSONObject("stretchingAligned");
        final String repeat;
        if(tilingAttrs!=null) {
            repeat = "repeat";
            final String ref = DrawFillImage.getRefPoint(tilingAttrs.optString("rectAlignment", null));
            if(ref!=null) {
                attrs.setValue(Namespaces.DRAW, "fill-image-ref-point", "draw:fill-image-ref-point", ref);
            }
        }
        else if(stretchingAttrs!=null) {
            int left = 0;
            int top = 0;
            int right = 0;
            int bottom = 0;
            Integer fillImageWidth = null;
            Integer fillImageHeight = null;
            final Object l = stretchingAttrs.opt("left");
            if(l instanceof Number) {
                left = ((Number)l).intValue();
            }
            final Object t = stretchingAttrs.opt("top");
            if(t instanceof Number) {
                top = ((Number)t).intValue();
            }
            final Object r = stretchingAttrs.opt("right");
            if(r instanceof Number) {
                right = ((Number)r).intValue();
            }
            final Object b = stretchingAttrs.opt("bottom");
            if(b instanceof Number) {
                bottom = ((Number)b).intValue();
            }
            if(left==0&&top==0&&right==0&&bottom==0) {
                repeat = "stretch";
            }
            else {
                repeat = "no-repeat";
                if(left>0&&right>0) {
                    if((left+right)<100) {
                        fillImageWidth = 100  - (left + right);
                    }
                }
                if(top>0&&bottom>0)  {
                    if((top+bottom)<100)
                        fillImageHeight = 100 - (top -bottom);
                }
            }
            if(fillImageWidth!=null) {
                attrs.setValue(Namespaces.DRAW, "fill-image-width", "draw:fill-image-width", fillImageWidth.toString() + "%");
            }
            if(fillImageHeight!=null) {
                attrs.setValue(Namespaces.DRAW, "fill-image-height", "draw:fill-image-height", fillImageHeight.toString() + "%");
            }
        }
        else if(stretchingAlignedAttrs!=null) {
            repeat = "no-repeat";
            final String ref = DrawFillImage.getRefPoint(stretchingAlignedAttrs.optString("rectAlignment", null));
            if(ref!=null) {
                attrs.setValue(Namespaces.DRAW, "fill-image-ref-point", "draw:fill-image-ref-point", ref);
            }
            final Object width = stretchingAlignedAttrs.opt("width");
            if(width instanceof Number) {
                attrs.setValue(Namespaces.DRAW, "fill-image-width", "draw:fill-image-width", ((Number)width).doubleValue() / 100.0 + "mm");
            }
            final Object scaleWidth = stretchingAlignedAttrs.opt("scaleX");
            if(scaleWidth instanceof Number) {
                attrs.setValue(Namespaces.DRAW, "fill-image-width", "draw:fill-image-width", ((Number)scaleWidth).doubleValue() + "%");
            }
            final Object height = stretchingAlignedAttrs.opt("height");
            if(height instanceof Number) {
                attrs.setValue(Namespaces.DRAW, "fill-image-height", "draw:fill-image-height", ((Number)height).doubleValue() / 100.0 + "mm");
            }
            final Object scaleHeight = stretchingAlignedAttrs.opt("scaleY");
            if(scaleHeight instanceof Number) {
                attrs.setValue(Namespaces.DRAW, "fill-image-height", "draw:fill-image-height", ((Number)scaleHeight).doubleValue() + "%");
            }
        }
        else {
            repeat = "no-repeat";
        }
        attrs.setValue(Namespaces.STYLE, "repeat", "style:repeat", repeat);
    }

    public static void createTilingAttrs(StyleManager styleManager, String imageUrl, String repeat, AttributesImpl sourceAttributes, OpAttrs fillAttrs) {
        final String imageWidth = sourceAttributes.getValue("draw:fill-image-width");
        Integer width = null;
        Integer height = null;
        boolean widthScale = false;
        boolean heightScale = false;
        if(imageWidth!=null&&!imageWidth.isEmpty()) {
            if(imageWidth.charAt(imageWidth.length()-1)=='%') {
                width = AttributesImpl.getPercentage(imageWidth, null);
                widthScale = true;
            }
            else {
                width = AttributesImpl.normalizeLength(imageWidth, false);
            }
        }
        final String imageHeight = sourceAttributes.getValue("draw:fill-image-height");
        if(imageHeight!=null&&!imageHeight.isEmpty()) {
            if(imageHeight.charAt(imageHeight.length()-1)=='%') {
                height = AttributesImpl.getPercentage(imageHeight, null);
                heightScale = true;
            }
            else {
                height = AttributesImpl.normalizeLength(imageHeight, false);
            }
        }
        if("repeat".equals(repeat)) {
            final OpAttrs tilingAttrs = fillAttrs.getMap("bitmap", true).getMap("tiling", true);
            tilingAttrs.put("rectAlignment", getRectAlignment(sourceAttributes.getValue("draw:fill-image-ref-point")));

            // pixel count of the graphic needed
            Double scaleX = null;
            Double scaleY = null;
            if((width!=null&&!widthScale)||(height!=null&&!heightScale)) {
                final Pair<Integer, Integer> pixelSize = styleManager.getPackage().getPixelSize(imageUrl);
                if(pixelSize!=null) {
                    if(width!=null&&!widthScale) {
                        scaleX = (72.0 / (pixelSize.getLeft() / (width / 2540.0))) * 100.0;
                    }
                    if(height!=null&&!heightScale) {
                        scaleY = (72.0 / (pixelSize.getRight() / (height / 2540.0))) * 100.0;
                    }
                }
            }
            if(width==null) {
                scaleX = (100.0 * 96) / 72.0;
            }
            else if(widthScale) {
                scaleX = (width * 96.0) / 72.0;
            }
            if(height==null) {
                scaleY = (100.0 * 96) / 72.0;
            }
            else if(heightScale) {
                scaleY = (height * 96.0) / 72.0;
            }
            if(scaleX!=null) {
                tilingAttrs.put("stretchX", scaleX);
            }
            if(scaleY!=null) {
                tilingAttrs.put("stretchY", scaleY);
            }
        }
        else if ("no-repeat".equals(repeat)) {
            final OpAttrs stretchingAttrs = fillAttrs.getMap("bitmap", true).getMap("stretchingAligned", true);
            stretchingAttrs.put("rectAlignment", getRectAlignment(sourceAttributes.getValue("draw:fill-image-ref-point")));
            if(width!=null) {
                stretchingAttrs.put(widthScale ? "scaleX" : "width", width);
            }
            if(height!=null) {
                stretchingAttrs.put(heightScale ? "scaleY" : "height", height);
            }
        }
        else if("stretch".equals(repeat)) {
            final OpAttrs stretchingAttrs = new OpAttrs(4);
            stretchingAttrs.put("left", 0);
            stretchingAttrs.put("right", 0);
            stretchingAttrs.put("top", 0);
            stretchingAttrs.put("bottom", 0);
            fillAttrs.getMap("bitmap", true).put("stretching", stretchingAttrs);
        }
    }

    private static String getRefPoint(String rectAlignment) {
        String ref = null;
        if(rectAlignment!=null) {
            switch(rectAlignment) {
                case "bottomLeft": ref = "bottom-left"; break;
                case "bottomRight": ref = "bottom-right"; break;
                case "topLeft": ref = "top-left"; break;
                case "topRight": ref = "top-right"; break;
                case "center": // PASSTROUGH INTENDED !!
                case "left":
                case "right":
                case "top": 
                case "bottom": ref = rectAlignment; break;
                default: ref = "center"; break;
            }
        }
        return ref;
    }

    private static String getRectAlignment(String refPoint) {
        String ref = "center";
        if(refPoint!=null) {
            switch(refPoint) {
                case "bottom-left": ref = "bottomLeft"; break;
                case "bottom-right": ref = "bottomRight"; break;
                case "top-left": ref = "topLeft"; break;
                case "top-right": ref = "topRight"; break;
                case "center":  // PASSTROUGH INTENDED !!
                case "left":
                case "right":
                case "top":
                case "bottom": ref = refPoint; break;
                default: ref = "center"; break;
            }
        }
        return ref;
    }
}
