/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.properties;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import com.openexchange.office.filter.odf.AttributeImpl;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Border;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.styles.StyleManager;

final public class TableCellProperties extends StylePropertiesBase {

    private DLList<IElementWriter> content;

    public TableCellProperties(AttributesImpl attributesImpl) {
        super(attributesImpl);
    }

    @Override
    public DLList<IElementWriter> getContent() {
        if(content==null) {
            content = new DLList<IElementWriter>();
        }
        return content;
    }

    @Override
    public String getQName() {
        return "style:table-cell-properties";
    }

    @Override
    public String getLocalName() {
        return "table-cell-properties";
    }

    @Override
    public String getNamespace() {
        return Namespaces.STYLE;
    }

    @Override
    public void applyAttrs(StyleManager styleManager, JSONObject attrs)
        throws JSONException {

        final JSONObject cellAttributes = attrs.optJSONObject("cell");
        if(cellAttributes!=null) {
            boolean createCellProtectAttribute = false;
            Boolean unlocked = null;
            Boolean hidden = null;
            final Iterator<Entry<String, Object>> characterIter = cellAttributes.entrySet().iterator();
            while(characterIter.hasNext()) {
                final Entry<String, Object> characterEntry = characterIter.next();
                final Object value = characterEntry.getValue();
                switch(characterEntry.getKey()) {
                    case "fillColor": {
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:background-color");
                        }
                        else {
                            attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", PropertyHelper.getColor((JSONObject)value, "transparent"));
                        }
                        break;
                    }
                    case "alignVert": {
                        if(value==JSONObject.NULL) {
                            attributes.remove("style:vertical-align");
                        }
                        else {
                            attributes.setValue(Namespaces.STYLE, "vertical-align", "style:vertical-align", (String)value);
                        }
                        break;
                    }
                    case "paddingBottom": {
                        if (value==JSONObject.NULL) {
                            attributes.remove("fo:padding-bottom");
                        } else {
                            attributes.setValue(Namespaces.FO, "padding-bottom", "fo:padding-bottom", (((Number)value).intValue() / 100) + "mm");
                        }
                        break;
                    }
                    case "paddingLeft": {
                        if (value==JSONObject.NULL) {
                            attributes.remove("fo:padding-left");
                        } else {
                            attributes.setValue(Namespaces.FO, "padding-left", "fo:padding-left", (((Number)value).intValue() / 100) + "mm");
                        }
                        break;
                    }
                    case "paddingRight": {
                        if (value==JSONObject.NULL) {
                            attributes.remove("fo:padding-right");
                        } else {
                            attributes.setValue(Namespaces.FO, "padding-right", "fo:padding-right", (((Number)value).intValue() / 100) + "mm");
                        }
                        break;
                    }
                    case "paddingTop": {
                        if (value==JSONObject.NULL) {
                            attributes.remove("fo:padding-top");
                        } else {
                            attributes.setValue(Namespaces.FO, "padding-top", "fo:padding-top", (((Number)value).intValue() / 100) + "mm");
                        }
                        break;
                    }
                    case "borderLeft": {
                        Border.applyFoBorderToSingleBorder(attributes);
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:border-left");
                        }
                        else {
                            final Border border = new Border();
                            border.applyJsonBorder((JSONObject)value);
                            final String foBorder = border.toString();
                            if(foBorder!=null) {
                                attributes.setValue(Namespaces.FO, "border-left", "fo:border-left", foBorder);
                            }
                        }
                        break;
                    }
                    case "borderTop": {
                        Border.applyFoBorderToSingleBorder(attributes);
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:border-top");
                        }
                        else {
                            final Border border = new Border();
                            border.applyJsonBorder((JSONObject)value);
                            final String foBorder = border.toString();
                            if(foBorder!=null) {
                                attributes.setValue(Namespaces.FO, "border-top", "fo:border-top", foBorder);
                            }
                        }
                        break;
                    }
                    case "borderRight": {
                        Border.applyFoBorderToSingleBorder(attributes);
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:border-right");
                        }
                        else {
                            final Border border = new Border();
                            border.applyJsonBorder((JSONObject)value);
                            final String foBorder = border.toString();
                            if(foBorder!=null) {
                                attributes.setValue(Namespaces.FO, "border-right", "fo:border-right", foBorder);
                            }
                        }
                        break;
                    }
                    case "borderBottom": {
                        Border.applyFoBorderToSingleBorder(attributes);
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:border-bottom");
                        }
                        else {
                            final Border border = new Border();
                            border.applyJsonBorder((JSONObject)value);
                            final String foBorder = border.toString();
                            if(foBorder!=null) {
                                attributes.setValue(Namespaces.FO, "border-bottom", "fo:border-bottom", foBorder);
                            }
                        }
                        break;
                    }
                    case "wrapText": {
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:wrap-option");
                        }
                        else {
                            attributes.setValue(Namespaces.FO, "wrap-option", "fo:wrap-option", ((Boolean)value).booleanValue() ? "wrap" : "no-wrap");
                        }
                        break;
                    }
                    case "unlocked": {
                        if(unlocked==JSONObject.NULL) {
                            unlocked = null;
                        }
                        else if (value instanceof Boolean) {
                            unlocked = (Boolean)value;
                        }
                        createCellProtectAttribute = true;
                        break;
                    }
                    case "hidden": {
                        if(hidden==JSONObject.NULL) {
                            hidden = null;
                        }
                        else if(value instanceof Boolean) {
                            hidden = (Boolean)value;
                        }
                        createCellProtectAttribute = true;                        
                        break;
                    }
                }
            }
            if(createCellProtectAttribute) {
                String newProtectValue = null;
                if(unlocked!=null) {
                    if(unlocked.booleanValue()) {
                        if(hidden!=null&&hidden.booleanValue()) {
                            newProtectValue = "formula-hidden";
                        }
                        else {
                            newProtectValue = "none";
                        }
                    }
                    else {
                        if(hidden!=null&&hidden.booleanValue()) {
                            newProtectValue = "hidden-and-protected";
                        }
                        else {
                            newProtectValue = "protected";
                        }
                    }
                }
                else if(hidden!=null&&hidden.booleanValue()) {
                    newProtectValue = "formula-hidden";
                }
                if(newProtectValue==null) {
                    attributes.remove("style:cell-protect");
                }
                else {
                    attributes.setValue(Namespaces.STYLE, "cell-protect", "style:cell-protect", newProtectValue);
                }
            }
        }
    }

    @Override
    public void createAttrs(StyleManager styleManager, boolean contentAutoStyle, OpAttrs attrs) {
        final Map<String, Object> cellAttrs = attrs.getMap("cell", true);
        final Integer defaultPadding = PropertyHelper.createDefaultPaddingAttrs(cellAttrs, attributes);
        Border.createDefaultBorderMapAttrsFromFoBorder(cellAttrs, defaultPadding, attributes);
        final Iterator<Entry<String, AttributeImpl>> propIter = attributes.getUnmodifiableMap().entrySet().iterator();
        final String styleCellProtect = getAttribute("style:cell-protect");
        if(styleCellProtect!=null&&!styleCellProtect.isEmpty()) {
            final String[] propValues = StringUtils.split(styleCellProtect);
            for(int i=0; i<propValues.length; i++) {
                final String val = propValues[i];
                if(val.equals("none")) {
                    cellAttrs.put("unlocked", Boolean.TRUE);
                }
                else if(val.equals("hidden-and-protected")||val.equals("formula-hidden")) {
                    cellAttrs.put("hidden", Boolean.TRUE);
                }
            }
        }
        while(propIter.hasNext()) {
            final Entry<String, AttributeImpl> propEntry = propIter.next();
            final String propName = propEntry.getKey();
            final String propValue = propEntry.getValue().getValue();
            switch(propName) {
                case "fo:padding-left": {
                    final Integer padding = AttributesImpl.normalizeLength(propValue, true);
                    if(padding!=null) {
                        cellAttrs.put("paddingLeft", padding);
                    }
                    break;
                }
                case "fo:padding-top": {
                    final Integer padding = AttributesImpl.normalizeLength(propValue, true);
                    if(padding!=null) {
                        cellAttrs.put("paddingTop", padding);
                    }
                    break;
                }
                case "fo:padding-right": {
                    final Integer padding = AttributesImpl.normalizeLength(propValue, true);
                    if(padding!=null) {
                        cellAttrs.put("paddingRight", padding);
                    }
                    break;
                }
                case "fo:padding-bottom": {
                    final Integer padding = AttributesImpl.normalizeLength(propValue, true);
                    if(padding!=null) {
                        cellAttrs.put("paddingBottom", padding);
                    }
                    break;
                }
                case "fo:border-left": {
                    final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                        cellAttrs.put("borderLeft", border);
                    }
                    break;
                }
                case "fo:border-top": {
                    final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                        cellAttrs.put("borderTop", border);
                    }
                    break;
                }
                case "fo:border-right": {
                    final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                        cellAttrs.put("borderRight", border);
                    }
                    break;
                }
                case "fo:border-bottom": {
                    final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                        cellAttrs.put("borderBottom", border);
                    }
                    break;
                }
                case "fo:background-color": {
                    final Map<String, String> color = PropertyHelper.createColorMap(propValue);
                    if(color!=null) {
                        cellAttrs.put("fillColor", color);
                    }
                    break;
                }
                case "style:vertical-align": {
                    cellAttrs.put("alignVert", propValue);
                    break;
                }
                case "fo:wrap-option": {
                    cellAttrs.put("wrapText", true);
                    break;
                }
            }
        }
    }

    @Override
    public TableCellProperties clone() {
        return (TableCellProperties)_clone();
    }
}
