/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.properties;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributeImpl;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Border;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.styles.StyleManager;

final public class ParagraphProperties extends StylePropertiesBase {

	private DLList<IElementWriter> content;
	private TabStops tabStops;

	public ParagraphProperties(AttributesImpl attributesImpl) {
		super(attributesImpl);
	}

	@Override
	public ParagraphProperties setAttributes(Attributes attributes) {
		super.setAttributes(attributes);
		return this;
	}

	@Override
	public DLList<IElementWriter> getContent() {
		if(content==null) {
			content = new DLList<IElementWriter>();
		}
		return content;
	}

	public TabStops getTabStops(boolean forceCreate) {
		if(tabStops==null&&forceCreate) {
			tabStops = new TabStops(new AttributesImpl());
		}
		return tabStops;
	}

	public void setTabStops(TabStops tabStops) {
		this.tabStops = tabStops;
	}

	@Override
	public String getQName() {
		return "style:paragraph-properties";
	}

	@Override
	public String getLocalName() {
		return "paragraph-properties";
	}

	@Override
	public String getNamespace() {
		return Namespaces.STYLE;
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs)
		throws JSONException {

		// TODO: addBorderProperties(key, attrs, propertiesElement, null);

		final JSONObject paragraphAttributes = attrs.optJSONObject("paragraph");
		if(paragraphAttributes!=null) {
			final Iterator<Entry<String, Object>> paragraphIter = paragraphAttributes.entrySet().iterator();
			while(paragraphIter.hasNext()) {
				final Entry<String, Object> paragraphEntry = paragraphIter.next();
				final Object value = paragraphEntry.getValue();
				switch(paragraphEntry.getKey()) {
					case "color": {
						break;
					}
					case "borderLeft": {
						Border.applyFoBorderToSingleBorder(attributes);
						if(value==JSONObject.NULL) {
							attributes.remove("fo:border-left");
						}
						else {
							final Border border = new Border();
							border.applyJsonBorder((JSONObject)value);
							final String foBorder = border.toString();
							if(foBorder!=null) {
								attributes.setValue(Namespaces.FO, "border-left", "fo:border-left", foBorder);
							}
						}
						break;
					}
					case "borderTop": {
						Border.applyFoBorderToSingleBorder(attributes);
						if(value==JSONObject.NULL) {
							attributes.remove("fo:border-top");
						}
						else {
							final Border border = new Border();
							border.applyJsonBorder((JSONObject)value);
							final String foBorder = border.toString();
							if(foBorder!=null) {
								attributes.setValue(Namespaces.FO, "border-top", "fo:border-top", foBorder);
							}
						}
						break;
					}
					case "borderRight": {
						Border.applyFoBorderToSingleBorder(attributes);
						if(value==JSONObject.NULL) {
							attributes.remove("fo:border-right");
						}
						else {
							final Border border = new Border();
							border.applyJsonBorder((JSONObject)value);
							final String foBorder = border.toString();
							if(foBorder!=null) {
								attributes.setValue(Namespaces.FO, "border-right", "fo:border-right", foBorder);
							}
						}
						break;
					}
					case "borderBottom": {
						Border.applyFoBorderToSingleBorder(attributes);
						if(value==JSONObject.NULL) {
							attributes.remove("fo:border-bottom");
						}
						else {
							final Border border = new Border();
							border.applyJsonBorder((JSONObject)value);
							final String foBorder = border.toString();
							if(foBorder!=null) {
								attributes.setValue(Namespaces.FO, "border-bottom", "fo:border-bottom", foBorder);
							}
						}
						break;
					}
					case "marginBottom": {
		                if(value==JSONObject.NULL) {
		                	attributes.remove("fo:margin-bottom");
		                }
		                else {
		                    attributes.setValue(Namespaces.FO, "margin-bottom", "fo:margin-bottom", (((Number)value).intValue() / 100) + "mm");
		                }
		                break;
					}
					case "indentLeft":
						// PASSTHROUGH INTENDED !!
					case "marginLeft": {
		                if(value==JSONObject.NULL) {
		                	attributes.remove("fo:margin-left");
		                }
		                else {
		                    attributes.setValue(Namespaces.FO, "margin-left", "fo:margin-left", (((Number)value).intValue() / 100) + "mm");
		                }
		                break;
					}
					case "indentRight":
						// PASSTHROUGH INTENDED !!
					case "marginRight": {
		                if(value==JSONObject.NULL) {
		                	attributes.remove("fo:margin-right");
		                }
		                else {
		                    attributes.setValue(Namespaces.FO, "margin-right", "fo:margin-right", (((Number)value).intValue() / 100) + "mm");
		                }
		                break;
					}
					case "marginTop": {
		                if(value==JSONObject.NULL) {
		                	attributes.remove("fo:margin-top");
		                }
		                else {
		                    attributes.setValue(Namespaces.FO, "margin-top", "fo:margin-top", (((Number)value).intValue() / 100) + "mm");
		                }
		                break;
					}
					case "paddingBottom": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-bottom");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-bottom", "fo:padding-bottom", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingLeft": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-left");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-left", "fo:padding-left", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingRight": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-right");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-right", "fo:padding-right", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "paddingTop": {
	                    if (value==JSONObject.NULL) {
	                        attributes.remove("fo:padding-top");
	                    } else {
	                        attributes.setValue(Namespaces.FO, "padding-top", "fo:padding-top", (((Number)value).intValue() / 100) + "mm");
	                    }
	                    break;
					}
					case "fillColor": {
                        if(value==JSONObject.NULL) {
                            attributes.remove("fo:background-color");
                        }
                        else {
                            attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", PropertyHelper.getColor((JSONObject)value, "transparent"));
                        }
                        break;
					}
					case "lineHeight": {
                        if(value==JSONObject.NULL) {
                        	attributes.remove("fo:line-height");
                        	attributes.remove("style:line-height-at-least");
                        	attributes.remove("style:line-spacing");
                        }
                        else {
                            PropertyHelper.setLineHeight((JSONObject)value, this);
                        }
                        break;
					}
					case "alignment": {
                        if(value==JSONObject.NULL) {
                        	attributes.remove("fo:text-align");
                        }
                        else {
                        	attributes.setValue(Namespaces.FO, "text-align", "fo:text-align", (String)value);
                        }
                        break;
					}
					case "indentFirstLine": {
                        if(value==JSONObject.NULL) {
                        	attributes.remove("fo:text-indent");
                        }
                        else {
                            attributes.setValue(Namespaces.FO, "text-indent", "fo:text-indent", (((Number)value).doubleValue() / 100.0) + "mm");
                        }
                        break;
					}
					case "pageBreakBefore": {
                        if(value==JSONObject.NULL) {
                        	attributes.remove("fo:break-before");
                        }
                        else if(((Boolean)value).booleanValue()) {
                        	attributes.setValue(Namespaces.FO, "break-before", "fo:break-before", "page");
                        }
                        // there can not be before and after break at the same paragraph
                        attributes.remove("fo:break-after");
                        break;
	                }
					case "pageBreakAfter": {
                        if(value==JSONObject.NULL) {
                        	attributes.remove("fo:break-after");
                        }
                        else if(((Boolean)value).booleanValue()) {
                        	attributes.setValue(Namespaces.FO, "break-after", "fo:break-after", "page");
                        }
                        // there can not be before and after break at the same paragraph
                        attributes.remove("fo:break-before");
                        break;
					}
					case "tabStops": {
						setTabStops(null);
						if(value instanceof JSONArray) {
							final JSONArray tabArray = (JSONArray)value;
							final TabStops newTabStops = getTabStops(true);
                            for(int i = 0; i < tabArray.length(); i++){
                            	final JSONObject tab = tabArray.getJSONObject(i);
                            	final TabStop tabStop = new TabStop(new AttributesImpl());
                            	newTabStops.getTabStopList().add(tabStop);
                                if(tab.has("pos") ){
                                    tabStop.getAttributes().setValue(Namespaces.STYLE, "position", "style:position", (tab.getInt("pos") / 1000F) + "cm");
                                }
                                if(tab.has("value")) {
                                    String tabValue = tab.getString("value");
                                    if(tabValue.equals("decimal")){
                                        tabValue = "char";
                                    } else if(tabValue.equals("bar")){
                                        tabValue = "left";
                                    } else if(tabValue.equals("clear")){
                                        continue; // clear unsupported
                                    }
                                    tabStop.getAttributes().setValue(Namespaces.STYLE, "type", "style:type", tabValue);
                                }
                            }
						}
						break;
					}
				}
			}
		}

        final JSONObject cellAttributes = attrs.optJSONObject("cell");
        if (cellAttributes != null) {
            Object alignHor = cellAttributes.opt("alignHor");
            if (null != alignHor) {
                if (alignHor == JSONObject.NULL) {
                    attributes.remove("fo:text-align");
                } else {
                    String textAlign = alignHor.toString();
                    switch (textAlign) {
                        case "left":
                            textAlign = "start";
                            break;
                        case "right":
                            textAlign = "end";
                            break;
                        default:
                            textAlign = "center";
                            break;
                    }
                    attributes.setValue(Namespaces.FO, "text-align", "fo:text-align", textAlign);
                }
            }
        }
	}

	@Override
	public void createAttrs(StyleManager styleManager, boolean contentAutoStyle, OpAttrs attrs) {
		final Map<String, Object> paragraphAttrs = attrs.getMap("paragraph", true);
		PropertyHelper.createDefaultMarginAttrs(paragraphAttrs, attributes);
        final Integer defaultPadding = PropertyHelper.createDefaultPaddingAttrs(paragraphAttrs, attributes);
        Border.createDefaultBorderMapAttrsFromFoBorder(paragraphAttrs, defaultPadding, attributes);
        final Iterator<Entry<String, AttributeImpl>> propIter = attributes.getUnmodifiableMap().entrySet().iterator();
        while(propIter.hasNext()) {
        	final Entry<String, AttributeImpl> propEntry = propIter.next();
        	final String propName = propEntry.getKey();
        	final String propValue = propEntry.getValue().getValue();
        	switch(propName) {
        		case "fo:margin-left": {
        			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
                    if(margin!=null) {
                    	paragraphAttrs.put("marginLeft", margin);
                        paragraphAttrs.put("indentLeft", margin);
                    }
        			break;
        		}
        		case "fo:margin-top": {
        			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
                    if(margin!=null) {
                    	paragraphAttrs.put("marginTop", margin);
                    }
        			break;
        		}
        		case "fo:margin-right": {
        			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
        			if(margin!=null) {
        				paragraphAttrs.put("marginRight", margin);
	                    paragraphAttrs.put("indentRight", margin);
        			}
        			break;
        		}
        		case "fo:margin-bottom": {
        			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
        			if(margin!=null) {
        				paragraphAttrs.put("marginBottom", margin);
        			}
        			break;
        		}
        		case "fo:padding-left": {
        			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
        			if(padding!=null) {
        				paragraphAttrs.put("paddingLeft", padding);
        			}
        			break;
        		}
        		case "fo:padding-top": {
        			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
        			if(padding!=null) {
        				paragraphAttrs.put("paddingTop", padding);
        			}
        			break;
        		}
        		case "fo:padding-right": {
        			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
        			if(padding!=null) {
        				paragraphAttrs.put("paddingRight", padding);
        			}
        			break;
        		}
        		case "fo:padding-bottom": {
        			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
        			if(padding!=null) {
        				paragraphAttrs.put("paddingBottom", padding);
        			}
        			break;
        		}
        		case "fo:border-left": {
        			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                    	paragraphAttrs.put("borderLeft", border);
                    }
        			break;
        		}
        		case "fo:border-top": {
        			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                    	paragraphAttrs.put("borderTop", border);
                    }
        			break;
        		}
        		case "fo:border-right": {
        			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                    	paragraphAttrs.put("borderRight", border);
                    }
        			break;
        		}
        		case "fo:border-bottom": {
        			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
                    if(border!=null) {
                    	paragraphAttrs.put("borderBottom", border);
                    }
        			break;
        		}
        		case "fo:background-color": {
	                final Map<String, String> color = PropertyHelper.createColorMap(propValue);
	                if(color!=null) {
	                	paragraphAttrs.put("fillColor", color);
	                }
	                break;
        		}
        		case "fo:line-height": {
	                if(!propValue.isEmpty()) {
	                    paragraphAttrs.put("lineHeight", PropertyHelper.createLineHeightMap(propValue));
	                }
	                break;
        		}
        		case "style:line-spacing": {
	                if(!propValue.isEmpty()) {
	                	final Map<String, Object> lineSpacing = new HashMap<String, Object>(2);
	                	lineSpacing.put("type", "leading");
	                	lineSpacing.put("value", AttributesImpl.normalizeLength(propValue));
	                    paragraphAttrs.put("lineHeight", lineSpacing);
	                }
	                break;
        		}
        		case "style:line-height-at-least": {
	                if(!propValue.isEmpty()) {
	                	final Map<String, Object> atLeast = new HashMap<String, Object>(2);
	                	atLeast.put("type", "atLeast");
	                	atLeast.put("value", AttributesImpl.normalizeLength(propValue));
	                    paragraphAttrs.put("lineHeight", atLeast);
	                }
	                break;
        		}
        		case "fo:text-align": {
                    paragraphAttrs.put("alignment", PropertyHelper.mapFoTextAlign(propValue));
                    break;
        		}
        		case "fo:text-indent": {
        			final Integer firstLineIndent = AttributesImpl.normalizeLength(propValue, true);
        			if(firstLineIndent!=null) {
	                	paragraphAttrs.put("indentFirstLine", firstLineIndent);
	                }
        		}
        		case "style:tab-stop-distance": {
        			break;
        		}
        		case "fo:break-before": {
	                if(propValue.equals("page")) {
	                	paragraphAttrs.put("pageBreakBefore", Boolean.TRUE);
	                }
	                break;
        		}
        		case "fo:break-after": {
	                if (propValue.equals("page")) {
	                	paragraphAttrs.put("pageBreakAfter", Boolean.TRUE);
	                }
	                break;
        		}
            }
       	}
        if(tabStops!=null) {
            int tabIndent = 0;
        	final boolean hasTabsRelativeToIndent = styleManager.hasTabsRelativeToIndent();
        	if(hasTabsRelativeToIndent) {
        		Object leftMargin = paragraphAttrs.get("marginLeft");
        		if(leftMargin instanceof Number) {
        			tabIndent = ((Number)leftMargin).intValue();
        		}
        	}
        	final List<TabStop> tabStopList = tabStops.getTabStopList();
        	final JSONArray tabArray = new JSONArray(tabStopList.size());
        	for(TabStop tabStop:tabStopList) {
        		try {
	        		boolean hasPosition = false;

	        		final JSONObject newTab = new JSONObject();
	        		final String leaderText = tabStop.getAttribute("style:leader-text");
	        		if(leaderText!=null) {
	        			String fillChar = null;
	        			if(leaderText.equals(".")) {
	        				fillChar = "dot";
	        			}
	        			else if(leaderText.equals("-")) {
	        				fillChar = "hyphen";
	        			}
	        			else if(leaderText.equals("_")) {
	        				fillChar = "underscore";
	        			}
	        			else if(leaderText.equals(" ")) {
	        				fillChar = "none";
	        			}
	        			if(fillChar!=null) {
	        				newTab.put("fillChar", fillChar);
	        			}
	        		}
	        		final String tabPos = tabStop.getAttribute("style:position");
	        		if(tabPos!=null) {
	        			final Integer pos = AttributesImpl.normalizeLength(tabPos, true);
	        			if(pos!=null) {
	        				newTab.put("pos", pos + tabIndent);
	        			}
	        			hasPosition = true;
	        		}
	                final String tabType = tabStop.getAttribute("style:type");
	                if(tabType!=null) {
	                	if(tabType.equals("char")) {
	                		newTab.put("value", "decimal");
	                	}
	                	else if(!tabType.isEmpty()) {
	                		newTab.put("value", tabType);
	                	}
	                }
	                if(hasPosition) {
	                	tabArray.put(newTab);
	                }
        		}
        		catch(JSONException e) {
        		    //
        		}
        	}
        	paragraphAttrs.put("tabStops", tabArray);
        }
        if(styleManager.getDocumentType().equals("presentation")) {
            final Object marginTop = paragraphAttrs.remove("marginTop");
            if(marginTop instanceof Number) {
                final OpAttrs lineSpacing = new OpAttrs(2);
                lineSpacing.put("type", "fixed");
                lineSpacing.put("value", marginTop);
                paragraphAttrs.put("spacingBefore", lineSpacing);
            }
            final Object marginBottom = paragraphAttrs.remove("marginBottom");
            if(marginBottom instanceof Number) {
                final OpAttrs lineSpacing = new OpAttrs(2);
                lineSpacing.put("type", "fixed");
                lineSpacing.put("value", marginBottom);
                paragraphAttrs.put("spacingAfter", lineSpacing);
            }
        }
        if(paragraphAttrs.isEmpty()) {
        	attrs.remove("paragraph");
        }
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		if((tabStops!=null&&!tabStops.isEmpty())||!attributes.isEmpty()||(content!=null&&!content.isEmpty())||!getTextContent().isEmpty()) {
			output.startElement(getNamespace(), getLocalName(), getQName());
			attributes.write(output);
			if(!text.isEmpty()) {
				output.characters(text);
			}
			if(tabStops!=null) {
				tabStops.writeObject(output);
			}
			if(content!=null) {
				final Iterator<IElementWriter> childIter = content.iterator();
				while(childIter.hasNext()) {
					childIter.next().writeObject(output);
				}
			}
			output.endElement(getNamespace(), getLocalName(), getQName());
		}
	}

	@Override
	public int hashCode() {
		int hash = super.hashCode();
		if(tabStops!=null) {
			hash = hash * 31 + tabStops.hashCode();
		}
		return hash;
	}

	@Override
	public boolean equals(Object obj) {
		if(!super.equals(obj)) {
			return false;
		}
		final TabStops otherTabStops = ((ParagraphProperties)obj).getTabStops(false);
		if(tabStops!=null) {
			if(otherTabStops==null) {
				return false;
			}
			if(!tabStops.equals(otherTabStops)) {
				return false;
			}
		}
		else if(otherTabStops!=null) {
			return false;
		}
		return true;
	}

	@Override
	public ParagraphProperties clone() {
		final ParagraphProperties clone = (ParagraphProperties)_clone();
		if(tabStops!=null) {
			clone.setTabStops(tabStops.clone());
		}
		return clone;
	}

	public void applyHardDefaultAttributes(Set<String> paraKeyset) {
        final Iterator<String> keyIter = paraKeyset.iterator();
        while(keyIter.hasNext()) {
            switch(keyIter.next()) {
                case "color": {
                    break;
                }
                case "borderLeft": {
                    attributes.setValue(Namespaces.FO, "border-left", "fo:border-left", "none");
                    break;
                }
                case "borderTop": {
                    attributes.setValue(Namespaces.FO, "border-top", "fo:border-top", "none");
                    break;
                }
                case "borderRight": {
                    attributes.setValue(Namespaces.FO, "border-right", "fo:border-right", "none");
                    break;
                }
                case "borderBottom": {
                    attributes.setValue(Namespaces.FO, "border-bottom", "fo:border-bottom", "none");
                    break;
                }
                case "marginBottom": {
                    attributes.setValue(Namespaces.FO, "margin-bottom", "fo:margin-bottom",  "0mm");
                    break;
                }
                case "indentLeft":
                    // PASSTHROUGH INTENDED !!
                case "marginLeft": {
                    attributes.setValue(Namespaces.FO, "margin-left", "fo:margin-left", "0mm");
                    break;
                }
                case "indentRight":
                    // PASSTHROUGH INTENDED !!
                case "marginRight": {
                    attributes.setValue(Namespaces.FO, "margin-right", "fo:margin-right", "0mm");
                    break;
                }
                case "marginTop": {
                    attributes.setValue(Namespaces.FO, "margin-top", "fo:margin-top", "0mm");
                    break;
                }
                case "paddingBottom": {
                    attributes.setValue(Namespaces.FO, "padding-bottom", "fo:padding-bottom", "0mm");
                    break;
                }
                case "paddingLeft": {
                    attributes.setValue(Namespaces.FO, "padding-left", "fo:padding-left", "0mm");
                    break;
                }
                case "paddingRight": {
                    attributes.setValue(Namespaces.FO, "padding-right", "fo:padding-right", "0mm");
                    break;
                }
                case "paddingTop": {
                    attributes.setValue(Namespaces.FO, "padding-top", "fo:padding-top", "0mm");
                    break;
                }
                case "fillColor": {
                    attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", "transparent");
                    break;
                }
                case "lineHeight": {
                    attributes.setValue(Namespaces.FO, "line-height", "fo:line-height", "100%");
                    attributes.remove("style:line-height-at-least");
                    attributes.remove("style:line-spacing");
                    break;
                }
                case "alignment": {
                    attributes.setValue(Namespaces.FO, "text-align", "fo:text-align", "left");
                    break;
                }
                case "indentFirstLine": {
                    attributes.setValue(Namespaces.FO, "text-indent", "fo:text-indent", "0mm");
                    break;
                }
                case "pageBreakBefore": {
                    break;
                }
                case "pageBreakAfter": {
                    break;
                }
                case "tabStops": {
                    break;
                }
            }
        }
	}
}
