/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.properties;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import com.openexchange.office.filter.odf.AttributeImpl;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Border;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.styles.DrawFillImage;
import com.openexchange.office.filter.odf.styles.Gradient;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleManager;

final public class DrawingPageProperties extends StylePropertiesBase {

	private DLList<IElementWriter> content;

	public DrawingPageProperties(AttributesImpl attributesImpl) {
		super(attributesImpl);
	}

	@Override
	public DLList<IElementWriter> getContent() {
		if(content==null) {
			content = new DLList<IElementWriter>();
		}
		return content;
	}

	@Override
	public String getQName() {
		return "style:drawing-page-properties";
	}

	@Override
	public String getLocalName() {
		return "drawing-page-properties";
	}

	@Override
	public String getNamespace() {
		return Namespaces.STYLE;
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs)
	    throws JSONException {

	    final JSONObject fillAttributes = attrs.optJSONObject("fill");
        if(fillAttributes!=null) {
            final Object type = fillAttributes.opt("type");
            if(type!=null) {
                if(type==JSONObject.NULL) {
                    attributes.remove("draw:fill");
                    attributes.remove("draw:fill-color");
                    attributes.remove("fo:background-color");
                    attributes.remove("draw:fill-image-name");
                    attributes.remove("draw:fill-gradient-name");
                    attributes.remove("presentation:background-visible");
                    attributes.remove("style:repeat");
                }
                else if(type instanceof String) {
                    String drawFill = "solid";
                    if("none".equals(type)) {
                        drawFill = "none";
                    }
                    else if("bitmap".equals(type)) {
                        drawFill = "bitmap";
                    }
                    else if("gradient".equals(type)) {
                        drawFill = "gradient";
                    }
                    attributes.setValue(Namespaces.DRAW, "fill", "draw:fill", drawFill);
                }
            }
            final JSONObject color = fillAttributes.optJSONObject("color");
            if(color!=null) {
                if (color==JSONObject.NULL) {
                    attributes.remove("fo:color");
                    attributes.remove("fo:background-color");
                }
                else {
                    final String colorType = color.optString("type", null);
                    if(colorType!=null) {
                        if(colorType.equals("auto")) {
                            attributes.setValue(Namespaces.DRAW, "fill-color", "draw:fill-color", "#ffffff");
                            attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", "#ffffff");
                        }
                        else {
                            attributes.setValue(Namespaces.DRAW, "fill-color", "draw:fill-color", PropertyHelper.getColor(color, null));
                            attributes.setValue(Namespaces.FO, "background-color", "fo:background-color", PropertyHelper.getColor(color, null));
                        }
                    }
                }
            }
            final JSONObject bitmapAttrs = fillAttributes.optJSONObject("bitmap");
            if(bitmapAttrs!=null) {
                attributes.setValue(Namespaces.DRAW,  "fill-image-name",  "draw:fill-image-name", DrawFillImage.getFillImage(styleManager, attributes.getValue("draw:fill-image-name"), bitmapAttrs));
                DrawFillImage.applyTilingAttrs(bitmapAttrs, getAttributes());
            }
            final JSONObject gradientAttrs = fillAttributes.optJSONObject("gradient");
            if(gradientAttrs!=null) {
                attributes.setValue(Namespaces.DRAW, "fill-gradient-name", "draw:fill-gradient-name", Gradient.getGradient(styleManager, attributes.getValue("draw:fill-gradient-name"), fillAttributes));
            }
        }
	    final JSONObject slideAttrs = attrs.optJSONObject("slide");
	    if(slideAttrs!=null) {
	        final Iterator<Entry<String, Object>> slideAttrsIter = slideAttrs.entrySet().iterator();
	        while(slideAttrsIter.hasNext()) {
	            final Entry<String, Object> entry = slideAttrsIter.next();
	            final Object o = entry.getValue();
	            switch(entry.getKey()) {
	                case "followMasterShapes" : {
	                    if(o instanceof Boolean) {
	                        getAttributes().setBooleanValue(Namespaces.PRESENTATION, "background-objects-visible", "presentation:background-objects-visible", (Boolean)o);
	                    }
	                    else if (o==JSONObject.NULL){
	                        getAttributes().remove("presentation:background-objects-visible");
	                    }
	                    break;
	                }
                    case "isSlideNum" : {
                        if(o instanceof Boolean) {
                            getAttributes().setBooleanValue(Namespaces.PRESENTATION, "display-page-number", "presentation:display-page-number", (Boolean)o);
                        }
                        else if(o== JSONObject.NULL) {
                            getAttributes().remove("presentation:display-page-number");
                        }
                        break;
                    }
	                case "isHeader" : {
                        if(o instanceof Boolean) {
                            getAttributes().setBooleanValue(Namespaces.PRESENTATION, "display-header", "presentation:display-header", (Boolean)o);
                        }
                        else if(o== JSONObject.NULL) {
                            getAttributes().remove("presentation:display-header");
                        }
                        break;
                    }
                    case "isFooter" : {
                        if(o instanceof Boolean) {
                            getAttributes().setBooleanValue(Namespaces.PRESENTATION, "display-footer", "presentation:display-footer", (Boolean)o);
                        }
                        else if(o== JSONObject.NULL) {
                            getAttributes().remove("presentation:display-footer");
                        }
                        break;
                    }
                    case "isDate" : {
                        if(o instanceof Boolean) {
                            getAttributes().setBooleanValue(Namespaces.PRESENTATION, "display-date-time", "presentation:display-date-time", (Boolean)o);
                        }
                        else if(o== JSONObject.NULL) {
                            getAttributes().remove("presentation:display-date-time");
                        }
                        break;
                    }
                    case "hidden" : {
                        if(o instanceof Boolean&&((Boolean)o).booleanValue()) {
                            getAttributes().setValue(Namespaces.PRESENTATION, "visibility", "presentation:visibility", "hidden");
                        }
                        else {
                            getAttributes().remove("presentation:visibility");
                        }
                        break;
                    }
	            }
	        }
	    }
	}
	
	@Override
	public void createAttrs(StyleManager styleManager, boolean contentAutoStyle, OpAttrs attrs) {
	    final OpAttrs pageAttrs = attrs.getMap("page", true);
        final OpAttrs fillAttrs = attrs.getMap("fill",  true);
        final OpAttrs slideAttrs = attrs.getMap("slide", true);

        boolean disableMasterBackground = false;
		PropertyHelper.createDefaultMarginAttrs(pageAttrs, attributes);
        final Integer defaultPadding = PropertyHelper.createDefaultPaddingAttrs(pageAttrs, attributes);
        Border.createDefaultBorderMapAttrsFromFoBorder(pageAttrs, defaultPadding, attributes);
        final Iterator<Entry<String, AttributeImpl>> propIter = attributes.getUnmodifiableMap().entrySet().iterator();
        while(propIter.hasNext()) {
        	final Entry<String, AttributeImpl> propEntry = propIter.next();
        	final String propName = propEntry.getKey();
        	final String propValue = propEntry.getValue().getValue();
        	switch(propName) {
	    		case "fo:margin-left": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	                if(margin!=null) {
	                	pageAttrs.put("marginLeft", margin);
	                	pageAttrs.put("indentLeft", margin);
	                }
	    			break;
	    		}
	    		case "fo:margin-top": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	                if(margin!=null) {
	                	pageAttrs.put("marginTop", margin);
	                }
	    			break;
	    		}
	    		case "fo:margin-right": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	    			if(margin!=null) {
	    				pageAttrs.put("marginRight", margin);
	    				pageAttrs.put("indentRight", margin);
	    			}
	    			break;
	    		}
	    		case "fo:margin-bottom": {
	    			final Integer margin = AttributesImpl.normalizeLength(propValue, true);
	    			if(margin!=null) {
	    				pageAttrs.put("marginBottom", margin);
	    			}
	    			break;
	    		}
	    		case "fo:padding-left": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				pageAttrs.put("paddingLeft", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-top": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				pageAttrs.put("paddingTop", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-right": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				pageAttrs.put("paddingRight", padding);
	    			}
	    			break;
	    		}
	    		case "fo:padding-bottom": {
	    			final Integer padding = AttributesImpl.normalizeLength(propValue, true);
	    			if(padding!=null) {
	    				pageAttrs.put("paddingBottom", padding);
	    			}
	    			break;
	    		}
	    		case "fo:border-left": {
	    			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
	                if(border!=null) {
	                	pageAttrs.put("borderLeft", border);
	                }
	    			break;
	    		}
	    		case "fo:border-top": {
	    			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
	                if(border!=null) {
	                	pageAttrs.put("borderTop", border);
	                }
	    			break;
	    		}
	    		case "fo:border-right": {
	    			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
	                if(border!=null) {
	                	pageAttrs.put("borderRight", border);
	                }
	    			break;
	    		}
	    		case "fo:border-bottom": {
	    			final Map<String, Object> border = Border.createBordermapFromFoBorder(propValue, defaultPadding);
	                if(border!=null) {
	                	pageAttrs.put("borderBottom", border);
	                }
	    			break;
	    		}
	    		case "fo:page-width": {
    				pageAttrs.put("width", AttributesImpl.normalizeLength(propValue));
	    			break;
	    		}
	    		case "fo:page-height": {
    				pageAttrs.put("height", AttributesImpl.normalizeLength(propValue));
	    			break;
	    		}
	    		case "style:print-orientation": {
	    			pageAttrs.put("printOrientation", propValue);
	       			break;
	       	  	}
	    		case "style:num-format": {
	                pageAttrs.put("numberFormat", propValue);
	                break;
	            }
                case "draw:fill": {
                    if(propValue.equals("none")) {
                        fillAttrs.put("type", "none");
                    }
                    else if(propValue.equals("bitmap")) {
                        fillAttrs.put("type", "bitmap");
                    }
                    else if(propValue.equals("gradient")) {
                        fillAttrs.put("type", "gradient");
                    }
                    else {
                        fillAttrs.put("type", "solid");
                    }
                    break;
                }
                case "draw:fill-image-name": {
                    final StyleBase styleBase = styleManager.getStyle(propValue, "fill-image", contentAutoStyle);
                    if(styleBase!=null) {
                        styleBase.createAttrs(styleManager, attrs);
                        final Object imageUrl = fillAttrs.opt("bitmap", "imageUrl");
                        if(imageUrl instanceof String) {
                            String repeat = attributes.getValue("style:repeat");
                            if(repeat==null) {
                                repeat = "repeat";
                            }
                            DrawFillImage.createTilingAttrs(styleManager, (String)imageUrl, repeat, attributes, fillAttrs);
                        }
                    }
                    if(!attributes.containsKey("draw:fill")) {
                        fillAttrs.put("type", "bitmap");
                    }
                    break;
                }
                case "draw:fill-gradient-name": {
                    StyleBase styleBase = styleManager.getStyle(propValue, "gradient", contentAutoStyle);
                    if(styleBase==null) {
                        styleBase = new Gradient(null);
                    }
                    styleBase.createAttrs(styleManager, attrs);
                    if(!attributes.containsKey("draw:fill")) {
                        fillAttrs.put("type", "gradient");
                    }
                    break;
                }
                case "fo:background-color": {
                    final Map<String, String> color = PropertyHelper.createColorMap(propValue);
                    if(color!=null) {
                        fillAttrs.put("color", color);
                    }
                    if(!attributes.containsKey("draw:fill")) {
                        fillAttrs.put("type", "solid");
                    }
                    break;
                }
                case "draw:fill-color": {
                    final Map<String, String> color = PropertyHelper.createColorMap(propValue);
                    if(color!=null) {
                        fillAttrs.put("color", color);
                    }
                    if(!attributes.containsKey("draw:fill")) {
                        fillAttrs.put("type", "solid");
                    }
                    break;
                }
                case "presentation:background-visible": {
                    if(!AttributesImpl.getBoolean(propValue, Boolean.TRUE).booleanValue()) {
                        disableMasterBackground = true;
                    }
                    break;
                }
                case "presentation:background-objects-visible": {
                    if(!AttributesImpl.getBoolean(propValue, Boolean.TRUE).booleanValue()) {
                        slideAttrs.put("followMasterShapes", false);
                    }
                    break;
                }
                case "presentation:display-header": {
                    slideAttrs.put("isHeader", AttributesImpl.getBoolean(propValue, Boolean.FALSE));
                    break;
                }
                case "presentation:display-footer": {
                    slideAttrs.put("isFooter", AttributesImpl.getBoolean(propValue, Boolean.FALSE));
                    break;
                }
                case "presentation:display-page-number": {
                    slideAttrs.put("isSlideNum", AttributesImpl.getBoolean(propValue, Boolean.FALSE));
                    break;
                }
                case "presentation:display-date-time": {
                    slideAttrs.put("isDate", AttributesImpl.getBoolean(propValue, Boolean.FALSE));
                    break;
                }
                case "presentation:visibility": {
                    if(propValue.equals("hidden")) {
                        slideAttrs.put("hidden", Boolean.TRUE);
                    }
                    break;
                }
        	}
        }
        if(disableMasterBackground) {
            final Object masterBackground = fillAttrs.get("type");
            if(!(masterBackground instanceof String)) {
                fillAttrs.put("type", "solid");
            }
        }
        if(slideAttrs.isEmpty()) {
            attrs.remove("slide");
        }
        if(pageAttrs.isEmpty()) {
            attrs.remove("page");
        }
        if(fillAttrs.isEmpty()) {
            attrs.remove("fill");
        }
	}

	@Override
	public DrawingPageProperties clone() {
		return (DrawingPageProperties)_clone();
	}
}
