/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.draw;

import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Namespaces;

public class Transformer {

    private double x;
    private double y;
    private double width;
    private double height;
    private Double rotation;
    private boolean isLineMode;
    private boolean modified;

    public Transformer(boolean isLineMode) {
        this.isLineMode = isLineMode;
        modified = false;
    }

    public Transformer(AttributesImpl attributes, boolean isLineMode) {
        this.isLineMode = isLineMode;
        modified = false;
        if(isLineMode) {
            init(attributes.getValue("svg:x1"), attributes.getValue("svg:y1"), attributes.getValue("svg:x2"), attributes.getValue("svg:y2"), attributes.getValue("draw:transform"), true);
        }
        else {
            init(attributes.getValue("svg:x"), attributes.getValue("svg:y"), attributes.getValue("svg:width"), attributes.getValue("svg:height"), attributes.getValue("draw:transform"), false);
        }
    }

    public void init(String sX, String sY, String sWidth, String sHeight, String sTransformation, boolean sIsLineMode) {
        x = sX != null ? AttributesImpl.normalizeLength(sX, false) : 0.0d;
        y = sY != null ? AttributesImpl.normalizeLength(sY, false) : 0.0d;
        width = sWidth != null ? AttributesImpl.normalizeLength(sWidth, false) : 1.0d;
        height = sHeight != null ? AttributesImpl.normalizeLength(sHeight, false) : 1.0d;
        rotation = null;
        isLineMode = sIsLineMode;
        if(sTransformation!=null) {
            double centerX = width * 0.5d + x;
            double centerY = height * 0.5d + y;
            final List<Pair<String, List<Double>>> transformations = getTransformations(sTransformation);
            for(Pair<String, List<Double>> entry:transformations) {
                final List<Double> values = entry.getValue();
                switch(entry.getKey()) {
                    case "rotate" : {
                        if(!values.isEmpty()) {
                            final double alpha = -values.get(0);
                            rotation = (alpha * 180.0d) / Math.PI;
                            if(rotation!=0.0d) {
                                final double x2 = centerX * Math.cos(alpha) - centerY * Math.sin(alpha);
                                final double y2 = centerY * Math.cos(alpha) + centerX * Math.sin(alpha);
                                centerX = x2;
                                centerY = y2;
                            }
                        }
                        break;
                    }
                    case "skewX" : {
                        break;
                    }
                    case "skewY" : {
                        break;
                    }
                    case "matrix" : {
                        break;
                    }
                    case "scale" : {
                        break;
                    }
                    case "translate" : {
                        if(values.size()>=1) {
                            centerX += values.get(0);
                        }
                        if(values.size()>=2) {
                            centerY += values.get(1);
                        }
                        break;
                    }
                }
            }
            x = centerX - width * 0.5d;
            y = centerY - height * 0.5d;
        }
        modified = false;

        if(isLineMode) {
            width -= x;
            height -= y;
        }
    }

    private List<Pair<String, List<Double>>> getTransformations(String sTransformation) {
        final List<Pair<String, List<Double>>> transformations = new ArrayList<Pair<String, List<Double>>>();
        int index = 0;
        while(index>=0) {
            index = getNextTransformationEntry(transformations, index, sTransformation);
        }
        return transformations;
    }

    private int getNextTransformationEntry(List<Pair<String, List<Double>>> transformations, int index, String sTransformation) {
        if(index<0) {
            return -1;
        }
        final int o = sTransformation.indexOf('(', index + 1);
        if(o<0) {
            return -1;
        }
        final int c = sTransformation.indexOf(')', o + 2);
        if(c<0) {
            return -1;
        }
        final String command = sTransformation.substring(index, o).trim();
        if(!command.isEmpty()) {
            final Pair<String, List<Double>> pair = new ImmutablePair<String, List<Double>>(command, new ArrayList<Double>());
            final String[] parameters = sTransformation.substring(o+1, c).split("\\s+", -1);
            for(String p:parameters) {
                Double v = 0.0d;
                try {
                    if(command.equals("translate")) {
                        v = AttributesImpl.normalizeLength(p, false).doubleValue();
                    }
                    else {
                        v = Double.parseDouble(p);
                    }
                }
                catch(NumberFormatException e) {
                    //
                }
                pair.getValue().add(v);
            }
            transformations.add(pair);
        }
        if(c+1==sTransformation.length()) {
            return -1;
        }
        return c + 1;
    }

    public double getX() {
        return x;
    }

    public double getY() {
        return y;
    }

    public double getWidth() {
        return width;
    }

    public double getHeight() {
        return height;
    }

    public Rectangle2D getRect2D() {
        return new Rectangle2D.Double(x, y, width, height);
    }

    public Double getRotation() {
        return rotation;
    }

    public void setX(double x) {
        this.x = x;
        modified = true;
    }

    public void setY(double y) {
        this.y = y;
        modified = true;
    }

    public void setWidth(double width) {
        this.width = width;
        modified = true;
    }

    public void setHeight(double height) {
        this.height = height;
        modified = true;
    }

    public void setRotation(Double sRotation) {
        rotation = sRotation;
        modified = true;
    }

    void updateAttributes(AttributesImpl attributes) {
        if(modified) {
            attributes.remove("draw:transform");
            if(isLineMode) {
                // dollar, paragraph sign, percent= line special case
                attributes.setValue(Namespaces.SVG, "x1", "svg:x1", (x / 100.0 + "mm"));
                attributes.setValue(Namespaces.SVG, "y1", "svg:y1", (y / 100.0 + "mm"));
                attributes.setValue(Namespaces.SVG, "x2", "svg:x2", ((width + x) / 100.0 + "mm"));
                attributes.setValue(Namespaces.SVG, "y2", "svg:y2", ((height + y) / 100.0 + "mm"));
                // removing eventual connections
                attributes.remove("draw:start-shape");
                attributes.remove("draw:start-glue-point");
                attributes.remove("draw:end-shape");
                attributes.remove("draw:end-glue-point");
            }
            else {

                if(rotation!=null&&rotation!=0.0d) {
                    attributes.remove("svg:x");
                    attributes.remove("svg:y");
                    double centerX = width * 0.5d;
                    double centerY = height * 0.5d;
                    final double alpha = rotation * Math.PI / 180.0d;
                    final double x2 = centerX * Math.cos(alpha) - centerY * Math.sin(alpha);
                    final double y2 = centerY * Math.cos(alpha) + centerX * Math.sin(alpha);
                    double translateX = -(x2 - width * 0.5d);
                    double translateY = -(y2 - height * 0.5d);
                    translateX += x;
                    translateY += y;
                    final StringBuffer buffer = new StringBuffer();
                    buffer.append("rotate (");
                    buffer.append(Double.valueOf(-alpha).toString());
                    buffer.append(") translate (");
                    buffer.append(Double.valueOf(translateX / 100d).toString());
                    buffer.append("mm ");
                    buffer.append(Double.valueOf(translateY / 100d).toString());
                    buffer.append("mm)");
                    attributes.setValue(Namespaces.DRAW, "transform", "draw:transform", buffer.toString());
                }
                else {
                    attributes.setValue(Namespaces.SVG, "x", "svg:x", (x / 100.0 + "mm"));
                    attributes.setValue(Namespaces.SVG, "y", "svg:y", (y / 100.0 + "mm"));
                }
                attributes.setValue(Namespaces.SVG, "width", "svg:width", (width / 100.0 + "mm"));
                attributes.setValue(Namespaces.SVG, "height", "svg:height", (height / 100.0 + "mm"));
            }
        }
    }
}
