/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.draw;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import javax.imageio.ImageIO;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.Names;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.odftoolkit.odfdom.pkg.manifest.OdfFileEntry;
import org.odftoolkit.odfdom.type.Base64Binary;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Length;
import com.openexchange.office.filter.odf.Length.Unit;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.styles.StyleManager;

public class DrawImage implements IDrawing {

	private final DrawFrame drawFrame;
	private final AttributesImpl attributes;
    private final DLList<Object> childs = new DLList<Object>();

	public DrawImage(DrawFrame drawFrame) {
		this.drawFrame = drawFrame;
		this.attributes = new AttributesImpl();
	}

	public DrawImage(DrawFrame drawFrame, Attributes attributes) {
		this.drawFrame = drawFrame;
		this.attributes = new AttributesImpl(attributes);
	}

	@Override
	public DLList<Object> getContent() {
		return childs;
	}

    @Override
    public AttributesImpl getAttributes() {
        return attributes;
    }

    @Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		SaxContextHandler.startElement(output, Namespaces.DRAW, "image", "draw:image");
		attributes.write(output);
		final Iterator<Object> childIter = childs.iterator();
		while(childIter.hasNext()) {
		    ((IElementWriter)childIter.next()).writeObject(output);
		}
		SaxContextHandler.endElement(output, Namespaces.DRAW, "image", "draw:image");
	}

	@Override
	public DrawingType getType() {
		return DrawingType.IMAGE;
	}

	public String getFormat() {
	    final String href = attributes.getValue("xlink:href");
	    return href!=null&&href.length()>3 ? href.substring(href.length()-3) : "";
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle) {
		final JSONObject imageProps = attrs.optJSONObject("image");
		if(imageProps==null) {
			return;
		}
		final String href = imageProps.optString("imageUrl", null);
		if(href!=null) {
    		if(transferResource(href, operationDocument.getDocument().getStyleManager())) {
                attributes.setValue(Namespaces.XLINK, "type", "xlink:type", "simple");
                final Integer height = drawFrame.getAttributes().getIntValue("svg:height");
                final Integer width = drawFrame.getAttributes().getIntValue("svg:widtht");
                if(height==null||width==null) {
                    try {
                        final BufferedImage image = ImageIO.read(operationDocument.getDocument().getPackage().getInputStream(href));
                        drawFrame.getAttributes().setValue(Namespaces.SVG, "width", "svg:width", Length.mapToUnit(String.valueOf(image.getWidth(null)) + "px", Unit.CENTIMETER));
                        drawFrame.getAttributes().setValue(Namespaces.SVG, "height", "svg:height", Length.mapToUnit(String.valueOf(image.getHeight(null)) + "px", Unit.CENTIMETER));
                    }
                    catch(IOException ohoh) {
                        //
                    }
                }
    		}
            attributes.setValue(Namespaces.XLINK, "href", "xlink:href", href);
		}
	    else if (imageProps.has("imageData")) {
            String imageData = imageProps.optString("imageData");
            if (imageData != null && !imageData.isEmpty()) {
                //expected header is
                //		"data:image/png;base64,
                String[] header = imageData.split("base64,");
                String suffix = "png";
                String mediaTypeString = "image/png";
                mediaTypeString = header[0].substring(header[0].indexOf(":") + 1, header[0].indexOf(";"));
                suffix = header[0].substring(header[0].indexOf("/") + 1, header[0].indexOf(";"));
                String fileName = "img_" + new Random().nextInt() + "." + suffix;
                Base64Binary base64Binary = Base64Binary.valueOf(header[1]);
                operationDocument.getDocument().getPackage().insert(base64Binary.getBytes(), "Pictures/" + fileName, mediaTypeString);
                attributes.setValue(Namespaces.XLINK, "href", "xlink:href", "Pictures/" + fileName);
            }
        }
        if (imageProps.has("imageXmlId")) {
            String xmlId = imageProps.optString("imageXmlId");
            if (xmlId != null && !xmlId.isEmpty()) {
            	attributes.setValue(Namespaces.XML, "id", "xml:id", xmlId);
            }
        }
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs, boolean contentAutoStyle) {
		final String hRef = attributes.getValue("xlink:href");
		if(hRef!=null) {
			final Map<String, Object> imageProps = attrs.getMap("image", true);
            imageProps.put("imageUrl", hRef);
            if (imageProps.containsKey("cropLeft")||imageProps.containsKey("cropTop")||imageProps.containsKey("cropRight")||imageProps.containsKey("cropBottom")) {
            	calculateCrops(operationDocument, hRef, imageProps);
            }
        }
		final String id = attributes.getValue("xml:id");
		if(id!=null) {
			attrs.getMap("drawing", true).put("imageXmlId", id);
        }
	}

	public static boolean transferResource(String href, StyleManager styleManager) {
        if(href!=null&&!href.isEmpty()) {
            if (href.contains("uid")) {
                int uidStart = href.indexOf("uid") + 3;
                if(uidStart!=3&&uidStart<href.length()&&uidStart<href.indexOf('.')) {
                    String uidString = null;
                    if (href.contains(".")) {
                        uidString = href.substring(uidStart, href.indexOf('.'));
                    } else {
                        uidString = href.substring(uidStart);
                    }
                    if(uidString!=null) {
                        final long uid = Long.parseLong(uidString, 16);
                        final Map<Long, byte[]> resourceMap = styleManager.getResourceMap();
                        if(resourceMap!=null) {
                            final byte[] fileBytes = resourceMap.get(uid);
                            if(fileBytes!=null) {
                                try {
                                    styleManager.getPackage().insert(fileBytes, href, OdfFileEntry.getMediaTypeString(href));
                                    return true;
                                }
                                catch(Exception ohoh) {
                                    //
                                }
                            }
                        }
                    }
                }
            }
        }
        return false;
	}

	private void calculateCrops(OdfOperationDoc operationDocument, String href, Map<String, Object> imageProps) {
    	try {
            // ToDo: Although the streams are cached we might cache the clipping for known href, help if images occure more than once
    		final OdfPackage pkg = operationDocument.getDocument().getPackage();
            InputStream is = pkg.getInputStream(href);
            if (is != null) {
                BufferedImage bimg = ImageIO.read(is);
                if (bimg != null) {
                    double width = AttributesImpl.normalizeLength((bimg.getWidth() / Names.DOTS_PER_INCH) + "in");
                    double height = AttributesImpl.normalizeLength((bimg.getHeight() / Names.DOTS_PER_INCH) + "in");

                	// 2nd half of absolute fo:clip to relative crop (OX API) mapping
                    if (imageProps.containsKey("cropRight")) {
                        Number cropRight = (Number) imageProps.get("cropRight");
                        if (cropRight != null) {
                            if (cropRight.doubleValue() != 0.0) {
                                imageProps.put("cropRight", cropRight.doubleValue() * 100.0 / width);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropRight");
                            }
                        }
                    }
                    if (imageProps.containsKey("cropLeft")) {
                        Number cropLeft = (Number) imageProps.get("cropLeft");
                        if (cropLeft != null) {
                            if (cropLeft.doubleValue() != 0.0) {
                                imageProps.put("cropLeft", cropLeft.doubleValue() * 100.0 / width);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropLeft");
                            }
                        }
                    }
                    // 2nd half of absolute fo:clip to relative crop (OX API) mapping
                    if (imageProps.containsKey("cropTop")) {
                        Number cropTop = (Number) imageProps.get("cropTop");
                        if (cropTop != null) {
                            if (cropTop.doubleValue() != 0.0) {
                                imageProps.put("cropTop", cropTop.doubleValue() * 100.0 / height);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropTop");
                            }
                        }
                    }
                    if (imageProps.containsKey("cropBottom")) {
                        Number cropBottom = (Number) imageProps.get("cropBottom");
                        if (cropBottom != null) {
                            if (cropBottom.doubleValue() != 0.0) {
                                imageProps.put("cropBottom", cropBottom.doubleValue() * 100.0 / height);
                            } else {
                                // do not set explicitly with 0
                                imageProps.remove("cropBottom");

                            }
                        }
                    }
                }
            }
        } catch (IOException ex) {
            //
        }
    }
}
