/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.draw;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.doc.OdfPresentationDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.odf.AttributeImpl;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.properties.GraphicProperties;
import com.openexchange.office.filter.odf.styles.IGraphicProperties;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleManager;

public class DrawFrame extends Shape {

    public boolean multiImages = false;

    public DrawFrame(Attributes attributes, boolean rootShape) {
        super(attributes, Namespaces.DRAW, "frame", "draw:frame", rootShape);
	}

	public DrawFrame(String documentType, boolean rootShape) {
        super(documentType, Namespaces.DRAW, "frame", "draw:frame", rootShape);

        if(documentType.equals("text")) {
            this.attributes.setValue(Namespaces.TEXT, "anchor-type", "text:anchor-type", "as-char");
        }
	}

	public IDrawing getDrawing() {
		if(!childs.isEmpty()) {
		    final Iterator<Object> childIter = childs.iterator();
		    while(childIter.hasNext()) {
		        final Object o = childIter.next();
		        if(multiImages && o instanceof DrawImage) {
		            final String format = ((DrawImage)o).getFormat();
		            if(!format.equals("svg")) {
		                return (IDrawing)o;
		            }
		        }
		        else if(o instanceof IDrawing) {
	                return (IDrawing)o;
	            }
		    }
		}
		return null;
	}

	@Override
	public DLList<Object> getContent() {
		final IDrawing iDrawing = getDrawing();
		if(iDrawing!=null) {
		    return iDrawing.getContent();
		}
		return childs;
	}

	public void addContent(IElementWriter element) {
        childs.add(element);
	}

	@Override
    public DrawingType getType() {
		final IDrawing drawing = getDrawing();
		if(drawing!=null) {
			return drawing.getType();
		}
		return DrawingType.UNDEFINED;
	}

	@Override
    public AttributesImpl getAttributes() {
		return attributes;
	}

	@Override
    public String getStyleName() {
		return attributes.getValue("draw:style-name");
	}

	@Override
    public void setStyleName(String name) {
		attributes.setValue(Namespaces.DRAW, "style-name", "draw:style-name", name);
	}

	public String getTextStyleName() {
	    return attributes.getValue("draw:text-style-name");
	}

	public void setTextStyleName(String name) {
	    attributes.setValue(Namespaces.DRAW, "text-style-name", "draw:text-style-name", name);
	}

	public String getPresentationStyleName() {
	    return attributes.getValue("presentation:style-name");
	}

	public void setPresentationStyleName(String name) {
	    attributes.setValue(Namespaces.PRESENTATION, "style-name", "presentation:style-name", name);
	}

	public String getPresentationClass() {
	    return attributes.getValue("presentation:class");
	}

	public void setPresentationClass(String presentationClass) {
	    attributes.setValue(Namespaces.PRESENTATION, "class", "presentation:class", presentationClass);
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		SaxContextHandler.startElement(output, Namespaces.DRAW, "frame", "draw:frame");
		final IDrawing drawing = getDrawing();
		if(drawing instanceof DrawTextBox&&attributes.containsKey("presentation:class")) {
		    if(!((DrawTextBox)drawing).isSaveContent()||drawing.getContent().isEmpty()) {
		        attributes.setBooleanValue(Namespaces.PRESENTATION, "placeholder", "presentation:placeholder", Boolean.TRUE);
		    }
		    else {
	            attributes.remove("presentation:placeholder");
	        }
		}
        transformer.updateAttributes(attributes);
		attributes.write(output);
		for(Object child:childs) {
			((IElementWriter)child).writeObject(output);
		}
		SaxContextHandler.endElement(output, Namespaces.DRAW, "frame", "draw:frame");
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle)
		throws JSONException {

	    final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
        final String presentationStyleName = getPresentationStyleName();

        JSONObject drawingAttrs = attrs.optJSONObject("drawing");
	    if(drawingAttrs!=null) {
	        final boolean flipV = drawingAttrs.optBoolean("flipV", false);
	        if(flipV) {

	            // odf is not supporting vertical flip for other objects than custom-shapes :-( 
    	        final HashMap<String, Object> destAttrs = new HashMap<String, Object>();
    	        StyleManager.deepCopy(attrs.asMap(), destAttrs);
    	        attrs = new JSONObject(destAttrs);
    	        drawingAttrs = attrs.optJSONObject("drawing");
    	        drawingAttrs.remove("flipV");

    	        Double rotation = null;
    	        final Object o = drawingAttrs.opt("rotation");
    	        if(o instanceof Number) {
    	            rotation = ((Number)o).doubleValue();
    	        }
    	        else if(o!=JSONObject.NULL) {
    	            if(transformer!=null) {
    	                rotation = transformer.getRotation();
    	            }
    	        }
    	        if(rotation==null) {
    	            rotation = 180.0;
    	        }
    	        else {
    	            rotation = ((-rotation) + 180.0) % 360.0;
    	            if(rotation < 0) {
    	                rotation = 360 + rotation;
    	            }
    	        }
	            drawingAttrs.put("rotation", rotation);

	            Boolean flipH = null;
	            final Object fH = drawingAttrs.opt("flipH");
	            if(fH instanceof Boolean) {
	                flipH = (Boolean)fH;
	            }
	            else if(fH!=JSONObject.NULL) {
	                StyleBase styleBase = null;
	                if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
	                    styleBase = styleManager.getStyle(presentationStyleName, "presentation", contentAutoStyle);
	                }
	                else {
                        styleBase = styleManager.getStyle(getStyleName(), "graphic", contentAutoStyle);
	                }
	                if(styleBase!=null) {
	                    final GraphicProperties graphicProperties = ((IGraphicProperties)styleBase).getGraphicProperties();
	                    final String mirror = graphicProperties.getAttribute("style:mirror");
	                    if(mirror!=null&&mirror.contains("horizontal")) {
	                        flipH = Boolean.TRUE;
	                    }
	                }
	                if(flipH==null||!flipH.booleanValue()) {
	                    flipH = Boolean.TRUE;
	                }
	                else {
	                    flipH = Boolean.FALSE;
	                }
	                drawingAttrs.put("flipH", flipH);
	            }
	        }
	    }
        final JSONObject presentationProps = attrs.optJSONObject("presentation");
        if(presentationProps!=null) {
            final Object phType = presentationProps.opt("phType");
            if(phType==JSONObject.NULL) {
                attributes.remove("presentation:class");
                attributes.remove("presentation:placeholder");
                attributes.remove("presentation:user-transformed");
            }
            else if(phType instanceof String) {
                String presentationClass = null;
                switch((String)phType) {
                    case "chart" : presentationClass = "chart"; break;
                    case "pic" : presentationClass = "graphic"; break;
                    case "obj" : presentationClass = "object"; break;
                    case "dgm" : presentationClass = "orgchart"; break;
                    case "body" : presentationClass = "outline"; break;
                    case "subTitle" : presentationClass = "subtitle"; break;
                    case "tbl" : presentationClass = "table"; break;
//                  case "body" : presentationClass = "text"; break;
                    case "title": presentationClass = "title"; break;
                    case "dt" :  presentationClass = "date-time"; break;
                    case "ftr" : presentationClass = "footer"; break;
                    case "hdr" : presentationClass = "header"; break;
                    case "sldNum" : presentationClass = "page-number"; break;
                }
                if(presentationClass!=null) {
                    attributes.setValue(Namespaces.PRESENTATION, "class", "presentation:class", presentationClass);
                }
            }
            final Object userTransformed = presentationProps.opt("userTransformed");
            if(userTransformed==JSONObject.NULL) {
                attributes.remove("presentation:user-transformed");
            }
            else if(userTransformed instanceof Boolean) {
                attributes.setBooleanValue(Namespaces.PRESENTATION, "user-transformed", "presentation:user-transformed", (Boolean)userTransformed);
            }
            setPresentationStyleName(styleManager.applyAttributes("presentation", getPresentationStyleName(), contentAutoStyle, attrs));
        }
        else {
            if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
                setPresentationStyleName(styleManager.applyAttributes("presentation", presentationStyleName, contentAutoStyle, attrs));
            }
            else {
                setStyleName(styleManager.applyAttributes("graphic", getStyleName(), contentAutoStyle, attrs));
            }
        }

		final JSONObject drawingProps = attrs.optJSONObject("drawing");
        if(drawingProps != null) {
            if (drawingProps.has("name")) {
                String name = drawingProps.optString("name");
                if (name != null && !name.isEmpty()) {
                	attributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
                }
            }
            if (drawingProps.has("width")) {
                transformer.setWidth(drawingProps.optInt("width"));
            }
            if (drawingProps.has("height")) {
                transformer.setHeight(drawingProps.optInt("height"));
            }
            final Object rotation = drawingProps.opt("rotation");
            if(rotation!=null) {
                transformer.setRotation(rotation instanceof Number ? ((Number)rotation).doubleValue() : null);
            }
            if(operationDocument.getDocument() instanceof OdfTextDocument) {
                if(rootShape) {
                    if (drawingProps.has("anchorHorOffset")) {
                        transformer.setX(drawingProps.optInt("anchorHorOffset"));
                    }
                    if (drawingProps.has("anchorVertOffset")) {
                        transformer.setY(drawingProps.optInt("anchorVertOffset"));
                    }
                    applyTextRootShapeProps(drawingProps);
                }
                else {
                    if (drawingProps.has("left")) {
                        transformer.setX(drawingProps.optInt("left"));
                    }
                    if (drawingProps.has("top")) {
                        transformer.setY(drawingProps.optInt("top"));
                    }
                }
            }
            else {
                if (drawingProps.has("left")) {
                    transformer.setX(drawingProps.optInt("left"));
                }
                if (drawingProps.has("top")) {
                    transformer.setY(drawingProps.optInt("top"));
                }
            }
        }
        final JSONObject shapeProps = attrs.optJSONObject("shape");
        if(shapeProps!=null) {
            if (shapeProps.has("autoResizeHeight") && shapeProps.optBoolean("autoResizeHeight") == true) {
            	attributes.remove("svg:height");
            }
        }

		final IDrawing iDrawing = getDrawing();
		if(iDrawing!=null) {
			iDrawing.applyAttrsFromJSON(operationDocument, attrs, contentAutoStyle);
		}
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs, boolean contentAutoStyle) {

	    final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
		final Map<String, AttributeImpl> frameAttributes = getAttributes().getUnmodifiableMap();
        final String textStyleName = getTextStyleName();
        if(textStyleName!=null&&!textStyleName.isEmpty()) {
            styleManager.createAutoStyleAttributes(attrs, textStyleName, "paragraph", contentAutoStyle);
        }
        if(operationDocument.getDocument() instanceof OdfPresentationDocument) {
            final String presentationStyleName = getPresentationStyleName();
            if(presentationStyleName!=null&&!presentationStyleName.isEmpty()) {
                boolean createAutoStyles = false;
                if(contentAutoStyle) {
                    final String presentationClass = getPresentationClass();
                    if(("title").equals(presentationClass)||("outline").equals(presentationClass)) {
                        createAutoStyles = true;
                    }
                }
                if(createAutoStyles) {
                    styleManager.createAutoStyleAttributes(attrs, presentationStyleName, "presentation", contentAutoStyle);
                }
                else {
                    styleManager.createPresentationStyle(attrs, presentationStyleName, contentAutoStyle);
                }
            }
            else {
                final String styleName = getStyleName();
                if(styleName!=null&&!styleName.isEmpty()) {
                    styleManager.createAutoStyleAttributes(attrs, styleName, "graphic", contentAutoStyle);
                }
            }

            attrs.remove("styleId");

            final AttributeImpl presentationClassAttr = frameAttributes.get("presentation:class");
            if(presentationClassAttr!=null) {
                final String presentationClass = presentationClassAttr.getValue();
                if(presentationClass!=null) {
                    String phType = null;
                    switch(presentationClass) {
                        case "chart": phType = "chart"; break;
                        case "graphic": phType = "pic"; break;
                        case "object": phType = "obj"; break;
                        case "orgchart": phType = "dgm"; break;
                        case "outline": phType = "body"; break;
                        case "subtitle": phType = "subTitle"; break;
                        case "table": phType = "tbl"; break;
                        case "text":  phType = "body"; break;
                        case "title": phType = "title"; break;
                        // following classes are only usable with shapes on master pages
                        case "date-time": phType = "dt"; break;
                        case "footer": phType = "ftr"; break;
                        case "header": phType = "hdr"; break;
                        case "page-number": phType = "sldNum"; break;
                        // following classes are allowed on notes and handout pages only
                        case "page":
                        case "handout":
                        case "notes":
                            //
                        break;
                    }
                    if(phType!=null) {
                        attrs.getMap("presentation", true).put("phType", phType);
                    }
                }
                final AttributeImpl userDefined = frameAttributes.get("presentation:user-transformed");
                if(userDefined!=null) {
                    attrs.getMap("presentation", true).put("userTransformed", AttributesImpl.getBoolean(userDefined.getValue(), Boolean.FALSE));
                }
            }
        }
        else {
            final String styleName = getStyleName();
            if(styleName!=null&&!styleName.isEmpty()) {
                styleManager.createAutoStyleAttributes(attrs, styleName, "graphic", contentAutoStyle);
            }
        }

        final OpAttrs drawingAttrs = attrs.getMap("drawing", true);
        final boolean isTextRootShape = rootShape && operationDocument.getDocument() instanceof OdfTextDocument;
        
        drawingAttrs.put("width", Double.valueOf(transformer.getWidth()).intValue());
        drawingAttrs.put("height", Double.valueOf(transformer.getHeight()).intValue());
        drawingAttrs.put(isTextRootShape ? "anchorHorOffset" : "left", Double.valueOf(transformer.getX()).intValue());
        drawingAttrs.put(isTextRootShape ? "anchorVertOffset" : "top", Double.valueOf(transformer.getY()).intValue());

        final Double rotation = transformer.getRotation();
        if(rotation!=null) {
            drawingAttrs.put("rotation", rotation);
        }
        if(isTextRootShape) {
            createTextRootShapeProps(drawingAttrs);
        }
        final IDrawing iDrawing = getDrawing();
        if(iDrawing!=null) {
        	iDrawing.createAttrs(operationDocument, attrs, contentAutoStyle);
        }
	}
}
