/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.odf;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument.OdfXMLFile;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.DocumentProperties;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.FilterException.ErrorCode;
import com.openexchange.office.filter.core.DocumentComplexity;
import com.openexchange.office.filter.odf.styles.DocumentStyles;
import com.openexchange.office.filter.ods.dom.SpreadsheetContent;
import com.openexchange.office.imagemgr.ResourceManager;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.common.memory.MemoryObserver.MemoryListener;
import com.openexchange.office.tools.config.ConfigurationHelper;
import com.openexchange.session.Session;

/**
 * The access point to the ODF Toolkit, obfuscating all OX Office dependencies
 * from the Toolkit.
 *
 */
public class OdfOperationDoc {

	private static final Logger LOG = LoggerFactory.getLogger(OdfOperationDoc.class);
    private static final String STATUS_ID_FILE = "debug/statusid.txt";
    
    final private long FACTOR_FOR_REQUIRED_HEAPSPACE = 80;
    private static final String MODULE = "//module/";
    private static final String SPREADSHEET = "//spreadsheet/";
    private static final String DEBUG_OPERATIONS = "debugoperations";
    private static final String MAX_TABLE_COLUMNS = "maxTableColumns";
    private static final String MAX_TABLE_ROWS = "maxTableRows";
    private static final String MAX_TABLE_CELLS = "maxTableCells";
    private static final String MAX_SHEETS = "maxSheets";

    private static final String ORIGNAL_ODT_FILE = "debug/original.odt";
    private static final String OPERATION_REVISON_FILE = "debug/revision.txt";
    private static final String OPERATION_TEXT_FILE_PREFIX = "debug/operationUpdates_";

	private static boolean lowMemoryAbort = false;
	private static int successfulAppliedOperations = 0;
    private boolean isMetadataUpdated = false;

	private OdfDocument mDocument;

    private MemoryObserver memoryObserver = null;
	private MemoryListener memoryListener = null;
    private  DocumentProperties documentProperties;

    private boolean mSaveDebugOperations;
    private int mMaxTableColumnCount;
    private int mMaxTableRowCount;
    private int mMaxTableCellCount;
    private int mMaxSheetCount;

    public OdfOperationDoc(InputStream aInputDocumentStm) throws Exception {
        mDocument = OdfDocument.loadDocument(aInputDocumentStm);
        documentProperties = new DocumentProperties();
    }

    public OdfOperationDoc(InputStream documentStream, DocumentProperties documentProperties) throws Exception {
        this(null, documentStream, null, documentProperties);
    }

    public OdfOperationDoc(Session session, InputStream _inputDocumentStream, ResourceManager resourceManager, DocumentProperties configuration) throws Exception {
    	try {
    		documentProperties = configuration;

    		registerMemoryListener();

    		mSaveDebugOperations = Boolean.parseBoolean(ConfigurationHelper.getOfficeConfigurationValue(null, session, MODULE + DEBUG_OPERATIONS, "false"));
	        mMaxTableColumnCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, MODULE + MAX_TABLE_COLUMNS, 15);
	        mMaxTableRowCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, MODULE + MAX_TABLE_ROWS, 1500);
	        mMaxTableCellCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, MODULE + MAX_TABLE_CELLS, 1500);
	        mMaxSheetCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, SPREADSHEET + MAX_SHEETS, 256);

	        // complexity check... TODO: distinguish between odt and odf
	        final long maxAllowedXmlSizeMatchingComplexity = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, "//spreadsheet/maxCells", 500000) * 100;
        	final long maxAllowedXmlSizeMatchingMemory = MemoryObserver.calcMaxHeapSize(50) / FACTOR_FOR_REQUIRED_HEAPSPACE;
        	final InputStream inputDocumentStream = DocumentComplexity.checkDocumentComplexity(_inputDocumentStream, maxAllowedXmlSizeMatchingComplexity, maxAllowedXmlSizeMatchingMemory);

            mDocument = OdfDocument.loadDocument(inputDocumentStream, configuration);
            mDocument.getStyleManager().setResourceManager(resourceManager);
    	}
    	catch(Throwable e) {
    		removeMemoryListener();
    		throw getRethrowException(e);
    	}
    }

	public static boolean isLowMemoryAbort() {
	    return lowMemoryAbort;
	}
	public static void setLowMemoryAbort(boolean l) {
	    lowMemoryAbort = l;
	}
	public static void abortOnLowMemory() {
	    if(lowMemoryAbort) {
	        throw new RuntimeException();
	    }
	}
	public static void setSuccessfulAppliedOperations(int opCount) {
		successfulAppliedOperations = opCount;
	}
	public static int getSuccessfulAppliedOperations() {
		return successfulAppliedOperations;
	}
    public void registerMemoryListener() {
        MemoryObserver memObserver = getMemoryObserver();
        if(memObserver!=null) {
            if(memoryListener==null) {
                memoryListener = new MemoryListener() {
                    @Override
                    public void memoryTresholdExceeded(long usedMemory, long maxMemory) {
                        notifyLowMemory();
                    }
                };
                memObserver.addListener(memoryListener);
            }
            if(memoryObserver.isUsageThresholdExceeded()) {
                notifyLowMemory();
            }
        }
    }

    public void removeMemoryListener() {
        if(memoryListener!=null) {
            MemoryObserver memObserver = getMemoryObserver();
            if(memObserver!=null) {
                memObserver.removeListener(memoryListener);
                memoryListener = null;
            }
        }
        memoryObserver = null;
    }

    // take care... this function is called from the low-memory
    // thread, so breakpoints will not work... It is called before
    // gc is started and or an OutOfMemory Exception is thrown, so we
    // try to free up as much as possible ... the filter will then
    // most probably exit with an exception
    private void notifyLowMemory() {
        LOG.debug("ODF Filter: Low memory notification received");
        setLowMemoryAbort(true);
        if(mDocument!=null) {
	        // trying to free memory ...
        	try {
        		final DocumentStyles stylesDom = mDocument.getStylesDom();
        		stylesDom.abort();
        	}
        	catch(SAXException e) {
        	    //
        	}
        	if(mDocument instanceof OdfSpreadsheetDocument) {
        		try {
        			final SpreadsheetContent content = (SpreadsheetContent)((OdfSpreadsheetDocument)mDocument).getContentDom();
        			content.getSheets().clear();
        		}
        		catch(SAXException e) {
        		    //
        		}
        	}
        	final OdfPackage odfPackage = mDocument.getPackage();
	        if(odfPackage!=null) {
	            odfPackage.freeMemory();
	        }
        }
    }

    public static FilterException getRethrowException(Throwable e) {
    	FilterException ret;
        if(isLowMemoryAbort()) {
            ret = new FilterException("ODS FilterException...", ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
        }
        else if(e instanceof FilterException) {
    		ret = (FilterException)e;
    	}
    	else if (e instanceof java.util.zip.ZipException && e.getMessage().contains("only DEFLATED entries can have EXT descriptor")) {
            ret = new FilterException(e, ErrorCode.UNSUPPORTED_ENCRYPTION_USED);
        }
    	else if (e instanceof OutOfMemoryError) {
    		ret = new FilterException(e, ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
    	}
    	else {
    		ret = new FilterException(e, ErrorCode.CRITICAL_ERROR);
    	}
    	ret.setOperationCount(OdfOperationDoc.getSuccessfulAppliedOperations());
    	return ret;
    }

    /**
     * Receives the (known) operations of the ODF text document
     *
     * @return the operations as JSON
     */
    // ToDo OX - JSONObject is to be considered..
    public JSONObject getOperations()
    	throws SAXException, JSONException, FilterException {

        JSONObject ops = mDocument.getOperations(this);
        if (ops != null && ops.length() > 0) {
            LOG.debug("\n\n*** ALL OPERATIONS:\n{0}", ops.toString());
        } else {
            LOG.debug("\n\n*** ALL OPERATIONS:\nNo Operation have been extracted!");
        }
        return ops;
    }

    public int getMaxTableColumnsCount() {
        return mMaxTableColumnCount;
    }

    public int getMaxTableRowsCount() {
        return mMaxTableRowCount;
    }

    public int getMaxTableCellCount() {
        return mMaxTableCellCount;
    }

    public int getMaxSheetCount() {
        return mMaxSheetCount;
    }

    public DocumentProperties getDocumentProperties() {
        return documentProperties;
    }

    public void initPartitioning(DocumentProperties documentProperties) {
		mDocument.initPartitioning(documentProperties);
	}

    public Map<String, Object> getMetaData(DocumentProperties documentProperties)
		throws Exception {

		return mDocument.getMetaData(documentProperties);
	}

	public Map<String, Object> getActivePart(DocumentProperties documentProperties)
		throws Exception {

		return mDocument.getActivePart(documentProperties);
	}

	public Map<String, Object> getNextPart(DocumentProperties documentProperties)
		throws Exception {

		return mDocument.getNextPart(documentProperties);
	}

    /**
     * Applies the (known) operations to upon the latest state of the ODF text
     * document
     *
     * @param operationString ODF operations as String
     * @return the number of operations being accepted
     * @throws Exception
     */
    public int applyOperations(String operationString) throws Exception {
        JSONObject operations = new JSONObject();
        try {
            operations.put("operations", new JSONArray(new JSONTokener(operationString)));
        } catch (Exception ex) {
            LOG.error(null, ex);
        }
        int operationsCount = this.applyOperations(operations);
        // if the document was altered
        if (operationsCount > 0) {
            // remove the cached view
            removeCachedView();
        }
        return operationsCount;
    }

    private void removeCachedView() {
        final OdfPackage odfPackage = getDocument().getPackage();
        if(odfPackage!=null) {
            odfPackage.remove("Thumbnails/thumbnail.png");
        }
    }
    
    /**
     * Applies the (known) operations to upon the latest state of the ODF text
     * document
     *
     * @param operations ODF operations as JSONArray within an JSONObject with
     * "operations" key.
     * @return the number of operations being accepted
     */
    public int applyOperations(JSONObject operations) throws Exception {
        LOG.debug("\n*** EDIT OPERATIONS:\n{0}", operations.toString());
//      System.err.println("\n*** EDIT OPERATIONS:\n" + operations.toString());
        final JSONArray ops = operations.getJSONArray("operations");
        if (mSaveDebugOperations) {
            addOriginalOdfAsDebug();
            addOperationFileAsDebug(ops);
        }
        final int operationCount = getDocument().applyOperations(this, ops);
        if(operationCount > 0) {
            // remove the cached view
            removeCachedView();
            if (!isMetadataUpdated) {
                mDocument.updateMetaData();
                isMetadataUpdated = true;
            }
        }
        return operationCount;
    }

    private void addOriginalOdfAsDebug() 
        throws SAXException {

        OdfPackage pkg = mDocument.getPackage();
        // if there is not already an orignal file being stored
        if (!pkg.contains(ORIGNAL_ODT_FILE)) {
            LOG.debug("Adding original ODT document as debug within the zip at " + ORIGNAL_ODT_FILE);
            try {
                // ..from the ODF ZIP
                pkg.insert(pkg.getInputStream(), ORIGNAL_ODT_FILE, "application/vnd.oasis.opendocument.text");
            } catch (IOException ex) {
                LOG.error(null, ex);
            }
        }
    }

    private void addOperationFileAsDebug(JSONArray operations) {
        // serialize the operations as String (using ascii characters only) and indent a line for every new operations (heuristic: every array item will be split into new line)
        try {
            OdfPackage pkg = mDocument.getPackage();
            // start with zero to always increment (either read a default by file or new)
            int revisionNo = 0;
            // if there was already a revision, get it..
            if (pkg.contains(OPERATION_REVISON_FILE)) {
                // ..from the ODF ZIP
                byte[] revisionByteArray = pkg.getBytes(OPERATION_REVISON_FILE);
                if (revisionByteArray != null && revisionByteArray.length != 0) {
                    BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(revisionByteArray)));
                    // read the first line of the file, only containing one number
                    String firstLine = reader.readLine();
                    // map it to a number
                    revisionNo = Integer.parseInt(firstLine);
                    LOG.debug("Found an existing revision number:{0}", revisionNo);
                }
            } else {
                LOG.debug("Created a new revision number: 1");
            }
            // always increment, so even a new file starts with the revision number "1"
            revisionNo++;
            pkg.insert(operations.toString().getBytes(), OPERATION_TEXT_FILE_PREFIX + revisionNo + ".txt", "text/plain");
            pkg.insert(Integer.toString(revisionNo).getBytes(), OPERATION_REVISON_FILE, "text/plain");
        } catch (Exception ex) {
            LOG.error(null, ex);
        }
    }

    /**
     * writes the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    void writeStatusID(String statusId) {
        OdfPackage pkg = mDocument.getPackage();
        LOG.debug("Overwriting ODT document status ID " + STATUS_ID_FILE);
        pkg.insert(statusId.getBytes(), STATUS_ID_FILE, "text/plain");
    }

    /**
     * Read the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    String readStatusID() {
        OdfPackage pkg = mDocument.getPackage();
        String statusId = null;
        if (pkg != null && pkg.contains(STATUS_ID_FILE)) {
            // ..from the ODF ZIP
            byte[] revisionByteArray = pkg.getBytes(STATUS_ID_FILE);
            if (revisionByteArray != null && revisionByteArray.length != 0) {
                BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(revisionByteArray)));
                // read the first line of the file, only containing one number
                try {
                    statusId = reader.readLine();
                } catch (IOException e) {
                    LOG.debug("No existing status ID file found!");
                }
                LOG.debug("Found an existing status ID:{0}", statusId);
            }
        }
        return statusId;
    }

    public long getContentSize() {
        if(mDocument==null) {
            return 0;
        }
        final OdfPackage odfPackage = mDocument.getPackage();
        return odfPackage!=null ? odfPackage.getSize(OdfXMLFile.CONTENT.getFileName()) : 0;
    }

    /**
     * Returns the TextOperationDocument encapsulating the DOM view
     *
     * @return ODF text document
     */
    public OdfDocument getDocument() {
        return mDocument;
    }

    /**
     * Close the OdfPackage and release all temporary created data. After
     * execution of this method, this class is no longer usable. Do this as the
     * last action to free resources. Closing an already closed document has no
     * effect.
     */
    public void close() {
        mDocument.close();
    }

    private MemoryObserver getMemoryObserver () {
        if (memoryObserver == null) {
            memoryObserver = MemoryObserver.getMemoryObserver();
        }
        return memoryObserver;
    }
}
