/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core.chart;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONObject;

public class Chart extends AttributeSet {

    private final List<ChartSeries> series = new ArrayList<>();
    private final AttributeSet legend = new AttributeSet();
    private final Map<String, ChartAxis> axes = new HashMap<>();

    private final AttributeSet mainTitle = new AttributeSet();

    public Chart() {

    }

    public void createAttributes(JSONObject target) {
        JSONObject attrs = getAttributes();
        mergeAttributes(target, attrs);
    }

    public void createInsertOperations(int sheetIndex, JSONArray operationQueue, int drawingIndex) {
        JSONArray start = new JSONArray();
        start.put(sheetIndex).put(drawingIndex);

        int seriesIndex = 0;
        for (ChartSeries serie : series) {
            serie.createInsertOperation(start, operationQueue, seriesIndex);
            seriesIndex++;
        }

        if (!mainTitle.getAttributes().isEmpty()) {
            JSONObject op = new JSONObject();
            op.putSafe("start", start);
            op.putSafe("axis", "main");
            op.putSafe("attrs", mainTitle.getAttributes());
            op.putSafe("name", "setChartTitleAttributes");
            operationQueue.put(op);
        }

        if (!legend.getAttributes().isEmpty()) {
            JSONObject op = new JSONObject();
            op.putSafe("start", start);
            op.putSafe("attrs", legend.getAttributes());
            op.putSafe("name", "setChartLegendAttributes");
            operationQueue.put(op);
        }

        for (ChartAxis axis : axes.values()) {
            axis.createInsertOperation(start, operationQueue);
        }
    }

    public void setChartStyleId(int chartStyleId) {
        JSONObject chart = getAttributes().optJSONObject("chart");
        if (chart == null) {
            chart = new JSONObject();
            getAttributes().putSafe("chart", chart);
        }
        chart.putSafe("chartStyleId", chartStyleId);
    }

    public ChartSeries addDataSeries(ChartSeries serie) {
        series.add(serie);
        return serie;
    }

    public ChartSeries insertDataSeries(int seriesIndex, ChartSeries serie) {
        series.add(seriesIndex, serie);
        return serie;
    }

    public ChartSeries setDataSeriesAttributes(int seriesIndex, JSONObject attrs) {
        ChartSeries serie = series.get(seriesIndex);
        serie.setAttributes(attrs);
        return serie;
    }

    public ChartSeries deleteDataSeries(int seriesIndex) {
        return series.remove(seriesIndex);
    }

    public AttributeSet getLegend() {
        return legend;
    }

    public void setAxis(ChartAxis axis) {
        axes.put(axis.getAxis(), axis);
    }

    public ChartAxis getAxis(String axisId) {
        ChartAxis res = axes.get(axisId);
        if (res == null) {
            res = new ChartAxis();
            res.setAxis(axisId);
            setAxis(res);
        }
        return res;
    }

    public AttributeSet getTitle(String axisId) {
        if (axisId.equals("main")) {
            return mainTitle;
        }
        return getAxis(axisId).getTitle();
    }

    public List<ChartSeries> getSeries() {
        return Collections.unmodifiableList(series);
    }

    public boolean isScatter() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isScatter();
    }

    public boolean isBubble() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isBubble();
    }

    public boolean isPieOrDonut() {
        if (series.isEmpty()) {
            return false;
        }
        return series.get(0).isPie() || series.get(0).isDonut();
    }
}
