/*
 *  Copyright 2007-2008, Plutext Pty Ltd.
 *
 *  This file is part of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.

    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.

 */
package org.docx4j.jaxb;

import java.util.HashMap;
import java.util.Iterator;
import java.util.TreeSet;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import org.docx4j.dml.CTCustomGeometry2D;
import org.docx4j.dml.STShapeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Context {

    private static Thread jcThread;
    private static boolean jcThreadJoined = false;

    private static JAXBContext jc;

	private static JAXBContext jcDocPropsCore;
	private static JAXBContext jcDocPropsCustom;
	private static JAXBContext jcDocPropsExtended;
	private static JAXBContext jcRelationships;
	private static JAXBContext jcCustomXmlProperties;
	private static JAXBContext jcContentTypes;

	private static JAXBContext jcXmlPackage;

	private static JAXBContext jcSectionModel;

	private static JAXBContext jcXmlDSig;

	private static HashMap<STShapeType, CTCustomGeometry2D> customGeometries;

	/** @since 3.0.1 */
	private static JAXBContext jcMCE;

	private static Logger log = LoggerFactory.getLogger(Context.class);

	private static boolean lowMemoryAbort = false;

	private static int markupIdCurrent = 0;
	private static TreeSet<Integer> markupIdSet = new TreeSet<Integer>();

	static {
		// Diagnostics regarding JAXB implementation
		try {
			Object namespacePrefixMapper = NamespacePrefixMapperUtils.getPrefixMapper();
			if ( namespacePrefixMapper.getClass().getName().equals("org.docx4j.jaxb.NamespacePrefixMapperSunInternal") ) {
				// Java 6
				log.info("Using Java 6/7 JAXB implementation");
			} else {
				log.info("Using JAXB Reference Implementation");
			}
		} catch (JAXBException e) {
			log.error("PANIC! No suitable JAXB implementation available");
			log.error(e.getMessage(), e);
			e.printStackTrace();
		}

		jcThread = new Thread(
            new Runnable() {
                @Override
                public void run() {
                    try {
                        // JAXBContext.newInstance uses the context class loader of the current thread.
                        // To specify the use of a different class loader,
                        // either set it via the Thread.setContextClassLoader() api
                        // or use the newInstance method.
                        // JBOSS (older versions only?) might use a different class loader to load JAXBContext,
                        // which caused problems, so in docx4j we explicitly specify our class loader.
                        // IKVM 7.3.4830 also needs this to be done
                        // (java.lang.Thread.currentThread().setContextClassLoader doesn't seem to help!)
                        // and there are no environments in which this approach is known to be problematic

                        java.lang.ClassLoader classLoader = Context.class.getClassLoader();

                        jc = JAXBContext.newInstance("org.docx4j.wml:org.docx4j.w14:org.docx4j.w15:" +
                                "org.docx4j.schemas.microsoft.com.office.word_2006.wordml:" +
                                "org.docx4j.dml:org.docx4j.dml.chart:org.docx4j.dml.chartStyle2012:org.docx4j.dml.chartDrawing:org.docx4j.dml.compatibility:org.docx4j.dml.diagram:org.docx4j.dml.lockedCanvas:org.docx4j.dml.picture:org.docx4j.dml.wordprocessingDrawing:org.docx4j.dml.spreadsheetdrawing:org.docx4j.dml.diagram2008:" +
                                // All VML stuff is here, since compiling it requires WML and DML (and MathML), but not PML or SML
                                "org.docx4j.vml:org.docx4j.vml.officedrawing:org.docx4j.vml.wordprocessingDrawing:org.docx4j.vml.presentationDrawing:org.docx4j.vml.spreadsheetDrawing:org.docx4j.vml.root:" +
                                "org.docx4j.docProps.coverPageProps:" +
                                "org.opendope.xpaths:org.opendope.conditions:org.opendope.questions:org.opendope.answers:org.opendope.components:org.opendope.SmartArt.dataHierarchy:" +
                                "org.docx4j.math:" +
                                "org.docx4j.sharedtypes:org.docx4j.bibliography:org.docx4j.mce:" +
                            "org.docx4j.dml2010:org.docx4j.dml.wordprocessingCanvas2010:org.docx4j.dml.wordprocessingGroup2010:org.docx4j.dml.wordprocessingShape2010" + ":org.docx4j.dml.chartex2014", classLoader);

                        if (jc.getClass().getName().equals("org.eclipse.persistence.jaxb.JAXBContext")) {
                            log.debug("MOXy JAXB implementation is in use!");
                        } else {
                            log.debug("Not using MOXy; using " + jc.getClass().getName());
                        }
                        jcDocPropsCore = JAXBContext.newInstance("org.docx4j.docProps.core:org.docx4j.docProps.core.dc.elements:org.docx4j.docProps.core.dc.terms",classLoader );
                        jcDocPropsCustom = JAXBContext.newInstance("org.docx4j.docProps.custom",classLoader );
                        jcDocPropsExtended = JAXBContext.newInstance("org.docx4j.docProps.extended",classLoader );
                        jcXmlPackage = JAXBContext.newInstance("org.docx4j.xmlPackage",classLoader );
                        jcRelationships = JAXBContext.newInstance("org.docx4j.relationships",classLoader );
                        jcCustomXmlProperties = JAXBContext.newInstance("org.docx4j.customXmlProperties",classLoader );
                        jcContentTypes = JAXBContext.newInstance("org.docx4j.openpackaging.contenttype",classLoader );

                        jcSectionModel = JAXBContext.newInstance("org.docx4j.model.structure.jaxb",classLoader );

                        jcXmlDSig = JAXBContext.newInstance("org.plutext.jaxb.xmldsig",classLoader );

                        jcMCE = JAXBContext.newInstance("org.docx4j.mce",classLoader );

                   		log.debug(".. other contexts loaded ..");


                    } catch (Exception ex) {
                        log.error("Cannot initialize context", ex);
                    }
                }
            }
        );
        jcThread.start();
	}
	public static boolean isLowMemoryAbort() {
	    return lowMemoryAbort;
	}
	public static void setLowMemoryAbort(boolean l) {
	    lowMemoryAbort = l;
	}
	public static void abortOnLowMemory() {
	    if(lowMemoryAbort) {
	        throw new RuntimeException();
	    }
	}

	/*
	 * the jcThread is joined, additional it is checked if it should
	 * aborted because of low memory
	 *
	 */
	private static final void initializeContexts() {
	    abortOnLowMemory();
        if(!jcThreadJoined) {
            try {
                jcThread.join();
            } catch (InterruptedException e) {
                log.info("DOCX4J, could not get Context");
            }
            jcThreadJoined = true;
        }
	}

    public static final JAXBContext getJc() {
        initializeContexts();
        return jc;
    }

    public static final JAXBContext getJcDocPropsCore() {
        initializeContexts();
        return jcDocPropsCore;
    }

    public static final JAXBContext getJcDocPropsCustom() {
        initializeContexts();
        return jcDocPropsCustom;
    }

    public static final JAXBContext getJcDocPropsExtended() {
        initializeContexts();
        return jcDocPropsExtended;
    }

    public static final JAXBContext getJcRelationships() {
        initializeContexts();
        return jcRelationships;
    }

    public static final JAXBContext getJcCustomXmlProperties() {
        initializeContexts();
        return jcCustomXmlProperties;
    }

    public static final JAXBContext getJcContentTypes() {
        initializeContexts();
        return jcContentTypes;
    }

    public static final JAXBContext getJcXmlPackage() {
        initializeContexts();
        return jcXmlPackage;
    }

    public static final JAXBContext getJcSectionModel() {
        initializeContexts();
        return jcSectionModel;
    }

    public static final JAXBContext getJcXmlDSig() {
        initializeContexts();
        return jcXmlDSig;
    }

    public static final JAXBContext getJcMCE() {
        initializeContexts();
        return jcMCE;
    }

    public static final CTCustomGeometry2D getCustomShapeForPreset(STShapeType presetType) {
    	if(customGeometries==null) {
    		customGeometries = CustomGeometryFactory.createCustomGeometries();
    	}
    	return customGeometries.get(presetType);
    }

    public static final HashMap<STShapeType, CTCustomGeometry2D> getCustomShapePresets() {
    	if(customGeometries==null) {
    		customGeometries = CustomGeometryFactory.createCustomGeometries();
    	}
    	return customGeometries;
    }

    private static org.docx4j.wml.ObjectFactory wmlObjectFactory;
	public static org.docx4j.wml.ObjectFactory getWmlObjectFactory() {

	    abortOnLowMemory();
		if (wmlObjectFactory==null) {
			wmlObjectFactory = new org.docx4j.wml.ObjectFactory();
		}
		return wmlObjectFactory;
	}

    private static org.pptx4j.pml.ObjectFactory pmlObjectFactory;
    public static org.pptx4j.pml.ObjectFactory getpmlObjectFactory() {

        abortOnLowMemory();
        if (pmlObjectFactory==null) {
            pmlObjectFactory = new org.pptx4j.pml.ObjectFactory();
        }
        return pmlObjectFactory;
    }

    private static org.docx4j.dml.ObjectFactory dmlObjectFactory;
    public static org.docx4j.dml.ObjectFactory getDmlObjectFactory() {

        abortOnLowMemory();
        if (dmlObjectFactory==null) {
            dmlObjectFactory = new org.docx4j.dml.ObjectFactory();
        }
        return dmlObjectFactory;
    }

    private static org.docx4j.dml.chart.ObjectFactory dmlChartObjectFactory;
    public static org.docx4j.dml.chart.ObjectFactory getDmlChartObjectFactory() {

        abortOnLowMemory();
        if(dmlChartObjectFactory==null) {
            dmlChartObjectFactory = new org.docx4j.dml.chart.ObjectFactory();
        }
        return dmlChartObjectFactory;
    }

    private static org.docx4j.vml.wordprocessingDrawing.ObjectFactory vmlWordprocessingDrawingObjectFactory;
    public static org.docx4j.vml.wordprocessingDrawing.ObjectFactory getVmlWordprocessingDrawingObjectFactory() {

        abortOnLowMemory();
        if(vmlWordprocessingDrawingObjectFactory==null) {
            vmlWordprocessingDrawingObjectFactory = new org.docx4j.vml.wordprocessingDrawing.ObjectFactory();
        }
        return vmlWordprocessingDrawingObjectFactory;
    }

    private static org.docx4j.dml.spreadsheetdrawing.ObjectFactory dmlSpreadsheetDrawingObjectFactory;
    public static org.docx4j.dml.spreadsheetdrawing.ObjectFactory getDmlSpreadsheetDrawingObjectFactory() {

        abortOnLowMemory();
        if(dmlSpreadsheetDrawingObjectFactory==null) {
            dmlSpreadsheetDrawingObjectFactory = new org.docx4j.dml.spreadsheetdrawing.ObjectFactory();
        }
        return dmlSpreadsheetDrawingObjectFactory;
    }

	public static int getNextMarkupId() {
		final Iterator<Integer> markupIdIter = markupIdSet.tailSet(markupIdCurrent++).iterator();
		while(markupIdIter.hasNext()) {
			final Integer next = markupIdIter.next();
			if(markupIdCurrent < next.intValue()) {
				return markupIdCurrent;
			}
			markupIdCurrent++;
		}
		return markupIdCurrent;
	}

	public static void setMarkupId(Integer id) {
		markupIdSet.add(id);
	}

	public static void clearMarkupIdSet() {
		markupIdCurrent = 4000000;
		markupIdSet.clear();
	}
}
