package com.openexchange.office.tools.htmldoc;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;

public class Paragraph
        implements
        INode,
        ISplitpoint
{

    // private final static Logger LOG = LoggerFactory.getLogger(Paragraph.class);

    public final static String  EMPTYSPAN             = "<span nonjquerydata=\"{&quot;isempty&quot;:true}\"></span><br jquerydata=\"{&quot;dummy&quot;:true}\"/>";
    public final static String  PARASTART             = "<div class=\"p\" ";
    private static final String PARASTARTWITHOUTCLASS = "<div class=\"p";
    private final static String DEFPARAATTRS          = "text-align: left; border: none; padding: 0px 0px 0px 0mm; margin: 0mm 0mm 3.52mm; text-indent: 0mm; background-color: transparent;";
    private final static String DEFPARAATTRSINTABLE   = "text-align: left; border: none; padding: 0px 0px 0px 0mm; margin: 0mm 0mm 0mm; text-indent: 0mm; background-color: transparent;";

    // ///////////////////////////////////////////////////////////////////////////////////////////

    private final List<INode>   nodes                 = new ArrayList<INode>();
    private boolean             inTable               = false;
    private boolean             isImplicit            = false;
    private JSONObject          myAttrs               = null;
    private boolean             splitpoint;
    private boolean             manualPageBreak       = false;

    public Paragraph()
    {

    }

    @Override
    public final
            boolean
            isSplitpoint()
    {
        return splitpoint;
    }

    @Override
    public final
            void
            setSplitpoint(
                boolean splitpoint)
    {
        this.splitpoint = splitpoint;
    }

    @Override
    public final
            boolean
            setAttribute(
                JSONArray subStartIndex,
                JSONArray subEndIndex,
                JSONObject attrs)
                throws Exception
    {

        if (subEndIndex.length() == 0)
        {
            if (myAttrs != null)
            {
                throw new Exception("Paragraph cant be attributed twice");
            }
            myAttrs = attrs;
            return true;
        }
        else
        {

            int startTextIndex = subStartIndex.getInt(0);
            final int endTextIndex = subEndIndex.getInt(0);

            int current = 0;

            for (final INode node : nodes)
            {
                if (current == startTextIndex)
                {
                    final JSONObject na = node.getAttribute();
                    final int awaitedEnd = current + node.getTextLength() - 1;
                    if (endTextIndex < awaitedEnd)
                    {
                        throw new Exception(endTextIndex + " " + awaitedEnd + " " + nodes);
                    }
                    Validate.isTrue(endTextIndex >= awaitedEnd, "awaitedIndex is not correct!");

                    // Validate.isTrue(na == null, "node attributes is alreday set. you have to merge? " + node.getClass().getSimpleName() + " " + attrs + " " + na);

                    if (na != null && subStartIndex.length() == 1)
                    {
                        Set<String> keys = na.keySet();
                        for (String key : keys)
                        {
                            if (attrs.has(key) && !ObjectUtils.equals(attrs.get(key), na.get(key)))
                            {
                                Logger log = TextHtmlDocumentBuilder.getLogger();
                                log.debug("FASTLOAD: Merging of attributes not supported");
                                // return true;
                                throw new Exception("FASTLOAD: Merging of attributes not supported current: " + na + " new: " + attrs);
                            }
                            attrs.put(key, na.get(key));
                        }
                    }
                    if (endTextIndex == awaitedEnd)
                    {
                        return node.setAttribute(GenDocHelper.shiftedCopy(subStartIndex), GenDocHelper.shiftedCopy(subEndIndex), attrs);
                    }
                    else
                    {
                        node.setAttribute(GenDocHelper.shiftedCopy(subStartIndex), GenDocHelper.shiftedCopy(subEndIndex), attrs);
                        int nodeLength = node.getTextLength();
                        startTextIndex += nodeLength;
                        current += nodeLength;
                        continue;
                    }

                    // break;
                }
                else
                {
                    current += node.getTextLength();
                }
            }
        }
        return false;
    }

    @Override
    public
            void
            insert(
                JSONArray start,
                INode newChild)
                throws Exception
    {
        if (start.length() == 1)
        {
            nodes.add(newChild);

            if (newChild instanceof SubNode)
            {
                if (((SubNode) newChild).hasManualPageBreak())
                {
                    this.manualPageBreak = true;
                }
            }
        }
        else
        {
            int startTextIndex = start.getInt(0);

            int current = 0;

            for (final INode node : nodes)
            {
                if (current == startTextIndex)
                {
                    start = GenDocHelper.shiftedCopy(start);
                    node.insert(start, newChild);
                    return;
                }
                else
                {
                    current += node.getTextLength();
                }
            }
            throw new Exception("insert is not handled " + current + " " + newChild);
        }
    }

    public final
            void
            shrinkTextes()
                throws Exception
    {

        final int nodeCount = nodes.size();
        for (int i = nodeCount - 2; i >= 0; i--)
        {
            final INode currentNode = nodes.get(i);
            final INode afterNode = nodes.get(i + 1);

            if (currentNode instanceof Text && afterNode instanceof Text)
            {
                final Text currentText = (Text) currentNode;
                final Text afterText = (Text) afterNode;

                boolean equals = true;

                final JSONObject currentAttrs = currentNode.getAttribute();
                final JSONObject afterAttrs = afterNode.getAttribute();
                if (currentAttrs != afterAttrs)
                {
                    if (currentAttrs == null || afterAttrs == null)
                    {
                        equals = false;
                    }
                    else
                    {
                        equals = currentAttrs.isEqualTo(afterAttrs);
                    }
                }

                if (equals)
                {
                    nodes.remove(i);
                    nodes.remove(i);
                    final Text newText = new Text(currentText.getText() + afterText.getText(), currentText.getParagraphPos());
                    newText.setAttribute(null, null, currentNode.getAttribute());
                    nodes.add(i, newText);
                }

            }

        }
    }

    protected
            void
            appendHTMLCLasses(
                StringBuilder document)
                throws Exception
    {
        if (splitpoint && manualPageBreak)
        {
            document.append(" splitpoint manual-page-break");
        }
        else if (manualPageBreak)
        {
            document.append(" manual-page-break");
        }
        else if (splitpoint)
        {
            document.append(" splitpoint");
        }
    }

    protected
            void
            appendHTMLAttributes(
                StringBuilder document)
                throws Exception
    {

    }

    protected
            void
            appendHTMLStyles(
                StringBuilder document)
                throws Exception
    {

        if (inTable)
        {
            document.append(DEFPARAATTRSINTABLE);
        }
        else
        {
            document.append(DEFPARAATTRS);
        }

    }

    @Override
    public final
            boolean
            appendContent(
                StringBuilder document)
                throws Exception
    {

        document.append(PARASTARTWITHOUTCLASS);

        appendHTMLCLasses(document);

        document.append("\"");

        appendHTMLAttributes(document);

        GenDocHelper.appendAttributes(myAttrs, document, isImplicit);

        document.append(" style=\"");
        appendHTMLStyles(document);
        document.append("\">");

        int len = getTextLength();

        if (len > 0)
        {
            shrinkTextes();

            // now iterate through the paragraph nodes and create the node
            // specific html parts
            boolean addEmptySpan = true;
            boolean contentAdded = false;
            boolean firstNode = true;

            for (final INode node : nodes)
            {
                if (!firstNode && !contentAdded && node.needsEmptySpan())
                {
                    // In case our previous node didn't add content and the
                    // current node MUST have a previous node, we have to force
                    // the creation now setting addEmptySpan to true.
                    // e.g. happens in the case: <drawing node><tab node> where
                    // the drawing node doesn't add an empty span and is inserted
                    // later via operations where the code assumes that there is
                    // an empty text node before the tab!
                    addEmptySpan = true;
                }
                addEmptySpan = GenDocHelper.addEmptySpanForNecessaryNode(document, node, addEmptySpan);
                contentAdded |= node.appendContent(document);
                firstNode = false;
            }
            if (addEmptySpan || !contentAdded)
            {
                addEmptySpan(document);
            }
        }
        else
        {
            document.append(EMPTYSPAN);
        }

        document.append("</div>");
        return true;
    }

    protected void addEmptySpan(StringBuilder document) {
        GenDocHelper.addEmptySpan(document, true);
    }

    @Override
    public
            String
            toString()
    {
        return "Paragraph " + nodes;
    }

    public final
            void
            setInTable(
                boolean inTable)
    {
        this.inTable = inTable;
    }

    public final
            void
            setImplicit(
                boolean isImplicit)
    {
        this.isImplicit = isImplicit;
    }

    public final
            List<JSONObject>
            getReducedOps(
                JSONArray paraPos)
                throws Exception
    {
        final int currentCount = nodes.size();

        shrinkTextes();

        if (nodes.size() < currentCount)
        {
            final List<JSONObject> newOps = new ArrayList<JSONObject>();

            // old great working code

            int start = 0;
            for (final INode node : nodes)
            {

                final JSONObject insert = new JSONObject();
                final JSONArray insertPos = new JSONArray(paraPos.asList());
                insertPos.put(start);
                insert.put("start", insertPos);

                start += node.getTextLength();

                if (node instanceof Text)
                {
                    final Text text = (Text) node;
                    insert.put("text", text.getText());
                    insert.put("name", "insertText");
                }
                else if (node instanceof HardBreak)
                {
                    final HardBreak hardbreak = (HardBreak) node;
                    insert.put("type", hardbreak.getType());
                    insert.put("name", "insertHardBreak");
                }
                else
                {
                    String opName = "insert" + node.getClass().getSimpleName();
                    insert.put("name", opName);
                }

                newOps.add(insert);

                if (node.getAttribute() != null)
                {
                    final JSONObject setAtt = new JSONObject();
                    setAtt.put("start", insertPos);

                    final JSONArray end = new JSONArray(paraPos.asList());
                    end.put(start - 1);
                    setAtt.put("end", end);
                    setAtt.put("attrs", node.getAttribute());
                    setAtt.put("name", "setAttributes");

                    newOps.add(setAtt);
                }
            }

            return newOps;
        }

        return null;
    }

    @Override
    public final
            int
            getTextLength()
    {
        int res = 0;
        for (INode sn : nodes)
        {
            res += sn.getTextLength();
        }
        return res;
    }

    @Override
    public final
            boolean
            needsEmptySpan()
    {
        return false;
    }

    @Override
    public final
            JSONObject
            getAttribute()
    {
        return myAttrs;
    }
}
