/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.doc;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Defines certain paper sizes and helper functions.
 *
 * @author Carsten Driesner
 * @since 7.8.3
 */
public enum PaperSize {

    A4,
    LETTER;

    /**
     * Static unmodifiable map language to default paper size.
     * Don't forget to update getPageSize() if you introduce
     * a new paper size entry!!
     */
    public static final Map<String, PaperSize> PAPERSIZE_LANGMAP;
    static {
        // supported locales with the default format
        final Map<String, PaperSize> tmp = new HashMap<String, PaperSize>();
        tmp.put("en-US", PaperSize.LETTER);
        tmp.put("en-GB", PaperSize.A4);
        tmp.put("de-DE", PaperSize.A4);
        tmp.put("fr-FR", PaperSize.A4);
        tmp.put("fr-CA", PaperSize.LETTER);
        tmp.put("es-ES", PaperSize.A4);
        tmp.put("es-MX", PaperSize.LETTER);
        tmp.put("nl-NL", PaperSize.A4);
        tmp.put("pl-PL", PaperSize.A4);
        tmp.put("ja-JP", PaperSize.A4);
        tmp.put("it-IT", PaperSize.A4);
        tmp.put("zh-CN", PaperSize.A4);
        tmp.put("zh-TW", PaperSize.A4);
        tmp.put("hu-HU", PaperSize.A4);
        tmp.put("sk-SK", PaperSize.A4);
        tmp.put("cs-CZ", PaperSize.A4);
        tmp.put("lv-LV", PaperSize.A4);
        tmp.put("ro-RO", PaperSize.A4);
        tmp.put("pt-BR", PaperSize.LETTER);
        tmp.put("sv-SE", PaperSize.A4);
        tmp.put("da-DK", PaperSize.A4);
        tmp.put("ru-RU", PaperSize.A4);

        PAPERSIZE_LANGMAP = Collections.unmodifiableMap(tmp);
    }

    /**
     * Retrieves the default page size for a specific language.
     *
     * @param sLanguageCode the language code (ISO)
     * @return the default page size in that country or PaperSize.A4 if language is not mapped
     */
    public static PageSize getPageSize(final String sLanguageCode) {
        PaperSize aPaperSize = PAPERSIZE_LANGMAP.get(sLanguageCode);
        return getPageSize((aPaperSize == null) ? PaperSize.A4 : aPaperSize);
    }

    /**
     * Retrieves the page size of paper size.
     *
     * @param ePaperSize a paper size
     * @return the page size in 1/100th mm
     */
    public static PageSize getPageSize(PaperSize ePaperSize) {
        PageSize aPageSize = null;

        switch (ePaperSize) {
            case A4: aPageSize = PageSize.A4; break;
            case LETTER: aPageSize = PageSize.LETTER; break;
            // fallback is A4
            default: aPageSize = PageSize.A4;
        }
        return aPageSize;
    }
};
