/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.presenter.impl;

import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.config.ConfigurationService;
import com.openexchange.presenter.osgi.Services;
import com.openexchange.realtime.Component;
import com.openexchange.realtime.ComponentHandle;
import com.openexchange.realtime.LoadFactorCalculator;
import com.openexchange.realtime.cleanup.AbstractRealtimeJanitor;
import com.openexchange.realtime.cleanup.GlobalRealtimeCleanup;
import com.openexchange.realtime.exception.RealtimeException;
import com.openexchange.realtime.packet.ID;
import com.openexchange.server.ServiceLookup;

/**
 * A very simple component that initializes a {@link PresenterConnection} as needed.
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class PresenterComponent extends AbstractRealtimeJanitor implements Component {
    private static final Logger LOG = LoggerFactory.getLogger(PresenterConnection.class);

    private static final int MAX_PARTICIPANTS_FOR_PRESENTATION = 100;
    private final ConcurrentHashMap<ID, ComponentHandle> m_openConnections = new ConcurrentHashMap<ID, ComponentHandle>();
    private final ServiceLookup services;
    private static final String componentID = "presenter";
    private int maxParticipants = MAX_PARTICIPANTS_FOR_PRESENTATION;

    /**
     * Initializes a new {@link PresenterComponent}.
     * @param chineseRoomActivator
     */
    public PresenterComponent(ServiceLookup services) {
        super();
        this.services = services;
        
        try {
		    final ConfigurationService configService = services.getService(ConfigurationService.class);
		    
		    if (null != configService) {
		    	maxParticipants = configService.getIntProperty("com.openexchange.presenter.maxParticipants", MAX_PARTICIPANTS_FOR_PRESENTATION);
		    }
        } catch (IllegalStateException e) {
        	LOG.warn("Presenter: Unable to retrieve configuration service - using default for maxNumberOfParticipants", e);
        }
    }

    /**
     * Create the connection, say synthetic.presenter://room1
     */
    @Override
    public ComponentHandle create(ID id) throws RealtimeException {
        final ComponentHandle connection = new PresenterConnection(id, services, componentID, maxParticipants);

        if (connection != null) {
            m_openConnections.put(id, connection);
        }

        return connection;
    }

    /**
     * Handler that is called by the RT-framework when a ComponentHandle
     * should be removed from the component.
     *
     * @param id
     *  The id of the connection be removed from the component.
     */
    @Override
    public void cleanupForId(final ID id) {
        // We are only interested in IDs of office components for now
        if (componentID.equals(id.getComponent())) {
            final ConcurrentHashMap<ID, ComponentHandle> presenters = getOpenConnections();
            final ComponentHandle removedConnection = presenters.remove(id);

            if (null == removedConnection) {
                LOG.debug("Presenter: Couldn't remove connection for ID: " + id);
            } else {
                LOG.debug("Presenter: Removed document with ID: " + id.toString() + ", presenter count: " + presenters.size());
            }
        }
    }

    /**
     * Returns the open connections controlled by this component instance.
     *
     * @return
     *  A map filled with the open connections of this component instance.
     */
    protected ConcurrentHashMap<ID, ComponentHandle> getOpenConnections() {
        return m_openConnections;
    }

    /**
     * Now we can address connections as synthetic.presenter://room1
     */
    @Override
    public String getId() {
        return componentID;
    }

    /**
     * No automatic shutdown of connection instances.
     */
    @Override
    public EvictionPolicy getEvictionPolicy() {
        return Component.NONE;
    }

    @Override
    public void setLoadFactorCalculator(LoadFactorCalculator loadFactorCalculator) {
        // TODO Auto-generated method stub
    }

    /**
     * Tries to shutdown the presenter component gracefully, including to cleanup the
     * global entries for the open presenter instances.
     */
    public void shutdown() {
        final GlobalRealtimeCleanup globalRealtimeCleanup = Services.optService(GlobalRealtimeCleanup.class);
        if (globalRealtimeCleanup == null) {
            LOG.error("Unable to issue cluster wide cleanup for Presenter IDs due to missing GlobalRealtimeCleanup service.");
        }
        
        if (globalRealtimeCleanup != null) {
        	for(ID id: m_openConnections.keySet()) {
                globalRealtimeCleanup.cleanForId(id);
            }
        }
    }
}
