/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.dml.*;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.tools.ThemeFonts;
import com.openexchange.office.ooxml.tools.json.JSONHelper;
import com.openexchange.office.tools.doc.DocumentImageHelper;

final public class DMLHelper {

	public static void createJsonFromPositiveSize2D(JSONObject drawingProperties, CTPositiveSize2D positiveSize2D)
        throws JSONException {

        if(positiveSize2D!=null) {
            drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
            drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
        }
    }

    public static JSONObject createJsonFromBlipFillProperties(Part contextPart, JSONObject attrs, CTBlipFillProperties blipFillProperties)
        throws JSONException {

        if(blipFillProperties!=null) {
            attrs.put("imageUrl", getBlipUrl(contextPart, blipFillProperties.getBlip(false)));

            final Boolean rotateWithShape = blipFillProperties.isRotWithShape();
            if(rotateWithShape!=null) {
            	attrs.put("rotateWithShape", rotateWithShape);
            }
            final CTRelativeRect srcRect = blipFillProperties.getSrcRect(false);
            if (srcRect!=null) {
                final int l = srcRect.getL() / 1000;
                final int t = srcRect.getT() / 1000;
                final int r = srcRect.getR() / 1000;
                final int b = srcRect.getB() / 1000;

                // add possible crop properties
                if (l != 0)
                    attrs.put("cropLeft", l);
                if (t != 0)
                    attrs.put("cropTop", t);
                if (r != 0)
                    attrs.put("cropRight", r);
                if (b != 0)
                    attrs.put("cropBottom", b);
            }
            final CTBlip blip = blipFillProperties.getBlip(false);
            if(blip!=null) {
            	final Iterator<Object> blipEffectIter = blip.getAlphaBiLevelOrAlphaCeilingOrAlphaFloor().iterator();
            	while(blipEffectIter.hasNext()) {
            		final Object blipEffect = blipEffectIter.next();
            		switch(blipEffect.getClass().getSimpleName()) {
            			case "CTAlphaModulateFixedEffect": {
            				attrs.put("transparency", (100000 - ((CTAlphaModulateFixedEffect)blipEffect).getAmt()) / 100000.0);
                			break;
            			}
            		}
            	}
            }
            if(blipFillProperties.getTile(false)!=null) {
            	final CTTileInfoProperties tiling = blipFillProperties.getTile(false);
            	final JSONObject json = new JSONObject();
            	final STRectAlignment align = tiling.getAlgn();
            	if(align!=null) {
            		String a = null;
            		switch(align) {
					case B: 	a = "bottom"; break;
					case BL:	a = "bottomLeft"; break;
					case BR:	a = "bottomRight"; break;
					case CTR:	a = "center"; break;
					case L:		a = "left"; break;
					case R:		a = "right"; break;
					case T:		a = "top"; break;
					case TL:	a = "topLeft"; break;
					case TR:	a = "topRight"; break;
            		}
            		if(a!=null) {
            			json.put("rectAlignment", a);
            		}
            	}
            	final STTileFlipMode flip = tiling.getFlip();
            	String flipMode = null;
            	if(flip==STTileFlipMode.X) {
            		flipMode = "x";
            	}
            	else if(flip==STTileFlipMode.Y) {
            		flipMode = "y";
            	}
            	else if(flip==STTileFlipMode.XY) {
            		flipMode = "xy";
            	}
            	else if(flip==STTileFlipMode.NONE) {
            		flipMode = "none";
            	}
            	if(flipMode!=null) {
            		json.put("flipMode", flipMode);
            	}
            	if(tiling.getSx()!=null) {
            		json.put("stretchX", ((double)tiling.getSx()) / 1000.0);
            	}
            	if(tiling.getSy()!=null) {
            		json.put("stretchY", ((double)tiling.getSx()) / 1000.0);
            	}
            	if(tiling.getTx()!=null) {
            		json.put("offX", (int)convertEmuTo100thmm(tiling.getTx()));
            	}
            	if(tiling.getTy()!=null) {
            		json.put("offY", (int)convertEmuTo100thmm(tiling.getTy()));
            	}
            	if(!json.isEmpty()) {
            		attrs.put("tiling", json);
            	}
            }
            else if(blipFillProperties.getStretch(false)!=null) {
            	final CTStretchInfoProperties stretching = blipFillProperties.getStretch(false);
            	final JSONObject json = new JSONObject();
            	final CTRelativeRect fillRect = stretching.getFillRect();
            	if(fillRect!=null) {
            		if(fillRect.getL()!=0) {
            			json.put("left",((double)fillRect.getL()) / 1000.0);
            		}
            		if(fillRect.getT()!=0) {
            			json.put("top",((double)fillRect.getT()) / 1000.0);
            		}
            		if(fillRect.getR()!=0) {
            			json.put("right",((double)fillRect.getR()) / 1000.0);
            		}
            		if(fillRect.getB()!=0) {
            			json.put("bottom",((double)fillRect.getB()) / 1000.0);
            		}
            	}
            	if(!json.isEmpty()) {
            		attrs.put("stretching", json);
            	}
            }
        }
        return attrs;
    }

    public static void applyGradFillPropertiesFromJson(CTGradientFillProperties gradientFill, OperationDocument operationDocument, Part part, JSONObject gradientAttrs)
    	throws JSONException {
		final String gradientType = gradientAttrs.optString("type", null);
		if(gradientType!=null) {
			if(gradientType.equals("linear")) {
				if(gradientFill.getLin()==null) {
					gradientFill.setLin(Context.getDmlObjectFactory().createCTLinearShadeProperties());
				}
				gradientFill.setPath(null);
			}
			else if(gradientType.equals("path")) {
				if(gradientFill.getPath()==null) {
					gradientFill.setPath(Context.getDmlObjectFactory().createCTPathShadeProperties());
				}
				gradientFill.setLin(null);
			}
		}
		if(gradientFill.getLin()!=null) {
			final Object gradientRotate = gradientAttrs.opt("rotation");
			if(gradientRotate instanceof Number) {
				final Double angle = ((Number)gradientRotate).doubleValue() * 60000.0;
				gradientFill.getLin().setAng(angle.intValue());
			}
			else if(gradientRotate!=null){
				gradientFill.getLin().setAng(null);
			}
			final Object gradientScaled = gradientAttrs.opt("isScaled");
			if(gradientScaled instanceof Boolean) {
				gradientFill.getLin().setScaled((Boolean)gradientScaled);
			}
			else if(gradientScaled!=null) {
				gradientFill.getLin().setScaled(null);
			}
		}
		else if(gradientFill.getPath()!=null) {
			final CTPathShadeProperties pathShadeProperties = gradientFill.getPath();
			final Object pathType = gradientAttrs.opt("pathType");
			if(pathType!=null) {
				if(pathType instanceof String) {
					if(((String)pathType).equals("circle")) {
						pathShadeProperties.setPath(STPathShadeType.CIRCLE);
					}
					else if(((String)pathType).equals("rect")) {
						pathShadeProperties.setPath(STPathShadeType.RECT);
					}
					else {
						pathShadeProperties.setPath(STPathShadeType.SHAPE);
					}
				}
				else {
					pathShadeProperties.setPath(STPathShadeType.SHAPE);
				}
			}
			final CTRelativeRect relativeRect = pathShadeProperties.getFillToRect(true);
			final Object pathFillTop = gradientAttrs.opt("pathFillTop");
			if(pathFillTop!=null) {
				relativeRect.setT(pathFillTop instanceof Number ? convertPercentToRel((Number)pathFillTop): null);
			}
			final Object pathFillLeft = gradientAttrs.opt("pathFillLeft");
			if(pathFillLeft!=null) {
				relativeRect.setL(pathFillLeft instanceof Number ? convertPercentToRel((Number)pathFillLeft): null);
			}
			final Object pathFillRight = gradientAttrs.opt("pathFillRight");
			if(pathFillRight!=null) {
				relativeRect.setR(pathFillRight instanceof Number ? convertPercentToRel((Number)pathFillRight): null);
			}
			final Object pathFillBottom = gradientAttrs.opt("pathFillBottom");
			if(pathFillBottom!=null) {
				relativeRect.setB(pathFillBottom instanceof Number ? convertPercentToRel((Number)pathFillBottom): null);
			}
			if(relativeRect.getT()==0&&relativeRect.getL()==0&&relativeRect.getR()==0&&relativeRect.getB()==0) {
				pathShadeProperties.setFillToRect(null);
			}
		}
		final CTRelativeRect tileRect = gradientFill.getTileRect(true);
		final Object tileRectTop = gradientAttrs.opt("top");
		if(tileRectTop!=null) {
			tileRect.setT(tileRectTop instanceof Number ? convertPercentToRel((Number)tileRectTop) : null);
		}
		final Object tileRectLeft = gradientAttrs.opt("left");
		if(tileRectLeft!=null) {
			tileRect.setL(tileRectLeft instanceof Number ? convertPercentToRel((Number)tileRectLeft) : null);
		}
		final Object tileRectRight = gradientAttrs.opt("right");
		if(tileRectRight!=null) {
			tileRect.setR(tileRectRight instanceof Number ? convertPercentToRel((Number)tileRectRight) : null);
		}
		final Object tileRectBottom = gradientAttrs.opt("bottom");
		if(tileRectBottom!=null) {
			tileRect.setB(tileRectBottom instanceof Number ? convertPercentToRel((Number)tileRectBottom) : null);
		}
		if(tileRect.getT()==0&&tileRect.getL()==0&&tileRect.getR()==0&&tileRect.getB()==0) {
			gradientFill.setTileRect(null);
		}
		final Object flipH = gradientAttrs.opt("flipH");
		final Object flipV = gradientAttrs.opt("flipV");
		if(flipH!=null||flipV!=null) {
			boolean fH = false;
			boolean fV = false;
			final STTileFlipMode flipMode = gradientFill.getFlip();
			if(flipMode==STTileFlipMode.X) {
				fH = true;
			}
			else if(flipMode==STTileFlipMode.Y) {
				fV = true;
			}
			else if(flipMode==STTileFlipMode.XY) {
				fH = true;
				fV = true;
			}
			if(flipH!=null) {
				if(flipH instanceof Boolean) {
					fH = ((Boolean)flipH).booleanValue();
				}
				else {
					fH = false;
				}
			}
			if(flipV!=null) {
				if(flipV instanceof Boolean) {
					fV = ((Boolean)flipV).booleanValue();
				}
				else {
					fV = false;
				}
			}
			if(fH&&fV) {
				gradientFill.setFlip(STTileFlipMode.XY);
			}
			else if(fH) {
				gradientFill.setFlip(STTileFlipMode.X);
			}
			else if(fV) {
				gradientFill.setFlip(STTileFlipMode.Y);
			}
			else {
				gradientFill.setFlip(STTileFlipMode.NONE);
			}
		}
		final Object gradientRotateWithShape = gradientAttrs.opt("isRotateWithShape");
		if(gradientRotateWithShape instanceof Boolean) {
			gradientFill.setRotWithShape((Boolean)gradientRotateWithShape);
		}
		else if(gradientRotateWithShape!=null) {
			gradientFill.setRotWithShape(null);
		}
		final Object gradientStops = gradientAttrs.opt("colorStops");
		if(gradientStops instanceof JSONArray) {
			gradientFill.setGsLst(Context.getDmlObjectFactory().createCTGradientStopList());
			final CTGradientStopList gs = gradientFill.getGsLst();
			final List<CTGradientStop> gsLst = gs.getGs();
			final JSONArray stops = (JSONArray)gradientStops;
			for(int i = 0; i < stops.length(); i++) {
				final CTGradientStop gradientStop = Context.getDmlObjectFactory().createCTGradientStop();
				gsLst.add(gradientStop);
				final JSONObject stop = (JSONObject)stops.get(i);
				final Double p = (((Number)stop.opt("position")).doubleValue() * 100000);
				gradientStop.setPos(p.intValue());
				final JSONObject color = stop.optJSONObject("color");
				if(color!=null) {
					createSolidColorFillPropertiesFromJson(gradientStop, color);
				}
			}
		}
		else if(gradientStops!=null) {
			gradientFill.setGsLst(null);
		}
    }

    public static void applyPattFillPropertiesFromJson(CTPatternFillProperties pattFillProperties, OperationDocument operationDocument, Part part, JSONObject fillAttrs) {
    	try {
	    	STPresetPatternVal pattType = STPresetPatternVal.fromValue(fillAttrs.getString("pattern"));
	    	pattFillProperties.setPrst(pattType);

	    	CTColor bgClr = new CTColor();
	    	createSolidColorFillPropertiesFromJson(bgClr, fillAttrs.getJSONObject("backgroundColor"));
	    	pattFillProperties.setBgClr(bgClr);

	    	CTColor fgClr = new CTColor();
	    	createSolidColorFillPropertiesFromJson(fgClr, fillAttrs.getJSONObject("color"));
	    	pattFillProperties.setFgClr(fgClr);
    	} catch (Exception ex) {
    	}
    }

    public static void applyBlipFillPropertiesFromJson(CTBlipFillProperties blipFillProperties, OperationDocument operationDocument, Part part, JSONObject imageAttrs)
    	throws InvalidFormatException, PartUnrecognisedException {

        if(imageAttrs==null) {
        	return;
        }
        String imageUrl = DocumentImageHelper.getImageUrlFromImageData(operationDocument.getResourceManager(), imageAttrs.optString("imageData"), operationDocument.getImagePath());
        if(imageUrl==null) {
            imageUrl = imageAttrs.optString("imageUrl", null);
        }
        if(imageUrl!=null) {
            final Relationship relationship = Commons.createGraphicRelation(operationDocument, part, imageUrl);
            if(relationship!=null) {
                final CTBlip blip = blipFillProperties.getBlip(true);
                final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
                if(linked) {
                    blip.setEmbed(null);
                    blip.setLink(relationship.getId());
                }
                else {
                    blip.setLink(null);
                    blip.setEmbed(relationship.getId());
                }
            }
        }
        final Object rotateWithShape = imageAttrs.opt("rotateWithShape");
        if(rotateWithShape!=null) {
        	if(rotateWithShape==JSONObject.NULL) {
        		blipFillProperties.setRotWithShape(null);
        	}
        	else {
        		blipFillProperties.setRotWithShape((Boolean)rotateWithShape);
        	}
        }
        final CTRelativeRect cropRect = blipFillProperties.getSrcRect(true);
        Object cropLeft = imageAttrs.opt("cropLeft");
        if(cropLeft instanceof Number) {
        	cropRect.setL(((Number)cropLeft).intValue() * 1000);
        }
        else if(cropLeft==JSONObject.NULL) {
        	cropRect.setL(null);
        }
        final Object cropTop = imageAttrs.opt("cropTop");
        if(cropTop instanceof Number) {
        	cropRect.setT(((Number)cropTop).intValue() * 1000);
        }
        else if(cropTop==JSONObject.NULL) {
        	cropRect.setT(null);
        }
        final Object cropRight = imageAttrs.opt("cropRight");
        if(cropRight instanceof Number) {
        	cropRect.setR(((Number)cropRight).intValue() * 1000);
        }
        else if(cropRight==JSONObject.NULL) {
        	cropRect.setR(null);
        }
        final Object cropBottom = imageAttrs.opt("cropBottom");
        if(cropBottom instanceof Number) {
        	cropRect.setB(((Number)cropBottom).intValue() * 1000);
        }
        else if(cropBottom==JSONObject.NULL) {
        	cropRect.setB(null);
        }
        if(cropRect.isEmpty()) {
        	blipFillProperties.setSrcRect(null);
        }
        final Object transparency = imageAttrs.opt("transparency");
        if(transparency!=null) {
	        if(transparency==JSONObject.NULL) {
	        	final CTBlip blip = blipFillProperties.getBlip(false);
	        	if(blip!=null) {
	        		final Iterator<Object> blipEffectIter = blip.getAlphaBiLevelOrAlphaCeilingOrAlphaFloor().iterator();
	        		while(blipEffectIter.hasNext()) {
	        			if(blipEffectIter.next() instanceof CTAlphaModulateFixedEffect) {
	        				blipEffectIter.remove();
	        				break;
	        			}
	        		}
	        	}
	        }
	        else {
	            final CTBlip blip = blipFillProperties.getBlip(true);
            	CTAlphaModulateFixedEffect alphaModulateFixedEffect = null;
            	final Iterator<Object> blipEffectIter = blip.getAlphaBiLevelOrAlphaCeilingOrAlphaFloor().iterator();
            	while(blipEffectIter.hasNext()) {
            		final Object blipEffect = blipEffectIter.next();
            		if(blipEffect instanceof CTAlphaModulateFixedEffect) {
            			alphaModulateFixedEffect = (CTAlphaModulateFixedEffect)blipEffect;
            			break;
            		}
            	}
            	if(alphaModulateFixedEffect==null) {
            		alphaModulateFixedEffect = new CTAlphaModulateFixedEffect();
            		blip.getAlphaBiLevelOrAlphaCeilingOrAlphaFloor().add(alphaModulateFixedEffect);
            	}
            	alphaModulateFixedEffect.setAmt(Double.valueOf(100000-((Number)transparency).doubleValue()*100000.0).intValue());
	        }
        }
        final JSONObject tilingAttrs = imageAttrs.optJSONObject("tiling");
        if(tilingAttrs!=null) {
        	if(tilingAttrs==JSONObject.NULL) {
        		blipFillProperties.setTile(null);
        	}
        	else {
        		final CTTileInfoProperties tileInfoProperties = blipFillProperties.getTile(true);
        		Object a = tilingAttrs.opt("rectAlignment");
        		if(a!=null) {
        			STRectAlignment rectAlign = null;
        			if(a instanceof String) {
        				switch((String)a) {
	    				case "bottom": 		rectAlign = STRectAlignment.B; break;
	    				case "bottomLeft":  rectAlign = STRectAlignment.L; break;
	    				case "bottomRight": rectAlign = STRectAlignment.BR; break;
	    				case "center": 		rectAlign = STRectAlignment.CTR; break;
	    				case "left": 		rectAlign = STRectAlignment.L; break;
	    				case "right": 		rectAlign = STRectAlignment.R; break;
	    				case "top": 		rectAlign = STRectAlignment.T; break;
	    				case "topLeft": 	rectAlign = STRectAlignment.TL; break;
	    				case "topRight": 	rectAlign = STRectAlignment.TR; break;
        				}
        			}
        			tileInfoProperties.setAlgn(rectAlign);
        		}
        		a = tilingAttrs.opt("flipMode");
        		if(a!=null) {
        			STTileFlipMode flipMode = null;
        			if(a instanceof String) {
        				switch((String)a) {
        				case "x": flipMode = STTileFlipMode.X; break;
        				case "y": flipMode = STTileFlipMode.Y; break;
        				case "xy": flipMode = STTileFlipMode.XY; break;
        				case "none": flipMode = STTileFlipMode.NONE;
        				}
        			}
        			tileInfoProperties.setFlip(flipMode);
        		}
        		a = tilingAttrs.opt("stretchX");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				tileInfoProperties.setSx(null);
        			}
        			else {
        				tileInfoProperties.setSx(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        		a = tilingAttrs.opt("stretchY");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				tileInfoProperties.setSy(null);
        			}
        			else {
        				tileInfoProperties.setSy(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        		a = tilingAttrs.opt("offX");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				tileInfoProperties.setTx(null);
        			}
        			else {
        				tileInfoProperties.setTx(Double.valueOf(((Number)a).doubleValue() * 1000.0).longValue());
        			}
         		}
        		a = tilingAttrs.opt("offY");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				tileInfoProperties.setTy(null);
        			}
        			else {
        				tileInfoProperties.setTy(Double.valueOf(((Number)a).doubleValue() * 1000.0).longValue());
        			}
        		}
        	}
        }
        final JSONObject stretchingAttrs = imageAttrs.optJSONObject("stretching");
        if(stretchingAttrs!=null) {
        	if(stretchingAttrs==JSONObject.NULL) {
        		blipFillProperties.setStretch(null);
        	}
        	else {
        		final CTStretchInfoProperties stretchInfoProperties = blipFillProperties.getStretch(true);
        		CTRelativeRect fillRect = stretchInfoProperties.getFillRect();
        		if(fillRect==null) {
        			fillRect = new CTRelativeRect();
        			stretchInfoProperties.setFillRect(fillRect);
        		}
        		Object a = stretchingAttrs.opt("left");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				fillRect.setL(null);
        			}
        			else {
        				fillRect.setL(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        		a = stretchingAttrs.opt("top");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				fillRect.setT(null);
        			}
        			else {
        				fillRect.setT(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        		a = stretchingAttrs.opt("bottom");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				fillRect.setB(null);
        			}
        			else {
        				fillRect.setB(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        		a = stretchingAttrs.opt("right");
        		if(a!=null) {
        			if(a==JSONObject.NULL) {
        				fillRect.setR(null);
        			}
        			else {
        				fillRect.setR(Double.valueOf(((Number)a).doubleValue() * 1000.0).intValue());
        			}
        		}
        	}
        }
        // either tiling or stretch is possible
        if(blipFillProperties.getTile(false)==null||blipFillProperties.getTile(false).isEmpty()) {
        	blipFillProperties.setTile(null);
        	blipFillProperties.getStretch(true);
        }
        else {
        	blipFillProperties.setStretch(null);
        }
    }

    public static void createJsonFromTransform2D(JSONObject attrs, ITransform2D xFrm, boolean convertTo100thmm)
    	throws JSONException {

    	if(xFrm!=null) {
			final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
			final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

            final CTPoint2D point2D = xFrm.getOff(false);
            if(point2D!=null) {
                if(!drawingAttrs.hasAndNotNull("left")) {
                	drawingAttrs.put("left", convertTo100thmm ? Commons.coordinateTo100TH_MM(point2D.getX()): point2D.getX());
                }
                if(!drawingAttrs.hasAndNotNull("top")) {
                	drawingAttrs.put("top", convertTo100thmm ? Commons.coordinateTo100TH_MM(point2D.getY()) :  point2D.getY());
                }
            }
            final CTPositiveSize2D positiveSize2D = xFrm.getExt(false);
            if(positiveSize2D!=null) {
                if(!drawingAttrs.hasAndNotNull("width")) {
                	drawingAttrs.put("width", convertTo100thmm ? Commons.coordinateTo100TH_MM(positiveSize2D.getCx()) :  positiveSize2D.getCx());
                }
                if(!drawingAttrs.hasAndNotNull("height")) {
                	drawingAttrs.put("height", convertTo100thmm ? Commons.coordinateTo100TH_MM(positiveSize2D.getCy()) :  positiveSize2D.getCy());
                }
            }
            if(xFrm.isFlipH()) {
                if(!drawingAttrs.hasAndNotNull("flipH")) {
                	drawingAttrs.put("flipH", true );
                }
            }
            if(xFrm.isFlipV()) {
                if(!drawingAttrs.hasAndNotNull("flipV")) {
                	drawingAttrs.put("flipV", true );
                }
            }
            final int rotation = xFrm.getRot();
            if(rotation!=0) {
            	drawingAttrs.put("rotation", ((double)rotation)/60000.0);
            }
	        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
				attrs.put("drawing", drawingAttrs);
			}
        }
    }

    public static void applyTransform2DFromJson(ITransform2DAccessor transformAccessor, JSONObject attrs, boolean convertToEmu) {

    	final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
    	if(drawingAttrs!=null) {

    	    final Object width = drawingAttrs.opt("width");
			if(width!=null) {
	            if(width instanceof Number) {
	                ITransform2D transform2D = transformAccessor.getXfrm(true);
	                transform2D.getExt(true).setCx(convertToEmu ? ((Number)width).intValue()*360 : ((Number)width).intValue());
	            }
			}
			final Object height = drawingAttrs.opt("height");
			if(height!=null) {
				if(height instanceof Number) {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.getExt(true).setCy(convertToEmu ? ((Number)height).intValue()*360 : ((Number)height).intValue());
				}
			}
			if(width==JSONObject.NULL&&height==JSONObject.NULL) {
			    ITransform2D transform2D = transformAccessor.getXfrm(true);
                transform2D.setExt(null);
			}
			final Object left = drawingAttrs.opt("left");
			if(left!=null) {
			    if(left instanceof Number) {
			        ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.getOff(true).setX(convertToEmu ? ((Number)left).longValue()*360 : ((Number)left).longValue());
			    }
			}
            final Object top = drawingAttrs.opt("top");
            if(top!=null) {
                if(top instanceof Number) {
                    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.getOff(true).setY(convertToEmu ? ((Number)top).longValue()*360 : ((Number)top).longValue());
                }
            }
            if(left==JSONObject.NULL&&top==JSONObject.NULL) {
                ITransform2D transform2D = transformAccessor.getXfrm(true);
                transform2D.setOff(null);
            }
			final Object flipH = drawingAttrs.opt("flipH");
			if(flipH!=null) {
				if(flipH instanceof Boolean) {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setFlipH((Boolean)flipH);
				}
				else {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setFlipH(null);
				}
			}
			final Object flipV = drawingAttrs.opt("flipV");
			if(flipV!=null) {
				if(flipV instanceof Boolean) {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setFlipV((Boolean)flipV);
				}
				else {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setFlipV(null);
				}
			}
			final Object rotation = drawingAttrs.opt("rotation");
			if(rotation!=null) {
				if(rotation instanceof Number) {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setRot(Integer.valueOf((int)((((Number)rotation).doubleValue()*60000)+0.5))); ;
				}
				else {
				    ITransform2D transform2D = transformAccessor.getXfrm(true);
                    transform2D.setRot(null);
				}
			}
			final ITransform2D transform2D = transformAccessor.getXfrm(false);
			if(transform2D!=null&&transform2D.getExt(false)==null&&transform2D.getOff(false)==null) {
			    transformAccessor.removeXfrm();
			}
    	}
    }

    public static void applyFillPropertiesFromJsonColor(IFillProperties fillProperties, Object jsonColor)
    	throws JSONException {

    	fillProperties.setNoFill(null);
		fillProperties.setBlipFill(null);
		fillProperties.setGradFill(null);
		fillProperties.setPattFill(null);
		fillProperties.setGrpFill(null);

		if(jsonColor instanceof JSONObject) {
			final CTSolidColorFillProperties fillColor = Context.getDmlObjectFactory().createCTSolidColorFillProperties();
			fillProperties.setSolidFill(fillColor);
			createSolidColorFillPropertiesFromJson(fillColor, (JSONObject)jsonColor);
		}
		else if (jsonColor==JSONObject.NULL) {
			fillProperties.setSolidFill(null);
		}
    }

    public static void applyFillPropertiesFromJson(IFillProperties fillProperties, JSONObject attrs, OperationDocument operationDocument, Part part)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException {

    	final JSONObject fillAttrs = attrs.optJSONObject("fill");
    	if(fillAttrs!=null) {

    		// applying fill type first
    		final Object fillType = fillAttrs.opt("type");
    		if(fillType instanceof String) {
    			if(fillType.equals("none")) {
    				fillProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
    			}
    			else {
        	    	fillProperties.setNoFill(null);
    			}
    			if(fillType.equals("solid")) {
    				if(fillProperties.getSolidFill()==null) {
    					fillProperties.setSolidFill(Context.getDmlObjectFactory().createCTSolidColorFillProperties());
    				}
    			}
    			else {
    				fillProperties.setSolidFill(null);
    			}
    			if(fillType.equals("gradient")) {
    				if(fillProperties.getGradFill()==null) {
    					fillProperties.setGradFill(Context.getDmlObjectFactory().createCTGradientFillProperties());
    				}
    			}
    			else {
    				fillProperties.setGradFill(null);
    			}
    			if(fillType.equals("pattern")) {
    				if(fillProperties.getPattFill()==null) {
    					fillProperties.setPattFill(Context.getDmlObjectFactory().createCTPatternFillProperties());
    				}
    			}
    			else {
    				fillProperties.setPattFill(null);
    			}
    			if(fillType.equals("bitmap")) {
    				if(fillProperties.getBlipFill()==null) {
            			fillProperties.setBlipFill(Context.getDmlObjectFactory().createCTBlipFillProperties());
    				}
    			}
    			else {
    				fillProperties.setBlipFill(null);
    			}
    			fillProperties.setGrpFill(null);
    		}
    		else if(fillType==JSONObject.NULL) {
    	    	fillProperties.setNoFill(null);
    			fillProperties.setSolidFill(null);
    			fillProperties.setGradFill(null);
    			fillProperties.setPattFill(null);
    			fillProperties.setBlipFill(null);
    			fillProperties.setGrpFill(null);
    		}
    		if(fillProperties!=null) {
        		if(fillProperties.getSolidFill()!=null) {
    				final JSONObject color = fillAttrs.optJSONObject("color");
    				if(color!=null) {
    					createSolidColorFillPropertiesFromJson(fillProperties.getSolidFill(), color);
    				}
        		}
        		else if(fillProperties.getGradFill()!=null) {
        			final JSONObject gradientAttrs = fillAttrs.optJSONObject("gradient");
        			if(gradientAttrs!=null) {
        				applyGradFillPropertiesFromJson(fillProperties.getGradFill(), operationDocument, part, gradientAttrs);
        			}
        		}
        		else if(fillProperties.getPattFill()!=null) {
        			applyPattFillPropertiesFromJson(fillProperties.getPattFill(), operationDocument, part, fillAttrs);
        		}
        		else if(fillProperties.getBlipFill()!=null) {
        			applyBlipFillPropertiesFromJson(fillProperties.getBlipFill(), operationDocument, part, fillAttrs.optJSONObject("bitmap"));
        		}
    		}
    	}
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromFillStyle(Object source, IStyleMatrixReference styleMatrixReference, JSONObject dest, Part part)
        throws JSONException {

        if(source instanceof CTSolidColorFillProperties) {
            dest.put("type", "solid");
            JSONObject color = createJsonColorFromSolidColorFillProperties((CTSolidColorFillProperties)source);
            if(styleMatrixReference!=null) {
            	color = createJsonColorFromSolidColorFillProperties(color, styleMatrixReference);
            }
            if(color!=null) {
            	dest.put("color", color);
            }
        }
        else if(source instanceof CTNoFillProperties) {
            dest.put("type", "none");
        }
        else if(source instanceof CTGradientFillProperties) {
            dest.put("type", "gradient");
            final JSONObject gradientAttrs = new JSONObject(5);

            final CTGradientFillProperties gradientFillProperties = (CTGradientFillProperties)source;
        	final CTLinearShadeProperties linearShadeProperties = gradientFillProperties.getLin();
        	if(linearShadeProperties!=null) {
        		gradientAttrs.put("type", "linear");
        		final Integer angle = linearShadeProperties.getAng();
        		if(angle!=null) {
        			gradientAttrs.put("rotation", angle.doubleValue() / 60000.0);
        		}
        		final Boolean isScaled = linearShadeProperties.isScaled();
        		if(isScaled!=null) {
        			gradientAttrs.put("isScaled", isScaled);
        		}
        	}
        	else if(gradientFillProperties.getPath()!=null) {
        		gradientAttrs.put("type", "path");
        		final CTPathShadeProperties pathShadeProperties = gradientFillProperties.getPath();
        		if(pathShadeProperties.getPath()==STPathShadeType.CIRCLE) {
        			gradientAttrs.put("pathType", "circle");
        		}
        		else if(pathShadeProperties.getPath()==STPathShadeType.RECT) {
        			gradientAttrs.put("pathType", "rect");
        		}
        		final CTRelativeRect relativeRect = pathShadeProperties.getFillToRect(false);
        		if(relativeRect!=null) {
	        		if(relativeRect.getB()!=0) {
	        			gradientAttrs.put("pathFillBottom", convertRelToPercent(relativeRect.getB()));
	        		}
	        		if(relativeRect.getL()!=0) {
	        			gradientAttrs.put("pathFillLeft", convertRelToPercent(relativeRect.getL()));
	        		}
	        		if(relativeRect.getR()!=0) {
	        			gradientAttrs.put("pathFillRight", convertRelToPercent(relativeRect.getR()));
	        		}
	        		if(relativeRect.getT()!=0) {
	        			gradientAttrs.put("pathFillTop", convertRelToPercent(relativeRect.getT()));
	        		}
        		}
        	}
        	final CTRelativeRect tileRect = gradientFillProperties.getTileRect(false);
        	if(tileRect!=null) {
        		if(tileRect.getB()!=0) {
        			gradientAttrs.put("bottom", convertRelToPercent(tileRect.getB()));
        		}
        		if(tileRect.getL()!=0) {
        			gradientAttrs.put("left", convertRelToPercent(tileRect.getL()));
        		}
        		if(tileRect.getR()!=0) {
        			gradientAttrs.put("right", convertRelToPercent(tileRect.getR()));
        		}
        		if(tileRect.getT()!=0) {
        			gradientAttrs.put("top", convertRelToPercent(tileRect.getT()));
        		}
        	}
        	final STTileFlipMode flipMode = gradientFillProperties.getFlip();
        	if(flipMode!=null) {
        		if(flipMode==STTileFlipMode.X) {
        			gradientAttrs.put("flipH", true);
        		}
        		else if(flipMode==STTileFlipMode.Y) {
        			gradientAttrs.put("flipV", true);
        		}
        		else if(flipMode==STTileFlipMode.XY) {
        			gradientAttrs.put("flipH", true);
        			gradientAttrs.put("flipV", true);
        		}
        	}
        	if(gradientFillProperties.isRotWithShape()==null || gradientFillProperties.isRotWithShape().booleanValue()) {
        		gradientAttrs.put("isRotateWithShape", true);
        	}
        	final CTGradientStopList gradientStopList = gradientFillProperties.getGsLst();
        	if(gradientStopList!=null) {
                final List<CTGradientStop> list = gradientStopList.getGs();
        		final JSONArray gradientStopArray = new JSONArray(list.size());
        		for(CTGradientStop gradientStop:list) {
        			final JSONObject colorStop = new JSONObject(2);
        			colorStop.put("position", ((double)gradientStop.getPos())/100000.0);
        			JSONObject color = createJsonColorFromSolidColorFillProperties(gradientStop);
                    if(styleMatrixReference!=null) {
                    	color = createJsonColorFromSolidColorFillProperties(color, styleMatrixReference);
                    }
        			if(color!=null) {
        				colorStop.put("color", color);
        			}
        			gradientStopArray.put(colorStop);
        		}
        		gradientAttrs.put("colorStops", gradientStopArray);
        	}
        	if(!gradientAttrs.isEmpty()) {
        		dest.put("gradient", gradientAttrs);
        	}
        }
        else if(source instanceof CTPatternFillProperties) {
        	final CTPatternFillProperties pattSource = (CTPatternFillProperties)source;
            final CTColor fgFill = pattSource.getFgClr();
            dest.put("type", "pattern");
            if(fgFill!=null) {
            	JSONObject color = createJsonColorFromSolidColorFillProperties(fgFill);
                if(styleMatrixReference!=null) {
                	color = createJsonColorFromSolidColorFillProperties(color, styleMatrixReference);
                }
            	if(color!=null) {
            		dest.put("color", color);
            	}
            }
            final CTColor bgFill = pattSource.getBgClr();
            if(bgFill!=null) {
            	JSONObject color = createJsonColorFromSolidColorFillProperties(bgFill);
                if(styleMatrixReference!=null) {
                	color = createJsonColorFromSolidColorFillProperties(color, styleMatrixReference);
                }
            	if(color!=null) {
            		dest.put("backgroundColor", color);
            	}
            }
        	final STPresetPatternVal pattType = pattSource.getPrst();
        	dest.put("pattern", (pattType!=null) ? pattType.value() : "none");
        }
        else if(source instanceof CTBlipFillProperties) {
            dest.put("type", "bitmap");
            final JSONObject bitmapProps = new JSONObject();
            createJsonFromBlipFillProperties(part, bitmapProps, (CTBlipFillProperties)source);
            dest.put("bitmap", bitmapProps);
        }
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromLineStyle(CTLineProperties lineProperties, IStyleMatrixReference styleMatrixReference, JSONObject dest, Part part)
        throws JSONException {

        if(lineProperties.getNoFill()!=null) {
            createJsonFromFillStyle(lineProperties.getNoFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getSolidFill()!=null) {
            createJsonFromFillStyle(lineProperties.getSolidFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getGradFill()!=null) {
            createJsonFromFillStyle(lineProperties.getGradFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getPattFill()!=null) {
            createJsonFromFillStyle(lineProperties.getPattFill(), styleMatrixReference, dest, part);
        }
        if(lineProperties.getW()!=null) {
            dest.put("width", convertEmuTo100thmm(lineProperties.getW()));
        }
        if(lineProperties.getPrstDash()!=null&&lineProperties.getPrstDash().getVal()!=null) {
            switch(lineProperties.getPrstDash().getVal()) {
            case SYS_DOT:
            case DOT:
                dest.put("style", "dotted");
                break;
            case LG_DASH:
            case SYS_DASH:
            case DASH:
                dest.put("style", "dashed");
                break;
            case LG_DASH_DOT:
            case SYS_DASH_DOT:
            case DASH_DOT:
                dest.put("style", "dashDot");
                break;
            case LG_DASH_DOT_DOT:
            case SYS_DASH_DOT_DOT:
                dest.put("style", "dashDotDot");
                break;
            case SOLID:
            default:
                break;
            }
        }
        final CTLineEndProperties headEnd = lineProperties.getHeadEnd(false);
        if(headEnd!=null) {
            final STLineEndType headEndType = headEnd.getType();
            if(headEndType!=null) {
                dest.put("headEndType", headEndType.value());
            }
            final STLineEndLength headEndLength = headEnd.getLen();
            if(headEndLength!=null) {
                if(headEndLength==STLineEndLength.SM) {
                    dest.put("headEndLength", "small");
                }
                else if(headEndLength==STLineEndLength.MED) {
                    dest.put("headEndLength", "medium");
                }
                else if(headEndLength==STLineEndLength.LG) {
                    dest.put("headEndLength", "large");
                }
            }
            final STLineEndWidth headEndWidth = headEnd.getW();
            if(headEndWidth!=null) {
                if(headEndWidth==STLineEndWidth.SM) {
                    dest.put("headEndWidth", "small");
                }
                else if(headEndWidth==STLineEndWidth.MED) {
                    dest.put("headEndWidth", "medium");
                }
                else if(headEndWidth==STLineEndWidth.LG) {
                    dest.put("headEndWidth", "large");
                }
            }
        }
        final CTLineEndProperties tailEnd = lineProperties.getTailEnd(false);
        if(tailEnd!=null) {
            final STLineEndType tailEndType = tailEnd.getType();
            if(tailEndType!=null) {
                dest.put("tailEndType", tailEndType.value());
            }
            final STLineEndLength tailEndLength = tailEnd.getLen();
            if(tailEndLength!=null) {
                if(tailEndLength==STLineEndLength.SM) {
                    dest.put("tailEndLength", "small");
                }
                else if(tailEndLength==STLineEndLength.MED) {
                    dest.put("tailEndLength", "medium");
                }
                else if(tailEndLength==STLineEndLength.LG) {
                    dest.put("tailEndLength", "large");
                }
            }
            final STLineEndWidth tailEndWidth = tailEnd.getW();
            if(tailEndWidth!=null) {
                if(tailEndWidth==STLineEndWidth.SM) {
                    dest.put("tailEndWidth", "small");
                }
                else if(tailEndWidth==STLineEndWidth.MED) {
                    dest.put("tailEndWidth", "medium");
                }
                else if(tailEndWidth==STLineEndWidth.LG) {
                    dest.put("tailEndWidth", "large");
                }
            }
        }
    }

    private static void createJsonFromFillStyleMatrix(ThemePart themePart, IStyleMatrixReference fillStyleReference, JSONObject dest)
        throws JSONException {

        if(fillStyleReference==null||themePart==null) {
            return;
        }
        final Theme theme = themePart.getJaxbElement();
        if(theme==null) {
        	return;
        }
        final BaseStyles themeElements = theme.getThemeElements();
        if(themeElements==null) {
            return;
        }
        final CTStyleMatrix fmtScheme = themeElements.getFmtScheme();
        if(fmtScheme==null) {
            return;
        }
        final long idx = fillStyleReference.getIdx();
        if(idx==0||idx==1000) {
            dest.put("type", "none");
            return;
        }
        else {
            List<Object> fillMatrix = null;
            if(idx>=1&&idx<1000) {
                final CTFillStyleList fillStyleList = fmtScheme.getFillStyleLst();
                if(fillStyleList!=null) {
                    fillMatrix = fillStyleList.getContent();
                }
            }
            else if(idx>1000) {
                final CTBackgroundFillStyleList bgFillStyleList = fmtScheme.getBgFillStyleLst();
                if(bgFillStyleList!=null) {
                    fillMatrix = bgFillStyleList.getContent();
                }
            }
            if(fillMatrix!=null) {
                final int index = (int)(idx>1000?idx-1001:idx-1);
                if(index<fillMatrix.size()) {
                    createJsonFromFillStyle(fillMatrix.get(index), fillStyleReference, dest, themePart);
                }
            }
        }
        return;
    }

    private static void createJsonFromLineStyleMatrix(ThemePart themePart, IStyleMatrixReference lineStyleReference, JSONObject dest)
        throws JSONException {

        if(lineStyleReference==null||themePart==null) {
            return;
        }
        final Theme theme = themePart.getJaxbElement();
        if(theme==null) {
        	return;
        }
        final BaseStyles themeElements = theme.getThemeElements();
        if(themeElements==null) {
            return;
        }
        final CTStyleMatrix fmtScheme = themeElements.getFmtScheme();
        if(fmtScheme==null) {
            return;
        }
       final int idx = (int)lineStyleReference.getIdx() - 1;
       final CTLineStyleList lineStyleList = fmtScheme.getLnStyleLst();
       if(idx >= 0 && lineStyleList!=null) {
           final List<CTLineProperties> lineMatrix = lineStyleList.getLn();
           if(idx<lineMatrix.size()) {
               createJsonFromLineStyle(lineMatrix.get(idx), lineStyleReference, dest, themePart);
           }
       }
       return;
    }

    public static void createJsonFromShapeProperties(JSONObject attrs, CTShapeProperties shapeProperties, ThemePart themePart, IShapeStyle shapeStyle, Part part, boolean createTransform, boolean rootShape)
            throws JSONException {

        if(shapeProperties!=null) {
            if(createTransform) {
                createJsonFromTransform2D(attrs, shapeProperties.getXfrm(false), rootShape);
            }
            if(shapeProperties.getPrstGeom(false)!=null) {
            	DMLGeometry.createJsonFromPresetGeometry(attrs, shapeProperties.getPrstGeom(false));
            }
            else if(shapeProperties.getCustGeom()!=null) {
            	DMLGeometry.createJsonFromCustomGeometry(attrs, shapeProperties.getCustGeom());
            }
        }
        if(shapeStyle!=null&&shapeStyle.getFontRef()!=null) {
        	createJsonFromFontRef(attrs, shapeStyle.getFontRef());
        }
        createJsonFromLineProperties(attrs, themePart, shapeStyle!=null ? shapeStyle.getLnRef() : null, shapeProperties!=null ? shapeProperties.getLn(): null, part);
        createJsonFromFillProperties(attrs, shapeProperties, themePart, shapeStyle!=null ? shapeStyle.getFillRef() : null, part);
    }

    public static void applyTableGridFromJson(CTTable table, long tableWidth, JSONArray tableGrid) {
		final CTTableGrid grid = table.getTblGrid(true);
		long totalGridWidth = 0;

		if(tableGrid!=null) {
			// applying new tableGrid
			grid.getGridCol().clear();
			Iterator<Object> gridIter = tableGrid.iterator();
			while(gridIter.hasNext()) {
				final Object o = gridIter.next();
				if(o instanceof Number) {
					totalGridWidth += ((Number)o).longValue();
				}
			}
			long remainingTableWidth = tableWidth;
			gridIter = tableGrid.iterator();
			while(gridIter.hasNext()) {
				final Object o = gridIter.next();
				if(o instanceof Number) {
					final CTTableCol tableCol = new CTTableCol();
					long destGrid = 0;
					if(!gridIter.hasNext()) {
						destGrid = remainingTableWidth;
					}
					else if(totalGridWidth!=0) {
						double w = ((Number)o).doubleValue();
						w = (w / totalGridWidth) * tableWidth;
						destGrid = Double.valueOf(w).longValue();
						remainingTableWidth -= destGrid;
					}
					tableCol.setW(destGrid);
					grid.getGridCol().add(tableCol);
				}
			}
		}
		else {	// the table width might have changed, we have to adapt the existing tableGrid
			Iterator<CTTableCol> tableColIter = grid.getGridCol().iterator();
			while(tableColIter.hasNext()) {
				totalGridWidth += tableColIter.next().getW();
			}
			if(tableWidth!=totalGridWidth) {
				long remainingTableWidth = tableWidth;
				tableColIter = grid.getGridCol().iterator();
				while(tableColIter.hasNext()) {
					final CTTableCol tableCol = tableColIter.next();
					long destGrid = 0;
					if(!tableColIter.hasNext()) {
						destGrid = remainingTableWidth;
					}
					else if(totalGridWidth!=0) {
						double w = tableCol.getW();
						w = (w / totalGridWidth) * tableWidth;
						destGrid = Double.valueOf(w).longValue();
						remainingTableWidth -= destGrid;
					}
					tableCol.setW(destGrid);
				}
			}
		}
    }

    public static void applyTableFromJson(JSONObject attrs, CTTable table, long width, OperationDocument operationDocument, Part part)
    	throws InvalidFormatException, PartUnrecognisedException, JSONException {

    	if(attrs==null) {
    		return;
    	}
		final String styleId = attrs.optString("styleId", null);
		if(styleId!=null) {
			final CTTableProperties tblPr = table.getTblPr(true);
			if(styleId instanceof String) {
				tblPr.setTableStyleId(styleId);
			}
			else if(styleId==JSONObject.NULL) {
				tblPr.setTableStyleId(null);
			}
		}
    	final JSONObject tableAttrs = attrs.optJSONObject("table");
		applyTableGridFromJson(table, width, tableAttrs!=null ? tableAttrs.optJSONArray("tableGrid") : null);
    	if(tableAttrs!=null) {
    		final CTTableProperties tblPr = table.getTblPr(true);
    		applyFillPropertiesFromJson(tblPr, tableAttrs, operationDocument, part);
    		final JSONArray tableLook = tableAttrs.optJSONArray("exclude");
    		if(tableLook!=null) {
    			tblPr.setBandCol(true);
    			tblPr.setBandRow(true);
    			tblPr.setFirstCol(true);
    			tblPr.setFirstRow(true);
    			tblPr.setLastCol(true);
    			tblPr.setLastRow(true);
    			for(int i=0; i<tableLook.length(); i++) {
    				final Object o = tableLook.get(i);
    				if(o instanceof String) {
	    				switch((String)o) {
	    					case "bandsVert": {
	    						tblPr.setBandCol(null);
	    						break;
	    					}
	    					case "bandsHor": {
	    						tblPr.setBandRow(null);
	    						break;
		                	}
	    					case "firstCol": {
	    						tblPr.setFirstCol(null);
	    						break;
		                	}
	    					case "firstRow": {
	    						tblPr.setFirstRow(null);
	    						break;
		                	}
	    					case "lastCol": {
	    						tblPr.setLastCol(null);
	    						break;
		                	}
	    					case "lastRow": {
	    						tblPr.setLastRow(null);
	    						break;
		                	}
	    				}
    				}
    			}
    		}
     	}
    }

    public static void createJsonFromTable(JSONObject attrs, CTTable table, ThemePart themePart, Part part)
    	throws JSONException {

        final JSONObject initialTableAttrs = attrs.optJSONObject("table");
        final JSONObject tableAttrs = initialTableAttrs!=null ? initialTableAttrs : new JSONObject(1);
        final CTTableGrid tableGrid = table.getTblGrid(false);
        if(tableGrid!=null&&!tableGrid.getGridCol().isEmpty()) {
        	final JSONArray jsonTableGrid = new JSONArray(tableGrid.getGridCol().size());
        	final Iterator<CTTableCol> gridIter = tableGrid.getGridCol().iterator();
        	while(gridIter.hasNext()) {
        		jsonTableGrid.put(gridIter.next().getW());
        	}
        	tableAttrs.put("tableGrid", jsonTableGrid);
        }
        final CTTableProperties tableProperties = table.getTblPr(false);
        if(tableProperties!=null) {
        	final String tableStyleId = tableProperties.getTableStyleId();
        	if(tableStyleId!=null&&!tableStyleId.isEmpty()) {
        		attrs.put("styleId", tableStyleId);
        	}

        	createJsonFromFillProperties(attrs, tableProperties, themePart, null, part);

        	final JSONArray jsonLook = new JSONArray();
        	if(!tableProperties.isBandCol()) {
        		jsonLook.put("bandsVert");
        	}
        	if(!tableProperties.isBandRow()) {
        		jsonLook.put("bandsHor");
        	}
        	if(!tableProperties.isFirstCol()) {
        		jsonLook.put("firstCol");
        	}
        	if(!tableProperties.isFirstRow()) {
        		jsonLook.put("firstRow");
        	}
        	if(!tableProperties.isLastCol()) {
        		jsonLook.put("lastCol");
        	}
        	if(!tableProperties.isLastRow()) {
        		jsonLook.put("lastRow");
        	}
        	if(!jsonLook.isEmpty()) {
        		tableAttrs.put("exclude", jsonLook);
        	}

        	final CTTableStyle tableStyle = tableProperties.getTableStyle();
        	if(tableStyle!=null) {
        		// TODO ...
        	}
        }
        if(initialTableAttrs==null&&!tableAttrs.isEmpty()) {
            attrs.put("table", tableAttrs);
        }
    }

    // returns true if the table height needs to be recalculated
    public static boolean applyTableRowFromJson(CTTableRow tableRow, JSONObject attrs) {
    	boolean tableHeightRecalcRequired = false;
    	final JSONObject rowAttrs = attrs.optJSONObject("row");
    	if(rowAttrs!=null) {
	    	final Object height = rowAttrs.opt("height");
	    	if(height instanceof Number) {
	    		tableRow.setH(convert100thmmToEmu((Number)height));
	    		tableHeightRecalcRequired = true;
	    	}
    	}
    	return tableHeightRecalcRequired;
    }

    public static void createJsonFromTableRow(JSONObject attrs, CTTableRow row,  ThemePart themePart, Part part)
    	throws JSONException {

    	final JSONObject jsonRowAttrs = new JSONObject(1);
    	jsonRowAttrs.put("height", (int)convertEmuTo100thmm(row.getH()));
    	attrs.put("row", jsonRowAttrs);
    }

    public static void createJsonFromTableCell(JSONObject attrs, CTTableCell cell,  ThemePart themePart, Part part)
    	throws JSONException {

		final JSONObject jsonCellObject = new JSONObject();
    	final CTTableCellProperties cellProperties = cell.getTcPr(false);
    	if(cellProperties!=null) {
			createJsonFromFillProperties(jsonCellObject, cellProperties, themePart, null, part);
			final Object o = jsonCellObject.remove("fill");
			if(o instanceof JSONObject) {
				final Object c = ((JSONObject)o).remove("color");
				if(c!=null) {
					jsonCellObject.put("fillColor", c);
				}
			}
			createJsonBorderFromLineStyle(jsonCellObject, "borderLeft", cellProperties.getLnL(false), themePart, part);
            createJsonBorderFromLineStyle(jsonCellObject, "borderTop", cellProperties.getLnT(false), themePart, part);
            createJsonBorderFromLineStyle(jsonCellObject, "borderRight", cellProperties.getLnR(false), themePart, part);
            createJsonBorderFromLineStyle(jsonCellObject, "borderBottom", cellProperties.getLnB(false), themePart, part);

            final Integer marL = cellProperties.getMarL();
			if(marL!=null) {
				jsonCellObject.put("paddingLeft", (int)(marL.intValue() / 360));
			}
			final Integer marR = cellProperties.getMarR();
			if(marR!=null) {
				jsonCellObject.put("paddingRight", (int)(marR.intValue() / 360));
			}
			final Integer marT = cellProperties.getMarT();
			if(marT!=null) {
				jsonCellObject.put("paddingTop", (int)(marT.intValue() / 360));
			}
			final Integer marB = cellProperties.getMarB();
			if(marB!=null) {
				jsonCellObject.put("paddingBottom", (int)(marB.intValue() / 360));
			}
    	}
    	if(cell.getGridSpan()>1) {
    		jsonCellObject.put("gridSpan", cell.getGridSpan());
    	}
    	if(!jsonCellObject.isEmpty()) {
    		attrs.put("cell", jsonCellObject);
    	}
    }

    public static void applyTableCellFromJson(CTTableCell tableCell, JSONObject attrs, ThemePart themePart, Part part)
    	throws JSONException {

    	final JSONObject cellAttrs = attrs.optJSONObject("cell");
    	if(cellAttrs!=null) {
    		final Object fillColor = cellAttrs.opt("fillColor");
    		if(fillColor!=null) {
    			CTTableCellProperties tableCellProperties = tableCell.getTcPr(false);
    			if(tableCellProperties!=null) {
					tableCellProperties.setBlipFill(null);
					tableCellProperties.setGradFill(null);
					tableCellProperties.setNoFill(null);
					tableCellProperties.setGrpFill(null);
					tableCellProperties.setPattFill(null);
					tableCellProperties.setSolidFill(null);
    			}
    			if(fillColor instanceof JSONObject) {
    				tableCellProperties = tableCell.getTcPr(true);

    				final CTSolidColorFillProperties solidFill = new CTSolidColorFillProperties();
					tableCellProperties.setSolidFill(solidFill);
					createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)fillColor);
    			}
    		}
    		final CTTableCellProperties cellPr = tableCell.getTcPr(true);
    		cellPr.setLnL(applyLinePropertiesFromJsonBorder(cellAttrs, "borderLeft", tableCell.getTcPr(true).getLnL(false)));
            cellPr.setLnT(applyLinePropertiesFromJsonBorder(cellAttrs, "borderTop", tableCell.getTcPr(true).getLnT(false)));
            cellPr.setLnR(applyLinePropertiesFromJsonBorder(cellAttrs, "borderRight", tableCell.getTcPr(true).getLnR(false)));
            cellPr.setLnB(applyLinePropertiesFromJsonBorder(cellAttrs, "borderBottom", tableCell.getTcPr(true).getLnB(false)));
    		final Object gridSpan = cellAttrs.opt("gridSpan");
    		if(gridSpan!=null) {
    			if(gridSpan==JSONObject.NULL) {
    				removeGridSpan(tableCell);
    			}
    			else {
    				removeGridSpan(tableCell);
    				if((Integer)gridSpan!=1) {
    					applyGridSpan(tableCell, (Integer)gridSpan);
    				}
    			}
    		}
			final Object marL = cellAttrs.opt("paddingLeft");
			if(marL!=null) {
	            if(marL instanceof Number) {
	            	tableCell.getTcPr(true).setMarL(Integer.valueOf(((Number)marL).intValue()*360));
	            }
	            else {
	            	tableCell.getTcPr(true).setMarL(null);
	            }
			}
			final Object marR = cellAttrs.opt("paddingRight");
			if(marR!=null) {
	            if(marR instanceof Number) {
	            	tableCell.getTcPr(true).setMarR(Integer.valueOf(((Number)marR).intValue()*360));
	            }
	            else {
	            	tableCell.getTcPr(true).setMarR(null);
	            }
			}
			final Object marT = cellAttrs.opt("paddingTop");
			if(marT!=null) {
	            if(marT instanceof Number) {
	            	tableCell.getTcPr(true).setMarT(Integer.valueOf(((Number)marT).intValue()*360));
	            }
	            else {
	            	tableCell.getTcPr(true).setMarT(null);
	            }
			}
			final Object marB = cellAttrs.opt("paddingBottom");
			if(marB!=null) {
	            if(marB instanceof Number) {
	            	tableCell.getTcPr(true).setMarB(Integer.valueOf(((Number)marB).intValue()*360));
	            }
	            else {
	            	tableCell.getTcPr(true).setMarB(null);
	            }
			}
    	}
    }

    private static CTLineProperties applyLinePropertiesFromJsonBorder(JSONObject cell, String borderName, CTLineProperties lineProperties)
        throws JSONException {

        final Object o = cell.opt(borderName);
        if(o!=null) {
            if(o instanceof JSONObject) {
                final JSONObject cellBorder = (JSONObject)o;
                if(lineProperties==null) {
                    lineProperties = new CTLineProperties();
                }
                applyBorderPropertiesFromJson(lineProperties, cellBorder);
            }
            else {
                lineProperties = null;
            }
        }
        return lineProperties;
    }

    private static void createJsonBorderFromLineStyle(JSONObject cell, String borderName, CTLineProperties lineProperties, ThemePart themePart, Part part)
        throws JSONException {

        if(lineProperties!=null) {
            final JSONObject border = new JSONObject();
            createJsonFromLineProperties(border, themePart, null, lineProperties, part);
            final Object o = border.remove("line");
            if(o instanceof JSONObject) {
                Object c = ((JSONObject)o).remove("color");
                if(c!=null) {
                    border.put("color", c);
                }
                c = ((JSONObject)o).remove("type");
                if(c!=null) {
                    border.put("style", c);
                }
                c = ((JSONObject)o).remove("width");
                if(c!=null) {
                    border.put("width", c);
                }
            }
            if(!border.isEmpty()) {
                cell.put(borderName, border);
            }
        }
    }

    private static void removeGridSpan(CTTableCell cell) {
    	int gridCount = cell.getGridSpan() - 1;
    	if(gridCount>0) {
        	final CTTableRow row = (CTTableRow)cell.getParent();
        	final IndexedNodeList<Object> tableContent = ((CTTable)row.getParent()).getContent();
        	IndexedNode<Object> rowNode = tableContent.getNode(tableContent.indexOf(row));
        	final int cellIndex = (((CTTableRow)rowNode.getData()).getContent()).indexOf(cell);
			final IndexedNodeList<Object> rowContent = ((CTTableRow)rowNode.getData()).getContent();
    		if(cellIndex>=0) {
    			rowContent.removeNodes(rowContent.getNextNode(rowContent.getNode(cellIndex)), gridCount);
    		}
    	}
    	cell.setGridSpan(null);
    }

    private static void applyGridSpan(CTTableCell cell, int gridSpan) {
    	int gridCount = gridSpan - 1;
    	if(gridCount>0) {
        	final CTTableRow row = (CTTableRow)cell.getParent();
        	final IndexedNodeList<Object> tableContent = ((CTTable)row.getParent()).getContent();
        	IndexedNode<Object> rowNode = tableContent.getNode(tableContent.indexOf(row));
        	final int cellIndex = (((CTTableRow)rowNode.getData()).getContent()).indexOf(cell);
			final IndexedNodeList<Object> rowContent = ((CTTableRow)rowNode.getData()).getContent();
			for(int i=0; i<gridCount; i++) {
				final CTTableCell mergedCell = new CTTableCell();
				mergedCell.setHMerge(true);
				mergedCell.setParent(row);
				rowContent.add(cellIndex+1, mergedCell);
			}
    	}
    	cell.setGridSpan(gridSpan);
    }

    final static public String[] tableStyleOperationNames = {"wholeTable", "band1Hor", "band2Hor", "band1Vert", "band2Vert", "lastCol", "firstCol", "lastRow", "southEastCell", "southWestCell", "firstRow", "northEastCell", "northWestCell"};

    public static void applyTableStyleFromJson(CTTableStyle tableStyle, JSONObject attrs, OperationDocument operationDocument, ThemePart themePart, Part part)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException {

    	final JSONObject backgroundStyle = attrs.optJSONObject("backgroundStyle");
    	if(backgroundStyle!=null) {
			final CTTableBackgroundStyle tableBackgroundStyle = new CTTableBackgroundStyle();
			tableStyle.setTblBg(tableBackgroundStyle);
			final CTFillProperties tableFill = new CTFillProperties();
			tableBackgroundStyle.setFill(tableFill);
			applyFillPropertiesFromJson(tableFill, backgroundStyle, operationDocument, part);
    	}
    	for(int i=0; i<CTTableStyle.tableStyleNames.length; i++) {
    		tableStyle.setTableStyleByName(CTTableStyle.tableStyleNames[i], applyTablePartStyleFromJson(attrs, tableStyleOperationNames[i],
    			tableStyle.getTableStyleByName(CTTableStyle.tableStyleNames[i], false), themePart, part));
    	}
    }

    public static void createJsonFromTableStyle(JSONObject attrs, CTTableStyle tableStyle, ThemePart themePart, Part part)
    	throws JSONException {

    	final CTTableBackgroundStyle backgroundStyle = tableStyle.getTblBg();
    	if(backgroundStyle!=null) {
    		if(backgroundStyle.getFill()!=null||backgroundStyle.getFillRef()!=null) {
    			final JSONObject back = new JSONObject();
    			createJsonFromFillProperties(back, backgroundStyle.getFill(), themePart, backgroundStyle.getFillRef(), part);
    			if(!back.isEmpty()) {
    				attrs.put("backgroundStyle", back);
    			}
    		}
    	}
    	for(int i=0; i<CTTableStyle.tableStyleNames.length; i++) {
        	createJsonFromTablePartStyle(attrs, tableStyleOperationNames[i], tableStyle.getTableStyleByName(CTTableStyle.tableStyleNames[i], false), themePart, part);
    	}
    }

    private static CTTablePartStyle applyTablePartStyleFromJson(JSONObject attrs, String condName, CTTablePartStyle tablePartStyle, ThemePart themePart, Part part)
    	throws JSONException {

    	Object o = attrs.optJSONObject(condName);
    	if(o!=null) {
    		if(o instanceof JSONObject) {
    			final JSONObject condTableStyle = (JSONObject)o;
				final JSONObject cellAttrs = condTableStyle.optJSONObject("cell");
				if(cellAttrs!=null) {
	    			if(tablePartStyle==null) {
	    				tablePartStyle = new CTTablePartStyle();
	    			}
	    			final CTTableStyleCellStyle tableStyleCellStyle = tablePartStyle.getTcStyle(true);
	    			final Object color = cellAttrs.opt("fillColor");
	    			if(color!=null) {
						tableStyleCellStyle.setFillRef(null);
	    				if(color instanceof JSONObject) {
	    					final CTFillProperties fillProperties = new CTFillProperties();
	    					tableStyleCellStyle.setFill(fillProperties);
	    					CTSolidColorFillProperties solidFill = new CTSolidColorFillProperties();
	    					fillProperties.setSolidFill(solidFill);
	    					createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)color);
	    				}
	    				else {
	    					tableStyleCellStyle.setFill(null);
	    				}
	    			}
	    			final CTTableCellBorderStyle tableCellBorderStyle = tableStyleCellStyle.getTcBdr(true);
	    			tableCellBorderStyle.setLeft(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderLeft", tableCellBorderStyle.getLeft(), themePart, part));
	    			tableCellBorderStyle.setRight(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderRight", tableCellBorderStyle.getRight(), themePart, part));
	    			tableCellBorderStyle.setBottom(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderBottom", tableCellBorderStyle.getBottom(), themePart, part));
	    			tableCellBorderStyle.setTop(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderTop", tableCellBorderStyle.getTop(), themePart, part));
	    			tableCellBorderStyle.setInsideH(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderInsideHor", tableCellBorderStyle.getInsideH(), themePart, part));
	    			tableCellBorderStyle.setInsideV(applyThemeableLineStyleFromJsonBorder(cellAttrs, "borderInsideVert", tableCellBorderStyle.getInsideV(), themePart, part));
	    			if(tableCellBorderStyle.getLeft()==null&&tableCellBorderStyle.getRight()==null&&tableCellBorderStyle.getBottom()==null
	    				&& tableCellBorderStyle.getTop()==null&&tableCellBorderStyle.getInsideH()==null&&tableCellBorderStyle.getInsideV()==null) {
	    				tableStyleCellStyle.setTcBdr(null);
	    			}
    			}
				final JSONObject characterAttrs = condTableStyle.optJSONObject("character");
				if(characterAttrs!=null) {
	    			if(tablePartStyle==null) {
	    				tablePartStyle = new CTTablePartStyle();
	    			}
					applyTableStyleTextStyleFromJson(characterAttrs, tablePartStyle.getTcTxStyle(true));
				}
    		}
    		else {
    			tablePartStyle = null;
    		}
    	}
    	return tablePartStyle;
    }

    private static void createJsonFromTablePartStyle(JSONObject attrs, String condName, CTTablePartStyle tablePartStyle, ThemePart themePart, Part part)
    	throws JSONException {

    	if(tablePartStyle!=null) {
    		final JSONObject condTableStyle = new JSONObject();
    		final CTTableStyleCellStyle tableStyleCellStyle = tablePartStyle.getTcStyle(false);
    		if(tableStyleCellStyle!=null) {
    			final JSONObject cell = new JSONObject();
    			final CTFillProperties fillProperties = tableStyleCellStyle.getFill();
    			final CTStyleMatrixReference fillStyleMatrixRef = tableStyleCellStyle.getFillRef();
    			if(fillProperties!=null||fillStyleMatrixRef!=null) {
    				createJsonFromFillProperties(cell, fillProperties, themePart, fillStyleMatrixRef, part);
    				final Object o = cell.remove("fill");
    				if(o instanceof JSONObject) {
    					final Object c = ((JSONObject)o).remove("color");
    					if(c!=null) {
    						cell.put("fillColor", c);
    					}
    				}
    			}
    			final CTTableCellBorderStyle tableCellBorderStyle = tableStyleCellStyle.getTcBdr(false);
    			if(tableCellBorderStyle!=null) {
    				createJsonBorderFromThemeableLineStyle(cell, "borderLeft", tableCellBorderStyle.getLeft(), themePart, part);
    				createJsonBorderFromThemeableLineStyle(cell, "borderRight", tableCellBorderStyle.getRight(), themePart, part);
    				createJsonBorderFromThemeableLineStyle(cell, "borderBottom", tableCellBorderStyle.getBottom(), themePart, part);
    				createJsonBorderFromThemeableLineStyle(cell, "borderTop", tableCellBorderStyle.getTop(), themePart, part);
    				createJsonBorderFromThemeableLineStyle(cell, "borderInsideHor", tableCellBorderStyle.getInsideH(), themePart, part);
    				createJsonBorderFromThemeableLineStyle(cell, "borderInsideVert", tableCellBorderStyle.getInsideV(), themePart, part);
    			}
    			if(!cell.isEmpty()) {
    				condTableStyle.put("cell", cell);
    			}
    		}
    		final CTTableStyleTextStyle tableStyleTextStyle = tablePartStyle.getTcTxStyle(false);
    		if(tableStyleTextStyle!=null) {
    			createJsonFromTableStyleTextStyle(condTableStyle, tableStyleTextStyle, themePart);
    		}
    		if(!condTableStyle.isEmpty()) {
    			attrs.put(condName, condTableStyle);
    		}
    	}
    }


    private static CTThemeableLineStyle applyThemeableLineStyleFromJsonBorder(JSONObject cell, String borderName, CTThemeableLineStyle lineStyle, ThemePart themePart, Part part)
    	throws JSONException {

    	final Object o = cell.opt(borderName);
    	if(o!=null) {
    		if(o instanceof JSONObject) {
    			final JSONObject cellBorder = (JSONObject)o;
    			if(lineStyle==null) {
    				lineStyle = new CTThemeableLineStyle();
    			}
    			lineStyle.setLn(applyBorderPropertiesFromJson(lineStyle.getLn(false), cellBorder));
    		}
    		else {
    			return null;
    		}
    	}
    	if(lineStyle!=null&&lineStyle.getLn(false)==null&&lineStyle.getLnRef()==null) {
    		return null;
    	}
    	return lineStyle;
    }

    private static void createJsonBorderFromThemeableLineStyle(JSONObject cell, String borderName, CTThemeableLineStyle lineStyle, ThemePart themePart, Part part)
    	throws JSONException {

    	if(lineStyle!=null) {
    		final JSONObject border = new JSONObject();
    		createJsonFromLineProperties(border, themePart, lineStyle.getLnRef(), lineStyle.getLn(false), part);
    		final Object o = border.remove("line");
    		if(o instanceof JSONObject) {
    			Object c = ((JSONObject)o).remove("color");
    			if(c!=null) {
    				border.put("color", c);
    			}
    			c = ((JSONObject)o).remove("type");
    			if(c!=null) {
        			border.put("style", c);
    			}
    			c = ((JSONObject)o).remove("width");
    			if(c!=null) {
        			border.put("width", c);
    			}
    		}
    		if(!border.isEmpty()) {
    			cell.put(borderName, border);
    		}
    	}
    }

    private static void applyTableStyleTextStyleFromJson(JSONObject characterAttrs, CTTableStyleTextStyle textStyle)
    	throws JSONException {

    	final Iterator<Entry<String, Object>> entries = characterAttrs.entrySet().iterator();
        while(entries.hasNext()) {
        	final Entry<String, Object> entry = entries.next();
        	final Object value = entry.getValue();
        	switch(entry.getKey()) {
        		case "bold": {
        			if(value==JSONObject.NULL) {
        				textStyle.setB(null);
        			}
        			else {
        				textStyle.setB(((Boolean)value).booleanValue() ? STOnOffStyleType.ON : STOnOffStyleType.OFF);
        			}
        			break;
        		}
        		case "color": {
        		    createSolidColorFillPropertiesFromJson(textStyle, (JSONObject)value);
        			break;
        		}
        	}
        }
    }

    private static void createJsonFromTableStyleTextStyle(JSONObject attrs, CTTableStyleTextStyle textStyle, ThemePart themePart)
    	throws JSONException {

    	final JSONObject character = new JSONObject();
    	final CTFontReference fontReference = textStyle.getFontRef(false);
    	if(fontReference!=null&&themePart!=null) {
    		final STFontCollectionIndex fontCollectionIndex = fontReference.getIdx();
    		if(fontCollectionIndex!=null) {
    			if(fontCollectionIndex==STFontCollectionIndex.MAJOR) {
    				character.put("fontName", new ThemeFonts(themePart).getMajorFont());
    			}
    			else if(fontCollectionIndex==STFontCollectionIndex.MINOR) {
    				character.put("fontName", new ThemeFonts(themePart).getMajorFont());
    			}
    		}
    		final JSONObject refColor = createJsonColorFromSolidColorFillProperties(fontReference);
    		if(refColor!=null) {
    			character.put("color", refColor);
    		}
    	}
    	final STOnOffStyleType bold = textStyle.getB();
    	if(bold!=null) {
    		character.put("bold", bold==STOnOffStyleType.ON);
    	}
    	final FontCollection fontCollection = textStyle.getFont();
    	if(fontCollection!=null) {

    	}
    	final JSONObject color = createJsonColorFromSolidColorFillProperties(textStyle);
    	if(color!=null) {
    		character.put("color", color);
    	}
    	if(!character.isEmpty()) {
    		attrs.put("character", character);
    	}
    }

    public static void createJsonFromFillProperties(JSONObject attrs, IFillProperties fillProperties, ThemePart themePart, IStyleMatrixReference fillStyleReference, Part part)
        throws JSONException {

        final JSONObject initialFillAttrs = attrs.optJSONObject("fill");
        final JSONObject fillAttrs = initialFillAttrs!=null ? initialFillAttrs : new JSONObject(1);

        if(fillStyleReference!=null) {
            // try to apply the style first...
            createJsonFromFillStyleMatrix(themePart, fillStyleReference, fillAttrs);
        }

        if(fillProperties!=null) {

        	// then each hard applied fill attribute
        	if(fillProperties.getNoFill()!=null) {
        	    createJsonFromFillStyle(fillProperties.getNoFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getSolidFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getSolidFill(), null, fillAttrs, part);
            }
	    	else if(fillProperties.getBlipFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getBlipFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getGradFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getGradFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getPattFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getPattFill(), null, fillAttrs, part);
	    	}
        	/*
        	if(shapeProperties.getGrpFill()!=null) {

        	}
        	*/
        }
        if(initialFillAttrs==null&&!fillAttrs.isEmpty()) {
            attrs.put("fill", fillAttrs);
        }
    }

    public static void applyConnectionPropertiesFromJson(JSONObject attrs, IConnectionAccessor connectionAccessor) {

        final JSONObject connector = attrs.optJSONObject("connector");
        if(connector!=null) {
            final Object startId = connector.opt("startId");
            if(startId!=null) {
                if(startId==JSONObject.NULL||((String)startId).isEmpty()) {
                    connectionAccessor.setStartConnection(null);
                }
                else {
                    connectionAccessor.getStartConnection(true).setId(Long.parseLong(((String)startId)));
                }
            }
            final Object startIndex = connector.opt("startIndex");
            if(startIndex!=null) {
                if(startIndex==JSONObject.NULL) {
                    connectionAccessor.setStartConnection(null);
                }
                else {
                    connectionAccessor.getStartConnection(true).setIdx(((Number)startIndex).longValue());
                }
            }
            final Object endId = connector.opt("endId");
            if(endId!=null) {
                if(endId==JSONObject.NULL||((String)endId).isEmpty()) {
                    connectionAccessor.setEndConnection(null);
                }
                else {
                    connectionAccessor.getEndConnection(true).setId(Long.parseLong(((String)endId)));
                }
            }
            final Object endIndex = connector.opt("endIndex");
            if(endIndex!=null) {
                if(endIndex==JSONObject.NULL) {
                    connectionAccessor.setEndConnection(null);
                }
                else {
                    connectionAccessor.getEndConnection(true).setIdx(((Number)endIndex).longValue());
                }
            }
        }
    }

    public static void createJsonFromConnectionProperties(JSONObject attrs, IConnectionAccessor connectionAccessor)
        throws JSONException {

        final CTConnection startConnection = connectionAccessor.getStartConnection(false);
        final CTConnection endConnection = connectionAccessor.getEndConnection(false);
        if(startConnection!=null||endConnection!=null) {
            final JSONObject connector = new JSONObject(2);
            if(startConnection!=null) {
                connector.put("startId", Long.toString(startConnection.getId()));
                connector.put("startIndex", startConnection.getIdx());
            }
            if(endConnection!=null) {
                connector.put("endId", Long.toString(endConnection.getId()));
                connector.put("endIndex", endConnection.getIdx());
            }
            attrs.put("connector", connector);
        }
    }

    public static CTShapeProperties applyShapePropertiesFromJson(CTShapeProperties shapeProperties, JSONObject attrs, OperationDocument operationDocument, Part part, boolean rootShape)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		if(shapeProperties==null) {
			shapeProperties = Context.getDmlObjectFactory().createCTShapeProperties();
		}
		applyTransform2DFromJson(shapeProperties, attrs, rootShape);
		final JSONObject lineAttrs = attrs.optJSONObject("line");
		if(lineAttrs!=null) {
			shapeProperties.setLn(applyLinePropertiesFromJson(shapeProperties.getLn(), lineAttrs));
		}
		applyFillPropertiesFromJson(shapeProperties, attrs, operationDocument, part);

		final JSONObject geometryAttrs = attrs.optJSONObject("geometry");
		if(geometryAttrs!=null) {
			DMLGeometry.applyGeometryPropertiesFromJson(geometryAttrs, shapeProperties);
		}
		return shapeProperties;
	}

    public static void createJsonFromFontRef(JSONObject attrs, IFontReference fontStyleReference)
    	throws JSONException {

    	final JSONObject initialCharacterAttrs = attrs.optJSONObject("character");
        final JSONObject characterAttrs = initialCharacterAttrs!=null ? initialCharacterAttrs : new JSONObject(2);

        final JSONObject fontColor = new JSONObject();
        createJsonColorFromSolidColorFillProperties(fontColor, fontStyleReference);
        if(fontColor!=null&&!fontColor.isEmpty()) {
        	characterAttrs.put("color", fontColor);
        }
        if(initialCharacterAttrs==null&&!characterAttrs.isEmpty()) {
       		attrs.put("character", characterAttrs);
       	}
    }

    public static void createJsonFromLineProperties(JSONObject attrs, ThemePart themePart, IStyleMatrixReference lineStyleReference, CTLineProperties lineProperties, Part part)
        throws JSONException {

        final JSONObject initialLineAttrs = attrs.optJSONObject("line");
        final JSONObject lineAttrs = initialLineAttrs!=null ? initialLineAttrs : new JSONObject(2);

        if(lineStyleReference!=null) {
            // try to apply the style first...
            createJsonFromLineStyleMatrix(themePart, lineStyleReference, lineAttrs);
        }
    	if(lineProperties!=null) {
    	    createJsonFromLineStyle(lineProperties, null, lineAttrs, part);
    	}

    	if(initialLineAttrs==null&&!lineAttrs.isEmpty()) {
            attrs.put("line", lineAttrs);
        }
    }

	public static CTLineProperties applyLinePropertiesFromJson(CTLineProperties lineProperties, JSONObject lineAttrs) throws JSONException {

		if(lineProperties==null) {
			lineProperties = Context.getDmlObjectFactory().createCTLineProperties();
		}
		final Object typeO = lineAttrs.opt("type");
		if(null!=typeO) {
			final String type = typeO instanceof String ? (String)typeO : "none";
            final CTSolidColorFillProperties oldSolidFill = lineProperties.getSolidFill();
		    lineProperties.setNoFill(null);
            lineProperties.setSolidFill(null);
            lineProperties.setGradFill(null);
            lineProperties.setPattFill(null);
            if(type.equals("none")) {
                lineProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
            }
            else if(type.equals("solid")) {
                if(oldSolidFill!=null) {
                    lineProperties.setSolidFill(oldSolidFill);
                } else {
                    lineProperties.setSolidFill(Context.getDmlObjectFactory().createCTSolidColorFillProperties());
                }
            }
		}
		final Object color = lineAttrs.opt("color");
		if(color!=null) {
		    // applying color only possible if fill type is solid
		    final CTSolidColorFillProperties solidFill = lineProperties.getSolidFill();
		    if(solidFill!=null) {
		        if(color instanceof JSONObject) {
		            createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)color);
                }
		        else {
                    final CTPresetColor presetColor = Context.getDmlObjectFactory().createCTPresetColor();
                    presetColor.setVal(STPresetColorVal.BLACK);
                    solidFill.setPrstClr(presetColor);
		        }
		    }
		}
		final Object width = lineAttrs.opt("width");
		if(width!=null) {
		    if(width instanceof Number) {
                lineProperties.setW(convert100thmmToEmu((Number)width));
		    }
		    else {
		        lineProperties.setW(null);
		    }
		}
		final Object style = lineAttrs.opt("style");
		if(style!=null) {
		    CTPresetLineDashProperties presetDash = null;
		    if(style instanceof String) {
		        if(((String)style).equals("dotted")) {
		            presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
		            presetDash.setVal(STPresetLineDashVal.DOT);
		        }
		        else if (((String)style).equals("dashed")) {
                    presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                    presetDash.setVal(STPresetLineDashVal.DASH);
		        }
		        else if (((String)style).equals("dashDot")) {
                    presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                    presetDash.setVal(STPresetLineDashVal.DASH_DOT);
		        }
		        else if (((String)style).equals("dashDotDot")) {
                    presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                    presetDash.setVal(STPresetLineDashVal.LG_DASH_DOT_DOT);
		        }
		    }
	        lineProperties.setPrstDash(presetDash);
		}
		final Object headEndType = lineAttrs.opt("headEndType");
		final Object headEndLength = lineAttrs.opt("headEndLength");
		final Object headEndWidth = lineAttrs.opt("headEndWidth");
		if(headEndType!=null||headEndLength!=null||headEndWidth!=null) {
	        applyLineEndType(lineProperties.getHeadEnd(true), headEndType, headEndLength, headEndWidth);
		}
        final Object tailEndType = lineAttrs.opt("tailEndType");
        final Object tailEndLength = lineAttrs.opt("tailEndLength");
        final Object tailEndWidth = lineAttrs.opt("tailEndWidth");
        if(tailEndType!=null||tailEndLength!=null||tailEndWidth!=null) {
            applyLineEndType(lineProperties.getTailEnd(true), tailEndType, tailEndLength, tailEndWidth);
        }
		return lineProperties;
	}

	private static void applyLineEndType(CTLineEndProperties lineEndProperties, Object endType, Object endLength, Object endWidth) {
	    if(endType!=null) {
	        if(endType==JSONObject.NULL) {
	            lineEndProperties.setType(STLineEndType.NONE);
	        }
	        else {
	            STLineEndType lineEndType = STLineEndType.NONE;
	            if(((String)endType).equals("triangle")) {
	                lineEndType = STLineEndType.TRIANGLE;
	            }
	            else if(((String)endType).equals("stealth")) {
	                lineEndType = STLineEndType.STEALTH;
	            }
	            else if(((String)endType).equals("diamond")) {
	                lineEndType = STLineEndType.DIAMOND;
	            }
	            else if(((String)endType).equals("oval")) {
	                lineEndType = STLineEndType.OVAL;
	            }
	            else if(((String)endType).equals("arrow")) {
	                lineEndType = STLineEndType.ARROW;
	            }
	            lineEndProperties.setType(lineEndType);
	        }
	    }
	    if(endLength!=null) {
	        if(endLength==JSONObject.NULL) {
	            lineEndProperties.setLen(STLineEndLength.MED);
	        }
	        else {
	            STLineEndLength lineEndLength = STLineEndLength.MED;
                if(((String)endLength).equals("small")) {
                    lineEndLength = STLineEndLength.SM;
                }
                else if(((String)endLength).equals("stealth")) {
                    lineEndLength = STLineEndLength.LG;
                }
                lineEndProperties.setLen(lineEndLength);
	        }
	    }
        if(endWidth!=null) {
            if(endWidth==JSONObject.NULL) {
                lineEndProperties.setW(STLineEndWidth.MED);
            }
            else {
                STLineEndWidth lineEndWidth = STLineEndWidth.MED;
                if(((String)endWidth).equals("small")) {
                    lineEndWidth = STLineEndWidth.SM;
                }
                else if(((String)endWidth).equals("stealth")) {
                    lineEndWidth = STLineEndWidth.LG;
                }
                lineEndProperties.setW(lineEndWidth);
            }
        }
	}

	public static CTLineProperties applyBorderPropertiesFromJson(CTLineProperties lineProperties, JSONObject borderAttrs) throws JSONException {

		if(lineProperties==null) {
			lineProperties = Context.getDmlObjectFactory().createCTLineProperties();
		}
		final Object typeO = borderAttrs.opt("style");
		if(null!=typeO) {
			final String type = typeO instanceof String ? (String)typeO : "none";
            final CTSolidColorFillProperties oldSolidFill = lineProperties.getSolidFill();
		    lineProperties.setNoFill(null);
            lineProperties.setSolidFill(null);
            lineProperties.setGradFill(null);
            lineProperties.setPattFill(null);
            if(type.equals("none")) {
                lineProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
            }
            else {
                if(oldSolidFill!=null) {
                    lineProperties.setSolidFill(oldSolidFill);
                } else {
                    lineProperties.setSolidFill(Context.getDmlObjectFactory().createCTSolidColorFillProperties());
                }
            }
		}
		final Object color = borderAttrs.opt("color");
		if(color!=null) {
		    // applying color only possible if fill type is solid
		    final CTSolidColorFillProperties solidFill = lineProperties.getSolidFill();
		    if(solidFill!=null) {
		        if(color instanceof JSONObject) {
		            createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)color);
                }
		        else {
                    final CTPresetColor presetColor = Context.getDmlObjectFactory().createCTPresetColor();
                    presetColor.setVal(STPresetColorVal.BLACK);
                    solidFill.setPrstClr(presetColor);
		        }
		    }
		}
		final Object width = borderAttrs.opt("width");
		if(width!=null) {
		    if(width instanceof Number) {
                lineProperties.setW(convert100thmmToEmu((Number)width));
		    }
		    else {
		        lineProperties.setW(null);
		    }
		}
		return lineProperties;
	}

    public static JSONObject createJsonColorFromSolidColorFillProperties(IColorChoice fillProperties)
    	throws JSONException {

    	return createJsonColorFromSolidColorFillProperties(null, fillProperties);
    }

    public static JSONObject createJsonColorFromSolidColorFillProperties(JSONObject sourceColor, IColorChoice fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
        	if (fillProperties.getScrgbClr()!= null) {
        		final CTScRgbColor color = fillProperties.getScrgbClr();
                final byte[] value = new byte[3];
                value[0] = JSONHelper.gammaChannel(color.getR());
                value[1] = JSONHelper.gammaChannel(color.getG());
                value[2] = JSONHelper.gammaChannel(color.getB());
                return JSONHelper.makeColor(sourceColor, "rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform());
            }
            else if(fillProperties.getSrgbClr()!= null) {
            	final CTSRgbColor color = fillProperties.getSrgbClr();
            	final byte[] value = color.getVal();
            	return JSONHelper.makeColor(sourceColor, "rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform());
            }
            else if (fillProperties.getHslClr()!= null) {
            	final CTHslColor color = fillProperties.getHslClr();
            	return JSONHelper.makeColor(sourceColor, "hsl", JSONHelper.makeJSON("h", color.getHue(), "s", color.getSat(), "l", color.getLum()), color.getEGColorTransform());
            }
            else if (fillProperties.getSchemeClr()!= null) {
            	final CTSchemeColor color = fillProperties.getSchemeClr();
                String value = null;
                final STSchemeColorVal sourveVal = color.getVal();
                if (sourveVal != null) {
                    switch (sourveVal) {
        			case ACCENT_1:
        			case ACCENT_2:
        			case ACCENT_3:
        			case ACCENT_4:
        			case ACCENT_5:
        			case ACCENT_6:
        				value = sourveVal.value();
        				break;
        			case BG_1:
        				value = "background1";
        				break;
        			case BG_2:
        				value = "background2";
        				break;
        			case DK_1:
        				value = "dark1";
        				break;
        			case DK_2:
        				value = "dark2";
        				break;
        			case TX_1:
        				value = "text1";
        				break;
        			case TX_2:
        				value = "text2";
        				break;
        			case LT_1:
        				value = "light1";
        				break;
        			case LT_2:
        				value = "light2";
        				break;
        			case HLINK:
        				value = "hyperlink";
        				break;
        			case FOL_HLINK:
        				value = "followedHyperlink";
        				break;
        			case PH_CLR:
        				value = "phClr";
        				break;
        			}
                }
                return JSONHelper.makeColor(sourceColor, "scheme", value, color.getEGColorTransform());
            }
            else if (fillProperties.getSysClr()!= null) {
            	final CTSystemColor color = fillProperties.getSysClr();
            	return JSONHelper.makeColor(sourceColor, "system", color.getVal(), color.getEGColorTransform());
            }
            else if (fillProperties.getPrstClr()!= null) {
            	final CTPresetColor color = fillProperties.getPrstClr();
            	return JSONHelper.makeColor(sourceColor, "preset", color.getVal().value(), color.getEGColorTransform());
            }
        }
        return null;
    }

	public static boolean createSolidColorFillPropertiesFromJson(IColorChoice docx, JSONObject json) throws JSONException {
		docx.setSysClr(null);
		docx.setScrgbClr(null);
		docx.setSrgbClr(null);
		docx.setHslClr(null);
		docx.setPrstClr(null);
		docx.setSchemeClr(null);


		final String type = json.getString("type");

		if (type.equals("auto")) {
			return false;
		}else if (type.equals("scheme")) {
			final CTSchemeColor schemeClr = new CTSchemeColor();
			createSchemeColorFromJson(schemeClr, json);
			docx.setSchemeClr(schemeClr);
		} else if (type.equals("rgb")) {
			final CTSRgbColor srgbClr = new CTSRgbColor();
			createSRgbColorFromJson(srgbClr, json);
			docx.setSrgbClr(srgbClr);
		} else {
			Logger.getAnonymousLogger().warning("colortype is not implemented " + type);
			// TODO: other colors?
		}
		return true;
	}

	public static void createSchemeColorFromJson(CTSchemeColor docx, JSONObject json) throws JSONException {
		final String value = json.getString("value");
		final STSchemeColorVal colorVal;
		if (value.equals("background1")) {
			colorVal = STSchemeColorVal.BG_1;
		} else if (value.equals("background2")) {
			colorVal = STSchemeColorVal.BG_2;
		} else if (value.equals("dark1")) {
			colorVal = STSchemeColorVal.DK_1;
		} else if (value.equals("dark2")) {
			colorVal = STSchemeColorVal.DK_2;
		} else if (value.equals("text1")) {
			colorVal = STSchemeColorVal.TX_1;
		} else if (value.equals("text2")) {
			colorVal = STSchemeColorVal.TX_2;
		} else if (value.equals("light1")) {
			colorVal = STSchemeColorVal.LT_1;
		} else if (value.equals("light2")) {
			colorVal = STSchemeColorVal.LT_2;
		} else if (value.equals("hyperlink")) {
			colorVal = STSchemeColorVal.HLINK;
		} else if (value.equals("followedHyperlink")) {
			colorVal = STSchemeColorVal.FOL_HLINK;
		} else {
			STSchemeColorVal tmpVal;
			try {
				tmpVal = STSchemeColorVal.fromValue(value);
			} catch (final IllegalArgumentException e) {
				Logger.getAnonymousLogger().warning("could not find STSchemeColorVal for " + value);
				tmpVal = STSchemeColorVal.TX_1;
			}
			colorVal = tmpVal;
		}
		docx.setVal(colorVal);
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

	public static void createSRgbColorFromJson(CTSRgbColor docx, JSONObject json) throws JSONException {
		docx.setVal(Commons.hexStringToBytes(json.getString("value")));
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

    public static void createJsonFromNonVisualDrawingShapeProperties(JSONObject attrs, INonVisualDrawingShapePropertyAccess iNonVisualDrawingShapePropertyAccess)
        throws JSONException {

    	final ILocksAccess iLocksAccess = iNonVisualDrawingShapePropertyAccess.getNonVisualDrawingShapeProperties(false);
        if(iLocksAccess==null) {
            return;
        }
        final CTLockingBase lockingBase = iLocksAccess.getLocks(false);
        if(lockingBase==null) {
        	return;
        }
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

		if(lockingBase.isNoChangeAspect()) {
			drawingAttrs.put("noChangeAspect", true);
		}
		if(lockingBase.isNoMove()) {
			drawingAttrs.put("noMove", true);
		}
		if(lockingBase.isNoResize()) {
			drawingAttrs.put("noResize", true);
		}
		if(lockingBase.isNoSelect()) {
			drawingAttrs.put("noSelect", true);
		}
		if(lockingBase.isNoGrp()) {
			drawingAttrs.put("noGroup", true);
		}
		if(lockingBase instanceof CTLockingBase.INoRot) {
			if(((CTLockingBase.INoRot)lockingBase).isNoRot()) {
				drawingAttrs.put("noRotation", true);
			}
		}
		if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
			attrs.put("drawing", drawingAttrs);
		}
    }

    public static void applyNonVisualDrawingShapeProperties(INonVisualDrawingShapePropertyAccess iNonVisualDrawingShapePropertyAccess, JSONObject drawingProperties) {
    	if(drawingProperties==null) {
    		return;
    	}
    	Object o = drawingProperties.opt("noChangeAspect");
    	if(o!=null) {
    		if(o instanceof Boolean) {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoChangeAspect((Boolean)o);
    		}
    		else {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoChangeAspect(null);
    		}
    	}
    	o = drawingProperties.opt("noMove");
    	if(o!=null) {
    		if(o instanceof Boolean) {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoMove((Boolean)o);
    		}
    		else {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoMove(null);
    		}
    	}
    	o = drawingProperties.opt("noResize");
    	if(o!=null) {
    		if(o instanceof Boolean) {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoResize((Boolean)o);
    		}
    		else {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoResize(null);
    		}
    	}
    	o = drawingProperties.opt("noSelect");
    	if(o!=null) {
    		if(o instanceof Boolean) {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoSelect((Boolean)o);
    		}
    		else {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoSelect(null);
    		}
    	}
    	o = drawingProperties.opt("noGroup");
    	if(o!=null) {
    		if(o instanceof Boolean) {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoGrp((Boolean)o);
    		}
    		else {
    			getLockingBase(iNonVisualDrawingShapePropertyAccess).setNoGrp(null);
    		}
    	}
    	o = drawingProperties.opt("noRotation");
    	if(o!=null) {
    		final CTLockingBase lockingBase = getLockingBase(iNonVisualDrawingShapePropertyAccess);
    		if(lockingBase instanceof CTLockingBase.INoRot) {
	    		if(o instanceof Boolean) {
	    			((CTLockingBase.INoRot)lockingBase).setNoRot((Boolean)o);
	    		}
	    		else {
	    			((CTLockingBase.INoRot)lockingBase).setNoRot(null);
	    		}
    		}
    	}
    }

    private static CTLockingBase getLockingBase(INonVisualDrawingShapePropertyAccess iNonVisualDrawingShapePropertyAccess) {
    	return iNonVisualDrawingShapePropertyAccess.getNonVisualDrawingShapeProperties(true).getLocks(true);
    }

    public static void createJsonFromNonVisualDrawingProperties(JSONObject attrs, INonVisualDrawingPropertyAccess iNonVisualDrawingProps)
        throws JSONException {

    	final CTNonVisualDrawingProps nonVisualDrawingProps = iNonVisualDrawingProps.getNonVisualDrawingProperties(false);
        if(nonVisualDrawingProps==null) {
            return;
        }
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

        final String name = nonVisualDrawingProps.getName();
        if(name!=null&&!name.isEmpty()) {
            drawingAttrs.put("name", name);
        }
        final String description = nonVisualDrawingProps.getDescr();
        if(description!=null&&!description.isEmpty()) {
            drawingAttrs.put("description", description);
        }
        if(nonVisualDrawingProps.isHidden()) {
        	drawingAttrs.put("hidden", true);
        }
        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
			attrs.put("drawing", drawingAttrs);
		}
    }

    public static void applyNonVisualDrawingProperties(INonVisualDrawingPropertyAccess iNonVisualDrawingProperties, JSONObject drawingProperties) {
    	if(drawingProperties==null) {
    		return;
    	}
    	final Object optName = drawingProperties.opt("name");
        final Object optDescription = drawingProperties.opt("description");
        if(optName!=null||optDescription!=null) {
        	CTNonVisualDrawingProps nonVisualDrawingProperties = iNonVisualDrawingProperties.getNonVisualDrawingProperties(true);
        	if(optName!=null) {
	            if(optName instanceof String) {
	            	nonVisualDrawingProperties.setName((String)optName);
	            }
	            else {
	                // name is a required xml attribute... so we are setting a dummy
	            	nonVisualDrawingProperties.setName("...");
	            }
        	}
        	if(optDescription!=null) {
	            if(optDescription instanceof String) {
	            	nonVisualDrawingProperties.setDescr((String)optDescription);
	            }
	            else {
	            	nonVisualDrawingProperties.setDescr(null);
	            }
	        }
        }
    }

    public static void createTransform2D(ITransform2D transform2D, int x100thmm, int y100thmm, int width100thmm, int height100thmm) {
        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTPoint2D point2D = dmlObjectFactory.createCTPoint2D();
        point2D.setX(Commons.coordinateFrom100TH_MM(x100thmm));
        point2D.setY(Commons.coordinateFrom100TH_MM(y100thmm));
        transform2D.setOff(point2D);
        final CTPositiveSize2D positiveSize2D = dmlObjectFactory.createCTPositiveSize2D();
        positiveSize2D.setCx(Commons.coordinateFrom100TH_MM(width100thmm));
        positiveSize2D.setCy(Commons.coordinateFrom100TH_MM(height100thmm));
    }

    public static CTPresetGeometry2D createPresetGeometry2D(STShapeType presetType) {
        final CTPresetGeometry2D presetGeometry2D = Context.getDmlObjectFactory().createCTPresetGeometry2D();
        presetGeometry2D.setPrst(presetType);
        return presetGeometry2D;
    }

    public static CTBlipFillProperties createBlipFillProperties(CTBlipFillProperties sourceBlipFill, OperationDocument operationDocument, Part part, String imageUrl)
    	throws InvalidFormatException, PartUnrecognisedException {

        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTBlipFillProperties blipFillProperties = sourceBlipFill != null ? sourceBlipFill : dmlObjectFactory.createCTBlipFillProperties();
        blipFillProperties.setBlip(createCTBlip(operationDocument, part, imageUrl));
        blipFillProperties.setStretch(dmlObjectFactory.createCTStretchInfoProperties());
        return blipFillProperties;
    }

    public static CTBlip createCTBlip(OperationDocument operationDocument, Part part, String imageUrl)
    	throws InvalidFormatException, PartUnrecognisedException {

    	final CTBlip blip = Context.getDmlObjectFactory().createCTBlip();
        final Relationship relationship = Commons.createGraphicRelation(operationDocument, part, imageUrl);
        if(relationship!=null) {
            final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
            if(linked) {
                blip.setLink(relationship.getId());
            }
            else {
                blip.setEmbed(relationship.getId());
            }
        }
        return blip;
    }

	public static void handleStandardTypes(CTTextBody txPr) {
		CTTextBodyProperties body = txPr.getBodyPr();
		if(body==null){
			body = new CTTextBodyProperties();
			txPr.setBodyPr(body);
		}

		CTTextListStyle lst2 = txPr.getLstStyle();
		if (lst2 == null) {
			lst2 = new CTTextListStyle();
			txPr.setLstStyle(lst2);
		}
	}

	/**
     * @param part (partName Uri should be /word/document.xml)
     * @return the imageUrl in following form: word/media/xxxx.png
     */
    public static String getBlipUrl(Part part, CTBlip blip) {
        String imageUrl = "";
        if(blip!=null) {
            imageUrl = Commons.getUrl(part, blip.getEmbed().length() > 0 ? blip.getEmbed() : blip.getLink() );
        }
        return imageUrl;
    }

    public static float fontSizeToPt(Integer sz) {
		if (sz == null) {
			return 10;
		}
		return ((float) sz) / 100f;
	}

    public static Integer ptTofontSize(double pt) {
		return (int) (pt * 100);
	}

    public static int lineWidthToThmm(Integer w) {
        if (w == null) {
            return 1;
        }
        return (int)((w/360f)+0.5);
    }

    public static int convert100thmmToEmu(Number e) {
    	return Double.valueOf(e.doubleValue() * 360.0).intValue();
    }

    public static double convertEmuTo100thmm(long e) {
    	return new BigDecimal(Double.valueOf(((double)e)/360.0).toString()).setScale(2, RoundingMode.HALF_UP).doubleValue();
    }

    public static double convertRelToPercent(long e) {
    	return new BigDecimal(Double.valueOf(((double)e)/100000.0).toString()).setScale(2, RoundingMode.HALF_UP).doubleValue();
    }

    public static int convertPercentToRel(Number e) {
    	return Double.valueOf(e.doubleValue() * 100000.0 + 0.5).intValue();
    }

    private static Integer thmmToLineWidth(Long w) {
    	return (int)(w*360);
	}

	public static JSONObject createJsonFromProperties(CTTextBody props) throws JSONException {
    	JSONObject character = null;
    	if (props != null) {

    		final List<Object> paras = props.getContent();
    		if (paras != null && paras.size() > 0) {
    			final CTTextParagraph para = (CTTextParagraph)paras.get(0);

    			final CTTextParagraphProperties prop = para.getPPr(false);
    			if (prop != null) {
    				final CTTextCharacterProperties defPr = prop.getDefRPr(false);
    				if (defPr != null) {

    					// color
    					if (defPr.getSolidFill() != null) {
    					    character = new JSONObject();
    					    final JSONObject color = createJsonColorFromSolidColorFillProperties(defPr.getSolidFill());
    					    if(color!=null) {
    					    	character.put("color", color);
    					    }
    					} else if (defPr.getNoFill() != null) {
    						character = new JSONObject();
    						Logger.getAnonymousLogger().warning("CtTextbody has a no fill property, result is set to null");
    					}
    					if (character == null) {
    						character = new JSONObject();
    					}

    					// font-family
    					final TextFont latn = defPr.getLatin();
    					if (latn != null) {
    						final String ff = latn.getTypeface();
    						if (ff != null && !ff.equals("+mn-lt")) {
    							character.put("fontName", ff);
    						}

    					}

    					final float size = fontSizeToPt(defPr.getSz());
    					character.put("fontSize", size);
    					final Boolean bold = defPr.isB();
    					final Boolean italic = defPr.isI();
    					if (bold != null && bold.booleanValue()) {
    						character.put("bold", true);
    					}
    					if (italic != null && italic.booleanValue()) {
    						character.put("italic", true);
    					}
    				}

    			}
    		}
    	}
    	return character;
    }

    public static void createJsonFromTextParagraphProperties(JSONObject attrs, CTTextParagraphProperties textParagraphProperties, Part part)
    	throws JSONException {

    	if(textParagraphProperties!=null) {
			final JSONObject initialParagraphAttrs = attrs.optJSONObject("paragraph");
			final JSONObject paragraphAttrs = initialParagraphAttrs!=null ? initialParagraphAttrs : new JSONObject();

			createJsonFromTextCharacterProperties(attrs, textParagraphProperties.getDefRPr(false), part);

			final Integer level = textParagraphProperties.getLvl();
			if(level!=null) {
				paragraphAttrs.put("level", level.intValue());
			}
			final JSONObject jsonLineHeight = createJsonFromCTTextSpacing(textParagraphProperties.getLnSpc());
			if(jsonLineHeight!=null) {
				paragraphAttrs.put("lineHeight", jsonLineHeight);
			}
			final JSONObject jsonSpacingBefore = createJsonFromCTTextSpacing(textParagraphProperties.getSpcBef());
			if(jsonSpacingBefore!=null) {
				paragraphAttrs.put("spacingBefore", jsonSpacingBefore);
			}
			final JSONObject jsonSpacingAfter = createJsonFromCTTextSpacing(textParagraphProperties.getSpcAft());
			if(jsonSpacingAfter!=null) {
				paragraphAttrs.put("spacingAfter", jsonSpacingAfter);
			}
			final Integer defTabSize = textParagraphProperties.getDefTabSz();
			if(defTabSize!=null) {
				paragraphAttrs.put("defaultTabSize", lineWidthToThmm(defTabSize.intValue()));
			}

			// bullet
			final JSONObject jsonBullet = new JSONObject(2);
			if(textParagraphProperties.getBuNone()!=null) {
				jsonBullet.put("type", "none");
			}
			else if(textParagraphProperties.getBuAutoNum()!=null) {
				final CTTextAutonumberBullet buAutoNumber = textParagraphProperties.getBuAutoNum();
				jsonBullet.put("type", "numbering");
				if(buAutoNumber.getStartAt()!=null) {
					jsonBullet.put("startAt", buAutoNumber.getStartAt());
				}
				if(buAutoNumber.getType()!=null) {
					jsonBullet.put("numType", buAutoNumber.getType().value());
				}
			}
			else if(textParagraphProperties.getBuChar()!=null) {
				final CTTextCharBullet buCharacter = textParagraphProperties.getBuChar();
				final String buChar = buCharacter.getChar();
				jsonBullet.put("type", "character");
				if(buChar!=null&&buChar.length()==1) {
					jsonBullet.put("character", buChar);
				}
			}
			else if(textParagraphProperties.getBuBlip()!=null) {
				final CTTextBlipBullet buBlip = textParagraphProperties.getBuBlip();
				jsonBullet.put("type", "bitmap");
				jsonBullet.put("imageUrl", getBlipUrl(part, buBlip.getBlip()));
			}
			if(!jsonBullet.isEmpty()) {
				paragraphAttrs.put("bullet", jsonBullet);
			}

			// bullet size
			if(textParagraphProperties.getBuSzTx()!=null) {
				final JSONObject buSize = new JSONObject(1);
				buSize.put("type", "followText");
				paragraphAttrs.put("bulletSize", buSize);
			}
			else if(textParagraphProperties.getBuSzPct()!=null) {
				final JSONObject buSize = new JSONObject(2);
				buSize.put("type", "percent");
				final Integer size = textParagraphProperties.getBuSzPct().getVal();
				if(size!=null) {
					buSize.put("size", (int)((size.intValue() + 500) / 1000));
				}
				paragraphAttrs.put("bulletSize", buSize);
			}
			else if(textParagraphProperties.getBuSzPts()!=null) {
				final JSONObject buSize = new JSONObject(2);
				buSize.put("type", "point");
				final Integer size = textParagraphProperties.getBuSzPts().getVal();
				if(size!=null) {
					buSize.put("size", (int)((size.intValue() + 500) / 100));
				}
				paragraphAttrs.put("bulletSize", buSize);
			}

			// bullet color
			if(textParagraphProperties.getBuClrTx()!=null) {
				final JSONObject bulletColor = new JSONObject(1);
				bulletColor.put("followText", true);
				paragraphAttrs.put("bulletColor", bulletColor);
			}
			else if(textParagraphProperties.getBuClr()!=null) {
				final JSONObject bulletColor = new JSONObject(2);
				bulletColor.put("followText", false);
				final JSONObject color = createJsonColorFromSolidColorFillProperties(textParagraphProperties.getBuClr());
				if(color!=null) {
					bulletColor.put("color", color);
				}
				paragraphAttrs.put("bulletColor", bulletColor);
			}

			// bullet font
			if(textParagraphProperties.getBuFontTx()!=null) {
				final JSONObject bulletFont = new JSONObject(1);
				bulletFont.put("followText", true);
				paragraphAttrs.put("bulletFont", bulletFont);
			}
			else if(textParagraphProperties.getBuFont()!=null) {
				final JSONObject bulletFont = new JSONObject(2);
				bulletFont.put("followText", false);
				final String bulletFontName = textParagraphProperties.getBuFont().getTypeface();
				if(bulletFontName!=null) {
					bulletFont.put("name", bulletFontName);
				}
				paragraphAttrs.put("bulletFont", bulletFont);
			}

			final STTextAlignType paragraphAlign = textParagraphProperties.getAlgn();
			if(paragraphAlign!=null) {
				String a = null;
				switch(paragraphAlign) {
					case CTR: {
						a = "center";
						break;
					}

					case THAI_DIST:
					case DIST:
					case JUST:
					case JUST_LOW: {
						a = "justify";
						break;
					}
					case L: {
						a = "left";
						break;
					}
					case R: {
						a = "right";
						break;
					}
				}
				if(a!=null) {
					paragraphAttrs.put("alignment", a);
				}
			}
			final Integer marL = textParagraphProperties.getMarL();
			if(marL!=null) {
				paragraphAttrs.put("indentLeft", lineWidthToThmm(marL.intValue()));
			}
			final Integer marR = textParagraphProperties.getMarR();
			if(marR!=null) {
				paragraphAttrs.put("indentRight", lineWidthToThmm(marR.intValue()));
			}
			final Integer indent = textParagraphProperties.getIndent();
			if(indent!=null) {
				paragraphAttrs.put("indentFirstLine", lineWidthToThmm(indent.intValue()));
			}
	        if(initialParagraphAttrs==null&&!paragraphAttrs.isEmpty()) {
				attrs.put("paragraph", paragraphAttrs);
			}
    	}
    }

    public static void applyTextParagraphPropertiesFromJson(CTTextParagraphProperties paragraphProperties, JSONObject attrs, OperationDocument operationDocument, Part part)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(attrs==null) {
    		return;
    	}
        final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");

        // change of paragraph level has to be applied first
        if(paragraphAttrs!=null) {
        	if(paragraphAttrs.has("level")) {
        		final Object level = paragraphAttrs.get("level");
        		if(level instanceof Integer) {
        			paragraphProperties.setLvl((Integer)level);
        		}
        		else if(level==JSONObject.NULL) {
        			paragraphProperties.setLvl(null);
        		}
        	}
        }
    	if(attrs.hasAndNotNull("character")) {
    		applyTextCharacterPropertiesFromJson(paragraphProperties.getDefRPr(true), attrs, part);
    	}
        if(paragraphAttrs==null) {
        	return;
        }
        final Iterator<Entry<String, Object>> entries = paragraphAttrs.entrySet().iterator();
        while(entries.hasNext()) {
        	final Entry<String, Object> entry = entries.next();
        	final Object value = entry.getValue();
        	switch(entry.getKey()) {
        		case "lineHeight": {
        			paragraphProperties.setLnSpc(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "spacingBefore": {
        			paragraphProperties.setSpcBef(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "spacingAfter": {
        			paragraphProperties.setSpcAft(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "defaultTabSize": {
        			if(value instanceof Number) {
        				paragraphProperties.setDefTabSz(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setDefTabSz(null);
        			}
        			break;
        		}
        		case "bullet": {
        			createBulletFromJson(paragraphProperties, value, operationDocument, part);
        			break;
        		}
        		case "bulletSize": {
        			createBulletSizeFromJson(paragraphProperties, value);
        			break;
        		}
        		case "bulletColor": {
        			createBulletColorFromJson(paragraphProperties, value);
        			break;
        		}
        		case "bulletFont": {
        			createBulletFontFromJson(paragraphProperties, value);
        			break;
        		}
        		case "alignment": {
        			if(value instanceof String) {
        				if(((String)value).equals("center")) {
        					paragraphProperties.setAlgn(STTextAlignType.CTR);
        				}
        				else if(((String)value).equals("justify")) {
        					paragraphProperties.setAlgn(STTextAlignType.JUST);
        				}
        				else if(((String)value).equals("right")) {
        					paragraphProperties.setAlgn(STTextAlignType.R);
        				}
        				else {
        					paragraphProperties.setAlgn(STTextAlignType.L);
        				}
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setAlgn(null);
        			}
        			break;
        		}
        		case "indentLeft": {
        			if(value instanceof Number) {
        				paragraphProperties.setMarL(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setMarL(null);
        			}
        			break;
        		}
        		case "indentRight": {
        			if(value instanceof Number) {
        				paragraphProperties.setMarR(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setMarR(null);
        			}
        			break;
        		}
        		case "indentFirstLine": {
        			if(value instanceof Number) {
        				paragraphProperties.setIndent(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setIndent(null);
        			}
        			break;
        		}
        	}
        }
    }

    // applying the properties to the listStyle
	public static void applyTextParagraphPropertiesFromJson(CTTextBody textBody, int level, JSONObject attrs, OperationDocument operationDocument, Part part)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		if(attrs==null) {
			return;
		}
		CTTextListStyle textListStyle = textBody.getLstStyle();
		if(textListStyle==null) {
			textListStyle = Context.getDmlObjectFactory().createCTTextListStyle();
			textBody.setLstStyle(textListStyle);
		}
		applyTextParagraphPropertiesFromJson(textListStyle.getPPr(level, true), attrs, operationDocument, part);
	}

    public static JSONObject createJsonFromCTTextSpacing(CTTextSpacing textSpacing)
    	throws JSONException {

    	if(textSpacing==null) {
    		return null;
    	}

    	final JSONObject jsonLineHeight = new JSONObject(2);
		if(textSpacing.getSpcPct()!=null) {
			final CTTextSpacingPercent spacing = textSpacing.getSpcPct();
			jsonLineHeight.put("type", "percent");
			jsonLineHeight.put("value", (int)(spacing.getVal() / 1000.0));
		}
		else if(textSpacing.getSpcPts()!=null) {
			final CTTextSpacingPoint spacing = textSpacing.getSpcPts();
			jsonLineHeight.put("type", "fixed");
			jsonLineHeight.put("value", (int)((spacing.getVal() * 25.4) / 72.0));
		}
		return jsonLineHeight;
    }

    public static CTTextSpacing createCTTextSpacingFromJson(Object attrs) {
    	if(attrs instanceof JSONObject) {
    		final String type = ((JSONObject)attrs).optString("type", null);
    		final Object value = ((JSONObject)attrs).opt("value");
    		if(type!=null&&value instanceof Integer) {
        		final CTTextSpacing spacing = Context.getDmlObjectFactory().createCTTextSpacing();
        		if(type.equals("percent")) {
        			final CTTextSpacingPercent textSpacingPercent = Context.getDmlObjectFactory().createCTTextSpacingPercent();
        			spacing.setSpcPct(textSpacingPercent);
        			textSpacingPercent.setVal(((Integer)value).intValue()*1000);
        			return spacing;
        		}
        		else if(type.equals("fixed")) {
        			final CTTextSpacingPoint textSpacingFixed = Context.getDmlObjectFactory().createCTTextSpacingPoint();
        			spacing.setSpcPts(textSpacingFixed);
        			textSpacingFixed.setVal((int)((((Integer)value).intValue() * 72.0) / 25.4));
        			return spacing;
        		}
        		return spacing;
    		}
    	}
    	return null;
    }

    public static void createBulletFromJson(CTTextParagraphProperties paragraphProperties, Object bulletAttrs, OperationDocument operationDocument, Part part)
    	throws InvalidFormatException, PartUnrecognisedException {

    	paragraphProperties.setBuNone(null);
    	paragraphProperties.setBuAutoNum(null);
    	paragraphProperties.setBuChar(null);
    	paragraphProperties.setBuBlip(null);
    	if(bulletAttrs instanceof JSONObject) {
    		final JSONObject bullet = (JSONObject)bulletAttrs;
    		final String buType = bullet.optString("type", "");
    		if(buType.equals("none")) {
    			paragraphProperties.setBuNone(Context.getDmlObjectFactory().createCTTextNoBullet());
    		}
    		else if(buType.equals("numbering")) {
    			paragraphProperties.setBuAutoNum(Context.getDmlObjectFactory().createCTTextAutonumberBullet());
    			final Object startAt = bullet.opt("startAt");
    			if(startAt instanceof Integer) {
    				paragraphProperties.getBuAutoNum().setStartAt((Integer)startAt);
    			}
    			final Object numType = bullet.opt("numType");
    			if(numType instanceof String) {
    				paragraphProperties.getBuAutoNum().setType(STTextAutonumberScheme.fromValue((String)numType));
    			}
    		}
    		else if(buType.equals("character")) {
    			paragraphProperties.setBuChar(Context.getDmlObjectFactory().createCTTextCharBullet());
    			final String buChar = bullet.optString("character", "");
    			if(buChar.length()==1) {
    				paragraphProperties.getBuChar().setChar(buChar);
    			}
    		}
    		else if(buType.equals("bitmap")) {
    			paragraphProperties.setBuBlip(Context.getDmlObjectFactory().createCTTextBlipBullet());
    			final String imageUrl = bullet.optString("imageUrl", "");
    			if(!imageUrl.isEmpty()) {
    				paragraphProperties.getBuBlip().setBlip(createCTBlip(operationDocument, part, imageUrl));
    			}
    		}
    	}
    }

    public static void createBulletSizeFromJson(CTTextParagraphProperties paragraphProperties, Object buSize) {
    	paragraphProperties.setBuSzTx(null);
    	paragraphProperties.setBuSzPct(null);
    	paragraphProperties.setBuSzPts(null);
    	if(buSize instanceof JSONObject) {
    		final String type = ((JSONObject)buSize).optString("type", "");
    		if(type.equals("followText")) {
    			paragraphProperties.setBuSzTx(Context.getDmlObjectFactory().createCTTextBulletSizeFollowText());
    		}
    		else if(type.equals("percent")) {
    			final Object value = ((JSONObject)buSize).opt("size");
    			if(value instanceof Number) {
	    			paragraphProperties.setBuSzPct(Context.getDmlObjectFactory().createCTTextBulletSizePercent());
	    			paragraphProperties.getBuSzPct().setVal((int)(((Number)value).doubleValue() * 1000));
    			}
    		}
    		else if(type.equals("point")) {
    			final Object value = ((JSONObject)buSize).opt("size");
    			if(value instanceof Number) {
    				paragraphProperties.setBuSzPts(Context.getDmlObjectFactory().createCTTextBulletSizePoint());
	    			paragraphProperties.getBuSzPts().setVal((int)(((Number)value).doubleValue() * 100));
    			}
    		}
    	}
    }

    public static void createBulletColorFromJson(CTTextParagraphProperties paragraphProperties, Object buColor)
    	throws JSONException {

    	paragraphProperties.setBuClr(null);
    	paragraphProperties.setBuClrTx(null);
    	if(buColor instanceof JSONObject) {
    		final boolean followText = ((JSONObject)buColor).optBoolean("followText", false);
    		if(followText) {
    			paragraphProperties.setBuClrTx(Context.getDmlObjectFactory().createCTTextBulletColorFollowText());
    		}
    		else {
    			final Object color = ((JSONObject)buColor).opt("color");
    			if(color instanceof JSONObject) {
	    			paragraphProperties.setBuClr(Context.getDmlObjectFactory().createCTColor());
	    			createSolidColorFillPropertiesFromJson(paragraphProperties.getBuClr(), (JSONObject)color);
    			}
    		}
    	}
    }

    public static void createBulletFontFromJson(CTTextParagraphProperties paragraphProperties, Object buFont) {

    	paragraphProperties.setBuFont(null);
    	paragraphProperties.setBuFontTx(null);
    	if(buFont instanceof JSONObject) {
    		final boolean followText = ((JSONObject)buFont).optBoolean("followText", false);
    		if(followText) {
    			paragraphProperties.setBuFontTx(Context.getDmlObjectFactory().createCTTextBulletTypefaceFollowText());
    		}
    		else {
    			final String name = ((JSONObject)buFont).optString("name", "");
    			paragraphProperties.setBuFont(Context.getDmlObjectFactory().createTextFont());
    			paragraphProperties.getBuFont().setTypeface(name);
    		}
    	}
    }

    public static void createJsonFromTextCharacterProperties(JSONObject attrs, CTTextCharacterProperties textCharacterProperties, Part part)
    	throws JSONException {

    	if(textCharacterProperties!=null) {
			final JSONObject initialCharacterAttrs = attrs.optJSONObject("character");
			final JSONObject characterAttrs = initialCharacterAttrs!=null ? initialCharacterAttrs : new JSONObject();

			final String lang = textCharacterProperties.getLang();
			if (lang != null) {
				characterAttrs.put("language", lang);
			}

			final Integer fontSize = textCharacterProperties.getSz();
			if(fontSize!=null) {
				characterAttrs.put("fontSize", new BigDecimal(fontSize.floatValue() / 100.0f).setScale(2, BigDecimal.ROUND_HALF_UP).floatValue());
			}
			final TextFont latinFont = textCharacterProperties.getLatin();
			if(latinFont!=null&&latinFont.getTypeface()!=null) {
				characterAttrs.put("fontName", latinFont.getTypeface());
			}
			final TextFont eastAsiaFont = textCharacterProperties.getEa();
			if(eastAsiaFont!=null) {
			    characterAttrs.put("fontNameEastAsia", eastAsiaFont.getTypeface());
			}
			final TextFont complexFont = textCharacterProperties.getCs();
			if(complexFont!=null) {
			    characterAttrs.put("fontNameComplex", complexFont.getTypeface());
			}
			final TextFont symFont = textCharacterProperties.getSym();
			if(symFont!=null) {
			    characterAttrs.put("fontNameSymbol", symFont.getTypeface());
			}
        	if(textCharacterProperties.getNoFill()!=null) {
        		// TODO: we need a fill object instead of a plain color
        		// createJsonFromFillStyle(textCharacterProperties.getNoFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getSolidFill()!=null) {
	    		final JSONObject color = createJsonColorFromSolidColorFillProperties(textCharacterProperties.getSolidFill());
	    		if(color!=null) {
	    			characterAttrs.put("color", color);
	    		}
            }
	    	else if(textCharacterProperties.getBlipFill()!=null) {
	    		// TODO: createJsonFromFillStyle(textCharacterProperties.getBlipFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getGradFill()!=null) {
	    	    // TODO: createJsonFromFillStyle(textCharacterProperties.getGradFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getPattFill()!=null) {
	    	    // TODO: createJsonFromFillStyle(textCharacterProperties.getPattFill(), null, characterAttrs, part);
	    	}
			final Boolean bold = textCharacterProperties.isB();
			if(bold!=null) {
				characterAttrs.put("bold", bold.booleanValue());
			}
			final STTextCapsType capsType = textCharacterProperties.getCap();
			if(capsType!=null) {
				characterAttrs.put("caps", capsType.value());
			}
			final Boolean italic = textCharacterProperties.isI();
			if(italic!=null) {
				characterAttrs.put("italic", italic.booleanValue());
			}
			final STTextUnderlineType underline = textCharacterProperties.getU();
			if(underline!=null) {
				switch(underline) {
				case DASH:
				case DASH_HEAVY:
				case DASH_LONG:
				case DASH_LONG_HEAVY:
				case DBL:
				case DOTTED:
				case DOTTED_HEAVY:
				case DOT_DASH:
				case DOT_DASH_HEAVY:
				case DOT_DOT_DASH:
				case DOT_DOT_DASH_HEAVY:
				case HEAVY:
				case SNG:
				case WAVY:
				case WAVY_DBL:
				case WAVY_HEAVY:
				case WORDS:
					characterAttrs.put("underline", true);
					break;
				default:
				case NONE:
					characterAttrs.put("underline", false);
					break;
				}
			}
			final STTextStrikeType strike = textCharacterProperties.getStrike();
			if(strike!=null) {
				switch(strike) {
				case NO_STRIKE:
					characterAttrs.put("strike", "none");
					break;
				case DBL_STRIKE:
					characterAttrs.put("strike", "double");
					break;
				case SNG_STRIKE:
					characterAttrs.put("strike", "single");
					break;
				}
			}
			final Integer baseline = textCharacterProperties.getBaseline();
			if(baseline!=null) {
				characterAttrs.put("baseline", (int)(baseline / 1000));
			}
			final CTHyperlink hyperlink = textCharacterProperties.getHlinkClick(false);
			if(hyperlink!=null) {
				final String rId = hyperlink.getId();
				if(rId!=null&&!rId.isEmpty()) {
					final String hLink = Commons.getUrl(part, rId);
					if(hLink!=null&&!hLink.isEmpty()) {
						characterAttrs.put("url", hLink);
					}
				}
			}
	        if(initialCharacterAttrs==null&&!characterAttrs.isEmpty()) {
				attrs.put("character", characterAttrs);
			}
    	}
    }

    public static void applyTextCharacterPropertiesFromJson(CTTextCharacterProperties characterProperties, JSONObject attrs, Part part)
    	throws JSONException {

    	if(attrs==null) {
    		return;
    	}
        final JSONObject characterAttrs = attrs.optJSONObject("character");
        if(characterAttrs==null) {
        	return;
        }
        final Iterator<Entry<String, Object>> entries = characterAttrs.entrySet().iterator();
        while(entries.hasNext()) {
        	final Entry<String, Object> entry = entries.next();
        	final Object value = entry.getValue();
        	switch(entry.getKey()) {
        		case "color" : {
                	applyFillPropertiesFromJsonColor(characterProperties, value);
        			break;
        		}
        		case "fontSize": {
        			if(value instanceof Number) {
        				characterProperties.setSz(new Float(((Number)value).floatValue()*100.0f).intValue());
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setSz(null);
        			}
        			break;
        		}
        		case "fontName": {
        			if(value instanceof String) {
        				characterProperties.setLatin(Context.getDmlObjectFactory().createTextFont());
        				characterProperties.getLatin().setTypeface((String)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setLatin(null);
        			}
        			break;
        		}
                case "fontNameEastAsia": {
                    if(value instanceof String) {
                        characterProperties.setEa(Context.getDmlObjectFactory().createTextFont());
                        characterProperties.getEa().setTypeface((String)value);
                    }
                    else if(value==JSONObject.NULL) {
                        characterProperties.setEa(null);
                    }
                    break;
                }
                case "fontNameComplex": {
                    if(value instanceof String) {
                        characterProperties.setCs(Context.getDmlObjectFactory().createTextFont());
                        characterProperties.getCs().setTypeface((String)value);
                    }
                    else if(value==JSONObject.NULL) {
                        characterProperties.setCs(null);
                    }
                    break;
                }
                case "fontNameSymbol": {
                    if(value instanceof String) {
                        characterProperties.setSym(Context.getDmlObjectFactory().createTextFont());
                        characterProperties.getSym().setTypeface((String)value);
                    }
                    else if(value==JSONObject.NULL) {
                        characterProperties.setSym(null);
                    }
                    break;
                }
        		case "bold": {
        			if(value instanceof Boolean) {
        				characterProperties.setB((Boolean)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setB(null);
        			}
        			break;
        		}
        		case "italic": {
        			if(value instanceof Boolean) {
        				characterProperties.setI((Boolean)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setI(null);
        			}
        			break;
        		}
        		case "underline": {
        			if(value instanceof Boolean) {
        				characterProperties.setU(((Boolean)value).booleanValue() ? STTextUnderlineType.SNG : STTextUnderlineType.NONE);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setU(null);
        			}
        			break;
        		}
        		case "caps": {
        			if(value instanceof String) {
        				characterProperties.setCap(STTextCapsType.fromValue((String)value));
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setCap(null);
        			}
        			break;
        		}
        		case "strike": {
        			if(value instanceof String) {
        				if(((String)value).equals("none")) {
        					characterProperties.setStrike(STTextStrikeType.NO_STRIKE);
        				}
        				else if(((String)value).equals("single")) {
        					characterProperties.setStrike(STTextStrikeType.SNG_STRIKE);
        				}
        				else if(((String)value).equals("double")) {
        					characterProperties.setStrike(STTextStrikeType.DBL_STRIKE);
        				}
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setStrike(null);
        			}
        			break;
        		}
        		case "baseline": {
        			if(value instanceof Number) {
        				characterProperties.setBaseline(((Number)value).intValue() * 1000);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setBaseline(null);
        			}
        			break;
        		}
        		case "url": {
        			if(value instanceof String) {
        				if(part!=null) {
	        				final CTHyperlink hyperlink = characterProperties.getHlinkClick(true);
	        				hyperlink.setId(Commons.setUrl(part, null, (String)value));
        				}
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setHlinkClick(null);
        			}
        			break;
        		}
        		case "language": {
        			if(value instanceof String) {
            			characterProperties.setLang((String)value);
        			}
        			break;
        		}
        	}
        }
    }

    public static void applyTextListStyleFromJson(CTTextBody textBody, JSONObject listStyle, Part part)
    	throws JSONException {

    	applyTextCharacterPropertiesFromJson(textBody, 0, listStyle.optJSONObject("l1"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 1, listStyle.optJSONObject("l2"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 2, listStyle.optJSONObject("l3"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 3, listStyle.optJSONObject("l4"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 4, listStyle.optJSONObject("l5"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 5, listStyle.optJSONObject("l6"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 6, listStyle.optJSONObject("l7"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 7, listStyle.optJSONObject("l8"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 8, listStyle.optJSONObject("l9"), part);
    	applyTextCharacterPropertiesFromJson(textBody, 9, listStyle.optJSONObject("default"), part);
    }

    // applying the properties to the listStyle
	public static void applyTextCharacterPropertiesFromJson(CTTextBody textBody, int level, JSONObject attrs, Part part)
		throws JSONException {

		if(attrs==null||attrs.optJSONObject("character")==null) {
			return;
		}
		CTTextListStyle textListStyle = textBody.getLstStyle();
		if(textListStyle==null) {
			textListStyle = Context.getDmlObjectFactory().createCTTextListStyle();
			textBody.setLstStyle(textListStyle);
		}
		applyTextCharacterPropertiesFromJson(textListStyle.getPPr(level, true).getDefRPr(true), attrs, part);
	}

	public static JSONObject createJsonFromTextListStyle(CTTextListStyle textListStyle, Part part)
		throws JSONException {

		if(textListStyle==null) {
			return null;
		}
		final JSONObject style = new JSONObject();
		createTextParagraphProperties(textListStyle.getDefPPr(), style, "default", part);
		createTextParagraphProperties(textListStyle.getLvl1PPr(), style, "l1", part);
		createTextParagraphProperties(textListStyle.getLvl2PPr(), style, "l2", part);
		createTextParagraphProperties(textListStyle.getLvl3PPr(), style, "l3", part);
		createTextParagraphProperties(textListStyle.getLvl4PPr(), style, "l4", part);
		createTextParagraphProperties(textListStyle.getLvl5PPr(), style, "l5", part);
		createTextParagraphProperties(textListStyle.getLvl6PPr(), style, "l6", part);
		createTextParagraphProperties(textListStyle.getLvl7PPr(), style, "l7", part);
		createTextParagraphProperties(textListStyle.getLvl8PPr(), style, "l8", part);
		createTextParagraphProperties(textListStyle.getLvl9PPr(), style, "l9", part);
		return style;
	}

	private static JSONObject createTextParagraphProperties(CTTextParagraphProperties textParagraphProperties, JSONObject destStyle, String level, Part part)
		throws JSONException {

		if(textParagraphProperties==null) {
			return null;
		}

		final JSONObject properties = new JSONObject(2);
		DMLHelper.createJsonFromTextParagraphProperties(properties, textParagraphProperties, part);
		if(!properties.isEmpty()) {
			destStyle.put(level, properties);
		}
		return null;
	}

	public static JSONObject createJsonFromTextBodyProperties(JSONObject attrs, CTTextBodyProperties textBodyProperties)
		throws JSONException {

		if(textBodyProperties!=null) {
			final JSONObject initialTextBodyAttrs = attrs.optJSONObject("shape");
			final JSONObject textBodyAttrs = initialTextBodyAttrs!=null ? initialTextBodyAttrs : new JSONObject();

			// top is default
			final STTextAnchoringType anchor = textBodyProperties.getAnchor();
			if(anchor!=null) {
				String a = null;
				switch(anchor) {
					case B: {
						a = "bottom";
						break;
					}
					case CTR: {
						a = "centered";
						break;
					}
					case DIST: {
						a = "distributed";
						break;
					}
					case JUST: {
						a = "justified";
						break;
					}
					case T: {
						a = "top";
						break;
					}
				}
				if(a!=null) {
					textBodyAttrs.put("anchor", a);
				}
			}

			// false is default
			final Boolean anchorCentered = textBodyProperties.isAnchorCtr();
			if(anchorCentered!=null) {
				textBodyAttrs.put("anchorCentered", anchorCentered.booleanValue());
			}

			// default 2,5mm
			final Integer marginLeft = textBodyProperties.getLIns();
			if(marginLeft!=null) {
				textBodyAttrs.put("paddingLeft", marginLeft.intValue() / 360);
			}

			// default 2,5mm
			final Integer marginRight = textBodyProperties.getRIns();
			if(marginRight!=null) {
				textBodyAttrs.put("paddingRight", marginRight.intValue() / 360);
			}

			// default 1,25mm
			final Integer marginTop = textBodyProperties.getTIns();
			if(marginTop!=null) {
				textBodyAttrs.put("paddingTop", marginTop.intValue() / 360);
			}

			// default 1,25mm
			final Integer marginBottom = textBodyProperties.getBIns();
			if(marginBottom!=null) {
				textBodyAttrs.put("paddingBottom", marginBottom.intValue() / 360);
			}

			// default is square
 			final STTextWrappingType wrap = textBodyProperties.getWrap();
			if(wrap!=null) {
				textBodyAttrs.put("wordWrap", wrap==STTextWrappingType.NONE ? false : true);
			}

			// default is overflow
			final STTextHorzOverflowType horzOverflow = textBodyProperties.getHorzOverflow();
			if(horzOverflow!=null) {
				textBodyAttrs.put("horzOverflow", horzOverflow==STTextHorzOverflowType.CLIP ? "clip" : "overflow");
			}

			final CTTextShapeAutofit spAutoFit = textBodyProperties.getSpAutoFit();
            if(spAutoFit!=null) {
            	textBodyAttrs.put("autoResizeHeight", true);
            }

            final CTTextNoAutofit noAutoFit = textBodyProperties.getNoAutofit();
            if(noAutoFit!=null) {
                textBodyAttrs.put("noAutoResize", true);
            }

            final CTTextNormalAutofit normAutoFit = textBodyProperties.getNormAutofit();
            if( normAutoFit != null) {
                textBodyAttrs.put("autoResizeText", true);

                textBodyAttrs.put("fontScale", ((double)normAutoFit.getFontScale()) / 100000.0);
                textBodyAttrs.put("lineReduction", ((double)normAutoFit.getLnSpcReduction()) / 100000.0);
            }

            final STTextVerticalType textVerticalType = textBodyProperties.getVert();
            if(textVerticalType!=null) {
            	textBodyAttrs.put("vert", textVerticalType.value());
            }
            // default is overflow
			final STTextVertOverflowType vertOverflow = textBodyProperties.getVertOverflow();
			if(vertOverflow!=null) {
				String o = null;
				switch(vertOverflow) {
					case CLIP: {
						o = "clip";
						break;
					}
					case ELLIPSIS: {
						o = "ellipsis";
						break;
					}
					case OVERFLOW : {
						o = "overflow";
						break;
					}
				}
				if(o!=null) {
					textBodyAttrs.put("vertOverflow", o);
				}
			}
	        if(initialTextBodyAttrs==null&&!textBodyAttrs.isEmpty()) {
				attrs.put("shape", textBodyAttrs);
			}
		}
		return attrs;
	}

	public static void applyTextBodyPropertiesFromJson(CTTextBodyProperties bodyPr, JSONObject attrs) {

		if(bodyPr==null) {
			return;
		}
		final JSONObject shapeAttrs = attrs.optJSONObject("shape");
	    if(shapeAttrs==null) {
	    	return;
	    }
	    final Iterator<Entry<String, Object>> iter = shapeAttrs.entrySet().iterator();
	    while(iter.hasNext()) {
	    	final Entry<String, Object> entry = iter.next();
	    	final Object value = entry.getValue();
	    	switch(entry.getKey()) {
	    		case "paddingLeft" : {
	                if(value instanceof Number) {
	                    bodyPr.setLIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setLIns(null);
	                }
	    			break;
	    		}
	    		case "paddingRight" : {
	                if(value instanceof Number) {
	                    bodyPr.setRIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setRIns(null);
	                }
	    			break;
	    		}
	    		case "paddingTop" : {
	                if(value instanceof Number) {
	                    bodyPr.setTIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setTIns(null);
	                }
	    			break;
	    		}
	    		case "paddingBottom" : {
	                if(value instanceof Number) {
	                    bodyPr.setBIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setBIns(null);
	                }
	    			break;
	    		}
	    		case "autoResizeHeight" : {
	                if(value instanceof Boolean && ((Boolean)value).booleanValue()) {
	                    bodyPr.setSpAutoFit(Context.getDmlObjectFactory().createCTTextShapeAutofit());
	                }
	                else {
	                    bodyPr.setSpAutoFit(null);
	                }
	    			break;
	    		}
	    		case "autoResizeText" : {
                    if(value instanceof Boolean && ((Boolean)value).booleanValue()) {
                        if (null == bodyPr.getNormAutofit()) {
                            bodyPr.setNormAutofit(Context.getDmlObjectFactory().createCTTextNormalAutofit());
                        }
                    }
                    else {
                        bodyPr.setNormAutofit(null);
                    }
                    break;
                }
	    		case "noAutoResize" : {
                    if(value instanceof Boolean && ((Boolean)value).booleanValue()) {
                        bodyPr.setNoAutofit(Context.getDmlObjectFactory().createCTTextNoAutofit());
                    }
                    else {
                        bodyPr.setNoAutofit(null);
                    }
                    break;
                }
	    		case "fontScale" : {
                    CTTextNormalAutofit fit = bodyPr.getNormAutofit();
                    if (null == fit) {
                        fit = Context.getDmlObjectFactory().createCTTextNormalAutofit();
                        bodyPr.setNormAutofit(fit);
                    }

                    if(value instanceof Number) {
                        fit.setFontScale((int) (((Number)value).doubleValue() * 100000.0));
                    }
                    else {
                        fit.setFontScale(null);
                    }
                    break;
                }
	    		case "lineReduction" : {
                    CTTextNormalAutofit fit = bodyPr.getNormAutofit();
                    if (null == fit) {
                        fit = Context.getDmlObjectFactory().createCTTextNormalAutofit();
                        bodyPr.setNormAutofit(fit);
                    }

                    if(value instanceof Number) {
                        fit.setLnSpcReduction((int) (((Number)value).doubleValue() * 100000.0));
                    }
                    else {
                        fit.setLnSpcReduction(null);
                    }
                    break;
                }
	    		case "anchor" : {
	    			STTextAnchoringType anchor = null;
	    			if(value instanceof String) {
	    				switch((String)value) {
	    					case "top" : {
	    						anchor = STTextAnchoringType.T;
	    						break;
	    					}
	    					case "centered" : {
	    						anchor = STTextAnchoringType.CTR;
	    						break;
	    					}
	    					case "distributed" : {
	    						anchor = STTextAnchoringType.DIST;
	    						break;
	    					}
	    					case "justified" : {
	    						anchor = STTextAnchoringType.JUST;
	    						break;
	    					}
	    					case "bottom" : {
	    						anchor = STTextAnchoringType.B;
	    						break;
	    					}
	    				}
	    			}
	    			bodyPr.setAnchor(anchor);
	    			break;
	    		}
	    		case "anchorCentered" : {
	    			Boolean anchorCentered = null;
	    			if(value instanceof Boolean) {
	    				anchorCentered = (Boolean)value;
	    			}
	    			bodyPr.setAnchorCtr(anchorCentered);
	    			break;
	    		}
	    		case "wordWrap" : {
	    			STTextWrappingType wordWrap = null;
	    			if(value instanceof Boolean) {
	    				wordWrap = ((Boolean)value).booleanValue() ? STTextWrappingType.SQUARE : STTextWrappingType.NONE;
	    			}
	    			bodyPr.setWrap(wordWrap);
	    			break;
	    		}
	    		case "vert" : {
	    			STTextVerticalType textVerticalType = null;
	                if(value instanceof String) {
	                	textVerticalType = STTextVerticalType.fromValue((String)value);
	                }
	                bodyPr.setVert(textVerticalType);
	                break;
	    		}
	    		case "horzOverflow" : {
	    			STTextHorzOverflowType horzOverflowType = null;
	    			if(value instanceof String) {
	    				if(((String)value).equals("clip")) {
	    					horzOverflowType = STTextHorzOverflowType.CLIP;
	    				}
	    				else if(((String)value).equals("overflow")) {
	    					horzOverflowType = STTextHorzOverflowType.OVERFLOW;
	    				}
	    			}
	    			bodyPr.setHorzOverflow(horzOverflowType);
	    			break;
	    		}
	    		case "vertOverflow" : {
	    			STTextVertOverflowType vertOverflowType = null;
	    			if(value instanceof String) {
	    				if(((String)value).equals("clip")) {
	    					vertOverflowType = STTextVertOverflowType.CLIP;
	    				}
	    				else if(((String)value).equals("overflow")) {
	    					vertOverflowType = STTextVertOverflowType.OVERFLOW;
	    				}
	    				else if(((String)value).equals("ellipsis")) {
	    					vertOverflowType = STTextVertOverflowType.ELLIPSIS;
	    				}
	    			}
	    			bodyPr.setVertOverflow(vertOverflowType);
	    			break;
	    		}
	    	}
	    }
	}
}
