/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.BookViews;
import org.xlsx4j.sml.CTBookView;
import org.xlsx4j.sml.CTCalcPr;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTDrawing;
import org.xlsx4j.sml.CTFont;
import org.xlsx4j.sml.CTHyperlink;
import org.xlsx4j.sml.CTHyperlinks;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTNumFmts;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTSheetPr;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.IDataValidation;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.WorkbookPr;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.XlsxOperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.AutoFilterHelper;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.ConditionalFormattings;
import com.openexchange.office.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.ooxml.xlsx.tools.MergeCellHelper;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.TableHelper;
import com.openexchange.office.ooxml.xlsx.tools.Utils;
import com.openexchange.office.ooxml.xlsx.tools.ValidationUtils;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class XlsxCreateOperationHelper extends CreateOperationHelper {

    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final Map<Long, String> indexToStyleId = new HashMap<Long, String>();
    final int maxCellCount;
    final int maxSheetCount;
    int currentCellCount = 0;

    public XlsxCreateOperationHelper(XlsxOperationDocument _operationDocument, JSONArray mOperationsArray)
    	throws FilterException {

    	super(_operationDocument, mOperationsArray);
        spreadsheetMLPackage = _operationDocument.getPackage();
        final ServiceLookup services = _operationDocument.getServiceLookup();
    	final Session session = _operationDocument.getSession();
        maxCellCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//spreadsheet/maxCells", 500000);
        maxSheetCount = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//spreadsheet/maxSheets", 256);

        final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        if(sheets!=null) {
            final List<Sheet> sheetList = sheets.getSheet();
            if(maxSheetCount>=0) {
                if(sheetList.size()>maxSheetCount) {
                	final FilterException filterException = new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                	filterException.setSubType(FilterException.ST_MAX_SHEET_COUNT_EXCEEDED);
                    throw filterException;
                }
            }
        }
        final CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {
            // creating styles
            final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                BuiltinIds.getUsedStyles(stylesheet, indexToStyleId, null);
            }
        }
    }

    @Override
    public XlsxOperationDocument getOperationDocument() {
        return (XlsxOperationDocument)operationDocument;
    }

    @Override
    public void createDocumentDefaults(String userLanguage) {

    }

    public void createStyleOperations()
        throws Exception {

        CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {

        	// create number formats
        	final CTNumFmts numFmts = stylesheet.getNumberFormats(false);
            if(numFmts!=null) {
            	for (CTNumFmt numFmt: numFmts.getNumFmt()) {
    	            final JSONObject operation = new JSONObject();
    	            operation.put("name", "insertNumberFormat");
    	            operation.put("id", numFmt.getNumFmtId());
    	            operation.put("code", numFmt.getFormatCode());
                	operationsArray.put(operation);
            	}
            }

        	// creating styles
            final CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                final List<CTXf> cellStyleXfList = cellStyleXfs.getXf();
                for(int i=0;i<cellStyleXfList.size();i++) {
                    final String styleId = indexToStyleId.get(new Long(i));
                    if(styleId!=null) {
                        final CTXf cellStyleXf = cellStyleXfList.get(i);
                        String styleName = styleId;
                        JSONObject jsonProperties = new JSONObject();
                        Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(cellStyleXf, stylesheet, true));
                        final CTFont ctFont = stylesheet.getFontByIndex(CellUtils.getAttributeIndex(cellStyleXf.getFontId(), cellStyleXf.isApplyFont(), true));
                        Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(new JSONObject(), stylesheet, ctFont));
                        final JSONObject applyProperties = createApplyProperties(cellStyleXf, true);
                        if(!applyProperties.isEmpty()) {
                            jsonProperties.put("apply", applyProperties);
                        }
                        JSONObject attrs = jsonProperties.length()>0?jsonProperties:null;
                        Boolean hidden = null;
                        Integer uiPriority = null;
                        Boolean isDefault = null;

                        // check for proper UI styleName
                        CTCellStyles cellStyles = stylesheet.getCellStyles();
                        if(cellStyles!=null) {
                            for(CTCellStyle cellStyle:cellStyles.getCellStyle()) {
                                if(cellStyle.getXfId()==i) {
                                    final String name = cellStyle.getName();
                                    if(name!=null&&!name.isEmpty()) {
                                        styleName = name;
                                    }
                                    if(i==0) {
                                        isDefault = new Boolean(true);
                                    }
                                    hidden = cellStyle.isHidden();
                                    break;
                                }
                            }
                        }
                        if(hidden==null||!hidden.booleanValue()) {
                        	operationsArray.put(createInsertStyleSheetOperation("cell", styleId, styleName, attrs, null, hidden, uiPriority, isDefault, null));
                        }
                    }
                }
            }

            // creating autostyles
            final CTCellXfs cellXfs = stylesheet.getCellXfs();
            if(cellXfs!=null) {
                final List<CTXf> cellXfsList = cellXfs.getXf();
                for(int i=0;i<cellXfsList.size();i++) {
                    final CTXf xf = cellXfsList.get(i);
                    final String styleId = "a" + Integer.toString(i);
                    JSONObject jsonProperties = new JSONObject();
                    Commons.mergeJsonObjectIfUsed("cell", jsonProperties, CellUtils.createCellProperties(xf, stylesheet, false));
                    final CTFont ctFont = stylesheet.getFontByIndex(CellUtils.getAttributeIndex(xf.getFontId(), xf.isApplyFont(), false));
                    Commons.mergeJsonObjectIfUsed("character", jsonProperties, CellUtils.createCharacterProperties(new JSONObject(), stylesheet, ctFont));
                    final JSONObject applyProperties = createApplyProperties(xf, false);                   
                    if(!applyProperties.isEmpty()) {
                    	jsonProperties.put("apply", applyProperties);
                   }
                   final Long xfId = xf.getXfId();
                    if(xfId!=null) {
                    	final String parentId = indexToStyleId.get(xfId);
                        if(parentId!=null) {
                        	jsonProperties.put("styleId", parentId);
                        }
                    }
                    operationsArray.put(createInsertAutoStyleOperation("cell", styleId, jsonProperties.length()>0?jsonProperties:null, i==0));
                }
            }
        }
    }

    final void createApplyProperty(JSONObject jsonApplyProps, String propName, Boolean xfProperty, boolean defaultValue)
    	throws JSONException {

    	// OOXML default value depends on XF type (true in styles, false in cells); operation default is always true 
    	boolean applyValue = (xfProperty == null) ? defaultValue : xfProperty.booleanValue();
    	if (!applyValue) { jsonApplyProps.put(propName, false); }
    }

    final JSONObject createApplyProperties(CTXf cellStyleXf, boolean defaultValue)
    	throws JSONException {

    	final JSONObject jsonApplyProperties = new JSONObject(0);
    	createApplyProperty(jsonApplyProperties, "border", cellStyleXf.isApplyBorder(), defaultValue);
    	createApplyProperty(jsonApplyProperties, "align", cellStyleXf.isApplyAlignment(), defaultValue);
    	createApplyProperty(jsonApplyProperties, "number", cellStyleXf.isApplyNumberFormat(), defaultValue);
    	createApplyProperty(jsonApplyProperties, "protect", cellStyleXf.isApplyProtection(), defaultValue);
    	createApplyProperty(jsonApplyProperties, "font", cellStyleXf.isApplyFont(), defaultValue);
    	createApplyProperty(jsonApplyProperties, "fill", cellStyleXf.isApplyFill(), defaultValue);
    	return jsonApplyProperties;
    }	    

    public void createOperations()
    	throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

	    createDocumentSettings(workbook);
	    createStyleOperations();
	    createThemeOperations(null, null);
	    createInsertNameOperations(workbook, null);
		final Sheets sheets = workbook.getSheets();
	    final List<Sheet> sheetList = sheets.getSheet();
	    for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
	    	final Sheet sheet = sheetList.get(sheetIndex);
	    	final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
	    	final String sheetType = XlsxOperationDocument.getSheetType(part);
	        createInsertSheetOperation(sheet, sheetType, part, sheetIndex);
	        if(sheetType.equals("worksheet")) {
	        	updateCellCount(((WorksheetPart)part).getJaxbElement());
			    createInsertNameOperations(workbook, new Long(sheetIndex));
        		createWorksheetOperations((WorksheetPart)part, sheetIndex);
    	        ((WorksheetPart)part).setJaxbElement((Worksheet)null);
	        }
	    }
    }

    public int createPreviewOperations(DocumentProperties documentProperties)
    	throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

	    final int activeSheetIndex = createDocumentSettings(workbook);
	    createStyleOperations();
	    createThemeOperations(null, null);
	    createInsertNameOperations(workbook, null);
		final Sheets sheets = workbook.getSheets();
	    final List<Sheet> sheetList = sheets.getSheet();
	    for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
	    	final Sheet sheet = sheetList.get(sheetIndex);
	    	final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
	    	final String sheetType = XlsxOperationDocument.getSheetType(part);
	        if(sheetType.equals("worksheet")&&(activeSheetIndex==sheetIndex)) {
	        	updateCellCount(((WorksheetPart)part).getJaxbElement());
		        createInsertSheetOperation(sheet, sheetType, part, sheetIndex);
	    	    createInsertNameOperations(workbook, new Long(sheetIndex));
        		createWorksheetOperations((WorksheetPart)part, sheetIndex);
		        ((WorksheetPart)part).setJaxbElement((Worksheet)null);
	        }
	        else {
		        createInsertSheetOperation(sheet, sheetType, null, sheetIndex);
	        }
	    }
        documentProperties.put(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, sheetList.size());
        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, activeSheetIndex);
	    return activeSheetIndex;
    }

    public void createOutstandingOperations(int activeSheetIndex)
        throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

		final Sheets sheets = workbook.getSheets();
	    final List<Sheet> sheetList = sheets.getSheet();
	    for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
	    	final Sheet sheet = sheetList.get(sheetIndex);
	    	final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
	    	final String sheetType = XlsxOperationDocument.getSheetType(part);
	        if(sheetType.equals("worksheet")&&(activeSheetIndex!=sheetIndex)) {
	        	final Worksheet worksheet = ((WorksheetPart)part).getJaxbElement();
	        	updateCellCount(worksheet);
		        createSetSheetAttributesOperation(sheet, worksheet, sheetIndex);
	    	    createInsertNameOperations(workbook, new Long(sheetIndex));
        		createWorksheetOperations((WorksheetPart)part, sheetIndex);
		        ((WorksheetPart)part).setJaxbElement((Worksheet)null);
	        }
	    }
    }

    public void createMetaOperations(DocumentProperties documentProperties)
    	throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

	    final int activeSheetIndex = createDocumentSettings(workbook);
	    createStyleOperations();
	    createThemeOperations(null, null);
	    createInsertNameOperations(workbook, null);
        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, activeSheetIndex);
        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX, Integer.valueOf(0));
    }

    public void createActivePartOperations(DocumentProperties documentProperties)
    	throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

        final int activeSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX);
		final Sheets sheets = workbook.getSheets();
	    final List<Sheet> sheetList = sheets.getSheet();
	    for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
	    	final Sheet sheet = sheetList.get(sheetIndex);
	    	final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
	    	final String sheetType = XlsxOperationDocument.getSheetType(part);
	        if(sheetType.equals("worksheet")&&(activeSheetIndex==sheetIndex)) {
	        	updateCellCount(((WorksheetPart)part).getJaxbElement());
		        createInsertSheetOperation(sheet, sheetType, part, sheetIndex);
	    	    createInsertNameOperations(workbook, new Long(sheetIndex));
        		createWorksheetOperations((WorksheetPart)part, sheetIndex);
		        ((WorksheetPart)part).setJaxbElement((Worksheet)null);
	        }
	        else {
		        createInsertSheetOperation(sheet, sheetType, null, sheetIndex);
	        }
	    }
        documentProperties.put(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, sheetList.size());
    }

    public boolean createNextPartOperations(DocumentProperties documentProperties)
    	throws Exception {

    	final WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
	    final Workbook workbook = workbookPart.getJaxbElement();

        final int activeSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX);
        int currentSheetIndex = (Integer)documentProperties.get(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX);

        final Sheets sheets = workbook.getSheets();
	    final List<Sheet> sheetList = sheets.getSheet();
	    for(; currentSheetIndex < sheetList.size(); currentSheetIndex++) {
	    	final Sheet sheet = sheetList.get(currentSheetIndex);
	    	final Part part = XlsxOperationDocument.getSheetPart(spreadsheetMLPackage, sheet);
	    	final String sheetType = XlsxOperationDocument.getSheetType(part);
	        if(sheetType.equals("worksheet")&&(activeSheetIndex!=currentSheetIndex)) {
	        	final Worksheet worksheet = ((WorksheetPart)part).getJaxbElement();
	        	updateCellCount(worksheet);
		        createSetSheetAttributesOperation(sheet, worksheet, currentSheetIndex);
	    	    createInsertNameOperations(workbook, new Long(currentSheetIndex));
        		createWorksheetOperations((WorksheetPart)part, currentSheetIndex);
		        ((WorksheetPart)part).setJaxbElement((Worksheet)null);
		        documentProperties.put(DocumentProperties.PROP_SPREADHSEET_CURRENT_SHEET_INDEX, Integer.valueOf(currentSheetIndex+1));
		        return true;
	        }
	    }
	    return false;
    }

    private void updateCellCount(Worksheet worksheet)
        throws FilterException {

    	if(maxCellCount>0) {
            final SheetData sheetData = worksheet.getSheetData();
            if(sheetData!=null) {
                final Iterator<Row> rowIterator = sheetData.createRowIterator();
                while(rowIterator.hasNext()) {
                    final Row row = rowIterator.next();
                    currentCellCount += row.getCellCount();
                    if(currentCellCount>maxCellCount) {
                        throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
                    }
                }
            }
    	}
    }

    /* returns the active sheet */

    private int createDocumentSettings(Workbook workbook)
    	throws JSONException {

    	long activeSheet = 0;
		final JSONObject documentAttrs = new JSONObject();
		documentAttrs.put("fileFormat", "ooxml");
		documentAttrs.put("cols", 16384);
		documentAttrs.put("rows", 1048576);
		final BookViews bookViews = workbook.getBookViews();
    	if(bookViews!=null) {
    		final List<CTBookView> bookViewList = bookViews.getWorkbookView();
    		if(!bookViewList.isEmpty()) {
    			final CTBookView bookView = bookViewList.get(0);
    			activeSheet = bookView.getActiveTab();
    			documentAttrs.put("activeSheet", activeSheet);
    		}
    	}

        // property nullDate: serial number 0 represents 1904-01-01 in "date1904" mode
        final WorkbookPr wbPr = workbook.getWorkbookPr(false);
        if (wbPr!=null && wbPr.isDate1904()) {
        	documentAttrs.put("nullDate", "1904-01-01");
        }

    	// property calcOnLoad: recalculate all formulas once after import
    	final CTCalcPr calcPr = workbook.getCalcPr(false);
        if (calcPr!=null && calcPr.isFullCalcOnLoad()) {
        	documentAttrs.put("calcOnLoad", true);
        }

		final JSONObject attrs = new JSONObject(1);
		attrs.put("document", documentAttrs);
		addSetDocumentAttributesOperation(attrs);

		return (int)activeSheet;
    }

    private void createInsertNameOperations(Workbook workbook, Long sheetIndex)
        throws JSONException {

        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            return;
        }
        final HashMap<String, CTDefinedName> definedNameMap = definedNames.getDefinedNameMap();
        for(CTDefinedName definedName:definedNameMap.values()) {
        	Long localSheetId = definedName.getLocalSheetId();
        	if ((sheetIndex == null) ? (localSheetId == null) : sheetIndex.equals(localSheetId)) {
        		addInsertNameOperation(sheetIndex, definedName.getName(), definedName.getValue()!=null ? definedName.getValue() : "", definedName.isHidden());
        	}
        }
    }

    private JSONObject getCellObject(Row row, Cell cell)
        throws Exception {

    	STCellType cellType = cell.getT();
    	if(cellType==null) {
    		cellType = STCellType.N;
    	}
    	final JSONObject cellObject = new JSONObject();
        final CTCellFormula cellFormula = cell.getF();
        if(cellFormula!=null) {
        	STCellFormulaType cellFormulaType = cellFormula.getT();
        	final String formulaExpr = cellFormula.getValue();
	    	if (formulaExpr!=null && !formulaExpr.isEmpty()) {
	    		cellObject.put("f", formulaExpr);
	    	}
        	if(cellFormulaType==STCellFormulaType.SHARED) {
                final Long si = cellFormula.getSi();
                if(si != null) {
					cellObject.put("si", si);
				}
				final String sr = cellFormula.getRef();
				if (sr != null) {
					cellObject.put("sr", sr);
				}
        	} else if (cellFormulaType==STCellFormulaType.ARRAY) {
				final String mr = cellFormula.getRef();
				if (mr != null) {
					cellObject.put("mr", mr);
				}
        	}
        }
        if(cellType==STCellType.INLINE_STR) {
            final CTRst is = cell.getIs();
            if(is!=null) {
                if(is.getT()!=null) {
                	cellObject.put("v", new String(is.getT().getValue()));;
                }
            }
        }
        else if(cell.getV()!=null) {
        	if(cellType==STCellType.N) {
                try {
                	cellObject.put("v", new Double(Double.parseDouble(cell.getV())));
                }
                catch(NumberFormatException e) {
                    // TODO: ups
                }
        	}
        	else if(cellType==STCellType.S) {
                String sharedString = getOperationDocument().getSharedString(cell.getV());
                if(sharedString!=null) {
                    cellObject.put("v", sharedString);
                }
        	}
        	else if(cellType==STCellType.B) {
                if(cell.getV().equals("0")||cell.getV().toLowerCase().equals("false")) {
                	cellObject.put("v", new Boolean(false));
                }
                else if(cell.getV().equals("1")||cell.getV().toLowerCase().equals("true")) {
                    cellObject.put("v", new Boolean(true));
                }
        	}        		
        	else if(cellType==STCellType.E) {
        		cellObject.put("e", cell.getV());
        	}
        	else {	// STCellType.STR, STCellType.F (ARRAY or DATA_TABLE format)
        		cellObject.put("v", cell.getV());
        	}
        }
        final long s = cell.getStyle();
        if ((s > 0) || cellObject.isEmpty()) {
        	cellObject.put("s", "a" + Long.toString(s));
        }
        return cellObject;
    }

    /*
     * part can be zero, then only a sheet insertion without attributes is done
     */
    private void createInsertSheetOperation(Sheet sheet, String sheetType, Part part, int sheetIndex)
    	throws JSONException {

        final JSONObject attrs = new JSONObject();
        if(part!=null&&sheetType.equals("worksheet")) {
            final WorksheetPart worksheetPart = (WorksheetPart)part;
            final Worksheet worksheet = worksheetPart.getJaxbElement();

            final JSONObject sheetProps = SheetUtils.createWorksheetProperties(sheet, worksheet);
            if(sheetProps!=null&&!sheetProps.isEmpty()) {
            	attrs.put("sheet", sheetProps);
            }
            final JSONObject columnProps = SheetUtils.createWorksheetColumnProperties(getOperationDocument(), worksheet);
            if(columnProps!=null&&!columnProps.isEmpty()) {
            	attrs.put("column", columnProps);
            }
            final JSONObject rowProps = SheetUtils.createWorksheetRowProperties(worksheet);
            if(rowProps!=null&&!rowProps.isEmpty()) {
            	attrs.put("row", rowProps);
            }
            addInsertSheetOperation(sheetIndex, sheet.getName(), sheetType, attrs);
        }
        else {
        	final JSONObject sheetProps = SheetUtils.createSheetProperties(sheet);
        	if(sheetProps!=null&&!sheetProps.isEmpty()) {
        		attrs.put("sheet", sheetProps);
        	}
            addInsertSheetOperation(sheetIndex, sheet.getName(), sheetType, attrs);
        }
    }

    private void createSetSheetAttributesOperation(Sheet sheet, Worksheet worksheet, int sheetIndex)
    	throws JSONException {

    	final JSONObject properties = new JSONObject();
        properties.put("sheet", SheetUtils.createWorksheetProperties(sheet, worksheet));
        properties.put("column", SheetUtils.createWorksheetColumnProperties(getOperationDocument(), worksheet));
        final JSONObject rowProperties = SheetUtils.createWorksheetRowProperties(worksheet);
        if(rowProperties!=null&&!rowProperties.isEmpty()) {
        	properties.put("row", rowProperties);
        }
        addSetSheetAttributesOperation(sheetIndex, properties);
    }

    private void createWorksheetOperations(final WorksheetPart worksheetPart, int sheetIndex)
        throws Exception {

    	final Worksheet worksheet = worksheetPart.getJaxbElement();

        // column attributes... row attributes ... cells...
        createColumnOperations(worksheet.getCols(), sheetIndex);
        createRowOperations(worksheet, sheetIndex);

        MergeCellHelper.createMergeCellOperations(operationsArray, sheetIndex, worksheet.getMergeCells());
        AutoFilterHelper.createAutoFilterOperations(operationsArray, sheetIndex, worksheet.getAutoFilter());
        TableHelper.createTableOperations(operationsArray, worksheetPart, sheetIndex, worksheet.getTableParts(false));

        // hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
            for(int i=0; i<hyperlinkList.size();i++) {
                final CTHyperlink hyperlink = hyperlinkList.get(i);
                final SmlUtils.CellRefRange cellRefRange = hyperlink.getCellRefRange(false);
                if(cellRefRange!=null) {
                    final String hyperlinkUrl = Commons.getUrl(worksheetPart, hyperlink.getId());
                    if(hyperlinkUrl!=null&&!hyperlinkUrl.isEmpty()) {
                        final JSONObject insertHyperlinkOperation = new JSONObject(5);
                        insertHyperlinkOperation.put("name", "insertHyperlink");
                        insertHyperlinkOperation.put("sheet", sheetIndex);
                        insertHyperlinkOperation.put("start", cellRefRange.getStart().getJSONArray());
                        if(!cellRefRange.getStart().equals(cellRefRange.getEnd())) {
                        	insertHyperlinkOperation.put("end", cellRefRange.getEnd().getJSONArray());
                        }
                        insertHyperlinkOperation.put("url", hyperlinkUrl);
                        operationsArray.put(insertHyperlinkOperation);
                    }
                }
            }
        }
        //drawings
        final CTDrawing drawing = worksheet.getDrawing();
        if(drawing!=null) {
            final String drawingId = drawing.getId();
            if(drawingId!=null&&!drawingId.isEmpty()) {
                final RelationshipsPart sourceRelationships = worksheetPart.getRelationshipsPart();
                if(sourceRelationships!=null) {
                    final Part part = sourceRelationships.getPart(drawingId);
                    if(part instanceof org.docx4j.openpackaging.parts.DrawingML.Drawing) {
                        Drawings.createDrawingOperations(this.getOperationDocument(), operationsArray, (org.docx4j.openpackaging.parts.DrawingML.Drawing)part, sheetIndex);
                    }
                }
            }
        }

        // validations
        final List<IDataValidations> validations = XlsxOperationDocument.getDataValidations(worksheet);
        int validationIndex = 0;
        for(IDataValidations dataValidations:validations) {
            for(IDataValidation dataValidation:dataValidations.getDataValidation()) {
                createDataValidation(sheetIndex, validationIndex++, dataValidation);
            }
        }

        // conditional formattings
        ConditionalFormattings.createOperations(getOperationDocument(), operationsArray, worksheet, sheetIndex);
    }

    private void createDataValidation(int sheetIndex, int validationIndex, IDataValidation dataValidation)
        throws JSONException {

        final List<SmlUtils.CellRefRange> cellRefRangeList = Utils.createCellRefRangeListFromSqrefList(dataValidation.getsqref());
        ValidationUtils.addInsertValidationOperation(operationsArray, sheetIndex, validationIndex, cellRefRangeList, dataValidation.getType(), dataValidation.getOperator(), dataValidation.getFormula1(), dataValidation.getFormula2(), dataValidation.isShowInputMessage(),
            dataValidation.getPromptTitle(), dataValidation.getPrompt(), dataValidation.isShowErrorMessage(), dataValidation.getErrorTitle(), dataValidation.getError(), dataValidation.getErrorStyle(),
                dataValidation.isShowDropDown()==false, dataValidation.isAllowBlank());
    }

    private boolean compareCell(Cell source, Cell dest) {
    	if(source.getF()!=null||dest.getF()!=null)
    		return false;
    	if(source.getIs()!=null||dest.getIs()!=null)
    		return false;
    	if(source.getT()!=dest.getT())
    		return false;
    	if(source.getVm()!=dest.getVm())
    		return false;
    	if(source.getCm()!=dest.getCm())
    		return false;
    	if(source.getStyle()!=dest.getStyle())
    		return false;
    	if(source.getV()==null&&dest.getV()==null)
    		return true;
    	if(source.getV()!=null&&dest.getV()!=null) {
    		return source.getV().equals(dest.getV());
    	}
    	else {
    		return false;
    	}
    }

    private void createRowOperations(Worksheet worksheet, int sheetIndex)
        throws Exception {

    	final SheetData sheetData = worksheet.getSheetData();
    	if(sheetData==null) {
    		return;
    	}

    	// first we will create row attributes
        Iterator<Row> rowIterator = sheetData.createRowIterator();

        Row lastRow = null;
        int lastRowNumber = 0;
        int rowCount = 0;

        while(rowIterator.hasNext()) {
            final Row row = rowIterator.next();

            if(lastRow!=null&&row.getRow()==lastRowNumber+rowCount&&RowUtils.compareRowProperties(lastRow, row)) {
                rowCount++;
            }
            else {
                if(lastRow!=null) {
                    final JSONObject attrs = RowUtils.createRowProperties(lastRow);
                    final Long style = lastRow.getS();
                    if(attrs!=null || style!=null) {
                        addChangeRows(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs, style);
                    }
                }
                lastRow = row;
                lastRowNumber = row.getRow();
                rowCount = 1;
            }
            if(!rowIterator.hasNext()&&lastRow!=null) {
                final JSONObject attrs = RowUtils.createRowProperties(lastRow);
                final Long style = lastRow.getS();
                if(attrs!=null || style!=null) {
                    addChangeRows(sheetIndex, lastRowNumber, (lastRowNumber - 1) + rowCount, attrs, style);
                }
            }
        }

        // create cell content for each row
        rowIterator = sheetData.createRowIterator();
        while(rowIterator.hasNext()) {
        	final Row row = rowIterator.next();

            JSONArray cellArray = new JSONArray();
            SmlUtils.CellRef topLeftCellRef = null;

            Cell 		lastCell = null;
            JSONObject 	lastCellObject = null;
            int 		lastCellX = -1;

            Iterator<Cell> cellIterator = row.createCellIterator();
            while(cellIterator.hasNext()) {
            	boolean repeat = false;
                Cell cell = cellIterator.next();
                final SmlUtils.CellRef cellRef = new SmlUtils.CellRef(cell.getColumn(), row.getRow());
                if(topLeftCellRef==null) {
                    topLeftCellRef = cellRef;
                }
                if(lastCell!=null) {
                	// checking empty cells
	                final int cellDifference = cellRef.getColumn() - lastCellX;
	                if(cellDifference > 1) {
	                	// we have empty cells ... at least one
                		cellArray.put(lastCellObject);
                		final JSONObject emptyCell = new JSONObject(1);
                		emptyCell.put("r", cellDifference-1);
                		cellArray.put(emptyCell);
	                }
	                else {
	                	if(compareCell(lastCell, cell)) {
	                		lastCellObject.put("r", lastCellObject.optInt("r", 1) + 1);
	                		repeat = true;
	                	}
	                	else {
	                		cellArray.put(lastCellObject);
	                	}
	                }
                }
            	lastCell = cell;
             	lastCellX = cellRef.getColumn();
            	if(repeat==false) {
	            	lastCellObject = getCellObject(row, cell);
            	}
            }
            if(lastCellObject!=null) {
                cellArray.put(lastCellObject);
            }
            if(!cellArray.isEmpty()) {
            	final JSONObject rowObject = new JSONObject();
                rowObject.put("c", cellArray);
            	final JSONArray rowArray = new JSONArray(1);
            	rowArray.put(rowObject);
                addChangeCellsOperation(sheetIndex, topLeftCellRef, rowArray);
            }
        }
    }

    private void createColumnOperations(List<Cols> colsIter, int sheetIndex)
        throws Exception {

        if(colsIter==null)
            return;

        for(Cols cols:colsIter) {
            List<Col> colIter = cols.getCol();
            if(colIter!=null) {
                for(Col col:colIter) {
                    final JSONObject attrs = ColumnUtils.createColumnProperties(getOperationDocument(), col);
                    final Long style = col.getStyle();
                    if(attrs!=null || style!=null) {
                        addChangeColumns(sheetIndex, col.getMin()-1, col.getMax()-1, attrs, style);
                    }
                }
            }
        }
    }

    @Override
	public void addSetDocumentAttributesOperation(JSONObject attrs)
    	throws JSONException {

    	if(attrs!=null&&!attrs.isEmpty()) {
	    	final JSONObject setDocumentAttributesObject = new JSONObject(2);
	    	setDocumentAttributesObject.put("name", "setDocumentAttributes");
	    	setDocumentAttributesObject.put("attrs", attrs);
	    	operationsArray.put(setDocumentAttributesObject);
    	}
    }

    public void addInsertSheetOperation(int sheetIndex, final String sheetName, final String type, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertSheetObject = new JSONObject();
        insertSheetObject.put("name", "insertSheet");
        insertSheetObject.put("sheet", sheetIndex);
        if(type!=null&&!type.equals("worksheet")) {
            insertSheetObject.put("type", type);
        }
        insertSheetObject.put("sheetName", sheetName);
        if(attrs!=null&&!attrs.isEmpty()) {
            insertSheetObject.put("attrs", attrs);
        }
        operationsArray.put(insertSheetObject);
    }

    public void addSetSheetAttributesOperation(int sheetIndex, final JSONObject attrs)
        throws JSONException {

        final JSONObject setSheetAttributesObject = new JSONObject();
        setSheetAttributesObject.put("name", "setSheetAttributes");
        setSheetAttributesObject.put("sheet", sheetIndex);
    	setSheetAttributesObject.put("attrs", attrs);
        operationsArray.put(setSheetAttributesObject);
    }

    final static List<Object> objectList = new ArrayList<Object>();

    public void addChangeCellsOperation(int sheetIndex, SmlUtils.CellRef cellRef, JSONArray cellContent)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject(4);
        insertCellsObject.put("name", "changeCells");
        insertCellsObject.put("sheet", sheetIndex);
        insertCellsObject.put("start", cellRef.getJSONArray());
        insertCellsObject.put("contents", cellContent);

        operationsArray.put(insertCellsObject);
    }

    public void addChangeRows(int sheetIndex, int start, int end, JSONObject attrs, Long style)
        throws JSONException {

        final JSONObject operation = new JSONObject();
        if (attrs!=null && attrs.length()>0) {
        	operation.put("attrs", attrs);
        }
        if (style!=null && style.longValue()>0) {
        	operation.put("s", "a" + Long.toString(style));
        }

        if (operation.length()>0) {
	        operation.put("name", "changeRows");
	        operation.put("sheet", sheetIndex);
	        operation.put("start", start);
	        if(end!=start) {
	            operation.put("end", end);
	        }
	        operationsArray.put(operation);
        }
    }

    public void addChangeColumns(int sheetIndex, long start, long end, JSONObject attrs, Long style)
        throws JSONException {

        final JSONObject operation = new JSONObject();
        if (attrs!=null && attrs.length()>0) {
        	operation.put("attrs", attrs);
        }
        if (style!=null && style.longValue()>0) {
        	operation.put("s", "a" + Long.toString(style));
        }

        if (operation.length()>0) {
	        operation.put("name", "changeColumns");
	        operation.put("sheet", sheetIndex);
	        operation.put("start", start);
	        if(end>start) {
	            operation.put("end", end);
	        }
	
	        operationsArray.put(operation);
        }
    }

    public void addInsertNameOperation(Long sheetIndex, String name, String formula, boolean hidden)
        throws JSONException {

        final JSONObject addInsertNameObject = new JSONObject(5);
        addInsertNameObject.put("name", "insertName");
        if(sheetIndex!=null) {
            addInsertNameObject.put("sheet", sheetIndex);
        }
        if(hidden) {
        	addInsertNameObject.put("hidden", true);
        }
        addInsertNameObject.put("label", name);
        addInsertNameObject.put("formula", formula);

        operationsArray.put(addInsertNameObject);
    }
}
