/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.xml.bind.JAXBException;
import javax.xml.stream.XMLStreamException;
import org.docx4j.XmlUtils;
import org.docx4j.dml.chart.CTChart;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.spreadsheetdrawing.ICellAnchor;
import org.docx4j.openpackaging.contenttype.ContentType;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.DefaultXmlPart;
import org.docx4j.openpackaging.parts.IPartFactory;
import org.docx4j.openpackaging.parts.JaxbXmlPart;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.DrawingML.Drawing;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.docx4j.relationships.Relationships;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.BookViews;
import org.xlsx4j.sml.CTAutoFilter;
import org.xlsx4j.sml.CTBookView;
import org.xlsx4j.sml.CTCalcPr;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTCellXfs;
import org.xlsx4j.sml.CTDefinedName;
import org.xlsx4j.sml.CTExtension;
import org.xlsx4j.sml.CTExtensionList;
import org.xlsx4j.sml.CTHyperlink;
import org.xlsx4j.sml.CTHyperlinks;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.CTSheetFormatPr;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTTablePart;
import org.xlsx4j.sml.CTTableParts;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.CfRule;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.DefinedNames;
import org.xlsx4j.sml.IDataValidation;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.SmlUtils.CellRef;
import org.xlsx4j.sml.SmlUtils.CellRefRange;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.WorkbookPr;
import org.xlsx4j.sml.Worksheet;
import com.google.common.collect.ImmutableSet;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.drawingml.ChartWrapper;
import com.openexchange.office.ooxml.drawingml.DMLChartSpace;
import com.openexchange.office.ooxml.drawingml.DMLGraphic;
import com.openexchange.office.ooxml.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.XlsxOperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.AutoFilterHelper;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils.ColRow;
import com.openexchange.office.ooxml.xlsx.tools.Drawings;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.TableHelper;
import com.openexchange.office.ooxml.xlsx.tools.Utils;

public class XlsxApplyOperationHelper extends ApplyOperationHelper {

	private final static Logger log = LoggerFactory.getLogger(ApplyOperationHelper.class);

    private final XlsxOperationDocument operationDocument;
    private final SpreadsheetMLPackage spreadsheetMLPackage;
    private final WorkbookPart workbookPart;
    private final Map<String, CTXf> styleXfMap;
    private final Map<String, Long> styleIdToIndex = new HashMap<String, Long>();

    public XlsxApplyOperationHelper(XlsxOperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        spreadsheetMLPackage = _operationDocument.getPackage();
        workbookPart = spreadsheetMLPackage.getWorkbookPart();
        styleXfMap = BuiltinIds.getUsedStyles(operationDocument.getStylesheet(false), null, styleIdToIndex);
    }

    @Override
    public XlsxOperationDocument getOperationDocument() {
        return operationDocument;
    }

    private CTCellFormula createCellFormula(Cell cell, STCellFormulaType type) {
		final CTCellFormula cellFormula = Context.getsmlObjectFactory().createCTCellFormula();
        cell.setF(cellFormula);
        if (type != STCellFormulaType.NORMAL) {
            cellFormula.setT(type);
        }
        return cellFormula;
    }

    private CTCellFormula createCellFormula(Cell cell) {
    	return createCellFormula(cell, STCellFormulaType.NORMAL);
    }

    public void setCellContent(Cell cell, JSONObject cellObject)
    	throws JSONException, InvalidFormatException {

    	final Object v = cellObject.opt("v");
    	final Object e = cellObject.opt("e");

    	if (v instanceof String) {
            cell.setT(STCellType.STR);
            cell.setV((String)v);
            cell.setIs(null);
		}
    	else if (v instanceof Boolean) {
			cell.setT(STCellType.B);
			cell.setV(((Boolean)v).booleanValue() ? "1" : "0");
            cell.setIs(null);
		}
		else if (v instanceof Number) {
            cell.setT(STCellType.N);
            cell.setV(((Number)v).toString());
            cell.setIs(null);
		}
		else if (v==JSONObject.NULL) {
            cell.setT(null);
            cell.setV(null);
            cell.setIs(null);
		}
		else if (e instanceof String) {
            cell.setT(STCellType.E);
            cell.setV((String)e);
            cell.setIs(null);
    	}

    	CTCellFormula formula = cell.getF();
    	final Object jsonF = cellObject.opt("f");
    	final Object jsonSi = cellObject.opt("si");
    	final Object jsonSr = cellObject.opt("sr");
    	final Object jsonMr = cellObject.opt("mr");

    	// the effective (old or new) formula expression
    	final String f =
    		(jsonF instanceof String) ? (String)jsonF :
    		(jsonF == JSONObject.NULL) ? null :
    		(formula != null) ? formula.getValue() :
    		null;

    	// the effective (old or new) shared index:
    	// - always delete the shared index when setting a matrix formula (property "mr")
    	// - "si" with "sr" represents the anchor cell of a shared formula
        // - "si" without "sr" represents any other child of a shared formula
    	Long si =
    		(jsonMr instanceof String) ? null :
    		(jsonSi instanceof Number) ? (Long)((Number)jsonSi).longValue() :
    		(jsonSi == JSONObject.NULL) ? null :
    		(formula != null) ? formula.getSi() :
    		null;

   		// the effective (old or new) bounding range of a shared formula or matrix formula:
    	// - both operation properties "sr" and "mr" will be mapped to the attribute @ref of the <formula> element
    	// - if both properties are contained in the operation, string value wins over null value
    	final String ref =
       		(jsonMr instanceof String) ? (String)jsonMr :
    		(jsonSr instanceof String) ? (String)jsonSr :
    		(jsonMr == JSONObject.NULL || jsonSr == JSONObject.NULL) ? null :
    		(formula != null) ? formula.getRef() :
    		null;

    	// whether the cell becomes or remains a formula cell
    	final boolean isFormula = (f != null) || (si != null) || (ref != null);

    	// create a <formula> element on demand; or remove an existing <formula> element
    	if (isFormula) {
    		if (formula == null) { formula = createCellFormula(cell); }
    		formula.setT((si != null) ? STCellFormulaType.SHARED : (ref != null) ? STCellFormulaType.ARRAY : null);
    		formula.setValue(f);
    		formula.setSi(si);
    		formula.setRef(ref);
    	} else if (formula != null) {
            cell.setF(null);
    	}

    	final String s = cellObject.optString("s", null);
    	if(s != null) {
        	Long index = Utils.parseAutoStyleId(s);
        	if (index != null) {
    			cell.setS(index.longValue() == 0 ? null : index);
        	}
    	}
    }

    public WorksheetPart getWorksheetPart(int sheetIndex) {
        final Sheet sheet = SheetUtils.getSheet(spreadsheetMLPackage, sheetIndex);
        final RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        return (WorksheetPart)relationshipPart.getPart(sheet.getId());
    }

    public Worksheet getWorksheet(int sheetIndex) {
        return getWorksheetPart(sheetIndex).getJaxbElement();
    }

    public void changeCells(int sheetIndex, JSONArray start, JSONArray rowArray)
        throws Exception {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::changeCells: size of start parameter != 2");

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        int y = start.getInt(1);
        for(int i=0; i<rowArray.length(); i++) {
        	final JSONObject rowObject = rowArray.getJSONObject(i);
            JSONArray cellArray = rowObject.optJSONArray("c");
            // property "c" may be an array index of another row object with a cell array
            if (cellArray == null) {
                int srcRowIndex = rowObject.optInt("c", -1);
                if (srcRowIndex >= 0) {
                	// use the "c" property of the other row object, but not the "r" property
                	cellArray = rowArray.getJSONObject(srcRowIndex).optJSONArray("c");
                }
            }
            final int rowRepeat = rowObject.optInt("r", 1);
            if (cellArray==null) {
            	y += rowRepeat;
            	continue;
            }
            for(int i2=0; i2<rowRepeat; i2++) {
	            int x = start.getInt(0);
	            for(int j=0; j<cellArray.length(); j++) {
	                final Row row = sheetData.getRow(y, true);
	                final JSONObject cellObject = cellArray.getJSONObject(j);
	            	int cellRepeat = 1;
	            	int emptyLength = 0;
	            	final Object r = cellObject.opt("r");
	            	if(r instanceof Number) {
	            		cellRepeat = ((Number)r).intValue();
	            		emptyLength = 1;
	            	}
	            	if(cellObject.optBoolean("u", false)) {
	        			row.clearCellRange(x, cellRepeat);
	        			x += cellRepeat;
	            	}
	            	else if (cellObject.length() > emptyLength) {
	                	for(int k=0; k<cellRepeat; k++) {
	                		setCellContent(row.getCell(x++, true), cellObject);
	                	}
	            	}
	            	else {
	            		x += cellRepeat;
	            	}
	            }
	            y++;
            }
        }
    }

    private boolean nullOnlyAttributes(JSONObject attrs) {
        final Iterator<Entry<String, Object>> iter = attrs.entrySet().iterator();
        while(iter.hasNext()) {
            final Entry<String, Object> entry = iter.next();
            if(entry.getValue() instanceof JSONObject) {
                if(nullOnlyAttributes((JSONObject)entry.getValue())==false) {
                    return false;
                }
            }
            else if(entry.getValue()!=JSONObject.NULL) {
                return false;
            }
        }
        return true;
    }

    public void changeRows(int sheetIndex, int start, int end, JSONObject attrs, String style)
        throws Exception {

        if(start<0 || end<start) {
            throw new RuntimeException("xlsx::changeRows: invalid interval");
        }

        final Worksheet worksheet = getWorksheet(sheetIndex);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        final boolean nullOnly = (attrs != null) && nullOnlyAttributes(attrs);
        for(int y=start; y<=end; y++) {
            final Row row = sheetData.getRow(y, !nullOnly);
            if(row!=null) {
                RowUtils.applyRowProperties(getOperationDocument(), attrs, style, row);
                sheetData.applyCachedRowData(row);
            }
        }
    }

    public void changeColumns(int sheetIndex, int start, int end, JSONObject attrs, String style)
        throws Exception {

    	if(start<0 || end<start) {
            throw new RuntimeException("xlsx::changeColumns: invalid interval");
        }

        final Worksheet worksheet = getWorksheet(sheetIndex);
        final List<Cols> colsList = worksheet.getCols();
        if(colsList.size()==0) {
            colsList.add(Context.getsmlObjectFactory().createCols());
        }

        // TODO: we only apply our attributes to the first Cols within the cols list, so additional
        // entries have to be merged ...
        start++;
        end++;

        final Cols cols = colsList.get(0);
        Utils.createColumnAttributeRanges(cols, Long.valueOf(start), Long.valueOf(end-start+1));
        for(final Col col:cols.getCol()) {
            if(col.getMin()>end)
                return;
            if(col.getMax()<start)
                continue;
            ColumnUtils.applyColumnProperties(getOperationDocument(), attrs, style, col, worksheet);
        }
    }

    public void insertRows(int sheetIndex, int start, int end, JSONObject attrs, String style)
        throws Exception {

        if(start<0 || end<start) {
            throw new RuntimeException("xlsx::insertRows: invalid interval");
        }
        int count = end-start+1;

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertRows(start, count);

        // set attributes and auto-style passed with the operation
        if (attrs!=null || style!=null) {
        	changeRows(sheetIndex, start, end, attrs, style);
        }

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            SmlUtils.insertRows(mergeCells.getMergeCell(), start, count, false);
            if(mergeCells.getMergeCell().isEmpty()) {
            	worksheet.setMergeCells(null);
            }
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            SmlUtils.insertRows(hyperlinks.getHyperlink(), start, count, false);
            if(hyperlinks.getHyperlink().isEmpty()) {
            	worksheet.setHyperlinks(null);
            }
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts(false);
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.insertRows(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, 1);
    }

    public void deleteRows(int sheetIndex, int start, int end) {

        if(start<0 || end<start) {
            throw new RuntimeException("xlsx::deleteRows: invalid interval");
        }
        int count = end-start+1;

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteRows, no sheetData available");
            return;
        }
        sheetData.deleteRows(start, count);

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
            SmlUtils.deleteRows(mergeCells.getMergeCell(), start, count, true);
            if(mergeCells.getMergeCell().isEmpty()) {
            	worksheet.setMergeCells(null);
            }
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
            SmlUtils.deleteRows(hyperlinks.getHyperlink(), start, count, false);
            if(hyperlinks.getHyperlink().isEmpty()) {
            	worksheet.setHyperlinks(null);
            }
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts(false);
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.deleteRows(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Row, -1);
    }

    public void insertColumns(int sheetIndex, int start, int end, JSONObject attrs, String style)
        throws Exception {

        if(start<0 || end<start) {
            throw new RuntimeException("xlsx::insertColumns: invalid interval");
        }

        int count = end-start+1;

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            sheetData = Context.getsmlObjectFactory().createSheetData();
            worksheet.setSheetData(sheetData);
        }
        sheetData.insertColumns(start, count);

        // move following columns (XML uses one-based indexes!)
        int cstart = start + 1;
        for(final Cols cols:worksheet.getCols()) {
        	final List<Col> colList = cols.getCol();
        	for (int i=colList.size()-1; i>=0; --i) {
        		final Col col = colList.get(i);
        		long min = col.getMin();
        		long max = col.getMax();
            	if (cstart<=min) {
            		// column will be moved completely (or deleted if it shifts outside the sheet)
            		min += count;
            		max = Math.min(max+count, 16384);
            		if (min<=max) {
            			col.setMin(min);
            			col.setMax(max);
            		} else {
            			colList.remove(i);
            		}
                } else if (cstart<=max) {
                	// column will be split into two parts
                    col.setMax(cstart-1);
                    final Col newCol = col.clone();
            		min = cstart+count;
            		max = Math.min(max+count, 16384);
            		if (min<=max) {
            			newCol.setMin(min);
            			newCol.setMax(max);
            			colList.add(i+1, newCol);
            		}
                }
            }
        }

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
        	SmlUtils.insertColumns(mergeCells.getMergeCell(), start, count, false);
        	if(mergeCells.getMergeCell().isEmpty()) {
        		worksheet.setMergeCells(null);
        	}
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
        	SmlUtils.insertColumns(hyperlinks.getHyperlink(), start, count, false);
        	if(hyperlinks.getHyperlink().isEmpty()) {
        		worksheet.setHyperlinks(null);
        	}
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts(false);
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.insertColumns(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, 1);

        // set attributes and auto-style passed with the operation
        if (attrs!=null || style!=null) {
        	changeColumns(sheetIndex, start, end, attrs, style);
        }
    }

    public void deleteColumns(int sheetIndex, int start, int end) {

        if(start<0 || end<start) {
            throw new RuntimeException("xlsx::deleteColumns: invalid interval");
        }
        int count = end-start+1;

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        worksheet.setDimension(null);
        final SheetData sheetData = worksheet.getSheetData();
        if(sheetData==null) {
            log.warn("xlsx export: deleteColumns, no sheetData available");
            return;
        }
        sheetData.deleteColumns(start, count);

        // taking care of column attributes
        final List<Cols> cols = worksheet.getCols();
        for(int ci = cols.size()-1; ci>=0; ci--) {
            final List<Col> colList = cols.get(ci).getCol();
            final int cstart = start + 1;
            final int cend = start + count;
        	for(int i = colList.size()-1; i>=0; i--) {

        		final Col col = colList.get(i);
        		long min = col.getMin();
        		long max = col.getMax();

        		if(max<cstart) {
        		 	break;
        		}
        		if(min>cend) {
        			col.setMin(min-count);
        			col.setMax(max-count);
        		}
        		else if (min>=cstart&&max<=cend) {
        			colList.remove(i);
        			if(colList.isEmpty()) {
        				cols.remove(ci);
        			}
        		}
        		else if (cstart>=min&&cend<=max) {
        			col.setMax(max-count);
        		}
        		else if (cend>max) {
        			col.setMax(cstart-1);
        		}
        		else if (cstart<min) {
        			col.setMin(cstart);
        			col.setMax(max-count);
        		}
        	}
        }

        // taking care of merged cells
        final CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells!=null) {
        	SmlUtils.deleteColumns(mergeCells.getMergeCell(), start, count, true);
        	if(mergeCells.getMergeCell().isEmpty()) {
        		worksheet.setMergeCells(null);
        	}
        }
        // taking care of hyperlinks
        final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks!=null) {
        	SmlUtils.deleteColumns(hyperlinks.getHyperlink(), start, count, false);
        	if(hyperlinks.getHyperlink().isEmpty()) {
        		worksheet.setHyperlinks(null);
        	}
        }
        // taking care of tables
        final CTTableParts tableParts = worksheet.getTableParts(false);
        if(tableParts!=null) {
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(int i=tablePartList.size()-1;i>=0;i--) {
        		TableHelper.deleteColumns(worksheetPart, tablePartList.get(i), start, count);
        	}
        }
        handleInsertDeleteColRow(sheetIndex, start, count, ColRow.Column, -1);
    }

    private SmlUtils.CellRefRange modifyCellRefRange(SmlUtils.CellRefRange cellRefRange, int start, int count, ColRow colrow, int rel, boolean append) {

        SmlUtils.CellRefRange newCellRefRange = null;
        if(rel>0) {
            if(colrow==ColRow.Column) {
                newCellRefRange = SmlUtils.CellRefRange.insertColumnRange(cellRefRange, start, count, append);
            }
            else {
                newCellRefRange = SmlUtils.CellRefRange.insertRowRange(cellRefRange, start, count, append);
            }
        }
        else {
            if(colrow==ColRow.Column) {
                newCellRefRange = SmlUtils.CellRefRange.deleteColumnRange(cellRefRange, start, count);
            }
            else {
                newCellRefRange = SmlUtils.CellRefRange.deleteRowRange(cellRefRange, start, count);
            }
        }
        return newCellRefRange;
    }

    private void handleInsertDeleteColRow(int sheetIndex, int start, int count, ColRow colrow, int rel){

        final Workbook workbook = workbookPart.getJaxbElement();
        final List<Sheet> allSheets = workbook.getSheets().getSheet();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();

        for (int i = 0; i < allSheets.size(); i++) {
            final Sheet sheet = allSheets.get(i);
            final Part part = workbookRelationshipsPart.getPart(sheet.getId());
            if(part instanceof WorksheetPart) {
                final WorksheetPart sheetPart = (WorksheetPart)part;
                final RelationshipsPart sheetRelationshipsPart = sheetPart.getRelationshipsPart();
                if(sheetRelationshipsPart!=null) {
                    final Worksheet worksheet = sheetPart.getJaxbElement();

                    if(sheetIndex==i) {

                        // check dataValidations
                        final List<IDataValidations> validations = XlsxOperationDocument.getDataValidations(worksheet);
                        for(final IDataValidations dataValidations:validations) {
                            final List<? extends IDataValidation> dataValidationList = dataValidations.getDataValidation();
                            for(int j = dataValidationList.size()-1;j>=0;j--) {
                                final IDataValidation dataValidation = dataValidationList.get(j);
                                final List<String> sqrefList = dataValidation.getsqref();
                                for(int k=sqrefList.size()-1; k>=0; k--) {
                                    final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqrefList.get(k));
                                    if(cellRefRange!=null) {
                                        final SmlUtils.CellRefRange newCellRefRange = modifyCellRefRange(cellRefRange, start, count, colrow, rel, true);
                                        if(newCellRefRange==null) {
                                            sqrefList.remove(k);
                                        }
                                        else if(!newCellRefRange.equals(cellRefRange)) {
                                            sqrefList.set(k, SmlUtils.getCellRefRange(newCellRefRange));
                                        }
                                    }
                                }
                                if(sqrefList.size()==0) {
                                	// TODO: we need to ensure that also calcengine + frontend is deleting this data validation
                                    dataValidationList.remove(j);
                                }
                            }
                        }

                        // check ConditionalFormattings
                        final Iterator<CfRule> conditionalRulesIter = worksheet.getConditionalFormattingRules().values().iterator();
                        while(conditionalRulesIter.hasNext()) {
                        	final CfRule cfRule = conditionalRulesIter.next();
                        	final List<String> sqrefList = cfRule.getRanges();
                            for(int k=sqrefList.size()-1; k>=0; k--) {
                                final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqrefList.get(k));
                                if(cellRefRange!=null) {
                                    final SmlUtils.CellRefRange newCellRefRange = modifyCellRefRange(cellRefRange, start, count, colrow, rel, true);
                                    if(newCellRefRange==null) {
                                        sqrefList.remove(k);
                                    }
                                    else if(!newCellRefRange.equals(cellRefRange)) {
                                        sqrefList.set(k, SmlUtils.getCellRefRange(newCellRefRange));
                                    }
                                }
                            }
                            if(sqrefList.size()==0) {
                            	conditionalRulesIter.remove();
                            }
                        }

                        // check autoFilter
                        final CTAutoFilter autoFilter = worksheet.getAutoFilter();
                        if(autoFilter!=null) {
                            final SmlUtils.CellRefRange cellRefRange = autoFilter.getCellRefRange(false);
                            if(cellRefRange!=null) {
                                final SmlUtils.CellRefRange newCellRefRange = this.modifyCellRefRange(cellRefRange, start, count, colrow, rel, false);
                                if(newCellRefRange==null) {
                                    AutoFilterHelper.deleteAutoFilter(this, sheetIndex);
                                }
                                else if(!newCellRefRange.equals(cellRefRange)) {
                                    autoFilter.setCellRefRange(newCellRefRange);
                                    AutoFilterHelper.insertDefinedTableName(this, Long.valueOf(sheetIndex), null, newCellRefRange);
                                }
                            }
                        }
                    }

                    // for each worksheet -->
                }
            }
        }
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
        throws InvalidFormatException, JAXBException, JSONException {

        // first we have to get a free relationship that can be used for our new sheet
        int sheetNumber = 1;

        Relationship relShip = null;
        PartName partName = null;
        do {
            partName = new PartName("/xl/worksheets/sheet" + Integer.toString(sheetNumber++) + ".xml");
            relShip = workbookPart.getRelationshipsPart().getRel(partName);
        }
        while(relShip!=null);

        // now we have a free partName... and we will look for a free SheetId
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();

        long sheetId = 1;
        do {
            boolean idFound = false;
            for(int i = 0; i < sheetList.size(); i++) {
                if(sheetList.get(i).getSheetId()==sheetId) {
                    idFound = true;
                    break;
                }
            }
            if(idFound==false)
                break;
            sheetId++;
        }
        while(true);

        final WorksheetPart worksheetPart = spreadsheetMLPackage.createWorksheetPart(partName, sheetName, sheetId);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        if(sheetFormatPr==null) {
            sheetFormatPr = Context.getsmlObjectFactory().createCTSheetFormatPr();
            worksheet.setSheetFormatPr(sheetFormatPr);
            sheetFormatPr.setBaseColWidth(new Long(10L));
            sheetFormatPr.setDefaultRowHeight(14.5d);
        }

        // setting the correct index position
        final Sheet sheet = sheetList.remove(sheetList.size() - 1);
        sheetList.add(sheetIndex, sheet);
        if(attrs!=null) {
            SheetUtils.applySheetProperties(workbook, sheetIndex, worksheet, attrs.optJSONObject("sheet"));
            SheetUtils.applyWorksheetColumnProperties(operationDocument, worksheet, attrs.optJSONObject("column"));
            SheetUtils.applyWorksheetRowProperties(operationDocument, worksheet, attrs.optJSONObject("row"));
        }

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            for(final CTDefinedName definedName:definedNames.getDefinedName()) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId>=sheetIndex) {
                        definedName.setLocalSheetId(localSheetId+1);
                    }
                }
            }
        }
    }

    public void moveSheet(int sheetIndex, int to) {
        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        sheetList.add(to, sheetList.remove(sheetIndex));

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(final CTDefinedName definedName:definedNameList) {
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    long newSheetId;
                    if(localSheetId==sheetIndex) {
                        newSheetId = to;
                    }
                    else {
                        newSheetId = localSheetId;
                        if(sheetIndex<newSheetId) {
                            newSheetId--;
                        }
                        if(newSheetId>=to) {
                            newSheetId++;
                        }
                    }
                    definedName.setLocalSheetId(newSheetId);
                }
            }
        }
    }

    public void deleteSheet(int sheetIndex) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheets sheets = workbook.getSheets();
        final List<Sheet> sheetList = sheets.getSheet();
        final Sheet sheet = sheetList.remove(sheetIndex);
        final String relId = sheet.getId();
        final RelationshipsPart relationshipsPart = workbookPart.getRelationshipsPart();
        relationshipsPart.removeRelationship(relationshipsPart.getRelationshipByID(relId));

        SheetUtils.updateBookView(workbook);

        // correcting definedName local sheet ids
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
            for(int i=0;i<definedNameList.size();i++) {
                final CTDefinedName definedName = definedNameList.get(i);
                final Long localSheetId = definedName.getLocalSheetId();
                if(localSheetId!=null) {
                    if(localSheetId==sheetIndex) {
                        definedNameList.remove(i--);
                    }
                    else if(localSheetId>sheetIndex) {
                        definedName.setLocalSheetId(localSheetId-1);
                    }
                }
            }
        }
    }

    public void copySheet(int sheet, int to, String sheetName, JSONObject tableNames)
    	throws Docx4JException, InvalidFormatException, JAXBException, JSONException, XMLStreamException {

    	insertSheet(to, sheetName, null);
    	if(to<=sheet) {
    		sheet++;
    	}
    	// create and insert a new copy of our worksheet
        final WorksheetPart sourceWorksheetPart = getWorksheetPart(sheet);
        final WorksheetPart destWorksheetPart   = getWorksheetPart(to);
        final Worksheet sourceWorksheet = sourceWorksheetPart.getJaxbElement();
        final Worksheet destWorksheet = XmlUtils.deepCopy(sourceWorksheet, Context.getJcSML());
        destWorksheetPart.setJaxbElement(destWorksheet);

        // now we have to clone the relationships
    	final Set<String> types = ImmutableSet.<String>builder()
    			.add("http://schemas.openxmlformats.org/officeDocument/2006/relationships/drawing")
    			.add("http://schemas.openxmlformats.org/officeDocument/2006/relationships/table")
    			.build();
    	final Set<String> ignoreTypes = ImmutableSet.<String>builder()
    			.add("http://schemas.microsoft.com/office/2007/relationships/slicer")
    			.build();

    	cloneRelationshipsPart(destWorksheetPart, sourceWorksheetPart, types, ignoreTypes);

        final Workbook workbook = workbookPart.getJaxbElement();

        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
        	// duplicating definedNames with localSheetId
        	final List<CTDefinedName> definedNameList = definedNames.getDefinedName();
        	for(int i=definedNameList.size()-1;i>=0;i--) {
        		final CTDefinedName sourceName = definedNameList.get(i);
        		if(sourceName.getLocalSheetId()!=null&&sourceName.getLocalSheetId().longValue()==sheet) {
        			final CTDefinedName destName = sourceName.clone();
        			destName.setLocalSheetId(Long.valueOf(to));
        			definedNameList.add(destName);
        		}
        	}
        }

        final CTExtensionList extLst = destWorksheet.getExtLst(false);
        if(extLst!=null) {
			final CTExtension ext = extLst.getExtensionByUri("{A8765BA9-456A-4dab-B4F3-ACF838C121DE}", false);
			if(ext!=null) {
				extLst.getExt().remove(ext);
			}
        }

        if(tableNames!=null) {
        	final Iterator<Entry<String, Object>> nameEntryIter = tableNames.entrySet().iterator();
        	while(nameEntryIter.hasNext()) {
        		final Entry<String, Object> nameEntry = nameEntryIter.next();
        		TableHelper.getTable(destWorksheetPart, nameEntry.getKey()).setDisplayName((String)nameEntry.getValue());
        	}
        }
    }

    private void cloneRelationshipsPart(Part destPart, Part sourcePart, Set<String> types, Set<String> ignoreTypes)
    	throws Docx4JException, InvalidFormatException, JAXBException, PartUnrecognisedException, XMLStreamException {

   		final RelationshipsPart sourceRelationshipsPart = sourcePart.getRelationshipsPart(false);
        if(sourceRelationshipsPart!=null) {
            final Relationships sourceRelationships = sourceRelationshipsPart.getRelationships();
	    	final List<Relationship> sourceRelationshipList = sourceRelationships.getRelationship();
	        for(int relIndex = 0; relIndex<sourceRelationshipList.size(); relIndex++) {
	        	final Relationship sourceRelationship = sourceRelationshipList.get(relIndex);
	        	final String sourceType = sourceRelationship.getType();
	        	final String targetMode = sourceRelationship.getTargetMode();
	        	if(ignoreTypes==null||!ignoreTypes.contains(sourceType)) {
		        	if((targetMode==null||targetMode.equalsIgnoreCase("Internal"))&&!sourceType.equals(Namespaces.HYPERLINK)) {
		        		if(types==null||types.contains(sourceType)) {
		        		    try {
	    						final Part part = sourceRelationshipsPart.getPart(sourceRelationship.getId());
	    						if(part instanceof JaxbXmlPart) {
	    							cloneXmlPart(destPart, (JaxbXmlPart<?>)part, sourceRelationship.getId());
	    							continue;
	    						}
	    						else if(part instanceof DefaultXmlPart) {
	    							cloneDefaultXmlPart(destPart, (DefaultXmlPart)part, sourceRelationship.getId());
	    							continue;
	    						}
		        		    }
		        		    catch(InvalidFormatException e) {
		        		        //
		        		    }
		        		}
		        	}
	        		destPart.getRelationshipsPart(true).getRelationships().getRelationship().add(XmlUtils.deepCopy(sourceRelationship, org.docx4j.jaxb.Context.getJcRelationships()));
	        	}
	        }
        }
    }

    public void insertHyperlink(int sheetIndex, JSONArray start, JSONArray optEnd, String url)
    	throws JSONException {

        if(optEnd==null) {
            optEnd = start;
        }
        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        CTHyperlinks hyperlinks = worksheet.getHyperlinks();
        if(hyperlinks==null) {
        	hyperlinks = Context.getsmlObjectFactory().createCTHyperlinks();
        	worksheet.setHyperlinks(hyperlinks);
        }
        final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
		final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
        for(int i=hyperlinkList.size()-1;i>=0;i--) {
        	final CTHyperlink hyperlink = hyperlinkList.get(i);
        	if((hyperlink.getCellRefRange(false)==null)||cellRefRange.contains(hyperlink.getCellRefRange(false))) {
        		hyperlinkList.remove(i);
        	}
        }
        final CTHyperlink hyperlink = Context.getsmlObjectFactory().createCTHyperlink();
        hyperlink.setCellRefRange(cellRefRange);
        hyperlink.setId(Commons.setUrl(worksheetPart, hyperlink.getId(), url));
        hyperlinks.getHyperlink().add(hyperlink);
    }

    public void deleteHyperlink(int sheetIndex, JSONArray start, JSONArray optEnd)
    	throws JSONException {

    	if(optEnd==null) {
            optEnd = start;
        }
    	final Worksheet worksheet = getWorksheet(sheetIndex);
    	final CTHyperlinks hyperlinks = worksheet.getHyperlinks();
    	if(hyperlinks!=null) {
    		final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
    		final List<CTHyperlink> hyperlinkList = hyperlinks.getHyperlink();
    		for(int i=hyperlinkList.size()-1;i>=0;i--) {
    			final CTHyperlink hyperlink = hyperlinkList.get(i);
    			if((hyperlink.getCellRefRange(false)==null)||cellRefRange.equals(hyperlink.getCellRefRange(false))) {
    				hyperlinkList.remove(i);
    			}
    		}
    		if(hyperlinkList.isEmpty()) {
    			worksheet.setHyperlinks(null);
    		}
    	}
    }

    private void cloneXmlPart(Part destPart, JaxbXmlPart<?> sourcePart, String relationshipId)
    	throws Docx4JException, InvalidFormatException, JAXBException, PartUnrecognisedException, XMLStreamException {

    	if(sourcePart instanceof IPartFactory) {
			final JaxbXmlPart<?> clonedPart = (JaxbXmlPart<?>)((IPartFactory)sourcePart).newPart(sourcePart.getPartName().getName());
	    	final ByteArrayOutputStream baos = new ByteArrayOutputStream();
	    	sourcePart.marshal(baos);
			clonedPart.unmarshal(new ByteArrayInputStream(baos.toByteArray()));
	        destPart.addTargetPart(clonedPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS, relationshipId);
	        cloneRelationshipsPart(clonedPart, sourcePart, null, null);
    	}
    }

    private void cloneDefaultXmlPart(Part destPart, DefaultXmlPart sourcePart, String relationshipId)
    	throws Docx4JException, JAXBException, XMLStreamException {

		final Document destDoc = (Document) sourcePart.getDocument().cloneNode(true);
		final DefaultXmlPart clonedPart = new DefaultXmlPart(sourcePart.getPartName());
		clonedPart.setContentType(new ContentType(sourcePart.getContentType()));
		clonedPart.setRelationshipType(sourcePart.getRelationshipType());
		destPart.addTargetPart(clonedPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS, relationshipId);
		clonedPart.setDocument(destDoc);
		cloneRelationshipsPart(clonedPart, sourcePart, null, null);
    }

    public void setSheetName(int sheetIndex, String newSheetName) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final Sheet _sheet = workbook.getSheets().getSheet().get(sheetIndex);
        _sheet.setName(newSheetName);
    }

    public static int getNameOffset(String content, int startOffset) {

        boolean stringLiteral = false;
        boolean complexString = false;
        for(int i=startOffset; i<content.length();i++) {
            final char n = content.charAt(i);
            if(complexString) {
                if(n=='\'') {
                    if(i+1<content.length()&&content.charAt(i+1)=='\'') {
                        i++;
                    }
                    else {
                        complexString = false;
                    }
                }
            }
            else if(stringLiteral) {
                if(n=='"') {
                    if(i+1<content.length()&&content.charAt(i+1)=='"') {
                        i++;
                    }
                    else {
                        stringLiteral = false;
                    }
                }
            }
            else if(n=='\'') {
                complexString = true;
            }
            else if(n=='"') {
                stringLiteral = true;
            }
            else if(n=='!') {
                return i;
            }
        }
        return -1;
    }

    private static String simpleCharPattern ="^[\\w.\\xa1-\\u2027\\u202a-\\uffff]+$";

    public static String updateSheetName(String content, String oldName, String newName) {
        String newContent = content;
        List<Integer> replaceOffsets = null;
        if(content!=null&&content.length()>oldName.length()) {
            for(int startOffset = 0; startOffset<content.length();) {
                final int nameOffset = getNameOffset(content, startOffset);
                if(nameOffset<0) {
                    break;
                }
                final int possibleLength = nameOffset - startOffset;
                if(possibleLength>=oldName.length()) {
                    boolean replace = false;
                    final boolean isComplex = content.charAt(nameOffset-1)=='\'';
                    final String subString = content.substring(nameOffset-oldName.length(), nameOffset);

                    final int mOffset = nameOffset - oldName.length();
                    if(isComplex&&oldName.charAt(0)=='\'') {
                        if(subString.equals(oldName)) {
                            replace = true;
                        }
                    }
                    else if(oldName.charAt(0)!='\'') {
                        if(subString.equals(oldName)) {
                            if(mOffset>startOffset) {
                                final String prev = content.substring(mOffset-1, mOffset);
                                if(!prev.matches(simpleCharPattern)) {
                                    replace = true;
                                }
                            }
                            else {
                                replace = true;
                            }
                        }
                    }
                    if(replace) {
                        if(replaceOffsets==null) {
                            replaceOffsets = new ArrayList<Integer>();
                        }
                        replaceOffsets.add(mOffset);
                    }
                }
                startOffset = nameOffset + 1;
            }
        }
        if(replaceOffsets!=null) {
            final StringBuffer buffer = new StringBuffer(content);
            for(int i=replaceOffsets.size()-1;i>=0;i--) {
                final int offset = replaceOffsets.get(i);
                buffer.replace(offset, offset + oldName.length(), newName);
            }
            newContent = buffer.toString();
        }
        return newContent;
    }

    public static String encodeSheetName(String sheetName) {
        final StringBuffer encodedName = new StringBuffer(sheetName.length());
        for(int i=0; i<sheetName.length();i++) {
            final char c = sheetName.charAt(i);
            encodedName.append(c);
            if(c=='\'') {
                encodedName.append(c);
            }
        }
        if(!sheetName.matches(simpleCharPattern)) {
            encodedName.insert(0, '\'');
            encodedName.append('\'');
        }
        return encodedName.toString();
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs)
    	throws JSONException {

    	if(attrs!=null) {
            final Workbook workbook = workbookPart.getJaxbElement();
            final Worksheet worksheet = getWorksheet(sheetIndex);
            final JSONObject sheetProperties = attrs.optJSONObject("sheet");
            if(sheetProperties!=null) {
                SheetUtils.applySheetProperties(workbook, sheetIndex, worksheet, attrs.optJSONObject("sheet"));
                final boolean visible = sheetProperties.optBoolean("visible", true);
                if(!visible) {
                    SheetUtils.updateBookView(workbook);
                }
            }
            final JSONObject columnProperties = attrs.optJSONObject("column");
            if(columnProperties!=null) {
                SheetUtils.applyWorksheetColumnProperties(getOperationDocument(), worksheet, columnProperties);
            }
            final JSONObject rowProperties = attrs.optJSONObject("row");
            if(rowProperties!=null) {
                SheetUtils.applyWorksheetRowProperties(getOperationDocument(), worksheet, rowProperties);
            }
        }
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
        throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        final Worksheet worksheet = getWorksheet(sheetIndex);
        CTMergeCells mergeCells = worksheet.getMergeCells();
        if(mergeCells==null) {
            mergeCells = Context.getsmlObjectFactory().createCTMergeCells();
            worksheet.setMergeCells(mergeCells);
        }
        // first we will remove each mergeCell that is covering our new mergeCellRange
        final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(new SmlUtils.CellRef(start.getInt(0), start.getInt(1)), new SmlUtils.CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            final CTMergeCell mergeCell = mergeCellList.get(i);
            final SmlUtils.CellRefRange mergedRefRange = mergeCell.getCellRefRange(false);
            if(mergedRefRange==null) {
                log.warn("xlsx export: mergeRange is invalid");
            }
            else {
                if(cellRefRange.intersects(mergedRefRange)) {
                    mergeCellList.remove(i);
                }
            }
        }
        if(type.equals("merge")) {
            final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
            mergeCell.setCellRefRange(new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1))));
            mergeCellList.add(mergeCell);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setCellRefRange(new CellRefRange(new CellRef(start.getInt(0), row), new CellRef(optEnd.getInt(0), row)));
                mergeCellList.add(mergeCell);
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                final CTMergeCell mergeCell = Context.getsmlObjectFactory().createCTMergeCell();
                mergeCell.setCellRefRange(new CellRefRange(new CellRef(column, start.getInt(1)), new CellRef(column, optEnd.getInt(1))));
                mergeCellList.add(mergeCell);
            }
        }
        if(mergeCells.getMergeCell().isEmpty()) {
            worksheet.setMergeCells(null);
        }
    }

    public void insertNumberFormat(long formatId, String formatCode) throws Exception {
    	operationDocument.getStylesheet(true).insertNumberFormat(formatId, formatCode);
    }

    public void deleteNumberFormat(long formatId) throws Exception {
    	final CTStylesheet stylesheet = operationDocument.getStylesheet(false);
    	if (stylesheet != null) {
    		stylesheet.deleteNumberFormat(formatId);
    	}
    }

    public Long getStyleIndex(String styleId) {
        if(styleId==null||styleId.isEmpty()) {
            return Long.valueOf(0);
        }
        return styleIdToIndex.get(styleId);
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, boolean hidden, int uiPriority, boolean defaultStyle)
        throws Exception {

        final CTStylesheet stylesheet = operationDocument.getStylesheet(true);
    	CTXf cellStyleXf = null;
		CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
        if(cellStyleXfs==null) {
            cellStyleXfs = Context.getsmlObjectFactory().createCTCellStyleXfs();
        }
        cellStyleXf = styleXfMap.get(styleId);
        if(cellStyleXf==null) {
            cellStyleXf = Context.getsmlObjectFactory().createCTXf();
            final List<CTXf> ctXfList = cellStyleXfs.getXf();
            final Long ctXfIndex = new Long(ctXfList.size());
            ctXfList.add(cellStyleXf);

            CTCellStyles cellStyles = stylesheet.getCellStyles();
            if(cellStyles==null) {
                cellStyles = Context.getsmlObjectFactory().createCTCellStyles();
            }
            final List<CTCellStyle> cellStyleList = cellStyles.getCellStyle();
            final CTCellStyle cellStyle = Context.getsmlObjectFactory().createCTCellStyle();
            cellStyleList.add(cellStyle);
            cellStyle.setBuiltinId(BuiltinIds.getBuiltinId(styleName));
            cellStyle.setName(styleName);
            cellStyle.setHidden(hidden);
            cellStyle.setXfId(ctXfIndex);
            styleXfMap.put(styleId, cellStyleXf);
            styleIdToIndex.put(styleId, ctXfIndex);
        }
        final JSONObject cellApplyProperties = attrs.optJSONObject("apply");
        if(cellApplyProperties!=null) {
        	// default for the "apply" flags in style XFs is "true"
            final Object applyBorder = cellApplyProperties.opt("border");
            if(applyBorder!=null) {
                cellStyleXf.setApplyBorder(applyBorder instanceof Boolean&&!((Boolean)applyBorder).booleanValue()?false:null);
            }
            final Object applyAlign = cellApplyProperties.opt("align");
            if(applyAlign!=null) {
                cellStyleXf.setApplyAlignment(applyAlign instanceof Boolean&&!((Boolean)applyAlign).booleanValue()?false:null);
            }
            final Object applyNumber = cellApplyProperties.opt("number");
            if(applyNumber!=null) {
                cellStyleXf.setApplyNumberFormat(applyNumber instanceof Boolean&&!((Boolean)applyNumber).booleanValue()?false:null);
            }
            final Object applyProtect = cellApplyProperties.opt("protect");
            if(applyProtect!=null) {
                cellStyleXf.setApplyProtection(applyProtect instanceof Boolean&&!((Boolean)applyProtect).booleanValue()?false:null);
            }
            final Object applyFont = cellApplyProperties.opt("font");
            if(applyFont!=null) {
                cellStyleXf.setApplyProtection(applyFont instanceof Boolean&&!((Boolean)applyFont).booleanValue()?false:null);
            }
            final Object applyFill = cellApplyProperties.opt("fill");
            if(applyFill!=null) {
                cellStyleXf.setApplyFill(applyFill instanceof Boolean&&!((Boolean)applyFill).booleanValue()?false:null);
            }
        }
    	if(cellStyleXf!=null) {
	        final JSONObject cellProperties = attrs.optJSONObject("cell");
	        if(cellProperties!=null) {
	            CellUtils.applyCellProperties(operationDocument, cellProperties, cellStyleXf, stylesheet);
	        }
	        final JSONObject characterProperties = attrs.optJSONObject("character");
	        if(characterProperties!=null) {
	            CellUtils.applyCharacterProperties(operationDocument, characterProperties, cellStyleXf, stylesheet);
	        }
    	}
    }

    private Boolean getAutoStyleApplyFlag(final JSONObject applyAttribs, String attrName) {
    	// default for the "apply" flags in cell XFs is "false"; default in operations is true (!)
    	final Object jsonValue = applyAttribs.opt(attrName);
    	return ((jsonValue == null) || ((jsonValue instanceof Boolean) && ((Boolean)jsonValue).booleanValue())) ? true : null;
    }

    private void applyAutoStyleAttributes(CTXf cellXf, JSONObject attrs, CTStylesheet stylesheet)
        throws Exception {

    	final JSONObject cellProperties = attrs.optJSONObject("cell");
        if(cellProperties!=null) {
            CellUtils.applyCellProperties(operationDocument, cellProperties, cellXf, stylesheet);
        }

        final JSONObject characterProperties = attrs.optJSONObject("character");
        if(characterProperties!=null) {
            CellUtils.applyCharacterProperties(operationDocument, characterProperties, cellXf, stylesheet);
        }

        final JSONObject cellApplyProperties = attrs.optJSONObject("apply");
        if (cellApplyProperties != null) {
            cellXf.setApplyBorder(getAutoStyleApplyFlag(cellApplyProperties, "border"));
            cellXf.setApplyAlignment(getAutoStyleApplyFlag(cellApplyProperties, "align"));
            cellXf.setApplyNumberFormat(getAutoStyleApplyFlag(cellApplyProperties, "number"));
            cellXf.setApplyProtection(getAutoStyleApplyFlag(cellApplyProperties, "protect"));
            cellXf.setApplyFont(getAutoStyleApplyFlag(cellApplyProperties, "font"));
            cellXf.setApplyFill(getAutoStyleApplyFlag(cellApplyProperties, "fill"));
        }
    }

    public void insertAutoStyle(String type, String styleId, JSONObject attrs, boolean defaultStyle)
        throws Exception {

    	Long index = Utils.parseAutoStyleId(styleId);
    	if (index == null) {
    		throw new FilterException("invalid auto-style identifier", ErrorCode.CRITICAL_ERROR);
    	}

        final CTStylesheet stylesheet = operationDocument.getStylesheet(true);
    	CTXf cellXf = null;
		final CTCellXfs ctXfs = stylesheet.getCellXfs();
		if(ctXfs != null) {
    		final List<CTXf> ctXfList = ctXfs.getXf();
			if (index != ctXfList.size()) {
	    		throw new FilterException("auto-style must be appended", ErrorCode.CRITICAL_ERROR);
			}
			cellXf = Context.getsmlObjectFactory().createCTXf();
			ctXfList.add(cellXf);
    		applyAutoStyleAttributes(cellXf, attrs, stylesheet);
			final String parentId = attrs.optString("styleId", "");
			if(!parentId.isEmpty()) {
				final CTCellStyles cellStyles = stylesheet.getCellStyles();
				if(cellStyles!=null) {
					for(CTCellStyle cellStyle:cellStyles.getCellStyle()) {
						if(parentId.equals(cellStyle.getName())) {
							cellXf.setXfId(cellStyle.getXfId());
							break;
						}
					}
				}
			}
		}
    }

    public void changeAutoStyle(String type, String styleId, JSONObject attrs)
        throws Exception {

    	Long index = Utils.parseAutoStyleId(styleId);
    	if (index == null) {
    		throw new FilterException("invalid auto-style identifier", ErrorCode.CRITICAL_ERROR);
    	}

        final CTStylesheet stylesheet = operationDocument.getStylesheet(true);
		final CTCellXfs ctXfs = stylesheet.getCellXfs();
		if(ctXfs != null) {
    		final List<CTXf> ctXfList = ctXfs.getXf();
    		int i = index.intValue();
			if ((i < 0) || (i >= ctXfList.size())) {
	        	throw new FilterException("auto-style does not exist", ErrorCode.CRITICAL_ERROR);
			}
    		applyAutoStyleAttributes(ctXfList.get(i), attrs, stylesheet);
		}
    }

    public void deleteAutoStyle(String type, String styleId)
        throws Exception {

        final Long index = Utils.parseAutoStyleId(styleId);
        if (index == null) {
    		throw new FilterException("invalid auto-style identifier", ErrorCode.CRITICAL_ERROR);
        }

    	final CTStylesheet stylesheet = operationDocument.getStylesheet(false);
		final CTCellXfs ctXfs = stylesheet.getCellXfs();
		if(ctXfs != null) {
    		final List<CTXf> ctXfList = ctXfs.getXf();
    		int i = index.intValue();
            if (i == 0) {
            	throw new FilterException("default auto-style cannot be deleted", ErrorCode.CRITICAL_ERROR);
            }
			if (index + 1 != ctXfList.size()) {
	        	throw new FilterException("only the last auto-style can be deleted", ErrorCode.CRITICAL_ERROR);
			}
			ctXfList.remove(i);
		}
    }

    private Drawing getDrawing(int sheetIndex)
        throws Exception {

        final WorksheetPart worksheetPart = getWorksheetPart(sheetIndex);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        final org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final String drawingsId = drawing.getId();
        if(drawingsId==null||drawingsId.isEmpty()) {
            throw new RuntimeException("xlsx:getDrawing: could not get drawing for sheet");
        }
        final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
        return (Drawing)worksheetRelationships.getPart(drawingsId);
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws Exception {

        Drawing drawingPart = null;
        final int sheetNumber = start.getInt(0);
        final WorksheetPart worksheetPart = getWorksheetPart(sheetNumber);
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        org.xlsx4j.sml.CTDrawing drawing = worksheet.getDrawing();
        if(drawing==null) { // -> drawing part needs to be created
            drawing = Context.getsmlObjectFactory().createCTDrawing();
            worksheet.setDrawing(drawing);

            drawingPart = new Drawing(new PartName("/xl/drawings/drawing"+ Integer.toString(sheetNumber+1) + ".xml"));
            drawingPart.setJaxbElement(org.docx4j.jaxb.Context.getDmlSpreadsheetDrawingObjectFactory().createCTDrawing());
            final Relationship dmlRelationship = worksheetPart.addTargetPart(drawingPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
            drawing.setId(dmlRelationship.getId());
        }
        else {
            drawingPart = (Drawing)worksheetPart.getRelationshipsPart().getPart(drawing.getId());
        }
        Drawings.insertDrawing(getOperationDocument(), start.getInt(1), drawingPart, type, attrs);

        applyDrawingAttributes(start, attrs);
    }

    public void applyDrawingAttributes(JSONArray start, JSONObject attrs)
        throws Exception {

        final org.docx4j.openpackaging.parts.DrawingML.Drawing drawingsPart = getDrawing(start.getInt(0));
        Drawings.applyDrawingAttributes(operationDocument, Drawings.applyAnchorAttributes(start.getInt(1), drawingsPart, attrs), drawingsPart, attrs);

        final org.docx4j.dml.spreadsheetdrawing.CTDrawing drawings = drawingsPart.getJaxbElement();
        final ICellAnchor cellAnchor = drawings.getEGAnchor().get(start.getInt(1));
        final HashMap<String, Object> drawingPropertyMap = DMLGraphic.getDrawingType(operationDocument, drawingsPart, cellAnchor);

        final JSONObject chartProperties = attrs.optJSONObject("chart");
        if(chartProperties!=null) {
            DMLChartSpace.applyChartSpaceProperties(chartProperties, Drawings.getChart(drawingPropertyMap), attrs.optJSONObject("fill"));
        }
    }

    public void deleteDrawing(JSONArray start)
        throws Exception {

        final Drawing drawing = getDrawing(start.getInt(0));
        final List<ICellAnchor> egAnchor = drawing.getJaxbElement().getEGAnchor();
        egAnchor.remove(start.getInt(1));
        if(egAnchor.isEmpty()) {
        	// removing the drawing relation, so the drawing will not be saved
        	final Worksheet worksheet = getWorksheet(start.getInt(0));
        	worksheet.setDrawing(null);
        }
    }

    public void insertChartDataSeries(JSONArray start, int series, JSONObject attrs)
        throws Exception {

        final Workbook workbook = workbookPart.getJaxbElement();
        final String sheetName = workbook.getSheets().getSheet().get(start.getInt(0)).getName();
        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.insertChartSpace(Drawings.getChart(drawingPropertyMap), sheetName, series, attrs);
    }

    public void setChartDataSeriesAttributes(JSONArray start, int series, JSONObject attrs)
            throws Exception {

            final Workbook workbook = workbookPart.getJaxbElement();
            final String sheetName = workbook.getSheets().getSheet().get(start.getInt(0)).getName();
            final Drawing drawing = getDrawing(start.getInt(0));
            final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
            DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
            DMLChartSpace.setDataSeriesAttributes(Drawings.getChart(drawingPropertyMap), sheetName, series, attrs);
        }

    public void deleteChartDataSeries(JSONArray start, int series)
        throws Exception {

        final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
        DMLChartSpace.deleteChartSpace(Drawings.getChart(drawingPropertyMap), series);
    }

    public void insertName(Long sheet, String label, String formula, Boolean hidden) {

    	if(sheet<0) {
    		sheet = null;
    	}

        final Workbook workbook = workbookPart.getJaxbElement();
        DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames==null) {
            definedNames = Context.getsmlObjectFactory().createDefinedNames();
            workbook.setDefinedNames(definedNames);
        }
        final HashMap<String, CTDefinedName> definedNameMap = definedNames.getDefinedNameMap();
        final CTDefinedName definedName = Context.getsmlObjectFactory().createCTDefinedName();
        definedName.setName(label);
        definedName.setLocalSheetId(sheet);
        definedName.setValue(formula);
        if(hidden!=null) {
            definedName.setHidden(hidden.booleanValue()?true:null);
        }
        definedNameMap.put(DefinedNames.getDefinedNameKey(sheet, label), definedName);
    }

    public void changeName(Long sheet, String label, String formula, String newLabel) {

    	if(sheet<0) {
    		sheet = null;
    	}
    	final Workbook workbook = workbookPart.getJaxbElement();
        final DefinedNames definedNames = workbook.getDefinedNames();
        final CTDefinedName oldName = definedNames.getDefinedNameMap().get(DefinedNames.getDefinedNameKey(sheet, label));
        definedNames.getDefinedNameMap().remove(DefinedNames.getDefinedNameKey(sheet, label));
        if(newLabel!=null) {
        	oldName.setName(newLabel);
        }
        if(formula!=null) {
        	oldName.setValue(formula);
        }
        definedNames.getDefinedNameMap().put(DefinedNames.getDefinedNameKey(sheet, newLabel!=null?newLabel:label), oldName);
    }

    public void deleteName(Long sheet, String name) {

        final Workbook workbook = workbookPart.getJaxbElement();
        final DefinedNames definedNames = workbook.getDefinedNames();
        if(definedNames!=null) {
            definedNames.getDefinedNameMap().remove(DefinedNames.getDefinedNameKey(sheet, name));
        }
        log.debug("xlsx:deleteName ... could not find name within definedNameList");
    }

	public void setChartAxisAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
		DMLChartSpace.setAxisAttributes(getChart(start), axis, attrs);
	}

	public void setChartGridlineAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
		DMLChartSpace.setChartGridlineAttributes(getChart(start), axis, attrs);
	}

	public void setChartTitleAttributes(JSONArray start, String axis, JSONObject attrs) throws Exception {
	    final ChartWrapper chart = getChart(start);

		if (axis.equals("main")) {
		    final CTChartSpace chartSpace = chart.getChartSpace();
			final CTChart ctChart = chartSpace.getChart();
			CTTitle title = ctChart.getTitle();
			if (title == null) {
				title = new CTTitle();
				ctChart.setTitle(title);
			}
			DMLChartSpace.setTitleFromAttrs(title, attrs);
            ctChart.setAutoTitleDeleted(DMLChartSpace.getBoolean(false));
		} else {
			DMLChartSpace.setChartTitleAttributes(chart, axis, attrs);
		}
	}

	public void setChartLegendAttributes(JSONArray start, JSONObject attrs) throws Exception {
		DMLChartSpace.setLegendFromAttrs(getChart(start), attrs);
	}

	private ChartWrapper getChart(JSONArray start) throws Exception {
	    final Drawing drawing = getDrawing(start.getInt(0));
        final HashMap<String, Object> drawingPropertyMap = new HashMap<String, Object>();
        DMLGraphic.getGraphicProperties(operationDocument, drawing, drawing.getJaxbElement().getEGAnchor().get(start.getInt(1)).getGraphicFrame().getGraphic(), drawingPropertyMap);
	    return Drawings.getChart(drawingPropertyMap);
	}

	public void setDocumentAttributes(JSONObject attrs) {
		final JSONObject documentAttrs = attrs.optJSONObject("document");
		if(documentAttrs!=null) {
	        final Workbook workbook = workbookPart.getJaxbElement();
			BookViews bookViews = workbook.getBookViews();
			if(bookViews==null) {
				bookViews = Context.getsmlObjectFactory().createBookViews();
				workbook.setBookViews(bookViews);
			}
    		final List<CTBookView> bookViewList = bookViews.getWorkbookView();
    		if(bookViewList.isEmpty()) {
    			final CTBookView newBookView = Context.getsmlObjectFactory().createCTBookView();
    			bookViewList.add(newBookView);
    		}
			final int activeSheet = documentAttrs.optInt("activeSheet", -1);
			if(activeSheet>=0) {
				final CTBookView bookView = bookViewList.get(0);
				bookView.setActiveTab(new Long(activeSheet));
			}

			// property nullDate: serial number 0 represents 1904-01-01 in "date1904" mode
			if (documentAttrs.has("nullDate")) {
				final String nullDate = documentAttrs.optString("nullDate", null);
				final boolean date1904 = "1904-01-01".equals(nullDate);
				// do not force to create the <workbookPr> element when resetting the flag
		        final WorkbookPr wbPr = workbook.getWorkbookPr(date1904);
		        if (wbPr!=null) { wbPr.setDate1904(date1904 ? true : null); }
			}

			// property calcOnLoad: recalculate all formulas when loading the document
			if (documentAttrs.has("calcOnLoad")) {
				final boolean calcOnLoad = documentAttrs.optBoolean("calcOnLoad", false);
				// do not force to create the <calcPr> element when resetting the flag
				final CTCalcPr calcPr = workbook.getCalcPr(calcOnLoad);
				if (calcPr!=null) { calcPr.setFullCalcOnLoad(calcOnLoad ? true : null); }
			}
		}
	}
}
