/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.components;

import java.text.ParseException;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.wml.ContentAccessor;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.CTPlaceholder;
import org.pptx4j.pml.STPlaceholderType;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.SldMaster;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.tools.PMLShapeHelper;

public class TextComponent extends PptxComponent {

    public TextComponent(ComponentContext parentContext, IndexedNode<Object> _textNode, int _componentNumber) {
        super(parentContext, _textNode, _componentNumber);
    }

    @Override
    public int getNextComponentNumber() {
        int textLength = ((CTRegularTextRun)getNode().getData()).getT().length();
        if (textLength==0) {
            System.out.println("Error: empty text string is no component... this results in problems with the component iterator");
            textLength++;       // increasing.. preventing an endless loop
        }
        return componentNumber + textLength;
    }

    @Override
    public void splitStart(int componentPosition, SplitMode splitMode) {
    	final int splitPosition = componentPosition-componentNumber;
    	if(splitPosition>0) {
    		splitText(splitPosition, true);
    		componentNumber+=splitPosition;
    	}
    }

    @Override
    public void splitEnd(int componentPosition, SplitMode splitMode) {
    	if(getNextComponentNumber()>++componentPosition) {
    		splitText(componentPosition-componentNumber, false);
    	}
    }

    private void splitText(int textSplitPosition, boolean splitStart) {
    	final CTRegularTextRun oldTextRun = (CTRegularTextRun)getNode().getData();
        if(textSplitPosition>0&&textSplitPosition<oldTextRun.getT().length()) {
        	final CTRegularTextRun newTextRun = XmlUtils.deepCopy(oldTextRun);
        	newTextRun.setParent(oldTextRun.getParent());
            final StringBuffer s = new StringBuffer(oldTextRun.getT());
        	final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)oldTextRun.getParent()).getContent();
            if(splitStart) {
            	parentContent.addNode(getNode(), new IndexedNode<Object>(newTextRun), true);
                newTextRun.setT(s.substring(0, textSplitPosition));
                oldTextRun.setT(s.substring(textSplitPosition));
            }
            else {
            	parentContent.addNode(getNode(), new IndexedNode<Object>(newTextRun), false);
                oldTextRun.setT(s.substring(0, textSplitPosition));
                newTextRun.setT(s.substring(textSplitPosition));
            }
        }
    }

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
			throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

		if(attrs!=null) {
			if(attrs.hasAndNotNull("character")) {
				final Component parentComponent = getParagraphComponent().getParentComponent();
				if(parentComponent instanceof IShapeComponent&&((IShapeComponent)parentComponent).isPresentationObject()) {
					final ShapeComponent shapeComponent = (ShapeComponent)parentComponent;
					final SlideComponent slideComponent = PMLShapeHelper.getSlideComponent(shapeComponent);
					final Part slidePart = (Part)slideComponent.getObject();
					if(slidePart instanceof SlideMasterPart) {
						final Shape shape = (Shape)shapeComponent.getObject();
						final CTPlaceholder placeholder = shape.getNvSpPr().getNvPr().getPh();
						if(placeholder.getType()==STPlaceholderType.TITLE) {
							final SldMaster sldMaster = ((SlideMasterPart)slidePart).getJaxbElement();
							applyStyleFormatting(operationDocument, sldMaster.getTxStyles().getTitleStyle(), attrs);
						}
						else if(placeholder.getType()==STPlaceholderType.BODY) {
							final SldMaster sldMaster = ((SlideMasterPart)slidePart).getJaxbElement();
							applyStyleFormatting(operationDocument, sldMaster.getTxStyles().getBodyStyle(), attrs);
						}
						else {
							// presentation object on master slide
							applyObjectStyleFormatting(operationDocument, shapeComponent, attrs);
						}
					}
					else if(slidePart instanceof SlideLayoutPart) {
						// presentation object on layout slide -> the object style is attributed
						applyObjectStyleFormatting(operationDocument, shapeComponent, attrs);
					}
					else {
						// presentation object on normal slide -> the object is hard formatted
						applyHardFormatting(operationDocument, attrs);
					}
				}
				else {
					applyHardFormatting(operationDocument, attrs);
				}
			}
		}
	}

	private void applyStyleFormatting(OperationDocument operationDocument, CTTextListStyle textListStyle, JSONObject attrs)
		throws InvalidFormatException, PartUnrecognisedException, JSONException {

		DMLHelper.applyTextCharacterPropertiesFromJson(textListStyle.getPPr(getParagraphComponent().getLevel(), true).getDefRPr(true), attrs, operationDocument.getContextPart());
	}

	private void applyObjectStyleFormatting(OperationDocument operationDocument, ShapeComponent shapeComponent, JSONObject attrs)
		throws JSONException {

		DMLHelper.applyTextCharacterPropertiesFromJson(shapeComponent.getTextBody(true), getParagraphComponent().getLevel(), attrs, operationDocument.getContextPart());
	}

	private void applyHardFormatting(OperationDocument operationDocument, JSONObject attrs)
		throws JSONException {

		DMLHelper.applyTextCharacterPropertiesFromJson(((CTRegularTextRun)getObject()).getRPr(true), attrs, operationDocument.getContextPart());
	}

	@Override
	public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException, ParseException, FilterException {

		DMLHelper.createJsonFromTextCharacterProperties(attrs, ((CTRegularTextRun)getObject()).getRPr(false), createOperationHelper.getOperationDocument().getContextPart());
		return attrs;
	}

	public ParagraphComponent getParagraphComponent() {
		return (ParagraphComponent)getParentComponent();
	}
}
