/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.components;

import java.util.List;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PresentationML.ICommonSlideAccessor;
import org.docx4j.openpackaging.parts.PresentationML.JaxbPmlPart;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.wml.ContentAccessor;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;
import org.pptx4j.Pptx4jException;
import org.pptx4j.pml.CTBackground;
import org.pptx4j.pml.CTGraphicalObjectFrame;
import org.pptx4j.pml.CTHeaderFooter;
import org.pptx4j.pml.CommonSlideData;
import org.pptx4j.pml.CxnSp;
import org.pptx4j.pml.GroupShape;
import org.pptx4j.pml.IHeaderFooterAccessor;
import org.pptx4j.pml.Pic;
import org.pptx4j.pml.Presentation;
import org.pptx4j.pml.Presentation.SldMasterIdLst.SldMasterId;
import org.pptx4j.pml.STSlideLayoutType;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.Sld;
import org.pptx4j.pml.SldLayout;
import org.pptx4j.pml.SldMaster;
import org.pptx4j.pml.SlideLayoutIdList.SldLayoutId;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.PptxOperationDocument;
import com.openexchange.office.ooxml.pptx.tools.PMLShapeHelper;

public class SlideComponent extends PptxComponent {

    public SlideComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
        final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

        Component nextComponent = null;
        final IndexedNode<Object> paragraphNode = getNode();
        final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)paragraphNode.getData()).getContent();
        for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
            final IndexedNode<Object> childNode = nodeList.getNode(i);
            final Object o = getContentModel(childNode, paragraphNode.getData());
            if(o instanceof GroupShape) {
            	nextComponent = new ShapeGroupComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof Shape) {
            	nextComponent = new ShapeComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CTGraphicalObjectFrame) {
            	nextComponent = new ShapeGraphicComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CxnSp) {
            	nextComponent = new ShapeConnectorComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof Pic) {
            	nextComponent = new ShapePicComponent(this, childNode, nextComponentNumber);
            }
/*
            else if(o instanceof CTRel) {
            	nextComponent = new ShapeRelComponent(this, childNode, nextComponentNumber);
            }
*/
        }
        return nextComponent;
    }

    @Override
    public void delete(OperationDocument operationDocument, int count)
    	throws InvalidFormatException {

    	final Part contextPart = operationDocument.getContextPart();
    	for(int i=(getComponentNumber()+count)-1; i>=getComponentNumber(); i--) {
			try {
				if(contextPart instanceof SlideLayoutPart) {
					
					final SlideLayoutPart slideLayoutPart = (SlideLayoutPart)contextPart;
					final SlideMasterPart slideMasterPart = slideLayoutPart.getSlideMasterPart();
					final RelationshipsPart slideMasterRelationshipsPart = slideMasterPart.getRelationshipsPart();
					final Relationship slideLayoutRel = slideMasterRelationshipsPart.getRel(slideLayoutPart.getPartName());
					slideMasterRelationshipsPart.removeRelationship(slideLayoutRel);
					final List<SldLayoutId> slideLayoutIdList = slideMasterPart.getJaxbElement().getSldLayoutIdLst(false).getSldLayoutId();
					for(int j = 0; j < slideLayoutIdList.size(); j++) {
						if(slideLayoutIdList.get(j).getRid().equals(slideLayoutRel.getId())) {
							slideLayoutIdList.remove(j);
							break;
						}
					}
				}
				else if(contextPart instanceof SlideMasterPart) {
					final SlideMasterPart slideMasterPart = (SlideMasterPart)contextPart;
					final MainPresentationPart mainPresentationPart = (((PptxOperationDocument)operationDocument).getPackage()).getMainPresentationPart();
					final Presentation mainPresentation = mainPresentationPart.getJaxbElement();
					final RelationshipsPart mainPresentationRelationshipsPart = mainPresentationPart.getRelationshipsPart();
					final Relationship slideMasterRel = mainPresentationRelationshipsPart.getRel(slideMasterPart.getPartName());
					mainPresentationRelationshipsPart.removeRelationship(slideMasterRel);
					final List<SldMasterId> sldMasterIdList = mainPresentation.getSldMasterIdLst().getSldMasterId();
					for(int j = 0; j < sldMasterIdList.size(); j++) {
						if(sldMasterIdList.get(j).getRid().equals(slideMasterRel.getId())) {
							sldMasterIdList.remove(j);
							break;
						}
					}
				}
				else {
					(((PptxOperationDocument)operationDocument).getPackage()).getMainPresentationPart().removeSlide(i);
				}
			} catch (Pptx4jException e) {
				throw new InvalidFormatException("");
			}
    	}
    }

    @Override
    public Component insertChildComponent(ComponentContext parentContext, IndexedNode<Object> contextNode, int number, Component child, Type type)
    	throws JAXBException {

    	IndexedNodeList<Object> indexedNodeList = (IndexedNodeList<Object>)((ContentAccessor)contextNode.getData()).getContent();
        IndexedNode<Object> referenceNode = child!=null ? child.getNode() : null;

        switch(type) {
        	case AC_SHAPE: {
	            final Child newChild = PMLShapeHelper.createShape();
	            newChild.setParent(contextNode.getData());
	            final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	            indexedNodeList.addNode(referenceNode, newChildNode, true);
	            return new ShapeComponent(parentContext, newChildNode, number);
        	}
        	case AC_GROUP: {
	            final Child newChild = PMLShapeHelper.createGroupShape();
	            newChild.setParent(contextNode.getData());
	            final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	            indexedNodeList.addNode(referenceNode, newChildNode, true);
	            return new ShapeGroupComponent(parentContext, newChildNode, number);
        	}
        	case AC_CONNECTOR: {
        		final Child newChild = PMLShapeHelper.createConnectorShape();
        		newChild.setParent(contextNode.getData());
	            final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	            indexedNodeList.addNode(referenceNode, newChildNode, true);
	            return new ShapeConnectorComponent(parentContext, newChildNode, number);
        	}
        	case AC_IMAGE: {
	            final Child newChild = PMLShapeHelper.createImage();
	            newChild.setParent(contextNode.getData());
	            final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	            indexedNodeList.addNode(referenceNode, newChildNode, true);
	            return new ShapePicComponent(parentContext, newChildNode, number);
        	}
        	case TABLE: {
        		final Child newChild = PMLShapeHelper.createGraphicalObjectFrame(Type.TABLE);
	            newChild.setParent(contextNode.getData());
	            final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	            indexedNodeList.addNode(referenceNode, newChildNode, true);
	            return new ShapeGraphicComponent(parentContext, newChildNode, number);
        	}
        	default : {
                throw new UnsupportedOperationException();
            }
        }
    }

    @Override
    public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JSONException {

		final ICommonSlideAccessor commonSlideAccess = (ICommonSlideAccessor)getObject();
		final CommonSlideData commonSlideData = commonSlideAccess.getCSld();

		final JSONObject slideAttrs = attrs.optJSONObject("slide");
		if(slideAttrs!=null) {
			final Object slideName = slideAttrs.opt("name");
			if(slideName!=null) {
				if(slideName==JSONObject.NULL) {
					commonSlideData.setName(null);
				}
				else {
					commonSlideData.setName((String)slideName);
				}
			}
			final Object sldElement = ((JaxbPmlPart<?>)getObject()).getJaxbElement();
			// hidden only available at normal slides
			if(sldElement instanceof Sld) {
				final Object hidden = slideAttrs.opt("hidden");
				if(hidden!=null) {
					if(hidden instanceof Boolean) {
						((Sld)sldElement).setShow(((Boolean)hidden).booleanValue() ? false : null);
					}
					else {
						((Sld)sldElement).setShow(null);
					}
				}
				final Object isFollowMasterShape = slideAttrs.opt("followMasterShapes");
				if(isFollowMasterShape!=null) {
					if(isFollowMasterShape instanceof Boolean&&!((Boolean)isFollowMasterShape).booleanValue()) {
						((Sld)sldElement).setShowMasterSp(false);
					}
					else {
						((Sld)sldElement).setShowMasterSp(null);
					}
				}
			}
			else if(sldElement instanceof SldLayout) {
				final Object isFollowMasterShape = slideAttrs.opt("followMasterShapes");
				if(isFollowMasterShape!=null) {
					if(isFollowMasterShape instanceof Boolean&&!((Boolean)isFollowMasterShape).booleanValue()) {
						((SldLayout)sldElement).setShowMasterSp(false);
					}
					else {
						((SldLayout)sldElement).setShowMasterSp(null);
					}
				}
				final Object type = slideAttrs.opt("type");
				if(type instanceof String) {
					((SldLayout)sldElement).setType(STSlideLayoutType.fromValue((String)type));
				}
				else if(type==JSONObject.NULL) {
					((SldLayout)sldElement).setType(null);
				}
				applyHeaderFooterAttrs((SldLayout)sldElement, slideAttrs);
			}
			else if(sldElement instanceof SldMaster) {
				applyHeaderFooterAttrs((SldMaster)sldElement, slideAttrs);
			}
		}
    	if(attrs.hasAndNotNull("fill")) {
    		commonSlideData.getBg(true).setBgRef(null);
			DMLHelper.applyFillPropertiesFromJson(commonSlideData.getBg(true).getBgPr(true), attrs, operationDocument, operationDocument.getContextPart());
		}
    }

    private void applyHeaderFooterAttrs(IHeaderFooterAccessor iHeaderFooterAccessor, JSONObject attrs) {
    	final Object isHeader = attrs.opt("isHeader");
    	final Object isFooter = attrs.opt("isFooter");
    	final Object isDate = attrs.opt("isDate");
    	final Object isSlideNum = attrs.opt("isSlideNum");
    	if(isHeader!=null||isFooter!=null||isDate!=null||isSlideNum!=null) {
    		final CTHeaderFooter hf = iHeaderFooterAccessor.getHf(true);
	    	if(isHeader!=null) {
	    		if(isHeader instanceof Boolean) {
	    			hf.setHdr((Boolean)isHeader);
	    		}
	    		else {
	    			hf.setHdr(null);
	    		}
	    	}
	    	if(isFooter!=null) {
	    		if(isFooter instanceof Boolean) {
	    			hf.setFtr((Boolean)isFooter);
	    		}
	    		else {
	    			hf.setFtr(null);
	    		}
	    	}
	    	if(isDate!=null) {
	    		if(isDate instanceof Boolean) {
	    			hf.setDt((Boolean)isDate);
	    		}
	    		else {
	    			hf.setDt(null);
	    		}
	    	}
	    	if(isSlideNum!=null) {
	    		if(isSlideNum instanceof Boolean) {
	    			hf.setSldNum((Boolean)isSlideNum);
	    		}
	    		else {
	    			hf.setSldNum(null);
	    		}
	    	}
    	}
    }

    @Override
    public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
    	throws JSONException {

		final JSONObject slideAttrs = new JSONObject();

    	final ICommonSlideAccessor commonSlideAccess = (ICommonSlideAccessor)getObject();
		final CommonSlideData commonSlideData = commonSlideAccess.getCSld();
		final String slideName = commonSlideData.getName();
		if(slideName!=null) {
			slideAttrs.put("name", slideName);
		}
		final Object sldElement = ((JaxbPmlPart<?>)getObject()).getJaxbElement();
		if(sldElement instanceof Sld) {
			if(!((Sld)sldElement).isShowMasterSp()) {
				slideAttrs.put("followMasterShapes", false);
			}
			if(!((Sld)sldElement).isShow()) {
				slideAttrs.put("hidden", true);
			}
		}
		else if(sldElement instanceof SldLayout) {
			final SldLayout sldLayout = (SldLayout)sldElement;
			if(!sldLayout.isShowMasterSp()) {
				slideAttrs.put("followMasterShapes", false);
			}
			if(sldLayout.getType()!=null) {
				slideAttrs.put("type", sldLayout.getType().value());
			}
			createHeaderFooterOperations((SldLayout)sldElement, slideAttrs);
		}
		else if(sldElement instanceof SldMaster) {
			createHeaderFooterOperations((SldMaster)sldElement, slideAttrs);
		}
		if(!slideAttrs.isEmpty()) {
			attrs.put("slide", slideAttrs);
		}
		final CTBackground background = commonSlideData.getBg(false);
		if(background!=null) {
			DMLHelper.createJsonFromFillProperties(attrs, background.getBgPr(true), createOperationHelper.getOperationDocument().getThemePart(false), background.getBgRef(), createOperationHelper.getOperationDocument().getContextPart());
		}
		return attrs;
    }

	private void createHeaderFooterOperations(IHeaderFooterAccessor iHeaderFooterAccessor, JSONObject attrs)
		throws JSONException {

		final CTHeaderFooter hf = iHeaderFooterAccessor.getHf(false);
		if(hf==null) {
			return;
		}
		if(hf.isDt()==null || hf.isDt().booleanValue()) {
			attrs.put("isDate", true);
		}
		else {
			attrs.put("isDate", false);
		}
		if(hf.isFtr()==null || hf.isFtr().booleanValue()) {
			attrs.put("isFooter", true);
		}
		else {
			attrs.put("isFooter", false);
		}
		if(hf.isHdr()==null || hf.isHdr().booleanValue()) {
			attrs.put("isHeader", true);
		}
		else {
			attrs.put("isHeader", false);
		}
		if(hf.isSldNum()==null || hf.isSldNum().booleanValue()) {
			attrs.put("isSlideNum", true);
		}
		else {
			attrs.put("isSlideNum", false);
		}
	}
}
