/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.components;

import java.text.ParseException;
import java.util.HashSet;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.P;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RStyle;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.docx.DocxOperationDocument;
import com.openexchange.office.ooxml.docx.tools.Character;
import com.openexchange.office.ooxml.docx.tools.TextUtils;
import com.openexchange.office.ooxml.docx.tools.Utils;
import com.openexchange.office.ooxml.tools.Commons;

public abstract class TextRun_Base extends DocxComponent {

    // change track flags, the  in the highword specifies the value that should be used
    final public static int CT_NONE  			= 0;
    final public static int CT_INSERTED  		= 1;	// corresponding HIWORD bit == VALUE
    final public static int CT_DELETED   		= 2;	// corresponding HIWORD bit == VALUE
    final public static int CT_MODIFIED 		= 4;	// corresponding HIWORD bit == VALUE

    // change tracking mode of this component
    private int changeTrackFlags = 0;

    protected IndexedNode<Object> textRunNode = null;
    protected IndexedNode<Object> hyperlinkNode = null;
    protected IndexedNode<Object> runInsNode = null;
    protected IndexedNode<Object> runDelNode = null;

    protected TextRun_Base(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);
        ComponentContext parent = this;
        do {
            parent = parent.getParentContext();
            if(parent instanceof TextRunContext) {
                textRunNode = parent.getNode();
            }
            else if(parent instanceof HyperlinkContext) {
                hyperlinkNode = parent.getNode();
            }
            else if(parent instanceof RunInsContext) {
                runInsNode = parent.getNode();
                changeTrackFlags |= (CT_INSERTED<<16)|CT_INSERTED;
            }
            else if(parent instanceof RunDelContext) {
                runDelNode = parent.getNode();
                changeTrackFlags |= (CT_DELETED<<16)|CT_DELETED;
            }
        }
        while(!(parent instanceof ParagraphComponent));
    }
	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
		return null;
	}
    @Override
    public void delete(com.openexchange.office.ooxml.OperationDocument operationDocument, int count)
    	throws InvalidFormatException {

    	IndexedNode<Object> previousHyperlinkNode = null;
    	if (hyperlinkNode!=null) {
			final ContentAccessor hyperlinkParent = (ContentAccessor)((Child)hyperlinkNode.getData()).getParent();
			final IndexedNodeList<Object> hyperlinkParentContent = (IndexedNodeList<Object>)hyperlinkParent.getContent();
			final IndexedNode<Object > previousNode = hyperlinkParentContent.getPreviousNode(hyperlinkNode);
			previousHyperlinkNode = previousNode!=null&&previousNode.getData() instanceof P.Hyperlink ? previousNode : null;
    	}

    	super.delete(operationDocument, count);

    	if(previousHyperlinkNode!=null) {
    		combineWithNextHyperlink(previousHyperlinkNode);
    	}
    }
	@Override
	public void splitStart(int n, SplitMode splitMode) {
		ParagraphComponent.splitAccessor(this, true, splitMode);
	}
	@Override
	public void splitEnd(int n, SplitMode splitMode) {
		ParagraphComponent.splitAccessor(this, false, splitMode);
	}
	@Override
	public void applyAttrsFromJSON(com.openexchange.office.ooxml.OperationDocument operationDocument, JSONObject attrs)
	    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

	    try {
			TextUtils.setTextRunAttributes((com.openexchange.office.ooxml.docx.DocxOperationDocument)operationDocument, attrs, this);
			if(hyperlinkNode!=null) {
				combineWithPreviousHyperlink(hyperlinkNode);
				combineWithNextHyperlink(hyperlinkNode);
			}
		}
        catch (Exception e) {
		}
	}

	private void combineWithPreviousHyperlink(IndexedNode<Object> hyperlinkNode) {
		final ComponentContext componentContext = getContext(HyperlinkContext.class);
		if(componentContext!=null) {
			final ContentAccessor hyperlinkParent = (ContentAccessor)((Child)hyperlinkNode.getData()).getParent();
			final IndexedNodeList<Object> hyperlinkParentContent = (IndexedNodeList<Object>)hyperlinkParent.getContent();

			final IndexedNode<Object> previousNode = hyperlinkParentContent.getPreviousNode(hyperlinkNode);
			if(previousNode!=null && previousNode.getData() instanceof P.Hyperlink) {
				final P.Hyperlink currentHyperlink = (P.Hyperlink)hyperlinkNode.getData();
				final P.Hyperlink previousHyperlink = (P.Hyperlink)previousNode.getData();
				if(currentHyperlink.getId() == null ? previousHyperlink.getId() == null : currentHyperlink.getId().equals(previousHyperlink.getId())) {
					final IndexedNodeList<Object> previousHyperlinkContent = (IndexedNodeList<Object>)previousHyperlink.getContent();
					final IndexedNodeList<Object> currentHyperlinkContent = (IndexedNodeList<Object>)currentHyperlink.getContent();
					previousHyperlinkContent.addNodes(currentHyperlinkContent, previousHyperlink);
					componentContext.setNode(previousNode);
					hyperlinkNode = previousNode;
					// empty hyperlink can now be removed
					hyperlinkParentContent.remove(currentHyperlink);
				}
			}
		}
	}

	private static void combineWithNextHyperlink(IndexedNode<Object> hyperlinkNode) {
		final ContentAccessor hyperlinkParent = (ContentAccessor)((Child)hyperlinkNode.getData()).getParent();
		final IndexedNodeList<Object> hyperlinkParentContent = (IndexedNodeList<Object>)hyperlinkParent.getContent();
		final IndexedNode<Object> nextNode = hyperlinkParentContent.getNextNode(hyperlinkNode);
		if(nextNode!=null && nextNode.getData() instanceof P.Hyperlink) {
			final P.Hyperlink currentHyperlink = (P.Hyperlink)hyperlinkNode.getData();
			final P.Hyperlink nextHyperlink = (P.Hyperlink)nextNode.getData();
			if(currentHyperlink.getId() == null ? nextHyperlink.getId() == null : currentHyperlink.getId().equals(nextHyperlink.getId())) {
				final IndexedNodeList<Object> nextHyperlinkContent = (IndexedNodeList<Object>)nextHyperlink.getContent();
				final IndexedNodeList<Object> currentHyperlinkContent = (IndexedNodeList<Object>)currentHyperlink.getContent();
				currentHyperlinkContent.addNodes(nextHyperlinkContent, currentHyperlink);
				// empty hyperlink can now be removed
				hyperlinkParentContent.remove(nextHyperlink);
			}
		}
	}

	@Override
	public JSONObject createJSONAttrs(com.openexchange.office.ooxml.operations.CreateOperationHelper createOperationHelper, JSONObject attrs)
		throws JSONException, ParseException, FilterException {

		JSONObject changes = null;
        JSONObject jsonCharacterProperties = null;

        final DocxOperationDocument operationDocument = (com.openexchange.office.ooxml.docx.DocxOperationDocument)createOperationHelper.getOperationDocument();
        final IndexedNode<Object> textRunNode = getTextRunNode();
        final R textRun = (R)textRunNode.getData();
        final RPr rPr = textRun.getRPr();
        if(rPr!=null) {
            final RStyle rStyle = rPr.getRStyle();
            if(rStyle!=null)
                attrs.put("styleId", rStyle.getVal());
            jsonCharacterProperties = Character.createCharacterProperties(createOperationHelper.getOperationDocument().getThemeFonts(), rPr);
            if(rPr.getRPrChange()!=null) {
            	changes = new JSONObject(2);
            	final JSONObject changedAttrs = new JSONObject();
            	final org.docx4j.wml.CTRPrChange.RPr rPrChange = rPr.getRPrChange().getRPr();
            	final JSONObject changedJsonCharacterAttributes = Character.createCharacterProperties(createOperationHelper.getOperationDocument().getThemeFonts(),  rPrChange);
            	if(changedJsonCharacterAttributes!=null) {
            		changedAttrs.put("character", changedJsonCharacterAttributes);
            	}
            	final JSONObject modifiedAttrs = Utils.createJSONFromTrackInfo(operationDocument, rPr.getRPrChange());
            	modifiedAttrs.put("attrs", changedAttrs);
            	changes.put("modified", modifiedAttrs);
            }
        }
        final int ctFlags = getCT();
        if(ctFlags!=CT_NONE) {
        	if(changes==null) {
        		changes = new JSONObject(2);
        	}
        	final IndexedNode<Object> runInsNode = getRunInsNode(false);
        	if(runInsNode!=null) {
        		changes.put("inserted", Utils.createJSONFromTrackInfo(operationDocument, (RunIns)runInsNode.getData()));
        	}
        	final IndexedNode<Object> runDelNode = getRunDelNode(false);
        	if(runDelNode!=null) {
        		changes.put("removed", Utils.createJSONFromTrackInfo(operationDocument, (RunDel)runDelNode.getData()));
        	}
        }
    	if(changes!=null&&!changes.isEmpty()) {
    		attrs.put("changes", changes);
    	}
        final IndexedNode<Object> hyperlinkNode = getHyperlinkNode(false);
        if(hyperlinkNode!=null) {
            String url = Commons.getUrl(operationDocument.getContextPart(), ((P.Hyperlink)hyperlinkNode.getData()).getId());
            if (url!=null&&url.length()>0) {
                if(jsonCharacterProperties==null)
                    jsonCharacterProperties = new JSONObject();
                jsonCharacterProperties.put("url", url);
            }
        }
        if(jsonCharacterProperties!=null&&!jsonCharacterProperties.isEmpty()) {
            attrs.put("character", jsonCharacterProperties);
        }
		return attrs;
	}
    public IndexedNode<Object> getTextRunNode() {
        return textRunNode;
    }
    public R getTextRun() {
        return (R)textRunNode.getData();
    }
    public int getCT() {
    	return changeTrackFlags;
    }
    public void setCT(int _changeTrackFlags) {
    	changeTrackFlags = _changeTrackFlags;
    }
    public IndexedNode<Object> getHyperlinkNode(boolean forceCreate) {
    	if(hyperlinkNode==null&&forceCreate) {
	        final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(1);
	        parentContextList.add(SdtParagraphContext.class);
	        hyperlinkNode = new IndexedNode<Object>(Context.getWmlObjectFactory().createPHyperlink());
	        insertContext(new HyperlinkContext(null, hyperlinkNode), parentContextList);
    	}
    	return hyperlinkNode;
    }
    public void removeHyperlink() {
    	if(hyperlinkNode!=null) {
    		removeContext(HyperlinkContext.class);
            hyperlinkNode = null;
    	}
    }
    public IndexedNode<Object> getRunInsNode(boolean forceCreate) {
    	if(runInsNode==null&&forceCreate) {
            final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(2);
            parentContextList.add(SdtParagraphContext.class);
            parentContextList.add(HyperlinkContext.class);
            runInsNode = new IndexedNode<Object>(Context.getWmlObjectFactory().createRunIns());
            insertContext(new RunInsContext(null, runInsNode), parentContextList);
    	}
    	return runInsNode;
    }
    public void removeRunInsNode() {
    	if(runInsNode!=null) {
    		removeContext(RunInsContext.class);
    		runInsNode = null;
    	}
    }
    public IndexedNode<Object> getRunDelNode(boolean forceCreate) {
    	if(runDelNode==null&&forceCreate) {
	        final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(3);
	        parentContextList.add(SdtParagraphContext.class);
	        parentContextList.add(HyperlinkContext.class);
	        parentContextList.add(RunInsContext.class);
	        runDelNode = new IndexedNode<Object>(Context.getWmlObjectFactory().createRunDel());
	        insertContext(new RunDelContext(null, runDelNode), parentContextList);
    	}
    	return runDelNode;
    }
    public void removeRunDelNode() {
    	if(runDelNode!=null) {
    		removeContext(RunDelContext.class);
    		runDelNode = null;
    	}
    }
}
