/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom.components;

import java.util.HashMap;

import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DLNode;
import com.openexchange.office.odf.DrawingType;
import com.openexchange.office.odf.INodeAccessor;
import com.openexchange.office.odf.IParagraph;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.styles.StyleBase;
import com.openexchange.office.odf.styles.StyleManager;
import com.openexchange.office.odt.dom.Annotation;
import com.openexchange.office.odt.dom.AnnotationEnd;
import com.openexchange.office.odt.dom.DrawFrame;
import com.openexchange.office.odt.dom.DrawImage;
import com.openexchange.office.odt.dom.DrawTextBox;
import com.openexchange.office.odt.dom.Paragraph;
import com.openexchange.office.odt.dom.Shape;
import com.openexchange.office.odt.dom.Text;
import com.openexchange.office.odt.dom.TextField;
import com.openexchange.office.odt.dom.TextLineBreak;
import com.openexchange.office.odt.dom.TextList;
import com.openexchange.office.odt.dom.TextListItem;
import com.openexchange.office.odt.dom.TextSpan;
import com.openexchange.office.odt.dom.TextTab;

public class ParagraphComponent extends Component implements IParagraph {

	final Paragraph paragraph;

	public ParagraphComponent(ComponentContext parentContext, DLNode<Object> paragraphNode, int componentNumber) {
		super(parentContext, paragraphNode, componentNumber);
		this.paragraph = (Paragraph)getObject();
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Paragraph)getObject()).getContent().getFirstNode();
        while(nextNode!=null) {
			final Object child = nextNode.getObject();
			if(child instanceof TextSpan) {
				final TextSpanContext textSpanContext =  new TextSpanContext(this, nextNode);
				final Component childComponent = textSpanContext.getNextChildComponent(null, previousChildComponent);
				if(childComponent!=null) {
					return childComponent;
				}
			}
			nextNode = nextNode.next;
		}
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		if(attrs!=null) {
			final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");
			final StyleManager styleManager = operationDocument.getPackage().getStyleManager();
			if(paragraphAttrs!=null) {

			    int level = -1;
                final Object listLevel = paragraphAttrs.opt("listLevel");
                if(listLevel!=null) {
                    if(listLevel instanceof Integer) {
                        level = ((Integer)listLevel).intValue();
                    }
                }
                else {
                    final TextListItem listItem = paragraph.getTextListItem();
                    if(listItem!=null) {
                        level = listItem.getListLevel();
                    }
                }

                boolean labelHidden = false;
                final Object listLabelHidden = paragraphAttrs.opt("listLabelHidden");
                if(listLabelHidden!=null) {
                    if(listLabelHidden instanceof Boolean) {
                        labelHidden = ((Boolean)listLabelHidden).booleanValue();
                    }
                    else {
                        labelHidden = false;
                    }
                }
                else {
                    final TextListItem listItem = paragraph.getTextListItem();
                    if(listItem!=null) {
                        labelHidden = listItem.isHeader();
                    }
                }

			    final Object listStyleId = paragraphAttrs.opt("listStyleId");
                if(listStyleId!=null) {
                    if(listStyleId instanceof String) {
                        final TextListItem textListItem = new TextListItem(new TextList(null));
                        paragraph.setTextListItem(textListItem);
                        textListItem.getParentTextList().setStyleName((String)listStyleId);
                        textListItem.getParentTextList().setContinueNumbering(true);
                        textListItem.setIsHeader(labelHidden);
                        paragraph.setListLevel(level);
                        // Insert listStyle operations are inserting the listStyle into content.xml... 
                        // If such a style is used within style.xml (such as header/footer are doing), then
                        // we need to clone the listStyle and also insert it into the style.xml...
                        // (great idea to put header footers into style.xml)
                        if(!isContentAutoStyle()) {
                            // check if this style is available in style.xml
                            StyleBase listStyle = styleManager.getAutoStyle((String)listStyleId, "list-style", false);
                            if(listStyle==null) {
                                // clone from content.xml ...
                                listStyle = styleManager.getAutoStyle((String)listStyleId, "list-style", true);
                                if(listStyle!=null) {
                                    final StyleBase listStyleClone = listStyle.clone();
                                    listStyleClone.setIsContentStyle(false);
                                    styleManager.addStyle(listStyleClone);
                                }
                            }
                        }
                    }
                    else if(listStyleId==JSONObject.NULL) {
                        paragraph.setListLevel(-1);
                    }
                }
                else {
                    if(listLevel!=null) {
                        paragraph.setListLevel(level);
                    }
                    if(listLabelHidden!=null) {
                        final TextListItem textListItem = paragraph.getTextListItem();
                        if(textListItem!=null) {
                            textListItem.setIsHeader(labelHidden);
                        }
                    }
                }
			}
			paragraph.setStyleName(styleManager.applyAttributes("paragraph", paragraph.getStyleName(), isContentAutoStyle(), attrs));
		}
	}

	@Override
	public void createAttrs(OdfOperationDocument operationDocument, OpAttrs attrs)
		throws SAXException {

		if(paragraph.getStyleName()!=null&&!paragraph.getStyleName().isEmpty()) {
			operationDocument.getPackage().getStyleManager().
				createAutoStyleAttributes(attrs, paragraph.getStyleName(), "paragraph", isContentAutoStyle());
		}
        final TextListItem textListItem = paragraph.getTextListItem();
		if(textListItem!=null) {
			final HashMap<String, Object> paragraphAttrs  = attrs.getMap("paragraph", true);
			TextList textList = textListItem.getParentTextList();
			final String styleId = textList.getId();
			if(styleId!=null&&!styleId.isEmpty()) {
				paragraphAttrs.put("listId", styleId);
			}
			if(textListItem.isHeader()) {
				paragraphAttrs.put("listLabelHidden", true);
			}
			paragraphAttrs.put("listLevel", textListItem.getListLevel());
			final Integer startValue = paragraph.getStartValue();
			if(startValue!=null) {
				paragraphAttrs.put("listStartValue", startValue);
			}
			final Integer outlineLevel = paragraph.getOutlineLevel();
			if(outlineLevel!=null) {
				paragraphAttrs.put("outlineLevel", outlineLevel);
			}
			final String styleName = textList.getStyleName(true);
			if(styleName!=null&&!styleName.isEmpty()) {
				paragraphAttrs.put("listStyleId", styleName);
			}
			if(paragraphAttrs.isEmpty()) {
				attrs.remove("paragraph");
			}
		}
	}

	@Override
	public void insertText(OdfOperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
		throws JSONException, SAXException {

    	if(text.length()>0) {
        	final Paragraph paragraph = (Paragraph)getObject();

            Text t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (Text)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getText());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setText(s.toString());
                    cRet = childComponent;
                }
                else {
                	t = new Text(text, true);
                	final TextSpanContext spanContext = (TextSpanContext)childComponent.getParentContext();
                    ((TextSpan)spanContext.getObject()).getContent().addNode(childComponent.getNode(), new DLNode<Object>(t), textPosition==0);
                    cRet = childComponent;

                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final TextSpan newRun = new TextSpan();
                paragraph.getContent().add(newRun);
/*
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
*/
                t = new Text(text, true);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
            if(attrs!=null) {
            	cRet.splitStart(textPosition);
            	cRet.splitEnd(textPosition+text.length()-1);
            	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

    @Override
    public Component insertChildComponent(OdfOperationDocument operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, SAXException {

        Object newChild = null;
        switch(childType) {
            case TAB : {
                newChild = new TextTab(null);
                break;
            }
            case HARDBREAK : {
            	newChild = new TextLineBreak();
            	break;
            }
            case FIELD : {
            	newChild = new TextField();
            	break;
            }
            case COMMENT_REFERENCE : {
            	newChild = new Annotation(null);
            	break;
            }
            case COMMENT_RANGE_END : {
            	newChild = new AnnotationEnd(null);
            	break;
            }
            case AC_IMAGE : {
            	final DrawFrame drawFrame = new DrawFrame();
            	drawFrame.getChilds().add(new DrawImage(drawFrame));
            	newChild = drawFrame;
            	break;
            }
            case AC_SHAPE : {
            	newChild = new Shape(DrawingType.SHAPE, Namespaces.DRAW, "rect", "draw:rect");
            	break;
            }
            case AC_GROUP : {
            	newChild = new Shape(DrawingType.GROUP, Namespaces.DRAW, "g", "draw:g");
            	break;
            }
            case AC_FRAME : {
            	final DrawFrame drawFrame = new DrawFrame();
            	drawFrame.getChilds().add(new DrawTextBox());
            	newChild = drawFrame;
            	break;
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
        final Paragraph paragraph = (Paragraph)getObject();
        final DLList<Object> paragraphContent = paragraph.getContent();

        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1);
            }
        	if(childComponent instanceof TextSpan_Base) {

        		// the new child can be added into an existing textRun
                final TextSpan textSpan = ((TextSpan_Base)childComponent).getTextSpan();
                textSpan.getContent().addNode(((TextSpan_Base)childComponent).getNode(), new DLNode<Object>(newChild), textPosition==0);
            }
            else {

            	// we need to create a new textRun, check if we can get referenceAttributes that we will clone for the new run. this is
            	final TextSpan newTextSpan = new TextSpan();
            	newTextSpan.getContent().add(newChild);

            	final INodeAccessor parentContextObject = (INodeAccessor)childComponent.getParentContext().getObject();
            	((DLList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new DLNode<Object>(newTextSpan), textPosition==0);
            }
        }
        else {
            final TextSpan newTextSpan = new TextSpan();
            paragraphContent.add(newTextSpan);
            newTextSpan.getContent().add(newChild);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition);
            childComponent.splitEnd(textPosition);
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }

	@Override
    public void splitParagraph(int textPosition) {
    	final Paragraph paragraph = (Paragraph)getObject();
        final Paragraph destParagraph = new Paragraph(
        	paragraph.getTextListItem()!=null
        	? new TextListItem(paragraph.getTextListItem().getParentTextList())
        	: null);

        // taking care of paragraphs sharing the same textListItem
        if(destParagraph.getTextListItem()!=null) {
        	DLNode<Object> nextParaNode = getNode().next;
        	while(nextParaNode!=null&&nextParaNode.getObject() instanceof Paragraph) {
	        	final Paragraph nextPara = (Paragraph)getNode().next.getObject();
	        	if(nextPara.getTextListItem()!=paragraph.getTextListItem()) {
	        		break;
	        	}
	        	nextPara.setTextListItem(destParagraph.getTextListItem());
    	        nextParaNode = nextParaNode.next;
        	}
        }

        final DLNode<Object> destParagraphNode = new DLNode<Object>(destParagraph);
        ((INodeAccessor)getParentContext().getObject()).getContent().addNode(getNode(), destParagraphNode, false);
        destParagraph.setStyleName(paragraph.getStyleName());

        Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition);

        	// moving text spans into the new paragraph
        	paragraph.getContent().moveNodes(component.getParentContext().getNode(), -1, destParagraph.getContent(), null, true);
        }
    }

	@Override
	public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final Paragraph paragraph = (Paragraph)getObject();
        	final DLList<Object> sourceContent = ((Paragraph)nextParagraphComponent.getObject()).getContent();
            paragraph.getContent().mergeNodes(sourceContent);
            ((INodeAccessor)getParentContext().getObject()).getContent().removeNode(nextParagraphComponent.getNode());
    	}
	}
}
