/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom.components;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DLNode;
import com.openexchange.office.odf.INodeAccessor;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odt.dom.Annotation;
import com.openexchange.office.odt.dom.AnnotationEnd;
import com.openexchange.office.odt.dom.Text;

public abstract class Component extends ComponentContext {

	protected int componentNumber;

    public enum Type {
    	PARAGRAPH,
    	TABLE,
    	TR,
    	TC,
    	TAB,
    	HARDBREAK,
    	FIELD,
    	COMMENT_REFERENCE,
    	COMMENT_RANGE_START,
    	COMMENT_RANGE_END,
    	AC_SHAPE,
    	AC_GROUP,
    	AC_IMAGE,
    	AC_FRAME
    }

    public Component(ComponentContext parentContext, DLNode<Object> o, int componentNumber) {
        super(parentContext, o);

        this.componentNumber = componentNumber;
    }

    public int getComponentNumber() {
        return componentNumber;
    }

    public int getNextComponentNumber() {
        return componentNumber + 1;
    }

    public Component getNextComponent() {
    	if(getParentContext()==null) {
    		return null;
    	}
        ComponentContext parentContext = this;
        Component nextComponent = null;
        do {
            ComponentContext previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }

    public Component getParentComponent() {
    	ComponentContext parentContext = getParentContext();
    	while(!(parentContext instanceof Component)) {
    		parentContext = parentContext.getParentContext();
    	}
    	return (Component)parentContext;
    }

    public ComponentContext getContextChild() {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    public abstract Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent);    

    public Component getChildComponent(int componentNumber) {
    	Component c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(componentNumber);
    	}
    	return c;
    }

    public Component getComponent(int componentNumber) {
    	Component c = this;
    	while(c!=null&&c.getNextComponentNumber()<=componentNumber) {
    		c = c.getNextComponent();
    	}
    	return c;
    }

    /*
     * returns the corresponding component for the given oxo position. if positionCount is zero a
     * RootComponent is returned. The contextPart is usually the MainDocumentPart and changed only
     * for Header & Footer.
     *
     */
    public static Component getComponent(Component rootComponent, JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return rootComponent;
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            Component c = rootComponent.getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    public void delete(OdfOperationDocument operationDocument, int count) {

    	final ComponentContext contextChild = getContextChild();

    	final int endComponent = (getComponentNumber()+count)-1;
        Component component = this;
        while(true) {
        	if(component.getNextComponentNumber()>endComponent) {
        		break;
        	}
        	component = component.getNextComponent();
        }
        component.splitEnd(endComponent);
        final DLList<Object> content = (DLList<Object>)((INodeAccessor)contextChild.getParentContext().getObject()).getContent();
        content.removeNodes(contextChild.getNode(), component.getContextChild().getNode());
    }

    public Component insertChildComponent(ComponentContext parentContext, DLNode<Object> contextNode, int number, Component child, Type type) {
    	return null;
    }

    public Component insertChildComponent(OdfOperationDocument operationDocument, int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JSONException, SAXException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type);
        if(attrs!=null) {
            c.applyAttrsFromJSON(operationDocument, attrs);
        }
        return c;
    }

    public abstract void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
        throws JSONException, SAXException;

    public abstract void createAttrs(OdfOperationDocument operationDocument, OpAttrs attrs)
    	throws SAXException;


	public static String componentToString(Component component)
    	throws SAXException {

    	final StringBuffer stringBuffer = new StringBuffer();
    	componentToString(component, stringBuffer);
    	return stringBuffer.toString();
    }

    private static void componentToString(Component component, StringBuffer stringBuffer) {
    	if(component instanceof RootComponent) {
    		stringBuffer.append("Root");
    	}
    	else if(component instanceof TableComponent) {
    		stringBuffer.append("Table");
    	}
    	else if(component instanceof RowComponent) {
    		stringBuffer.append("Row");
    	}
    	else if(component instanceof CellComponent) {
    		stringBuffer.append("Cell");
    	}
    	else if(component instanceof ParagraphComponent) {
    		stringBuffer.append("Para");
    	}
    	else if(component instanceof TextComponent) {
    		stringBuffer.append("Text");
			stringBuffer.append('\'' + ((Text)((TextComponent)component).getObject()).getText() + '\'');
    	}
    	else if(component instanceof ShapeRootComponent || component instanceof ShapeComponent) {
    		stringBuffer.append("Shape");
    	}
    	else if(component instanceof FrameComponent) {
    		stringBuffer.append("Frame");
    	}
    	else if(component instanceof TextFieldComponent) {
    		stringBuffer.append("Field");
    	}
    	else if(component instanceof TextTabComponent) {
    		stringBuffer.append("Tab");
    	}
    	else if(component instanceof AnnotationComponent) {
    		stringBuffer.append("Annotation-Ref");
    		final Annotation annotation = (Annotation)component.getObject();
    		if(annotation.getId()!=null&&!annotation.getId().isEmpty()) {
    			stringBuffer.append('[' + annotation.getId() + ']');
    		}
    	}
    	else if(component instanceof AnnotationEndComponent) {
    		stringBuffer.append("Annotation-End");
    		final AnnotationEnd annotationEnd = (AnnotationEnd)component.getObject();
    		if(annotationEnd.getId()!=null&&!annotationEnd.getId().isEmpty()) {
    			stringBuffer.append('[' + annotationEnd.getId() + ']');
    		}
    	}
    	Component childComponent = component.getChildComponent(0);
    	if(childComponent!=null) {
    		stringBuffer.append('(');
	    	while(childComponent!=null) {
	    		componentToString(childComponent, stringBuffer);
	    		childComponent = childComponent.getNextComponent();
	    		if(childComponent!=null) {
	    			stringBuffer.append(',');
	    		}
	    	}
	    	stringBuffer.append(')');
    	}
    }
}
